/* Native support code for HPUX IA64, for GDB the GNU debugger.

   Copyright 2005 Free Software Foundation, Inc.

   Free Software Foundation, Inc.

   This file is part of GDB.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  */

#include "defs.h"
#include "command.h"
#include "elf-bfd.h"
#include "gdbcore.h"
#include "ia64-tdep.h"
#include "inferior.h"
#include "objfiles.h"
#include "opcode/ia64.h"
#include "regcache.h"
#include "symfile.h"

/* Need to define the following macro in order to get the complete
   load_module_desc struct definition in dlfcn.h  Otherwise, it doesn't
   match the size of the struct the loader is providing us during load
   events.  */
#define _LOAD_MODULE_DESC_EXT

#include <dlfcn.h>
#include <elf.h>
#include <service_mgr.h>

struct so_list
{
  char *name;
  struct load_module_desc module_desc;

  struct so_list *next;
};

struct dld_info
{
  long long dld_flags;
  CORE_ADDR load_map;
};

static struct so_list *so_list_head = NULL;

/* If zero, then dld break events should be ignored.  */
static int handle_dld_break_events = 1;

/* Create a new so_list element.  The memory allocate should be deallocated
   using xfree_so_list_element.  */

static struct so_list *
xmalloc_so_list (char *name,
                 struct load_module_desc module_desc)
{
  struct so_list *new_so;

  new_so = (struct so_list *) xmalloc (sizeof (struct so_list));
  new_so->name = xstrdup (name);
  new_so->module_desc = module_desc;
  new_so->next = NULL;

  return new_so;
}

/* Deallocate SO.  */

static void
xfree_so_list_element (struct so_list *so)
{
  xfree (so->name);
}

/* Deallocate all so_list elements in SO_LIST_HEAD.  */

static void
xfree_so_list (void)
{
  struct so_list *so;

  while (so_list_head != NULL)
    {
      so = so_list_head;
      so_list_head = so_list_head->next;

      xfree_so_list_element (so);
      xfree (so);
    }
}

/* Insert SO at the head of SO_LIST_HEAD.  */

static void
prepend_to_so_list (struct so_list *so)
{
  so->next = so_list_head;
  so_list_head = so;
}

/* Update the value of the PC to point to the begining of the next
   instruction bundle.  */

static void
solib_ia64_hpux_move_pc_to_next_bundle (void)
{
  CORE_ADDR pc = read_pc ();

  pc -= pc & 0xf;
  pc += 16;
  write_pc (pc);
}

/* Find the address of the code and data segments in ABFD, and update
   TEXT_START and DATA_START accordingly.  */

static void
solib_ia64_hpux_find_start_vma (bfd *abfd,
                                CORE_ADDR *text_start,
                                CORE_ADDR *data_start)
{
  Elf_Internal_Ehdr *i_ehdrp = elf_elfheader (abfd);
  Elf64_Phdr phdr;
  int i;

  *text_start = 0;
  *data_start = 0;

  if (bfd_seek (abfd, i_ehdrp->e_phoff, SEEK_SET) == -1)
    error ("invalid program header offset in %s", abfd->filename);

  for (i = 0; i < i_ehdrp->e_phnum; i++)
    {
      if (bfd_bread ((PTR) & phdr, sizeof (phdr), abfd) != sizeof (phdr))
        error ("failed to read segment %d in %s", i, abfd->filename);
      
      if (phdr.p_flags & PF_X
          && (*text_start == 0 || phdr.p_vaddr < *text_start))
        *text_start = phdr.p_vaddr;

      if (phdr.p_flags & PF_W
          && (*data_start == 0 || phdr.p_vaddr < *data_start))
        *data_start = phdr.p_vaddr;
    }
}

/* Perform all necessary operations when the executable wasn't linked
   with any shared library.  */

void
no_shared_libraries (char *ignored, int from_tty)
{
  objfile_purge_solibs ();
  xfree_so_list ();
}

/* Callback soon after we have forked ourselves to create the inferior.  */

void
solib_ia64_hpux_acknowledge_created_inferior (void)
{
  /* We have just forked ourselves, and soon the child process will
     exec itself into a shell, which in turn will exec our inferior
     process.   Until the inferior process has been exec'ed, we should
     ignore all dld break events, as these will relevant to the shell,
     not to the inferior we're debugging.  */
  handle_dld_break_events = 0;
}

/* Callback after the real inferior has been created, ie the program
   we're trying to run has been loaded in memory and is ready to run.  */

void
solib_ia64_hpux_create_inferior_hook (int pid)
{
  /* At this point, we're just past the second exec, so the inferior
     has been started, and we should now process all dld break events.  */
  handle_dld_break_events = 1;
}

/* Return non-zero if the instruction at the current PC is a breakpoint
   part of the dynamic loading process.

   We identify such instructions by checking that the instruction at
   the current pc is a break insn where no software breakpoint has been
   inserted by us, and with a specific immediate value.  */

int
solib_ia64_hpux_dld_breakpoint (void)
{
  /* FIXME: brobecker/2005-06-08: The following is almost a copy/paste
     of the first half of ia64-dis.c:print_insn_ia64().  Can we extract
     that part off print_insn_ia64(), and then share the extracted
     function?  This is good enough for now.  */
  ia64_insn t0, t1, slot[3], template, insn;
  int slotnum;
  bfd_byte bundle[16];
  CORE_ADDR pc = read_pc ();

  /* If this is a regular breakpoint, then it can not be a dld one.  */
  if (breakpoint_inserted_here_p (pc))
    return 0;

  slotnum = ((long) pc) & 0xf;
  if (slotnum > 2)
    internal_error (__FILE__, __LINE__,
                    "invalid slot (%d) for address 0x%lx", slotnum, pc);

  pc -= (pc & 0xf);
  read_memory (pc, bundle, sizeof (bundle));

  /* bundles are always in little-endian byte order */
  t0 = bfd_getl64 (bundle);
  t1 = bfd_getl64 (bundle + 8);
  template = (t0 >> 1) & 0xf;
  slot[0] = (t0 >>  5) & 0x1ffffffffffLL;
  slot[1] = ((t0 >> 46) & 0x3ffff) | ((t1 & 0x7fffff) << 18);
  slot[2] = (t1 >> 23) & 0x1ffffffffffLL;

  if (template == 2 && slotnum == 1)
    {
      /* skip L slot in MLI template: */
      slotnum = 2;
    }

  insn = slot[slotnum];

  return ((insn == 0x1c0c9c0)       /* break.i 0x070327 */
          || (insn == 0x3c0c9c0));  /* break.i 0x0f0327 */
}

struct build_section_offset_data
{
  struct so_list *so;
  struct section_addr_info *section_addrs;
  int sect_num;
  /* FIXME: brobecker/2005-07-11: Perhaps it would be useful to save
     text_start and data_start in struct so_list.   One of the advantages
     is that we would have it available later if necessary, but it would
     also allow us to get rid of the following two fields in the struct.
     To be looked at one day.  */
  CORE_ADDR text_start;
  CORE_ADDR data_start;
};

/* Update the section information for section SECT_NUM in SECTION_ADDRS.
   SECT is the associated bfd asection.  ABFD is the bfd of the shared
   library we're processing.  TEXT_START and DATA_START are the base
   addresses of the text and data segments.  */
   
static void
build_section_data (struct so_list *so,
                    struct section_addr_info *section_addrs, int sect_num,
                    bfd *abfd, asection *sect,
                    CORE_ADDR text_start, CORE_ADDR data_start)
{
  CORE_ADDR sect_offset = bfd_section_vma (abfd, sect);

  if (((text_start < data_start) && (sect_offset < data_start)) ||
      ((text_start > data_start) && (sect_offset >= text_start)))
    {
      sect_offset += so->module_desc.text_base;
      sect_offset -= text_start;
    }
  else if (((text_start < data_start) && (sect_offset >= data_start)) ||
           ((text_start > data_start) && (sect_offset < text_start)))
    {
      sect_offset += so->module_desc.data_base;
      sect_offset -= data_start;
    }

  section_addrs->other[sect_num].name =
    xstrdup (bfd_section_name (abfd, sect));
  section_addrs->other[sect_num].addr = sect_offset;
}

/* Callback function for bfd_map_over_sections.  It updates the section
   information for the given section, using mostly build_section_data
   (it mostly unmarshals DATA1 and then use build_section_data).  */

static void
build_section_offset_callback (bfd *abfd, asection *sect, void *data1)
{
  struct build_section_offset_data *data =
    (struct build_section_offset_data *) data1;

  build_section_data (data->so, data->section_addrs, data->sect_num,
                      abfd, sect, data->text_start, data->data_start);
  data->sect_num++;
}

/* Return the section_addr_info array associated to the given shared
   library.  */

static struct section_addr_info *
solib_ia64_hpux_section_addrs (struct so_list *so)
{
  bfd *abfd;
  struct build_section_offset_data data;

  abfd = bfd_openr (so->name, gnutarget);
  if (abfd == NULL)
    perror_with_name (so->name);

  if (!bfd_check_format (abfd, bfd_object))
    {
      bfd_close (abfd);
      error ("invalid format for \"%s\" (%s)", so->name,
             bfd_errmsg (bfd_get_error ()));
    }

  /* Make some adjustments to the section offsets that corrects what
     the core symbol loader does incorrectly for this system.  */

  data.so = so;
  data.section_addrs = alloc_section_addr_info (bfd_count_sections (abfd));
  data.sect_num = 0;
  solib_ia64_hpux_find_start_vma (abfd, &data.text_start, &data.data_start);

  bfd_map_over_sections (abfd, build_section_offset_callback, &data);

  bfd_close (abfd);
  return data.section_addrs;
}

/* Load the symbols of the given shared library.  */

static void
solib_ia64_hpux_load_symbols (struct so_list *so,
                              int from_tty)
{
  struct section_addr_info *section_addrs;
  struct cleanup *old_chain;
  struct objfile *objfile;

  section_addrs = solib_ia64_hpux_section_addrs (so);
  old_chain = make_cleanup (xfree, section_addrs);

  objfile = symbol_file_add (so->name, from_tty, section_addrs, 0,
                             OBJF_SHARED);
}

/* Create a new so_list element for the given shared library referenced
   by SO_PATH.  Then read in its symbols.
   
   LOAD_MODULE_DESC is the module descriptor of the shared library.  */

static void
solib_ia64_hpux_add_to_solist (char *so_path,
                               struct load_module_desc module_desc,
                               int from_tty)
{
  struct so_list *so;

  so = xmalloc_so_list (so_path, module_desc);
  prepend_to_so_list (so);

  solib_ia64_hpux_load_symbols (so, from_tty);
}

/* Handler for library load event: Read the information provided by
   the loader, and then use it to read the shared library symbols.  */

static void
solib_ia64_hpux_handle_load_event (void)
{
  CORE_ADDR module_desc_addr;
  ULONGEST module_desc_size;
  CORE_ADDR so_path_addr;
  char so_path[MAXPATHLEN];
  struct load_module_desc module_desc;

  /* Extract the data provided by the loader as follow:
       - r33: Address of load_module_desc structure
       - r34: size of struct load_module_desc
       - r35: Address of string holding shared library path
   */
  regcache_cooked_read_unsigned (current_regcache, ia64_v32_regnum () + 1,
                                 &module_desc_addr);
  regcache_cooked_read_unsigned (current_regcache, ia64_v32_regnum () + 2,
                                 &module_desc_size);
  regcache_cooked_read_unsigned (current_regcache, ia64_v32_regnum () + 3,
                                 &so_path_addr);

  if (module_desc_size != sizeof (struct load_module_desc))
    warning ("load_module_desc size (%ld) != size returned by kernel (%ld)",
             sizeof (struct load_module_desc), module_desc_size);
  
  read_memory_string (so_path_addr, so_path, MAXPATHLEN);
  read_memory (module_desc_addr, (char *) &module_desc, sizeof (module_desc));

  solib_ia64_hpux_add_to_solist (so_path, module_desc, 0);
}

/* Handle loader events.  */

void
solib_ia64_hpux_handle_dld_breakpoint (void)
{
  ULONGEST arg0;

  if (handle_dld_break_events)
    {
      /* The type of event is provided by the loaded via r32.  */

      regcache_cooked_read_unsigned (current_regcache, ia64_v32_regnum (),
                                     &arg0);

      switch (arg0)
        {
          case BREAK_DE_SVC_LOADED:
            /* Currently, the only service loads are uld and dld,
               so we shouldn't need to do anything.  Just ignore.  */
            break;
          case BREAK_DE_LIB_LOADED:
            solib_ia64_hpux_handle_load_event ();
            break;
          case BREAK_DE_LIB_UNLOADED:
            /* Ignore for now.  */
            break;
          case BREAK_DE_LOAD_COMPLETE:
            /* Ignore for now.  */
            break;
          case BREAK_DE_BOR:
            /* Ignore for now.  */
            break;
        }
    }

  /* Now that we have handled the event, we can move the PC to
     the next instruction bundle, past the break instruction.  */
  solib_ia64_hpux_move_pc_to_next_bundle ();
}

/* Return the name of the shared library whose text range contain ADDR.
   Return NULL if no library matches.  */

char *
solib_ia64_hpux_pc_solib (CORE_ADDR addr)
{
  struct so_list *so = so_list_head;

  while (so)
    {
      if ((addr >= so->module_desc.text_base)
          && (addr < so->module_desc.text_base + so->module_desc.text_size))
        return so->name;

      so = so->next;
    }
 
  return NULL;
}

/* Scan the ".dynamic" section referenced by ABFD and DYN_SECT,
   a extract the information needed to fill in INFO.  */

static void
solib_ia64_hpux_read_dynamic_info (bfd *abfd, asection *dyn_sect,
                                   struct dld_info *info)
{
  int sect_size;
  char *buf;
  char *buf_end;
  
  sect_size = bfd_section_size (abfd, dyn_sect);
  buf = alloca (sect_size);
  buf_end = buf + sect_size;

  if (bfd_seek (abfd, dyn_sect->filepos, SEEK_SET) != 0
      || bfd_bread (buf, sect_size, abfd) != sect_size)
    error ("failed to read contents of .dynamic section");

  for (; buf < buf_end; buf += sizeof (Elf64_Dyn))
    {
      Elf64_Dyn *dynp = (Elf64_Dyn *) buf;
      Elf64_Sxword d_tag; 

      d_tag = bfd_h_get_64 (abfd, &dynp->d_tag);

      switch (d_tag)
        {
          case DT_HP_DLD_FLAGS:
            info->dld_flags = bfd_h_get_64 (abfd, &dynp->d_un);
            break;

          case DT_HP_LOAD_MAP:
            {
              CORE_ADDR load_map_addr = bfd_h_get_64 (abfd, &dynp->d_un.d_ptr);

              if (target_read_memory (load_map_addr, (char *) &info->load_map,
                                      sizeof (info->load_map)) != 0)
                error ("failed to read load map at 0x%lx", load_map_addr);
            }
            break;
        }
    }
}

/* Wrapper around target_read_memory used for dlgetmodinfo.  */

static void *
solib_ia64_hpux_read_tgt_mem (void *buffer, uint64_t ptr,
                              size_t bufsiz, int ident)
{
  if (target_read_memory (ptr, (char *) buffer, bufsiz) != 0)
    return 0;
  else
    return buffer;
}

/* Create a new so_list element for the given shared library referenced
   by its index (SO_INDEX).  INFO is the dld_info associated to this
   shared library.  */

static int
solib_ia64_hpux_add_to_solist_from_dld_info (struct dld_info info,
                                             int so_index, int from_tty)
{
  struct load_module_desc module_desc;
  uint64_t so_handle;
  char *so_path;

  so_handle = dlgetmodinfo (so_index, &module_desc, sizeof (module_desc),
                            solib_ia64_hpux_read_tgt_mem,
                            0, info.load_map);

  /* See if we haven't reached the end of the list.  The man page
     says that NULL is returned if the dlgetmodinfo call failed,
     but that causes a compiler warning if so_handle is compared
     to NULL directly, as NULL is a pointer.  So cast NULL to
     the same type as so_handle during the comparison.  */
  if (so_handle == (uint64_t) NULL)
    return 0;

  so_path = dlgetname (&module_desc, sizeof (module_desc),
                       solib_ia64_hpux_read_tgt_mem, 0, info.load_map);
  if (so_path == NULL)
    {
      /* Should never happen, but let's not crash if it does.  */
      warning ("unable to get shared library name, symbols not loaded");
      return 0;
    }

  solib_ia64_hpux_add_to_solist (so_path, module_desc, from_tty);
  return 1;
}

/* Load the symbols of all shared libraries mapped in memory.  */

/* FIXME: brobecker/2005-07-12: FILENAME is ignored for now.  */
/* FIXME: brobecker/2005-07-14: So is READ_SYMS.  */

void
solib_ia64_hpux_solib_add (char *filename, int from_tty,
                           struct target_ops *target, int read_syms)
{
  bfd *abfd;
  asection *dyn_sect;
  struct dld_info info;
  int i;

  if (symfile_objfile == NULL)
    return;
    
  abfd = symfile_objfile->obfd;
  dyn_sect = bfd_get_section_by_name (abfd, ".dynamic");

  if (dyn_sect == NULL || bfd_section_size (abfd, dyn_sect) == 0)
    return;

  solib_ia64_hpux_read_dynamic_info (abfd, dyn_sect, &info);

  if ((info.dld_flags & DT_HP_DEBUG_PRIVATE) == 0)
    {
      warning ("The shared libraries were not privately mapped; setting a\n\
breakpoint in a shared library will not work until you rerun the program.\n\
Use the following command to enable debugging of shared libraries.\n\
chatr +dbg enable a.out");
    }

  /* Read the symbols of the dynamic loader (dld.so).  */
  solib_ia64_hpux_add_to_solist_from_dld_info (info, -1, from_tty);

  /* Read the symbols of all the other shared libraries.  */
  for (i = 1; ; i++)
    if (! solib_ia64_hpux_add_to_solist_from_dld_info (info, i, from_tty))
      return;  /* End of list, we're done.  */
}

CORE_ADDR
solib_ia64_hpux_get_got_by_pc (CORE_ADDR faddr)
{ 
  struct so_list *so = so_list_head;
  
  while (so != NULL)
    { 
      if (so->module_desc.text_base <= faddr
          && (so->module_desc.text_base + so->module_desc.text_size) > faddr)
        return so->module_desc.linkage_ptr;
      so = so->next;
    }

  return 0;
} 
                   
/* Print a brief description of all shared libraries known by GDB.  */

static void
solib_ia64_hpux_shared_library_info (char *args, int from_tty)
{
  struct so_list *so = so_list_head;

  if (so_list_head == NULL)
    {
      printf_filtered ("No shared library loaded.");
      return;
    }

  printf_filtered ("Shared Libraries:\n");
  printf_filtered ("    %-19s%-19s%-19s%-19s\n",
                   "    tstart", "     tend", "    dstart",
                   "     dend");

  while (so != NULL)
    {
      printf_filtered ("%s\n", so->name);
      printf_filtered (" 0x%16lx", so->module_desc.text_base);
      printf_filtered (" 0x%16lx",
                       so->module_desc.text_base + so->module_desc.text_size);
      printf_filtered (" 0x%16lx", so->module_desc.data_base);
      printf_filtered (" 0x%16lx",
                       so->module_desc.data_base + so->module_desc.data_size);
      printf_filtered ("\n");

      so = so->next;
    }
}

void
_initialize_solib_ia64_hpux (void)
{
  add_info ("shared-library", solib_ia64_hpux_shared_library_info,
            "Print status of loaded shared libraries.");
}
