/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * GnoWavCut -- a GNOME/GTK+ based RIFF PCM Wave File splitter
 * Copyright (C) 2000 Yoichi Imai <yoichi@silver-forest.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "config.h"

#include <gnome.h>
#include "main.h"
#include "file.h"
#include "utils.h"

static GtkWidget *filesel_open = NULL;

static void file_data_clear(GnoWavCut *gnowavcut);
static void file_open_cb(GtkWidget *widget, gpointer data);

static gboolean file_check_header(WaveInfo *wave_info);

static void file_data_clear(GnoWavCut *gnowavcut) {
	utils_gnowavcut_if_playing(gnowavcut, FALSE);
	gtk_label_set_text(GTK_LABEL(gnowavcut->label_middle), "000:00");
	gtk_label_set_text(GTK_LABEL(gnowavcut->label_end), "000:00");
	gtk_entry_set_text(GTK_ENTRY(gnowavcut->entry_file), "");
	gtk_entry_set_text(GTK_ENTRY(gnowavcut->entry_rate), "");
	gtk_entry_set_text(GTK_ENTRY(gnowavcut->entry_bit), "");
	gtk_entry_set_text(GTK_ENTRY(gnowavcut->entry_channel), "");
	gtk_entry_set_text(GTK_ENTRY(gnowavcut->entry_base), "");
	gtk_label_set_text(GTK_LABEL(gnowavcut->label_mini), "");

	gtk_clist_clear(GTK_CLIST(gnowavcut->clist));
	if(gnowavcut->file_name != NULL)
	   g_free(gnowavcut->file_name);
	gnowavcut->file_name = NULL;

	gnowavcut->dsp_fd = -1;
	gnowavcut->wave_fd = -1;
	gnowavcut->input_tag = -1;
	gnowavcut->mini_play_size = 0;

	gnowavcut->now_pause = FALSE;
	/* memset(gnowavcut->wave_info, 0, sizeof(WaveInfo)); */
}

void file_dialog_open_cb(GtkWidget *widget, gpointer data)
{
	GnoWavCut *gnowavcut;

	gnowavcut = (GnoWavCut *)data;

	if (gnowavcut->now_playing == TRUE) {
		utils_msgbox_warning(_("The loaded wave file is playing."));
		return;
	}

	if (filesel_open) {
		gtk_widget_show(filesel_open);
		return;
	}

	if (!filesel_open) {
		filesel_open = gtk_file_selection_new(_("Open wave file..."));
		gtk_signal_connect(GTK_OBJECT(filesel_open), "delete_event",
				   GTK_SIGNAL_FUNC(gtk_widget_hide), NULL);
		gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel_open)->ok_button),
				   "clicked", GTK_SIGNAL_FUNC(file_open_cb), gnowavcut);
		gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel_open)->cancel_button),
					  "clicked", GTK_SIGNAL_FUNC(gtk_widget_hide), GTK_OBJECT(filesel_open));
		gtk_widget_show(filesel_open);
	}
}
void file_open(GnoWavCut *gnowavcut, gchar *path)
{
	FILE *fp;
	WaveInfo *wave_info;
	gchar *rate;
	gchar *time;
	gchar *basename;
	gchar *dirname;

	file_data_clear(gnowavcut);

	g_return_if_fail(path != NULL);

	wave_info = gnowavcut->wave_info;

	if ((fp = fopen(path, "r")) == NULL) {
		utils_msgbox_error(_("File open error"));
		return;
	}
	if (fread(wave_info, WAVE_HEADER_SIZE, 1, fp) < 1) {
		utils_msgbox_error(_("Header read error"));
		fclose(fp);
		return;
	}
	if (file_check_header(wave_info) == FALSE) {
		utils_msgbox_error(_("This file is not RIFF Wave file or broken."));
		fclose(fp);
		return;
	}
#if SIZEOF_OFF_T <= 4
	if(wave_info->riff_size > INT_MAX) {
		utils_msgbox_error(_("This system does not support files over 2GB."));
		fclose(fp);
		return;
	}
#endif
	fclose(fp);

	if(wave_info->bits_per_sample != 16) {
		utils_msgbox_error(_("GnoWavCut support 16 bit PCM file only"));
		return;
	}
	gtk_entry_set_text(GTK_ENTRY(gnowavcut->entry_bit), "16");

	if(wave_info->channels == 1)
		gtk_entry_set_text(GTK_ENTRY(gnowavcut->entry_channel), _("Mono"));
	else
		gtk_entry_set_text(GTK_ENTRY(gnowavcut->entry_channel), _("Stereo"));

	rate = g_strdup_printf("%d", wave_info->samples_per_sec);
	gtk_entry_set_text(GTK_ENTRY(gnowavcut->entry_rate), rate);
	g_free(rate);

	gnowavcut->file_name = g_strdup(path);
	gtk_entry_set_text(GTK_ENTRY(gnowavcut->entry_file),
			   gnowavcut->file_name);

	time = utils_get_time_from_seconds(wave_info->data_size / wave_info->size_per_sec);
	gtk_label_set_text(GTK_LABEL(gnowavcut->label_end), time);
	gtk_label_set_text(GTK_LABEL(gnowavcut->label_middle), "000:00");

	basename = g_strdup(g_basename(gnowavcut->file_name));
	if(strlen(basename) > 5 &&
	   strcmp(basename + strlen(basename) - 4, ".wav") == 0) {
		*(basename + strlen(basename)  - 4) = '\0';
	}
	gtk_entry_set_text(GTK_ENTRY(gnowavcut->entry_base), basename);

	dirname = g_dirname(gnowavcut->file_name);
	gtk_entry_set_text(GTK_ENTRY(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(gnowavcut->fentry_dir))), dirname);

	g_free(dirname);
	g_free(basename);
	g_free(time);
}	
static void file_open_cb(GtkWidget *widget, gpointer data)
{
	GnoWavCut *gnowavcut;

	gnowavcut = (GnoWavCut *)data;
	file_open(gnowavcut, gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel_open)));

	gtk_widget_hide(filesel_open);
}

static gboolean file_check_header(WaveInfo *wave_info)
{
	if(strncmp(wave_info->id_riff, "RIFF", 4)) return FALSE;
	if(strncmp(wave_info->type_riff, "WAVE", 4)) return FALSE;
	if(strncmp(wave_info->id_fmt, "fmt ", 4)) return FALSE;
	if(strncmp(wave_info->id_data, "data", 4)) return FALSE;
	if(wave_info->format_tag != FORMAT_PCM) return FALSE;
	if(wave_info->channels != 1 && wave_info->channels != 2) return FALSE;
	return TRUE;
}
