/*
     This file is part of GNUnet.
     (C) 2005, 2006, 2007 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/fs/namespace.c
 * @brief operations for deleting and adding to/updating namespaces
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include "fs.h"
#include "helper.h"
#include "meta.h"
#include "namespace.h"
#include "namespace_search.h"
#include "content_tracking.h"
#include <GNUnet/gnunet_util_crypto.h>
#include <GNUnet/gnunet_uritrack_lib.h>
#include <GNUnet/gnunet_namespace_lib.h>
#include <extractor.h>

/**
 * @brief linked list of pages in the namespace notebook
 */
typedef struct NL
{
  struct NL *next;
  GtkWidget *treeview;
  GtkWidget *namespacepage;
  GtkWidget *addButton;
  GtkWidget *updateButton;
  GtkTreeModel *model;
  char *name;
  HashCode512 id;
  struct ECRS_MetaData *meta;
} NamespaceList;

/**
 * Content selection in main list of available content.
 */
static GtkTreeSelection *content_selection;

static NamespaceList *head;

static GladeXML *metaXML;

/**
 * The user has changed the selection either in the
 * namespace content list or the global content list.
 * Update search button status values (add/change).
 */
static void
on_namespaceContentSelectionChanged (gpointer signal, gpointer cls)
{
  NamespaceList *list = head;
  int count;
  int ncount;
  GtkTreeSelection *ns;
  GtkTreeIter iter;
  char *freq;
  int ok;

  count = gtk_tree_selection_count_selected_rows (content_selection);
  while (list != NULL)
    {
      ns = gtk_tree_view_get_selection (GTK_TREE_VIEW (list->treeview));
      ncount = gtk_tree_selection_count_selected_rows (ns);
      gtk_widget_set_sensitive (list->addButton, count > 0);
      /* now check if update is legal */
      ok = 0;
      if ((count == 1) &&
          (ncount == 1) &&
          (TRUE == gtk_tree_selection_get_selected (ns, NULL, &iter)))
        {
          freq = NULL;
          gtk_tree_model_get (list->model,
                              &iter, IN_NAMESPACE_PUB_DATE_STRING, &freq, -1);
          if ((freq != NULL) && (0 != strcmp (freq, _("never"))))
            ok = 1;
          FREENONNULL (freq);
        }
      gtk_widget_set_sensitive (list->updateButton, ok);
      list = list->next;
    }
}


static void
makeNamespaceFrame (NamespaceList * entry)
{
  GtkWidget *child;
  GtkWidget *resultList;
  GtkCellRenderer *renderer;
  GtkListStore *model;
  GladeXML *namespaceXML;
  GtkTreeViewColumn *column;
  int col;

  DEBUG_BEGIN ();
  namespaceXML
    = glade_xml_new (getGladeFileName (),
                     "namespaceContentFrame", PACKAGE_NAME);
  connectGladeWithPlugins (namespaceXML);
  child = extractMainWidgetFromWindow (namespaceXML, "namespaceContentFrame");
  resultList = glade_xml_get_widget (namespaceXML,
                                     "namespaceContentFrameTreeView");
  entry->addButton = glade_xml_get_widget (namespaceXML, "addButton");
  entry->updateButton = glade_xml_get_widget (namespaceXML,
                                              "namespaceUpdateButton");
  entry->treeview = GTK_WIDGET (GTK_TREE_VIEW (resultList));
  model = gtk_list_store_new (IN_NAMESPACE_NUM, G_TYPE_STRING,  /* (file)name */
                              G_TYPE_UINT64,    /* size */
                              G_TYPE_STRING,    /* human-readable size */
                              G_TYPE_STRING,    /* description */
                              G_TYPE_STRING,    /* mime-type */
                              G_TYPE_STRING,    /* last-ID */
                              G_TYPE_STRING,    /* next-ID */
                              G_TYPE_STRING,    /* pub-freq */
                              G_TYPE_STRING,    /* next pub date */
                              G_TYPE_POINTER,   /* URI */
                              G_TYPE_POINTER);  /* META */
  entry->model = GTK_TREE_MODEL (model);
  gtk_tree_view_set_model (GTK_TREE_VIEW (resultList),
                           GTK_TREE_MODEL (model));
  gtk_tree_selection_set_mode (gtk_tree_view_get_selection
                               (GTK_TREE_VIEW (resultList)),
                               GTK_SELECTION_SINGLE);
  g_signal_connect_data (gtk_tree_view_get_selection
                         (GTK_TREE_VIEW (resultList)), "changed",
                         G_CALLBACK (&on_namespaceContentSelectionChanged),
                         NULL, NULL, 0);



  renderer = gtk_cell_renderer_text_new ();
  col =
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (resultList),
                                                 -1, _("Filename"), renderer,
                                                 "text",
                                                 IN_NAMESPACE_FILENAME, NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (resultList), col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, IN_NAMESPACE_FILENAME);
  /*gtk_tree_view_column_set_sort_indicator(column, TRUE); */

  gtk_tree_view_column_set_resizable (gtk_tree_view_get_column
                                      (GTK_TREE_VIEW (resultList), col - 1),
                                      TRUE);
  renderer = gtk_cell_renderer_text_new ();
  g_object_set (renderer, "xalign", 1.00, NULL);
  col =
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (resultList),
                                                 -1, _("Filesize"), renderer,
                                                 "text", IN_NAMESPACE_HSIZE,
                                                 NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (resultList), col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, IN_NAMESPACE_SIZE);
  /*gtk_tree_view_column_set_sort_indicator(column, TRUE); */
  gtk_tree_view_column_set_resizable (gtk_tree_view_get_column
                                      (GTK_TREE_VIEW (resultList), col - 1),
                                      TRUE);
  renderer = gtk_cell_renderer_text_new ();
  col =
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (resultList),
                                                 -1, _("Description"),
                                                 renderer, "text",
                                                 IN_NAMESPACE_DESCRIPTION,
                                                 NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (resultList), col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, IN_NAMESPACE_DESCRIPTION);
  /*gtk_tree_view_column_set_sort_indicator(column, TRUE); */
  gtk_tree_view_column_set_resizable (gtk_tree_view_get_column
                                      (GTK_TREE_VIEW (resultList), col - 1),
                                      TRUE);
  renderer = gtk_cell_renderer_text_new ();
  col =
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (resultList),
                                                 -1, _("Mime-type"), renderer,
                                                 "text",
                                                 IN_NAMESPACE_MIMETYPE, NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (resultList), col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, IN_NAMESPACE_MIMETYPE);
  /*gtk_tree_view_column_set_sort_indicator(column, TRUE); */
  gtk_tree_view_column_set_resizable (gtk_tree_view_get_column
                                      (GTK_TREE_VIEW (resultList), col - 1),
                                      TRUE);
  renderer = gtk_cell_renderer_text_new ();
  col =
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (resultList),
                                                 -1,
                                                 _("Publication Frequency"),
                                                 renderer, "text",
                                                 IN_NAMESPACE_PUB_FREQ_STRING,
                                                 NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (resultList), col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column,
                                           IN_NAMESPACE_PUB_FREQ_STRING);
  /*gtk_tree_view_column_set_sort_indicator(column, TRUE); */
  gtk_tree_view_column_set_resizable (gtk_tree_view_get_column
                                      (GTK_TREE_VIEW (resultList), col - 1),
                                      TRUE);
  renderer = gtk_cell_renderer_text_new ();
  col =
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (resultList),
                                                 -1,
                                                 _("Next Publication Date"),
                                                 renderer, "text",
                                                 IN_NAMESPACE_PUB_DATE_STRING,
                                                 NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (resultList), col - 1);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column,
                                           IN_NAMESPACE_PUB_DATE_STRING);
  /*gtk_tree_view_column_set_sort_indicator(column, TRUE); */
  gtk_tree_view_column_set_resizable (gtk_tree_view_get_column
                                      (GTK_TREE_VIEW (resultList), col - 1),
                                      TRUE);
  renderer = gtk_cell_renderer_text_new ();
  col =
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (resultList),
                                                 -1, _("Last ID"), renderer,
                                                 "text",
                                                 IN_NAMESPACE_LAST_STRING,
                                                 NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (resultList), col - 1);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_resizable (gtk_tree_view_get_column
                                      (GTK_TREE_VIEW (resultList), col - 1),
                                      TRUE);
  renderer = gtk_cell_renderer_text_new ();
  col =
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (resultList),
                                                 -1, _("Next ID"), renderer,
                                                 "text",
                                                 IN_NAMESPACE_NEXT_STRING,
                                                 NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (resultList), col - 1);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_resizable (gtk_tree_view_get_column
                                      (GTK_TREE_VIEW (resultList), col - 1),
                                      TRUE);


  UNREF (namespaceXML);
  DEBUG_END ();

  entry->namespacepage = child;
}



/**
 * Update the model that lists the content of a namespace:
 * add this content.
 *
 * @param uri URI of the last content published
 * @param lastId the ID of the last publication
 * @param nextId the ID of the next update
 * @param publicationFrequency how often are updates scheduled?
 * @param nextPublicationTime the scheduled time for the
 *  next update (0 for sporadic updates)
 * @return OK to continue iteration, SYSERR to abort
 */
static int
addNamespaceContentToModel (void *cls,
                            const ECRS_FileInfo * fi,
                            const HashCode512 * lastId,
                            const HashCode512 * nextId,
                            TIME_T publicationFrequency,
                            TIME_T nextPublicationTime)
{
  GtkListStore *model = GTK_LIST_STORE (cls);
  GtkTreeIter iter;
  char *filename;
  char *desc;
  char *mime;
  char *uriString;
  EncName last;
  EncName next;
  char *freq;
  char *date;
  unsigned long long size;
  char *size_h;

  DEBUG_BEGIN ();
  filename = ECRS_getFirstFromMetaData (fi->meta,
                                        EXTRACTOR_FILENAME,
                                        EXTRACTOR_TITLE,
                                        EXTRACTOR_ARTIST,
                                        EXTRACTOR_AUTHOR,
                                        EXTRACTOR_PUBLISHER,
                                        EXTRACTOR_CREATOR,
                                        EXTRACTOR_PRODUCER,
                                        EXTRACTOR_UNKNOWN, -1);
  if (filename == NULL)
    filename = STRDUP (_("no name given"));
  else
    {
      char *dotdot;

      while (NULL != (dotdot = strstr (filename, "..")))
        dotdot[0] = dotdot[1] = '_';
    }
  desc = ECRS_getFirstFromMetaData (fi->meta,
                                    EXTRACTOR_DESCRIPTION,
                                    EXTRACTOR_GENRE,
                                    EXTRACTOR_ALBUM,
                                    EXTRACTOR_COMMENT,
                                    EXTRACTOR_SUBJECT,
                                    EXTRACTOR_FORMAT,
                                    EXTRACTOR_SIZE, EXTRACTOR_KEYWORDS, -1);
  if (desc == NULL)
    desc = STRDUP ("");
  mime = ECRS_getFromMetaData (fi->meta, EXTRACTOR_MIMETYPE);
  if (mime == NULL)
    mime = STRDUP (_("unknown"));
  if (ECRS_isFileUri (fi->uri))
    size = ECRS_fileSize (fi->uri);
  else
    size = 0;
  uriString = ECRS_uriToString (fi->uri);
  hash2enc (lastId, &last);
  if (nextId != NULL)
    hash2enc (nextId, &next);
  else
    memset (&next, 0, sizeof (EncName));
  if (publicationFrequency == ECRS_SBLOCK_UPDATE_SPORADIC)
    date = STRDUP (_("unspecified"));
  else if (publicationFrequency == ECRS_SBLOCK_UPDATE_NONE)
    date = STRDUP (_("never"));
  else
    date = GN_CTIME (&nextPublicationTime);
  if (date[strlen (date) - 1] == '\n')
    date[strlen (date) - 1] = '\0';

  freq = updateIntervalToString (publicationFrequency);
  size_h = string_get_fancy_byte_size (size);
  gtk_list_store_append (model, &iter);
  gtk_list_store_set (model,
                      &iter,
                      IN_NAMESPACE_FILENAME, filename,
                      IN_NAMESPACE_SIZE, size,
                      IN_NAMESPACE_HSIZE, size_h,
                      IN_NAMESPACE_DESCRIPTION, desc,
                      IN_NAMESPACE_MIMETYPE, mime,
                      IN_NAMESPACE_LAST_STRING, &last,
                      IN_NAMESPACE_NEXT_STRING, &next,
                      IN_NAMESPACE_PUB_FREQ_STRING, freq,
                      IN_NAMESPACE_PUB_DATE_STRING, date,
                      IN_NAMESPACE_URI, ECRS_dupUri (fi->uri),
                      IN_NAMESPACE_META, ECRS_dupMetaData (fi->meta), -1);
  FREE (size_h);
  FREE (filename);
  FREE (uriString);
  FREE (freq);
  FREE (date);
  FREE (mime);
  DEBUG_END ();
  return OK;
}

/**
 * Add a tab for the given namespace.
 */
int
addTabForNamespace (void *unused,
                    const char *namespaceName,
                    const HashCode512 * namespaceId,
                    const struct ECRS_MetaData *md, int rating)
{
  NamespaceList *list;
  GtkWidget *label;
  GtkWidget *notebook;
  GtkWidget *del_menu;

  if (OK != ECRS_testNamespaceExists (NULL, cfg, namespaceName, namespaceId))
    return OK;
  DEBUG_BEGIN ();
  label = gtk_label_new (namespaceName);
  list = MALLOC (sizeof (NamespaceList));
  list->name = STRDUP (namespaceName);
  list->id = *namespaceId;
  list->meta = ECRS_dupMetaData (md);
  makeNamespaceFrame (list);
  list->next = head;
  head = list;
  /* update sensitivity of add button */
  on_namespaceContentSelectionChanged (NULL, NULL);
  notebook = glade_xml_get_widget (getMainXML (), "localNamespacesNotebook");
  gtk_notebook_append_page (GTK_NOTEBOOK (notebook),
                            list->namespacepage, label);
  gtk_widget_show (notebook);
  NS_listNamespaceContent
    (ectx, cfg, namespaceName, &addNamespaceContentToModel, list->model);
  DEBUG_END ();
  /* enable "delete" menu entry */

  del_menu = glade_xml_get_widget (getMainXML (), "namespaceDelete");
  gtk_widget_set_sensitive (del_menu, TRUE);
  return OK;
}


static void
frame_destroy (GtkWidget * tree)
{
  GtkTreeIter iter;
  struct ECRS_URI *u;
  struct ECRS_MetaData *m;
  NamespaceList *prev;
  NamespaceList *pos;
  NamespaceList *next;
  GtkWidget *del_menu;

  pos = head;
  prev = NULL;
  while (pos != NULL)
    {
      next = pos->next;
      if (pos->treeview == tree)
        break;
      prev = pos;
      pos = next;
    }
  if (pos == NULL)
    {
      GE_BREAK (NULL, 0);
      return;
    }
  if (prev == NULL)
    head = pos->next;
  else
    prev->next = pos->next;
  FREE (pos->name);
  ECRS_freeMetaData (pos->meta);
  if (gtk_tree_model_get_iter_first (pos->model, &iter))
    {
      do
        {
          gtk_tree_model_get (pos->model,
                              &iter,
                              IN_NAMESPACE_URI, &u,
                              IN_NAMESPACE_META, &m, -1);
          gtk_list_store_set (GTK_LIST_STORE (pos->model),
                              &iter,
                              IN_NAMESPACE_URI, NULL,
                              IN_NAMESPACE_META, NULL, -1);
          if (u != NULL)
            ECRS_freeUri (u);
          if (m != NULL)
            ECRS_freeMetaData (m);
        }
      while (gtk_tree_model_iter_next (pos->model, &iter));
    }
  FREE (pos);
  del_menu = glade_xml_get_widget (getMainXML (), "namespaceDelete");
  gtk_widget_set_sensitive (del_menu, head != NULL);
}



void
namespaceDelete_clicked_fs (GtkWidget * dummy1, GtkWidget * dummy2)
{
  GtkWidget *notebook;
  NamespaceList *list;
  NamespaceList *prev;
  gint num;
  GtkWidget *page;
  GtkWidget *dialog;
  gint ret;

  DEBUG_BEGIN ();
  notebook = glade_xml_get_widget (getMainXML (), "localNamespacesNotebook");
  num = gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook));
  if (num == -1)
    {
      /* IMPROVE-ME: disable the menu item
         as long as this may happen! */
      dialog = gtk_message_dialog_new
        (NULL,
         GTK_DIALOG_MODAL,
         GTK_MESSAGE_ERROR,
         GTK_BUTTONS_CLOSE,
         _("No local namespaces available that could be deleted!"));
      gtk_dialog_run (GTK_DIALOG (dialog));
      gtk_widget_destroy (dialog);
      return;
    }
  page = gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), num);
  list = head;
  prev = NULL;
  while ((list != NULL) && (list->namespacepage != page))
    {
      prev = list;
      list = list->next;
    }
  if (list == NULL)
    {
      GE_BREAK (ectx, 0);
      return;
    }
  /* open window to ask for confirmation,
     only then delete */

  dialog = gtk_message_dialog_new
    (NULL,
     GTK_DIALOG_MODAL,
     GTK_MESSAGE_ERROR,
     GTK_BUTTONS_YES_NO,
     _("Should the namespace `%s' really be deleted?"), list->name);
  ret = gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
  if (GTK_RESPONSE_YES != ret)
    return;
  gtk_notebook_remove_page (GTK_NOTEBOOK (notebook), num);
  NS_deleteNamespace (ectx, cfg, list->name);
  frame_destroy (list->treeview);
  DEBUG_END ();
}

typedef struct
{
  unsigned int anonymityLevel;
  char *namespaceName;
  TIME_T updateInterval;
  HashCode512 *lastId;
  HashCode512 thisId;
  HashCode512 *nextId;
} IUC;

/**
 * Publish the selected file in the
 * selected namespace.
 */
static void
addToNamespaceCB (GtkTreeModel * model,
                  GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
  IUC *cls = data;
  struct ECRS_URI *resultURI;
  struct ECRS_URI *dst;
  struct ECRS_MetaData *meta;
  NamespaceList *list;
  ECRS_FileInfo fi;

  DEBUG_BEGIN ();
  dst = NULL;
  gtk_tree_model_get (model,
                      iter, NAMESPACE_URI, &dst, NAMESPACE_META, &meta, -1);
  if (dst == NULL)
    {
      GE_BREAK (ectx, 0);
      return;
    }
  resultURI = NS_addToNamespace (ectx, cfg, cls->anonymityLevel, 1000,  /* FIXME: priority */
                                 get_time () + 2 * cronYEARS,   /* FIXME: expiration */
                                 cls->namespaceName,
                                 cls->updateInterval,
                                 cls->lastId,
                                 &cls->thisId, cls->nextId, dst, meta);
  if (resultURI != NULL)
    {
      list = head;
      while ((list != NULL) && (0 != strcmp (cls->namespaceName, list->name)))
        list = list->next;
      if (list == NULL)
        {
          GE_BREAK (ectx, 0);
        }
      else
        {
          /* update namespace content list! */
          fi.uri = dst;
          fi.meta = meta;
          addNamespaceContentToModel (list->model,
                                      &fi,
                                      &cls->thisId,
                                      cls->nextId,
                                      cls->updateInterval,
                                      cls->updateInterval + TIME (NULL));
        }
      ECRS_freeUri (resultURI);
    }
  else
    {
      infoMessage (YES,
                   _("Failed to insert content into namespace "
                     "(consult logs).\n"));
    }
  DEBUG_END ();
}

void
on_namespaceInsertMetaDataDialogMetaDataAddButton_clicked_fs (GtkWidget *
                                                              dummy1,
                                                              GtkWidget *
                                                              dummy2)
{
  handleMetaDataListUpdate (metaXML,
                            "namespaceInsertMetaTypeComboBox",
                            "metaDataValueEntry", "metaDataTreeView");
}

/**
 * User clicked the "add" button, add content from
 * available content list to the currently selected
 * namespace.
 */
void
on_namespaceInsertButton_clicked_fs (GtkWidget * dummy1, GtkWidget * dummy2)
{

  const char *identifierName;
  NamespaceList *list;
  GtkWidget *nameLine;
  GtkWidget *page;
  GtkWidget *notebook;
  GtkWidget *dialog;
  GtkWidget *updateIntervalComboBox;
  HashCode512 nextId;
  IUC cls;
  gint num;

  notebook = glade_xml_get_widget (getMainXML (), "localNamespacesNotebook");
  num = gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook));
  GE_ASSERT (ectx, num != -1);
  page = gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), num);
  list = head;
  while ((list != NULL) && (list->namespacepage != page))
    list = list->next;
  if (list == NULL)
    {
      GE_BREAK (ectx, 0);
      return;
    }
  cls.namespaceName = list->name;

  metaXML
    = glade_xml_new (getGladeFileName (),
                     "namespaceInsertDialog", PACKAGE_NAME);
  connectGladeWithPlugins (metaXML);
  dialog = glade_xml_get_widget (metaXML, "namespaceInsertDialog");
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);

  updateIntervalComboBox = glade_xml_get_widget (metaXML,
                                                 "updateIntervalComboBoxEntry");
  gtk_combo_box_set_active (GTK_COMBO_BOX (updateIntervalComboBox), 0);


  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_OK)
    {
      if (OK != tryParseTimeInterval (metaXML,
                                      "updateIntervalComboBoxEntry",
                                      &cls.updateInterval))
        {
          /* this should be impossible - OK button is
             deactivated while parse errors are there */
          gtk_widget_destroy (dialog);
          UNREF (metaXML);
          metaXML = NULL;
          dialog = gtk_message_dialog_new
            (NULL,
             GTK_DIALOG_MODAL,
             GTK_MESSAGE_ERROR,
             GTK_BUTTONS_CLOSE, _("Failed to parse given time interval!"));
          gtk_dialog_run (GTK_DIALOG (dialog));
          gtk_widget_destroy (dialog);
          return;
        }
      cls.anonymityLevel
        = getSpinButtonValue (metaXML, "anonymitySpinButton");
      nameLine = glade_xml_get_widget (metaXML,
                                       "namespaceContentIdentifierEntry");
      identifierName = gtk_entry_get_text (GTK_ENTRY (nameLine));
      if (identifierName == NULL)
        identifierName = "";
      hash (identifierName, strlen (identifierName), &cls.thisId);
      cls.lastId = NULL;

      nameLine = glade_xml_get_widget (metaXML, "nextIdentifierEntry");
      identifierName = gtk_entry_get_text (GTK_ENTRY (nameLine));
      if ((identifierName == NULL) || (strlen (identifierName) == 0))
        {
          cls.nextId = NULL;
        }
      else
        {
          hash (identifierName, strlen (identifierName), &nextId);
          cls.nextId = &nextId;
        }
      ggc_tree_selection_selected_foreach
        (content_selection, &addToNamespaceCB, &cls);
    }
  gtk_widget_destroy (dialog);
  UNREF (metaXML);
  metaXML = NULL;
}


/**
 * User clicked on update; launch update dialog
 * and perform namespace content update.
 */
void
on_namespaceUpdateButton_clicked_fs (GtkWidget * dummy1, GtkWidget * dummy2)
{
  NamespaceList *list;
  GtkTreeIter iter;
  HashCode512 nextId;
  HashCode512 lastId;
  GtkTreeSelection *selection;
  IUC cls;
  char *last;
  char *next;
  char *freq;
  EncName nextnext;
  GtkWidget *nextEntryLine;
  GtkWidget *identifierLabel;
  GtkWidget *updateIntervalComboBox;
  GtkWidget *dialog;
  GtkWidget *mdialog;
  GtkWidget *notebook;
  GtkWidget *page;
  GtkTreeModel *model;
  gint num;
  const char *nn_str;

  DEBUG_BEGIN ();
  /* find out which namespace this is about */
  notebook = glade_xml_get_widget (getMainXML (), "localNamespacesNotebook");
  num = gtk_notebook_get_current_page (GTK_NOTEBOOK (notebook));
  GE_ASSERT (ectx, num != -1);
  page = gtk_notebook_get_nth_page (GTK_NOTEBOOK (notebook), num);
  list = head;
  while ((list != NULL) && (list->namespacepage != page))
    list = list->next;
  if (list == NULL)
    {
      GE_BREAK (ectx, 0);
      return;
    }
  cls.namespaceName = list->name;

  /* find out what we are updating */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (list->treeview));
  if (TRUE != gtk_tree_selection_get_selected (selection, NULL, &iter))
    {
      GE_BREAK (ectx, 0);
      return;
    }
  gtk_tree_model_get (list->model,
                      &iter,
                      IN_NAMESPACE_LAST_STRING, &last,
                      IN_NAMESPACE_NEXT_STRING, &next,
                      IN_NAMESPACE_PUB_FREQ_STRING, &freq, -1);
  if ((last == NULL) || (next == NULL) || (freq == NULL))
    {
      GE_BREAK (NULL, 0);
      return;
    }
  if (OK != parseTimeInterval (freq, &cls.updateInterval))
    {
      GE_BREAK (ectx, 0);
      cls.updateInterval = ECRS_SBLOCK_UPDATE_SPORADIC;
    }

  /* create update dialog */
  metaXML
    = glade_xml_new (getGladeFileName (),
                     "namespaceUpdateDialog", PACKAGE_NAME);
  connectGladeWithPlugins (metaXML);
  dialog = glade_xml_get_widget (metaXML, "namespaceUpdateDialog");
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
  identifierLabel = glade_xml_get_widget (metaXML, "identifierLabel");
  gtk_label_set_text (GTK_LABEL (identifierLabel), next);
  enc2hash (last, &lastId);
  cls.lastId = &lastId;
  enc2hash (next, &cls.thisId);

  nextEntryLine = glade_xml_get_widget (metaXML, "nextIdentifierEntry");
  if ((cls.updateInterval != ECRS_SBLOCK_UPDATE_SPORADIC) &&
      (OK == NS_computeNextId (NULL,
                               cfg,
                               cls.namespaceName,
                               &lastId,
                               &cls.thisId, cls.updateInterval, &nextId)))
    {
      hash2enc (&nextId, &nextnext);
      gtk_entry_set_text (GTK_ENTRY (nextEntryLine), (char *) &nextnext);
      gtk_widget_set_sensitive (nextEntryLine, FALSE);
    }
  /* set update interval in dialog to
     the existing update interval */
  updateIntervalComboBox = glade_xml_get_widget (metaXML,
                                                 "namespaceUpdateIntervalComboBoxEntry");
  model = gtk_combo_box_get_model (GTK_COMBO_BOX (updateIntervalComboBox));
  gtk_list_store_insert (GTK_LIST_STORE (model), &iter, 0);
  gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, freq, -1);
  gtk_combo_box_set_active (GTK_COMBO_BOX (updateIntervalComboBox), 0);

  /* run update dialog */
  if (gtk_dialog_run (GTK_DIALOG (dialog)) != GTK_RESPONSE_OK)
    goto CLEANUP;
  gtk_widget_hide (dialog);

  /* get data from update dialog */
  nn_str = gtk_entry_get_text (GTK_ENTRY (nextEntryLine));
  if (nn_str == NULL)
    nn_str = "";
  hash (nn_str, strlen (nn_str), &nextId);
  cls.nextId = &nextId;

  if (OK != tryParseTimeInterval (metaXML,
                                  "namespaceUpdateIntervalComboBoxEntry",
                                  &cls.updateInterval))
    {
      /* This should be impossible since the
         the OK button is deactivated while parse errors are present */
      mdialog = gtk_message_dialog_new
        (NULL,
         GTK_DIALOG_MODAL,
         GTK_MESSAGE_ERROR,
         GTK_BUTTONS_CLOSE, _("Failed to parse given time interval!"));
      gtk_dialog_run (GTK_DIALOG (mdialog));
      gtk_widget_destroy (mdialog);
      goto CLEANUP;
    }
  cls.anonymityLevel
    = getSpinButtonValue (metaXML, "namespaceUpdateAnonymitySpinButton");

  /* run actual update */
  ggc_tree_selection_selected_foreach
    (content_selection, &addToNamespaceCB, &cls);
CLEANUP:
  gtk_widget_destroy (dialog);
  UNREF (metaXML);
  metaXML = NULL;
  free (last);
  free (next);
  free (freq);
  DEBUG_END ();
}



void
fs_namespace_start ()
{
  GtkWidget *contentList;
  GtkListStore *model;
  GtkCellRenderer *renderer;
  GtkWidget *trackCheckButton;
  GtkTreeViewColumn *column;
  int col;

  DEBUG_BEGIN ();
  trackCheckButton
    = glade_xml_get_widget (getMainXML (), "trackingCheckButton");
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (trackCheckButton),
                                URITRACK_trackStatus (ectx,
                                                      cfg) ==
                                YES ? TRUE : FALSE);

  contentList = glade_xml_get_widget (getMainXML (), "availableContentList");

  model = gtk_list_store_new (NAMESPACE_NUM, G_TYPE_STRING,     /* name */
                              G_TYPE_UINT64,    /* size */
                              G_TYPE_STRING,    /* human-readable size */
                              G_TYPE_STRING,    /* uri-string */
                              G_TYPE_POINTER, G_TYPE_POINTER);  /* uri */
  gtk_tree_view_set_model (GTK_TREE_VIEW (contentList),
                           GTK_TREE_MODEL (model));
  content_selection =
    gtk_tree_view_get_selection (GTK_TREE_VIEW (contentList));
  gtk_tree_selection_set_mode (content_selection, GTK_SELECTION_MULTIPLE);

  g_signal_connect_data (content_selection,
                         "changed",
                         G_CALLBACK (&on_namespaceContentSelectionChanged),
                         NULL, NULL, 0);


  renderer = gtk_cell_renderer_text_new ();
  col =
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (contentList),
                                                 -1, _("Filename"), renderer,
                                                 "text", NAMESPACE_FILENAME,
                                                 NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (contentList), col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, NAMESPACE_FILENAME);
  /*gtk_tree_view_column_set_sort_indicator(column, TRUE); */

  gtk_tree_view_column_set_resizable (gtk_tree_view_get_column
                                      (GTK_TREE_VIEW (contentList), col - 1),
                                      TRUE);
  renderer = gtk_cell_renderer_text_new ();
  g_object_set (renderer, "xalign", 1.00, NULL);
  col =
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (contentList),
                                                 -1, _("Filesize"), renderer,
                                                 "text", NAMESPACE_HSIZE,
                                                 NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (contentList), col - 1);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_clickable (column, TRUE);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_sort_column_id (column, NAMESPACE_SIZE);
  /*gtk_tree_view_column_set_sort_indicator(column, TRUE); */
  gtk_tree_view_column_set_resizable (gtk_tree_view_get_column
                                      (GTK_TREE_VIEW (contentList), col - 1),
                                      TRUE);
  renderer = gtk_cell_renderer_text_new ();
  col =
    gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (contentList),
                                                 -1, _("URI"), renderer,
                                                 "text", NAMESPACE_URISTRING,
                                                 NULL);
  column = gtk_tree_view_get_column (GTK_TREE_VIEW (contentList), col - 1);
  gtk_tree_view_column_set_reorderable (column, TRUE);
  gtk_tree_view_column_set_resizable (column, TRUE);
  gtk_tree_view_column_set_resizable (gtk_tree_view_get_column
                                      (GTK_TREE_VIEW (contentList), col - 1),
                                      TRUE);
  URITRACK_registerTrackCallback (ectx, cfg, &updateViewSave, NULL);
  NS_listNamespaces (ectx, cfg, &addTabForNamespace, NULL);
  NS_registerDiscoveryCallback (ectx, cfg, &namespace_discovered_cb, NULL);
  DEBUG_END ();
}


void
fs_namespace_stop ()
{
  NS_unregisterDiscoveryCallback (&namespace_discovered_cb, NULL);
  while (head != NULL)
    frame_destroy (head->treeview);
  URITRACK_unregisterTrackCallback (&updateViewSave, NULL);
}

/* end of namespace.c */
