# plugs/user.py
#
#

""" user related commands """

__copyright__ = 'this file is in the public domain'

from gozerbot.generic import getwho, stripident
from gozerbot.commands import cmnds
from gozerbot.examples import examples
from gozerbot.users import users
from gozerbot.aliases import aliases, aliasdel
from gozerbot.plughelp import plughelp

plughelp.add('user', 'manage users')

def handle_getuserperm(bot, ievent):
    """ user-allperm <perm> .. list all users with permission <perm> """
    try:
        perm = ievent.args[0].upper()
    except IndexError:
        ievent.missing('<perm>')
        return
    result = []
    for i in users.data:
        if perm in i.perms:
            result.append(i.name)
    if result:
        ievent.reply('users with %s permission: ' % perm, result, dot=True)
    else:
        ievent.reply("no users with %s permission found" % perm)

cmnds.add('user-allperm', handle_getuserperm, 'OPER')
examples.add('user-allperm', 'user-allperm <perm> .. get users with \
<perm> permission', 'user-allperm rss')

def handle_getuserstatus(bot, ievent):
    """ user-allstatus <status> .. list users with status <status> """
    try:
        status = ievent.args[0].upper()
    except IndexError:
        ievent.missing('<status>')
        return
    result = []
    for i in users.data:
        if status in i.status:
            result.append(i.name)
    if result:
        ievent.reply("users with %s status: " % status, result, dot=True)
    else:
        ievent.reply("no users with %s status found" % status)

cmnds.add('user-allstatus', handle_getuserstatus, 'OPER')
examples.add('user-allstatus', 'user-allstatus <status> ..get users with \
<status> status', 'user-allstatus #dunkbots')

def handle_meet(bot, ievent):
    """ meet <nick> .. introduce a new user to the bot """
    try:
        nick = ievent.args[0].lower()
    except IndexError:
        ievent.missing('<nick>')
        return
    if users.exist(nick):
        ievent.reply('there is already a user with username %s' % nick)
        return
    userhost = getwho(bot, nick)
    if not userhost:
        ievent.reply("can't find userhost of %s" % nick)
        return
    username = users.getname(userhost)
    if username:
        ievent.reply('we already have a user with userhost %s (%s)' % \
(userhost, username))
        return
    if users.add(nick, [userhost, ], ['USER', ]):
        ievent.reply('%s added to user database' % nick)

cmnds.add('user-meet', handle_meet, ['OPER', 'MEET'])
examples.add('user-meet', 'meet <nick> .. introduce <nick> to the bot', \
'user-meet dunker')
aliases.data['meet'] = 'user-meet'

def handle_adduser(bot, ievent):
    """ user-add <name> <userhost> .. add a new user to the bot """
    try:
        (name, userhost) = ievent.args
    except ValueError:
        ievent.missing('<name> <userhost>')
        return
    username = users.getname(userhost)
    if username:
        ievent.reply('we already have a user with userhost %s (%s)' % \
(userhost, username))
        return
    if users.add(name.lower(), [userhost, ], ['USER', ]):
        ievent.reply('%s added to user database' % name)

cmnds.add('user-add', handle_adduser, 'OPER')
examples.add('user-add', 'user-add <name> <userhost> .. add <name> with \
<userhost> to the bot', 'user-add dunker bart@localhost')

def handle_merge(bot, ievent):
    """ user-merge <name> <nick> .. merge the userhost into an already \
        existing user """
    if len(ievent.args) != 2:
        ievent.missing('<name> <nick>')
        return  
    name, nick = ievent.args
    name = name.lower()
    if name == 'owner' and ievent.userhost != bot.owner:
         ievent.reply('only owneruserhost can merge with owner user')
         return 
    if users.gotperm(name, 'OPER') and not users.allowed(ievent.userhost, \
'OPER'):
        ievent.reply("only OPER perm can merge with OPER user")
        return
    if not users.exist(name):
        ievent.reply("we have no user %s" % name)
        return
    userhost = getwho(bot, nick)
    if not userhost:
        ievent.reply("can't find userhost of %s" % nick)
        return
    username = users.getname(userhost)
    if username:
        ievent.reply('we already have a user with userhost %s (%s)' % \
(userhost, username))
        return
    if users.merge(name, userhost):
        ievent.reply('%s merged' % nick)
    else:
        ievent.reply('merge failed')

cmnds.add('user-merge', handle_merge, ['OPER', 'MEET'])
examples.add('user-merge', 'user-merge <name> <nick> .. merge record with \
<name> with userhost from <nick>', 'merge bart dunker')
aliases.data['merge'] = 'user-merge'

def handle_delete(bot, ievent):
    """ user-del <name> .. remove user """
    if len(ievent.args) == 0:
        ievent.missing('<name>')
        return
    name = ' ' .join(ievent.args)
    if name == 'owner':
        ievent.reply("can't delete owner")
        return
    users.delete(name)
    ievent.reply('%s deleted' % name)

cmnds.add('user-del', handle_delete, 'OPER')
examples.add('user-del', 'user-del <name> .. delete user with <username>' , \
'user-del dunker')
aliases.data['delete'] = 'user-del'

def handle_us(bot, ievent):
    """ user-scan <txt> .. scan for user """
    try:
        name = ievent.args[0]
    except IndexError:
        ievent.missing('<txt>')
        return
    names = users.names()
    result = []
    for i in names:
        if i.find(name.lower()) != -1:
            result.append(i)
    if result:
        ievent.reply("users matching %s: " % name, result, dot=True)
    else:
        ievent.reply('no users matched')

cmnds.add('user-scan', handle_us, 'OPER')
examples.add('user-scan', 'user-scan <txt> .. search database for matching \
usernames', 'user-scan dunk')
aliases.data['us'] = 'user-scan'

def handle_names(bot, ievent):
    """ user-names .. show registered users """
    ievent.reply("usernames: ", users.names(), dot=True)

cmnds.add('user-names', handle_names, 'OPER')
examples.add('user-names', 'show names of registered users', 'user-names')
aliases.data['names'] = 'user-names'

def handle_name(bot, ievent):
    """ user-name .. show name of user giving the command """
    ievent.reply('your name is %s' % users.getname(ievent.userhost))

cmnds.add('user-name', handle_name, 'USER')
examples.add('user-name', 'show name of user giving the command', 'user-name')
aliases.data['name'] = 'user-name'

def handle_getname(bot, ievent):
    """ user-getname <nick> .. fetch name of nick """
    try:
        nick = ievent.args[0]
    except IndexError:
        ievent.missing("<nick>")
        return
    userhost = getwho(bot, nick)
    if not userhost:
        ievent.reply("can't find userhost of %s" % nick)
        return
    name = users.getname(userhost)
    if not name:
        ievent.reply("can't find user for %s" % userhost)
        return
    ievent.reply(name)

cmnds.add('user-getname', handle_getname, 'USER')
examples.add('user-getname', 'user-getname <nick> .. get the name of \
<nick>', 'user-getname dunker')
aliases.data['gn'] = 'user-getname'
aliases.data['getname'] = 'user-getname'

def handle_addperm(bot, ievent):
    """ user-addperm <name> <perm> .. add permission """
    if len(ievent.args) != 2:
        ievent.missing('<name> <perm>')
        return
    (name, perm) = ievent.args
    perm = perm.upper()
    user = users.byname(name)
    if not user:
        ievent.reply("can't find user %s" % name)
        return
    if perm in user.perms:
        ievent.reply('%s already has permission %s' % (name, perm))
        return
    else:
        user.perms.append(perm)
    users.save()
    ievent.reply('%s perm added' % perm)

cmnds.add('user-addperm', handle_addperm, 'OPER')
examples.add('user-addperm', 'add permission to user <name>', \
'user-addperm dunker rss')
aliases.data['setperms'] = 'user-addperm'
aliases.data['addperms'] = 'user-addperm'

def handle_getperms(bot, ievent):
    """ user-getperms <name> .. get permissions of name"""
    try:
        name = ievent.args[0]
    except IndexError:
        ievent.missing('<name>')
        return
    user = users.byname(name)
    if not user:
        ievent.reply("can't find user %s" % name)
        return
    if user.perms:
        ievent.reply("permissions of %s: " % name, user.perms, dot=True)
    else:
        ievent.reply('%s has no permissions set' % name)

cmnds.add('user-getperms', handle_getperms, 'OPER')
examples.add('user-getperms', 'user-getperms <name> .. get permissions of \
<name>', 'user-getperms dunker')
aliases.data['getperms'] = 'user-getperms'

def handle_perms(bot, ievent):
    """ user-perms .. get permission of the user given the command """
    if ievent.rest:
        ievent.reply('use getperms to get perms of somebody else')
        return
    user = users.getuser(ievent.userhost)
    if not user:
        ievent.reply("can't find user for %s" % ievent.userhost)
        return
    ievent.reply("you have permissions: ", user.perms, dot=True)

cmnds.add('user-perms', handle_perms, 'USER')
examples.add('user-perms', 'get permissions', 'user-perms')
aliases.data['perms'] = 'user-perms'

def handle_delperm(bot, ievent):
    """ user-delperm <name> <perm> .. delete permission of name """
    if len(ievent.args) != 2:
        ievent.missing('<name> <perm>')
        return
    name, perm = ievent.args
    perm = perm.upper()
    user = users.byname(name)
    if not user:
        ievent.reply("can't find user %s" % name)
        return
    if perm == 'OPER' and user.name == 'owner':
        ievent.reply("can't delete OPER from owner")
        return 
    if perm in user.perms:
        user.perms.remove(perm)
    else:
        ievent.reply("%s has no %s permission" % (name, perm))
        return
    users.save()
    ievent.reply('%s perm removed' % perm)

cmnds.add('user-delperm', handle_delperm, 'OPER')
examples.add('user-delperm', 'user-delperm <name> <perm> .. delete from user \
<name> permission <perm>', 'user-delperm dunker rss')

def handle_addpermall(bot, ievent):
    """ user-addpermall <perm> .. add permission to all users """
    try:
        perm = ievent.args[0].upper()
    except IndexError:
        ievent.missing('<perm>')
        return
    if perm == 'OPER':
        ievent.reply("can't add OPER permissions to all")
        return
    for i in users.data:
        i.perms.append(perm)
    users.save()
    ievent.reply('%s perm added' % perm)

cmnds.add('user-addpermall', handle_addpermall, 'OPER')
examples.add('user-addpermall', 'user-addpermall <perm> .. add <permission> to \
all users', 'user-addpermall USER')

def handle_delpermall(bot, ievent):
    """ user-delpermall <perm> .. delete permission from all users """
    try:
        perm = ievent.args[0].upper()
    except IndexError:
        ievent.missing('<perm>')
        return
    if perm == 'OPER':
        ievent.reply("can't delete OPER permissions from all")
        return
    for i in users.data:
        while 1:
            try:
                i.perms.remove(perm)
            except ValueError:
                break
    users.save()
    ievent.reply('%s perm deleted' % perm)

cmnds.add('user-delpermall', handle_delpermall, 'OPER')
examples.add('user-delpermall', 'user-delpermall <perm> .. delete <permission> \
from all users', 'user-delpersall BLA')

def handle_addstatus(bot, ievent):
    """ user-addstatus <name> <status> .. add status of name """
    if len(ievent.args) != 2:
        ievent.missing('<name> <status>')
        return
    name, status = ievent.args
    status = status.upper()
    user = users.byname(name)
    if not user:
        ievent.reply("can't find user %s" % name)
        return
    if status in user.status:
        ievent.reply("%s already has %s in status list" % \
(name, status))
        return
    user.status.append(status)
    users.save()
    ievent.reply('%s status added' % status)

cmnds.add('user-addstatus', handle_addstatus, 'OPER')
examples.add('user-addstatus', 'user-addstatus <name> <status>', \
'user-addstatus dunker #bartbot')
aliases.data['setstatus'] = 'user-addstatus'
aliases.data['addstatus'] = 'user-addstatus'

def handle_getstatus(bot, ievent):
    """ user-getstatus <name> .. get status of name """
    try:
        name = ievent.args[0]
    except IndexError:
        ievent.missing('<name>')
        return
    user = users.byname(name)
    if not user:
        ievent.reply("can't find user %s" % name)
        return
    if user.status:
        ievent.reply("status of %s: " % name, user.status, dot=True)
    else:
        ievent.reply('%s has no status set' % name)

cmnds.add('user-getstatus', handle_getstatus, 'OPER')
examples.add('user-getstatus', 'user-getstatus <name> .. get status of \
<name>', 'user-getstatus dunker')
aliases.data['getstatus'] = 'user-getstatus'

def handle_status(bot, ievent):
    """ user-status .. get status of user given the command """
    user = users.getuser(ievent.userhost)
    if user.status:
        ievent.reply("you have status: ", user.status, dot=True)
    else:
        ievent.reply('you have no status set')

cmnds.add('user-status', handle_status, 'USER')
examples.add('user-status', 'get status', 'user-status')
aliases.data['status'] = 'user-status'

def handle_delstatus(bot, ievent):
    """ user-delstatus <name> <status> .. delete status of name """
    if len(ievent.args) != 2:
        ievent.missing('<name> <status>')
        return
    name, status = ievent.args
    status = status.upper()
    user = users.byname(name)
    if not user:
        ievent.reply("can't find user %s" % name)
        return
    if status in user.status:
        user.status.remove(status)
    else:
        ievent.reply("%s has no %s status" % (name, status))
        return
    users.save()
    ievent.reply('%s status deleted' % status)

cmnds.add('user-delstatus', handle_delstatus, 'OPER')
examples.add('user-delstatus', 'user-delstatus <name> <status>', \
'user-delstatus dunker #dunkbots')
aliases.data['delstatus'] = 'user-delstatus'

def handle_adduserhost(bot, ievent):
    """ user-adduserhost <name> <userhost> .. add to userhosts of name """ 
    if len(ievent.args) != 2:
        ievent.missing('<name> <userhost>')
        return
    name, userhost = ievent.args
    name = name.lower()
    if name == 'owner' and ievent.stripped != bot.owner:
        ievent.reply('only owneruserhost can add userhost to owner')
        return
    userhost = stripident(userhost)
    user = users.byname(name)
    if not user:
        ievent.reply("can't find user %s" % name)
        return
    if userhost in user.userhosts:
        ievent.reply("%s already has userhosts %s" % \
(name,userhost))
        return
    user.userhosts.append(userhost)
    users.adduserhost(userhost, user)
    users.save()
    ievent.reply('userhost added')

cmnds.add('user-adduserhost', handle_adduserhost, 'OPER')
examples.add('user-adduserhost', 'user-adduserhost <name> <userhost> .. add \
<userhost> to userhosts of <name>', 'user-adduserhost dunker bart?@*.a2000.nl')
aliases.data['adduserhost'] = 'user-adduserhost'
aliases.data['adduserhosts'] = 'user-adduserhost'

def handle_deluserhost(bot, ievent):
    """ user-deluserhost <name> <userhost> .. remove from userhosts of \
        name """
    if len(ievent.args) != 2:
        ievent.missing('<name> <userhost>')
        return
    name, userhost = ievent.args
    name = name.lower()
    if name == 'owner' and ievent.stripped != bot.owner:
        ievent.reply('only owneruserhost can delete userhost from owner')
        return
    userhost = stripident(userhost)
    user = users.byname(name)
    if not user:
        ievent.reply("can't find user %s" % name)
        return
    if userhost in user.userhosts:
        users.deluserhost(userhost)
        user.userhosts.remove(userhost)
    else:
        ievent.reply("%s has no %s in userhost list" % (name, \
userhost))
        return
    users.save()
    ievent.reply('userhost removed')

cmnds.add('user-deluserhost', handle_deluserhost, 'OPER')
examples.add('user-deluserhost', 'user-deluserhost <name> <userhost> .. \
delete from usershosts of <name> userhost <userhost>', \
'user-deluserhost dunker bart1@bla.a2000.nl')
aliases.data['deluserhost'] = 'user-deluserhost'
aliases.data['deluserhosts'] = 'user-deluserhost'

def handle_getuserhosts(bot, ievent):
    """ user-getuserhosts <name> .. get userhosts of name """
    try:
        name = ievent.args[0]
    except IndexError:
        ievent.missing('<name>')
        return
    user = users.byname(name)
    if not user:
        ievent.reply("can't find user %s" % name)
        return
    if user.userhosts:
        ievent.reply("userhosts of %s: " % name, user.userhosts, dot=True)
    else:
        ievent.reply('%s has no userhosts set' % name)

cmnds.add('user-getuserhosts', handle_getuserhosts, 'OPER')
examples.add('user-getuserhosts', 'user-getuserhosts <name> .. get \
userhosts of <name>', 'user-getuserhosts dunker')
aliases.data['getuserhosts'] = 'user-getuserhosts'

def handle_userhosts(bot, ievent):
    """ user-userhosts .. get userhosts of user giving the command """
    user = users.getuser(ievent.userhost)
    if user.userhosts:
        ievent.reply("you have userhosts: ", user.userhosts, dot=True)
    else:
        ievent.reply('you have no userhosts set')

cmnds.add('user-userhosts', handle_userhosts, 'USER')
examples.add('user-userhosts', 'get userhosts', 'user-userhosts')
aliases.data['userhosts'] = 'user-userhosts'

def handle_getemail(bot, ievent):
    """ user-getemail <name> .. get email of name """
    try:
        name = ievent.args[0]
    except IndexError:
        ievent.missing('<name>')
        return
    user = users.byname(name)
    if not user:
        ievent.reply("can't find user %s" % name)
        return
    if user.email:
        ievent.reply('%s has email %s' % (name, user.email))
    else:
        ievent.reply('%s has no email set' % name)

cmnds.add('user-getemail', handle_getemail, 'USER')
examples.add('user-getemail', 'user-getemail <name> .. get email from user \
<name>', 'user-getemail dunker')
aliases.data['getemail'] = 'user-getemail'

def handle_setemail(bot, ievent):
    """ user-setemail <name> .. set email of name """
    try:
        name, email = ievent.args
    except ValueError:
        ievent.missing('<name> <email>')
        return
    user = users.byname(name)
    if not user:
        ievent.reply("can't find user %s" % name)
        return
    user.email = email
    users.save()
    ievent.reply('email set')

cmnds.add('user-setemail', handle_setemail, 'OPER')
examples.add('user-setemail', 'user-setemail <name> <email>.. set email of \
user <name>', 'user-setemail dunker bart@gozerbot.org')
aliases.data['setemail'] = 'user-setemail'

def handle_email(bot, ievent):
    """ user-email .. show email of user giving the command """
    if len(ievent.args) != 0:
        ievent.reply('use getemail to get the email address of an user .. \
email shows your own mail address')
        return
    user = users.getuser(ievent.userhost)
    if user.email:
        ievent.reply('%s' % user.email)
    else:
        ievent.reply('no email set')

cmnds.add('user-email', handle_email, 'USER')
examples.add('user-email', 'get email', 'user-email')
aliases.data['email'] = 'user-email'

def handle_delemail(bot, ievent):
    """ user-delemail .. reset email of user giving the command """
    user = users.getuser(ievent.userhost)
    user.email = ""
    users.save()
    ievent.reply('email removed')

cmnds.add('user-delemail', handle_delemail, 'OPER')
examples.add('user-delemail', 'reset email', 'user-delemail')
aliases.data['delemail'] = 'user-delemail'

def handle_usermatch(bot, ievent):
    """ user-match <userhost> .. show userrecord of user matching userhost """
    try:
        userhost = ievent.args[0]
    except IndexError:
        ievent.missing('<userhost>')
        return
    user = users.getuser(userhost)
    if not user:
        ievent.reply("can't find user for %s" % userhost)
        return
    ievent.reply("userrecord = %s" % str(user))

cmnds.add('user-match', handle_usermatch, ['USER', 'OPER'])
examples.add('user-match', 'user-match <userhost>', 'user-match test@test')
aliases.data['match'] = 'user-match'

def handle_usercheck(bot, ievent):
    """ user-check <nick> .. show userrecord of nick """
    try:
        nick = ievent.args[0]
    except IndexError:
        ievent.missing('<nick>')
        return
    userhost = getwho(bot, nick)
    if not userhost:
        ievent.reply("can't find userhost of %s" % nick)
        return
    user = users.getuser(userhost)
    if not user:
        ievent.reply("can't find user for %s" % userhost)
        return
    ievent.reply("userrecord = %s" % str(user))

cmnds.add('user-check', handle_usercheck, 'OPER')
examples.add('user-check', 'user-check <nick>', 'user-check dunker')
aliases.data['check'] = 'user-check'

def handle_addpermit(bot, ievent):
    """ user-addpermit <name> <permit> .. add permit to permit list of \
        <name> """
    try:
        who, what = ievent.args
    except ValueError:
        ievent.missing("<name> <what>")
        return
    whouser = users.byname(who)
    if not whouser:
        ievent.reply("can't find username of %s" % who)
        return
    user = users.getuser(ievent.userhost)
    if not user:
        ievent.reply("can't find user for %s" % ievent.userhost)
        return
    if (who, what) not in user.permit:
        user.permit.append((whouser.name, what))
        users.save()
        ievent.reply('permit added')
    else:
        ievent.reply('%s is already allowed to do %s' % (who, what))

cmnds.add('user-addpermit', handle_addpermit, 'USER')
examples.add('user-addpermit', 'user-addpermit <nick> <what> .. permit \
nick access to <what> .. use setperms to add permissions', \
'user-addpermit dunker todo')
aliasdel('allow')

def handle_permit(bot, ievent):
    """ user-permit .. get permit list of user giving the command """
    if ievent.rest:
        ievent.reply("use the allow command to allow somebody something .. \
use getname <nick> to get the username of somebody .. this command shows \
what permits you have")
        return
    user = users.getuser(ievent.userhost)
    if user.permit:
        result = []
        for i in user.permit:
            result.append(str(i))
        ievent.reply("you permit the following: ", result, dot=True)
    else:
        ievent.reply("you don't have any permits set")

cmnds.add('user-permit', handle_permit, 'USER')
examples.add('user-permit', 'show permit of user giving the command', \
'user-permit')
aliases.data['permit'] = 'user-permit'

def handle_userdelpermit(bot, ievent):
    """ user-deny <name> <permit> .. remove (name, permit) from permit list """
    try:
        who, what = ievent.args
    except ValueError:
        ievent.missing("<name> <what>")
        return
    whouser = users.byname(who)
    if not whouser:
        ievent.reply("can't find registered name of %s" % who)
        return
    user = users.getuser(ievent.userhost)
    got = 0
    for i in range(len(user.permit)-1, -1, -1):
        if (whouser.name, what) == user.permit[i]:
            del user.permit[i]
            got = 1
    if got:
        users.save()
        ievent.reply('%s denied' % what)
    else:
        ievent.reply('%s is already not allowed to do %s' % (who, what))

cmnds.add('user-delpermit', handle_userdelpermit, 'USER')
examples.add('user-delpermit', 'user-delpermit <name> <permit>', \
'user-delpermit dunker todo')
aliasdel('deny')

def handle_usersearch(bot, ievent):
    """ search user with matching userhost """
    try:
        what = ievent.args[0]
    except IndexError:
        ievent.missing('<what>')
        return
    got = []
    for i in users.data:
        for j in i.userhosts:
            if what in j:
                name = users.getname(j) 
                got.append((name, j))
    if got:
        result = []
        for i in got:
            result.append("(%s) %s" % (i[0], i[1]))
        ievent.reply(result, dot=True)
    else:
        ievent.reply('no userhost matching %s found' % what)

cmnds.add('user-search', handle_usersearch, 'OPER')
examples.add('user-search', 'search for matching userhosts', 'user-search \
gozerbot')
