/*
 *  gqcam.c 
 *
 *  Copyright (c) 2005 Giansalvo Gusinu <giansalvo at gusinu.net>
 *  Copyright (C) 1999 Cory Lueninghoener (cluenin1@bigred.unl.edu)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <pthread.h>
#include <semaphore.h>
#include <gtk/gtk.h>
#include <linux/types.h> 
#include <linux/videodev.h>
#include <signal.h>
#include <png.h>

#include "gqcam.h"
#include "frontend.h"
#include "save.h"
#include "filters.h"
#include "jpeg.h"
#include "yuv_to_rgb.h"

char version[] = VERSION;

void init_cam(struct Camera *camera)
{
  camera->greyscale = 0;
  camera->pic = NULL;
  camera->picbuff = NULL;
  camera->draw = 0;
  camera->pixmap = NULL;
  camera->frozen = 0;
  camera->update_camera = 0;
  camera->saving = 0;
  camera->inputtype = INPUT_NOT_FORCED;
  camera->savetype = SAVE_PNG;
  camera->capture = 1;
  camera->dev = 0;
  strcpy(camera->devname, "/dev/video");
  camera->docked = 1;
  camera->dump=0;
  camera->speed_fastest = 0;
  camera->currentsavepage = NULL;
  camera->timeout = 100;
  camera->on_timer = 0;
  camera->timer_struct.unit = SECONDS;
  camera->timer_struct.iscommand = 0;
  camera->swapcolors = 0;
  camera->save_struct.isinfo = 0;
  pthread_mutex_init( &camera->pref_mutex, NULL ); /* to modify pref/setting */
  pthread_mutex_init( &camera->freeze_mutex, NULL ); /* to freeze display */
  pthread_mutex_init( &camera->iscam_mutex, NULL ); /* is there an open cam? */
}

sem_t s_draw;
sem_t s_grab1, s_grab2;
int plsquit = 0;
int x_frames = 0;
int y_frames = 0;

void set_cam_info(struct Camera *camera)
{
  if (ioctl (camera->dev, VIDIOCSPICT, &camera->vid_pic) == -1) {
    perror ("ioctl (VIDIOCSPICT)");
  }
  if (ioctl (camera->dev, VIDIOCSWIN, &camera->vid_win) == -1) {
    perror ("ioctl (VIDIOCSWIN)");
  }
}

void get_cam_info(struct Camera *camera)
{
  int i;
  struct video_clip vid_clips[32];

  ioctl(camera->dev, VIDIOCGCAP, &camera->vid_caps);
  ioctl(camera->dev, VIDIOCGWIN, &camera->vid_win);
  ioctl(camera->dev, VIDIOCGPICT, &camera->vid_pic);

  for (i = 0; i < 32; i++) {
    vid_clips[i].x      = 0;
    vid_clips[i].y      = 0;
    vid_clips[i].width  = 0;
    vid_clips[i].height = 0;
  }
  camera->vid_win.clips = vid_clips;
  camera->vid_win.clipcount = 0;

  if (camera->vid_caps.type & VID_TYPE_MONOCHROME) {
    camera->greyscale = 1;
    camera->pic = realloc( camera->pic, camera->vid_caps.maxwidth*camera->vid_caps.maxheight);
    camera->picbuff = realloc( camera->picbuff, camera->vid_caps.maxwidth*camera->vid_caps.maxheight );
  }
  else {
    camera->greyscale = 0;
    camera->pic = realloc(camera->pic, camera->vid_caps.maxwidth*camera->vid_caps.maxheight*3);
    camera->picbuff = realloc(camera->pic, camera->vid_caps.maxwidth*camera->vid_caps.maxheight*3);
  }
}

/* FIXME this can be used for debugging. see caminfo() in frontend.c */
/*
void print_cam_info(struct Camera *camera)
{
  printf("Name: %s\n", camera->vid_caps.name);
  printf("Type: %i\n", camera->vid_caps.type);
  if (camera->vid_caps.type & VID_TYPE_CAPTURE) {
    printf("\tCan capture\n");
  }
  if (camera->vid_caps.type & VID_TYPE_TUNER) {
    printf("\tCan tune\n");
  }
  if (camera->vid_caps.type & VID_TYPE_TELETEXT) {
    printf("\tDoes teletext\n");
  }
  if (camera->vid_caps.type & VID_TYPE_OVERLAY) {
    printf("\tOverlay onto frame buffer\n");
  }
  if (camera->vid_caps.type & VID_TYPE_CHROMAKEY) {
    printf("\tOverlay by chromakey\n");
  }
  if (camera->vid_caps.type & VID_TYPE_CLIPPING) {
    printf("\tCan clip\n");
  }
  if (camera->vid_caps.type & VID_TYPE_FRAMERAM) {
    printf("\tUses the frame buffer memory\n");
  }
  if (camera->vid_caps.type & VID_TYPE_SCALES) {
    printf("\tScalable\n");
  }
  if (camera->vid_caps.type & VID_TYPE_MONOCHROME) {
    printf("\tMonochrome only\n");
  }
  if (camera->vid_caps.type & VID_TYPE_SUBCAPTURE) {
    printf("\tCan capture subareas of the image\n");
  }
  printf("Channels: %i\n", camera->vid_caps.channels);
  printf("Audios: %i\n", camera->vid_caps.audios);
  printf("Maxwidth: %i\n", camera->vid_caps.maxwidth);
  printf("Maxheight: %i\n", camera->vid_caps.maxheight);
  printf("Minwidth: %i\n", camera->vid_caps.minwidth);
  printf("Minheight: %i\n", camera->vid_caps.minheight);
  printf("---------\n");
  printf("X: %i\n", camera->vid_win.x);
  printf("Y: %i\n", camera->vid_win.y);
  printf("Width: %i\n", camera->vid_win.width);
  printf("Height: %i\n", camera->vid_win.height);
  printf("Chromakey: %i\n", camera->vid_win.chromakey);
  printf("Flags: %i\n", camera->vid_win.flags);
  printf("---------\n");
  printf("Brightness:\t%i (%i)\n", camera->vid_pic.brightness, camera->vid_pic.brightness/256);
  printf("Hue:\t\t%i (%i)\n", camera->vid_pic.hue, camera->vid_pic.hue/256);
  printf("Color:\t\t%i (%i)\n", camera->vid_pic.colour, camera->vid_pic.colour/256);
  printf("Contrast:\t%i (%i)\n", camera->vid_pic.contrast, camera->vid_pic.contrast/256);
  printf("Whiteness:\t%i (%i)\n", camera->vid_pic.whiteness, camera->vid_pic.whiteness/256);
  printf("Depth:\t\t%i\n", camera->vid_pic.depth);
  printf("Palette:\t%i\n",camera->vid_pic.palette);
  return;
}
*/

void open_cam(struct Camera *camera) 
{
  if((camera->dev<=0)){
    camera->dev = open(camera->devname, O_RDWR);
    //    printf("Opening: %d\n", camera->dev);
    if (camera->dev < 0) {
      perror(camera->devname);
      exit(1);
    }
  }
  pthread_mutex_unlock( &camera->iscam_mutex );
}

void close_cam(struct Camera *camera, int quiting)
{
  int debug = 0;
  pthread_mutex_lock( &camera->iscam_mutex );
  if(camera->dev > 0){
    close(camera->dev);
    camera->dev = 0;
  }
}


void display(struct Camera *camera) 
{
  GdkDrawable *drawable;
  int xpos=0, ypos=0;
  GdkRectangle update_rec;
  char tbuff[1024];
  static int frameno = 0;

  while( !plsquit ) {
    if( !sem_wait( &s_draw ) ) {
      unsigned char *tmp;
      tmp = camera->pic;
      camera->pic = camera->picbuff;
      camera->picbuff = tmp;

      sem_post( &s_grab1 );

      update_rec.x = 0;
      update_rec.y = 0;
      update_rec.width = camera->vid_caps.maxwidth;
      update_rec.height = camera->vid_caps.maxheight;
      
      xpos = (camera->vid_caps.maxwidth - camera->vid_win.width)/2;
      ypos = (camera->vid_caps.maxheight - camera->vid_win.height)/2;
      
      drawable = camera->drawing_area->window;
      
      gdk_threads_enter();
      /* Run filters */
      if (camera->swapcolors)
	swap_rgb24(camera);
      if (camera->autobright)
	auto_bright(camera);

      if (camera->greyscale)
	gdk_draw_gray_image (camera->pixmap, camera->drawing_area->style->white_gc, xpos, ypos, camera->vid_win.width, camera->vid_win.height, GDK_RGB_DITHER_NORMAL, camera->pic, camera->vid_win.width);
      else
	gdk_draw_rgb_image (camera->pixmap, camera->drawing_area->style->white_gc, xpos, ypos, camera->vid_win.width, camera->vid_win.height, GDK_RGB_DITHER_NORMAL, camera->pic, camera->vid_win.width*3);
      gdk_threads_leave();

      gdk_threads_enter();
      gtk_statusbar_pop( GTK_STATUSBAR( camera->statusbar ), 12 );
      sprintf( tbuff, "Speed (fps) - Average: %d  Current: %d. Frame no %d", 
	       camera->fps_avg, camera->fps_current, frameno++);
      gtk_statusbar_push( GTK_STATUSBAR( camera->statusbar ), 12, tbuff );
      gtk_widget_draw (camera->drawing_area, &update_rec);
      gdk_threads_leave();
    }
    x_frames++;
    y_frames++;
  }
  return;
}

void grab_image(struct Camera *camera)
{
  const JPEG_HEADER_SIZE = 2;

  GdkRectangle update_rec;
  GdkEventExpose *event;
  int input_type;
  struct ov511_frame temp;

  get_cam_info(camera);

  update_rec.x = 0;
  update_rec.y = 0;
  update_rec.width = camera->vid_caps.maxheight;
  update_rec.height = camera->vid_caps.maxwidth;

  while ( !plsquit ) {
    // order matters! the sem_waits MUST be before the mutex lock
    
    if( !sem_wait( &s_grab1 ) && ( camera->speed_fastest || !sem_wait( &s_grab2 ) ) && !pthread_mutex_lock( &camera->freeze_mutex ) && !pthread_mutex_lock( &camera->iscam_mutex )){

      pthread_mutex_lock( &camera->pref_mutex );
      if (camera->update_camera){
	set_cam_info(camera);
	get_cam_info(camera);

	camera->update_camera = 0;
      }
      pthread_mutex_unlock( &camera->pref_mutex );

      if( camera->dev ){

	/* 
	 * select actual input type based on user option, device capabilities and default.
	 * FIXME this info should rather be kept updated (when changing camera or settings)
	 * and not calculated here!
	 */
	if (camera->inputtype != INPUT_NOT_FORCED)
	  input_type = camera->inputtype;
	else
	  switch (camera->vid_pic.palette) {
	  case VIDEO_PALETTE_RGB24:  /* TESTED, it works! */
	  case VIDEO_PALETTE_RGB565: /* not tested, will it work like RGB24? */
	  case VIDEO_PALETTE_RGB555: /* not tested, will it work like RGB24? */
	  case VIDEO_PALETTE_RGB32:  /* not tested, will it work like RGB24? */
	    input_type = INPUT_RGB;
	    break;
	  case VIDEO_PALETTE_YUV420: /* TESTED, it works! and it is used as default */
	  default:
	    input_type = INPUT_YUV;
	    break;
	  }

	  switch(input_type) {
	  case INPUT_YUV:
            camera->img_size = read (camera->dev, camera->rawbuffer, camera->vid_caps.maxwidth * camera->vid_caps.maxheight * 3);
	    temp.width  = temp.rawwidth  = camera->vid_caps.maxwidth;
	    temp.height = temp.rawheight = camera->vid_caps.maxheight;
	    yuv420p_to_rgb(&temp, camera->rawbuffer, camera->picbuff, 24);
	    break;
	  case INPUT_RGB:
            camera->img_size = read (camera->dev, camera->picbuff, camera->vid_caps.maxwidth * camera->vid_caps.maxheight * 3);
	    break;
	  case INPUT_JPEG:
	    camera->img_size = read (camera->dev, camera->rawbuffer, sizeof(camera->rawbuffer)) - JPEG_HEADER_SIZE; // TODO error management
	    camera->img_size = decompress_JPEG(camera->rawbuffer + JPEG_HEADER_SIZE, camera->img_size, camera->picbuff); // TODO error management
	    break;
	  default:
	    //TODO internal error
	    fprintf(stderr, "grab_image: incorrect input_type %d\n", input_type);
	    break;
	  }
      }
      if (camera->img_size <= 0)
	fprintf(stderr, "Error reading image...\n");      
    }
    pthread_mutex_unlock( &camera->freeze_mutex );
    pthread_mutex_unlock( &camera->iscam_mutex );
    if( camera->dump ) {
      //fprintf(stderr, "gqcam.c: camera->dump\n");
      return;
    }
    if(camera->on_timer){
      savefile_append_time(camera);
      if(camera->timer_struct.beep)
	gdk_beep();
      switch (camera->savetype) {
      case SAVE_PPM:
	ppm_save(camera);
	break; 
    
      case SAVE_JPEG:
	jpeg_save(camera);	
	break;
	
      case SAVE_PNG:
	png_save(camera);
	break;
      }
      if(camera->timer_struct.iscommand){
	printf("Commanding...\n");
	system(camera->timer_struct.command);
      }
    }
    sem_post( &s_draw );
  } 
}
  
void delete_event(GtkWidget *widget, struct Camera *camera)
{
  FILE *preffile;
  char savefile[255];

  sprintf(savefile, "%s/.gqcamrc", getenv("HOME"));
  preffile = fopen(savefile, "w");
  save_pref_file(preffile, camera);
  fclose(preffile);

  gtk_main_quit ();
}

int next_frame(struct Camera *camera) {
  int val;
  
  sem_getvalue( &s_grab2, &val );
  if( !val )
     sem_post( &s_grab2 );
  return 1;
}

int increment_second_counter(struct Camera *camera) {
  static int x_seconds = 0;
  
  x_seconds++;
  // the following just prevents integer wrap
  if( ( x_seconds % 512 ) == 0 ) {
    x_seconds /= 2;
    x_frames /= 2;
  }
  camera->fps_avg = x_frames / x_seconds;
  camera->fps_current = y_frames;
  y_frames = 0;
  return;
}

int main(int argc, char *argv[])
{
  static struct Camera camera;
  pthread_t grab_thread;
  pthread_t draw_thread;
  guint timeoutid;
  int i, brightness=180, contrast=104, whiteness=155;
  unsigned char buff[3];
  char *filename = NULL, readfile[255];
  int done = 0;
  FILE *preffile;

  init_cam(&camera);
/*
  g_thread_init(NULL);
  gtk_init (&argc, &argv);
  gdk_rgb_init();
*/

  sprintf(readfile, "%s/.gqcamrc", getenv("HOME"));
  preffile = fopen(readfile, "r");
  if(preffile != NULL){
    read_pref_file(preffile, &camera);
    fclose(preffile);
  }

  while( !done ) {
    static struct option long_options[] = 
    {
      { "help", no_argument, NULL, 'h' },
      { "version", no_argument, NULL, 'V' },
      { "swap", no_argument, NULL, 's' },
      { "autobright", no_argument, NULL, 'a' },
      { "video", required_argument, NULL, 'v' },
      { "dump", required_argument, NULL, 'd' },
      { "type", required_argument, NULL, 't' },
      { "bright", required_argument, NULL, 'b' },
      { "white", required_argument, NULL, 'w' },
      { "contrast", required_argument, NULL, 'c' },
      { "maxfps", required_argument, NULL, 'm' },
      { "fullspeed", no_argument, NULL, 'F' },
      { "input", required_argument, NULL, 'i' },
      { 0, 0, 0, 0 }
    };
    int c;

    c = getopt_long( argc, argv, "FhVsav:d:b:w:c:m:t:i:", long_options, NULL );
    switch ( c ) {
    case 'h' :
      print_usage();
      exit(0);
      break;
    case 'V' :
      printf( "gqcam version %s\n", version );
      exit(0);
      break;
    case 's':
      camera.swapcolors = 1;
      break;
    case 'a':
      camera.autobright = 1;
      break;
    case 'v' :
      sprintf(camera.devname, "%s", optarg );
      break;
    case 'd' :
      camera.dump = 1;
      if( strcmp( optarg, "-" ) != 0 ) /* leave alone if stdout */
	filename = optarg;
      break;
    case 't' :
      if(!strcmp(optarg, "PPM"))
	camera.savetype = SAVE_PPM;
      else if(!strcmp(optarg, "PNG"))
	camera.savetype = SAVE_PNG;
      else if(!strcmp(optarg, "JPEG"))
	camera.savetype = SAVE_JPEG;
      else if(!strcmp(optarg, "RAW"))
	camera.savetype = SAVE_RAW;
      break;
    case 'b' :
      brightness = atoi( optarg );
      break;
    case 'w' :
      whiteness = atoi( optarg );
      break;
    case 'c' :
      contrast = atoi( optarg );
      break;
    case 'm' :
      if( atoi(optarg) > 0 ) /* Avoid weird values */
	camera.timeout = 1000/atoi(optarg);
      break;
    case 'F' :
      camera.speed_fastest = 1;
      camera.timeout = 0;
      break;
    case 'i' :
      if(!strcmp(optarg, "JPEG"))
	camera.inputtype = INPUT_JPEG;
      else if (!strcmp(optarg, "RGB"))
	camera.inputtype = INPUT_RGB;
      else if (!strcmp(optarg, "YUV"))
	camera.inputtype = INPUT_YUV;
      break;
    case '?' :
      fprintf( stderr, "invalid option, or ambiguous argument\n" );
      print_usage();
      exit(1);
      break;
    case EOF :
      done = 1;
      break;
    default :
      print_usage();
      exit(1);
    };
  }


  camera.fps_avg = 0;
  camera.fps_current = 0;
  if( camera.dump ){   
    exit( dump_pict( &camera, filename, brightness, contrast, whiteness ) );
  }

  g_thread_init(NULL);
  gtk_init (&argc, &argv);
  gdk_rgb_init();

  sem_init( &s_draw, 0, 0 );
  sem_init( &s_grab1, 0, 1 );
  sem_init( &s_grab2, 0, 1 );
  open_cam(&camera);
  get_cam_info(&camera);
  //camera.vid_pic.brightness = brightness*256;
  //camera.vid_pic.contrast = contrast*256;
  //camera.vid_pic.whiteness = whiteness*256;
  //set_cam_info(&camera);
  create_frontend(&camera);

  if( !camera.speed_fastest )
    camera.timeoutid = gtk_timeout_add(camera.timeout, (GtkFunction)next_frame, (gpointer)&camera);

  timeoutid = gtk_timeout_add(1000, (GtkFunction)increment_second_counter, (gpointer)&camera);
  
  pthread_create(&grab_thread, NULL, (void *)&grab_image, (void*)&camera);
  pthread_create(&draw_thread, NULL, (void *)&display, (void*)&camera);

  gdk_threads_enter();
  gtk_main();
  gdk_threads_leave();

  plsquit = 1; // ask our threads to quit
  // update semiphores and mutexes so threads can run and finish
  pthread_mutex_unlock( &camera.freeze_mutex );
  sem_post( &s_draw );
  sem_post( &s_grab1 );
  sem_post( &s_grab2 );
  // join the threads cleanly
  pthread_join( grab_thread, NULL );
  pthread_join( draw_thread, NULL );
  pthread_mutex_unlock( &camera.freeze_mutex );
  close_cam(&camera, 1);
  exit(0);
}

void print_usage() {
      fprintf(stderr, "gqcam %s - GTK QuickCam Control\n", VERSION);
      fprintf(stderr, "Copyright (C) 2005  Giansalvo Gusinu <giansalvo at gusinu.net>\n");
      fprintf(stderr, "Copyright (C) 1999  Cory Lueninghoener <cluenin1@bigred.unl.edu>\n\n");
      fprintf(stderr, "usage: gqcam [-hVFs] [-d file] [-c contrast] [-w whiteness] [-b brightness] [-m maxfps] [-d filename] [-t type] [-i format]\n\n");
      fprintf(stderr, "\t-h --help\t\tdisplay this help screen\n");
      fprintf(stderr, "\t-V --version\t\tdisplay version and exit\n");
      fprintf(stderr, "\t-s --swap\t\tswap RGB into BGR\n");
      fprintf(stderr, "\t-a --autobright\tperform autobrightness\n");
      fprintf(stderr, "\t-v --video <video dev>\tgrab frames from <video dev> [/dev/video]\n");//FIXME
      fprintf(stderr, "\t-d --dump <filename>\tdump image to <filename> \n\t\t\t\t(or stdout if filename is \"-\")\n");
      fprintf(stderr, "\t-t --type <type>\ttype of image to dump: [PNG], PPM,JPEG, RAW)\n");
      fprintf(stderr, "\t-b --brightness <num>\tset brightness to <num>\n");
      fprintf(stderr, "\t-w --whiteness <num>\tset white balance to <num>\n");
      fprintf(stderr, "\t-c --contrast <num>\tset contrast to <num>\n");
      fprintf(stderr, "\t-m --maxfps <num>\tset the maximum frames per second [%d]\n", 10 );//FIXME
      fprintf(stderr, "\t-F --fullspeed\t\tdisplay as many frames per second as you can\n" );
      fprintf(stderr, "\t-i --input\t\tforce input format: YUV, RGB, JPEG\n" );
}

int dump_pict( struct Camera *camera, char *filename, int brightness, int contrast, int whiteness )
{
  FILE *outf = NULL;
  if( filename != NULL ){
    strcpy(camera->savefile, filename);
    strcpy(camera->savefileclean, filename);
  }
  /*
    outf = fopen( filename, "w" );
    if( !outf ) {
      fprintf( stderr, "Couldn't open %s for writing!\n" );
      return 1;
    }
  }
  else {
    outf = stdout;
  }
*/
  fprintf( stderr, "Dumping...\n" );
  open_cam(camera);
  get_cam_info(camera);
  camera->vid_pic.brightness = brightness*256;
  camera->vid_pic.contrast = contrast*256;
  camera->vid_pic.whiteness = whiteness*256;
  set_cam_info(camera);
  sem_post( &s_grab1 );
  sem_post( &s_grab2 );
  grab_image(camera);
  camera->pic = camera->picbuff;
  close_cam(camera, 1);
  
  gdk_threads_enter();
  if(camera->savetype == SAVE_JPEG){
    camera->save_struct.smoothness = 0;
    camera->save_struct.quality = 75;
    camera->save_struct.optimize = 0;
    jpeg_save(camera);
  }
  else if(camera->savetype == SAVE_PNG){
    camera->save_struct.interlace = PNG_INTERLACE_NONE;
    camera->save_struct.compression = 6;
    png_save(camera);
  }
  else if(camera->savetype == SAVE_PPM){
    camera->save_struct.format = PPM_RAW;
    ppm_save(camera);
  }
  else if(camera->savetype == SAVE_RAW) {
	  raw_save(camera);
  }

  gdk_threads_leave();

  return 0;
}
  
