//*****************************************************************************
//                                 NetList.cpp                                *
//                                -------------                               *
//  Started     : 01/09/2003                                                  *
//  Last Update : 28/06/2007                                                  *
//  Copyright   : (C) 2003 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "netlist/NetList.hpp"

//*****************************************************************************
// Constructor.

NetList::NetList( void )
{
  m_ofnLoad.Clear( );
  m_ofnSave.Clear( );
  m_osSchems.Empty( );
}

//*****************************************************************************
// Destructor.

NetList::~NetList( )
{
}

//*****************************************************************************
// Compare algorithm for sorting strings alpha/numerically.
//
// Arguments:
//   ros1 - a reference to the first  string to be compared
//   ros2 - a reference to the second string to be compared
//
// Return Values:
//   >0 - ros1 is greater than ros2
//   =0 - ros1 is equal to     ros2
//   <0 - ros1 is less    than ros2

int  NetList::iCompare( const wxString & ros1, const wxString & ros2 )
{
  unsigned long  ulNum1, ulNum2;
  size_t    sz1, sz2;
  wxString  osNum1;
  wxString  osNum2;

  for( sz1=0; sz1<ros1.Length( ) && sz1<ros2.Length( ); sz1++ )
  {
    // Compare numbers as a whole
    if( isdigit( ros1[ sz1 ] ) && isdigit( ros2[ sz1 ] ) )
    {
      // Extract the number from each string
      osNum1 += osNum2 = wxT("");
      for( sz2=sz1; sz2<ros1.Length( ); sz2++ ) osNum1 += ros1[ sz2 ];
      for( sz2=sz1; sz2<ros2.Length( ); sz2++ ) osNum2 += ros2[ sz2 ];
      // Convert to unsigned long integers
      osNum1.ToULong( &ulNum1 );
      osNum2.ToULong( &ulNum2 );
      // Compare the two numbers
      if( ulNum1 > ulNum2 ) return(  1 );
      if( ulNum1 < ulNum2 ) return( -1 );
      // Increment the loop counter
      sz1 = osNum1.Length( ) - 1;
      continue;
    }

    // Compare characters one at a time
    if( ros1[ sz1 ] > ros2[ sz1 ] ) return(  1 );
    if( ros1[ sz1 ] < ros2[ sz1 ] ) return( -1 );
  }

  // If characters match compare string length
  if( ros1.Length( ) > ros2.Length( ) ) return(  1 );
  if( ros1.Length( ) < ros2.Length( ) ) return( -1 );

  // The strings are identical
  return( 0 );
}

//*****************************************************************************
// This function indicates whether a line in a sequence of line falls within a 
// sub-circuit definition block.
//
// Argument List:
//   A line in a sequence of lines to be tested
//
// Return Values:
//   TRUE  - The line does    falls within a su-circuit definition block
//   FALSE - The line doesn't falls within a su-circuit definition block

bool  NetList::bIsSubCkt( wxString & roLine )
{
  static  bool  bIsSubCkt=FALSE;
  wxString  os1;

  os1 = roLine;
  os1.MakeUpper( );
  if( os1.StartsWith( wxT(".SUBCKT") ) ) bIsSubCkt = TRUE;
  if( os1.StartsWith( wxT(".ENDS") ) )   bIsSubCkt = FALSE;

  return( bIsSubCkt );
}

//*****************************************************************************
// Extract the title for the circuit description which is traditionally the
// first line in the file. An '*' seems to be used as a comment designator.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractTitle( void )
{
  // Check that the circuit isn't empty
  if( IsEmpty( ) )     return( FALSE );

  // If the first line is empty assume there is no title
  wxString  os1 = Item( 0 );
  if( os1.IsEmpty( ) ) return( FALSE );

  // If the first line doesn't begin with an '*' assume first line is the title
  if( os1.GetChar( 0 ) != wxT('*') )
  {
    os1 = wxT("* ") + os1;
    m_oasTitle.Add( os1 );
    return( TRUE );
  }
  else if( ! os1.Contains( wxT("gnetlist") ) )
  { // Take all lines beginning with an '*' or ' ' as the title
    for( size_t sz1=0; sz1<GetCount( ); sz1++ )
    {
      os1 = Item( sz1 );
      if( os1.IsEmpty( ) )               break;
      if( os1.GetChar( 0 ) != wxT('*') ) break;
      if( os1.Contains( APP_NAME ) ) 
      { // Skip gSpiceUI banner
        m_oasTitle.Empty( );
        sz1 += 2;
        continue;
      }
      m_oasTitle.Add( os1 );
    }
  }

  return( TRUE );
}

//*****************************************************************************
// Extract any include directives.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractIncludes( void )
{
  wxString  os1, os2;
  size_t    sz1;

  // Need at least a title line, 2 components and an include directive
  if( GetCount( ) < 4 ) return( TRUE ); // There are no include directives

  // Scan circuit description for include directives
  for( sz1=0; sz1<GetCount( ); sz1++ )
  {
    os1 = Item( sz1 );

    if( bIsSubCkt( os1 ) ) continue;
    if( os1.IsEmpty( )   ) continue;

    os2 = os1;
    os2.MakeUpper( );

    if( os2.StartsWith( wxT(".INCLUDE ") ) ) m_oasIncludes.Add( os1 );
  }

  return( TRUE );
}

//*****************************************************************************
// Extract all the component description lines from the circuit description.
// A component line is identified if the first character in the line is an
// alpha character and its length is greater than 8. The following examples
// shows a component description of minimum length (a resistor connected to
// nodes 1 and 0 of value 9 Ohm):
//
//   R1 1 0 9
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractCpnts( void )
{
  wxString   os1, os2;
  size_t     sz1;
  Component  oCpnt;

  // Need at least a title line and 2 components
  if( GetCount( ) < 3 ) return( FALSE ); // There are insufficient lines

  // Reset sub-circuit detection function
  os1 = wxT(".ENDS");
  bIsSubCkt( os1 );

  // Scan circuit description for components
  for( sz1=1; sz1<GetCount( ); sz1++ )
  {
    // Retrieve the next line from the net list
    os1 = Item( sz1 );

    // Ignore all component definitions inside a sub-circuit block
    if( bIsSubCkt( os1 ) )        continue;

    // Determine if this line is a valid component description
    if( ! oCpnt.bSetDefn( os1 ) ) continue;

    // A component description was found
    os1.Trim( );
    m_oasCpnts.Add( os1 );
  }

  // Circuit description must have components
  if( m_oasCpnts.IsEmpty( ) ) return( FALSE );

  m_oasCpnts.Sort( &iCompare );

  return( TRUE );
}

//*****************************************************************************
// Extract all the model description lines from the circuit description.
// A model description can consist of 1 or more lines and is identified if the
// line begins with ".MODEL ". Subsequent lines beginning with a '+' character
// are appended to the model description. The format is illustrated in the
// following example:
//
//   .MODEL CMOSN NMOS (LEVEL=2 LD=0.265073u TOX=418.0e-10
//   + NSUB=1.53142e+16 VTO=0.844345 KP=4.15964e-05 GAMMA=0.863074
//   + CJ=0.0003844 MJ=0.488400 CJSW=5.272e-10 MJSW=0.300200 PB=0.700000)
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractModels( void )
{
  wxString  osModel;
  wxString  os1, os2;
  size_t    sz1;

  // Need at least a title line, 2 components and a model description line
  if( GetCount( ) < 4 ) return( TRUE ); // There are no model descriptions

  // Reset sub-circuit detection function
  os1 = wxT(".ENDS");
  bIsSubCkt( os1 );

  // Scan circuit description for models
  for( sz1=0; sz1<GetCount( ); sz1++ )
  {
    os1 = Item( sz1 );

    if( bIsSubCkt( os1 ) )                     continue;
    if( os1.IsEmpty( ) && osModel.IsEmpty( ) ) continue;

    os2 = os1;
    os2.MakeUpper( );

    if( os2.StartsWith( wxT(".MODEL ") ) )
    { // First line of model description found
      if( ! osModel.IsEmpty( ) ) m_oasModels.Add( osModel );
      osModel = os1;
    }
    else if( os1.GetChar( 0 ) == wxT('+') )
    { // Intermediate line of model description found
      if( osModel.Length( ) > 1 ) osModel << wxT('\n') << os1;
    }
    else if( ! osModel.IsEmpty( ) )
    { // Last line of model description found
      m_oasModels.Add( osModel );
      osModel.Empty( );
    }
  }

  return( TRUE );
}

//*****************************************************************************
// Extract any sub-circuit descriptions from the circuit description (net list).
// The format of a sub-circuit description is illustrated in the following
// example:
//
//   .SUBCKT CCTNAME 1 5
//   R1 1 2 1K
//   R2 2 3 2K
//   R3 3 4 3K
//   R4 4 5 4K
//   .ENDS CCTNAME
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractSubCcts( void )
{
  wxString  osSubCct;
  wxString  os1, os2;
  size_t    sz1;

  // Need at least a title line, 2 components and a sub-circuit description
  // containing 2 lines
  if( GetCount( ) < 7 ) return( TRUE ); // There are no sub-cct descriptions

  // Scan circuit description for sub-circuits
  for( sz1=0; sz1<GetCount( ); sz1++ )
  {
    os1 = Item( sz1 );

    if( os1.IsEmpty( ) && osSubCct.IsEmpty( ) ) continue;

    os2 = os1;
    os2.MakeUpper( );

    if( os2.StartsWith( wxT(".SUBCKT ") ) )
    { // First line in sub-circuit description found
      osSubCct = os1;
    }
    else if( os2.StartsWith( wxT(".ENDS") ) )
    { // Last line in sub-circuit description found
      if( ! osSubCct.IsEmpty( ) )
      {
        osSubCct << wxT('\n') << os1;
        m_oasSubCcts.Add( osSubCct );
        osSubCct.Empty( );
      }
    }
    else if( ! osSubCct.IsEmpty( ) )
    { // Intermediate line in sub-circuit description found
      osSubCct << wxT('\n') << os1;
    }
  }

  return( TRUE );
}

//*****************************************************************************
// Extract all nodes labels from the component line string array.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractNodeLbls( void )
{
  // Check that there are some components
  if( m_oasCpnts.GetCount( ) <= 0 ) return( FALSE );

  wxStringTokenizer  ostk1;
  wxString           os1;
  size_t             sz1;
  wxChar             cCpnt;

  for( sz1=0; sz1<m_oasCpnts.GetCount( ); sz1++ )
  {
    os1 = m_oasCpnts.Item( sz1 );
    ostk1.SetString( os1 );

    // Need at least 3 fields the first is assumed to be the component label
    if( ostk1.CountTokens( ) < 3 ) continue;

    os1 = ostk1.GetNextToken( );     // Discard component label
    cCpnt = wxToupper( os1.GetChar( 0 ) );

    os1 = ostk1.GetNextToken( );     // First node
    if( m_oasNodeLbls.Index( os1 ) == wxNOT_FOUND ) m_oasNodeLbls.Add( os1 );

    os1 = ostk1.GetNextToken( );     // Second node
    if( m_oasNodeLbls.Index( os1 ) == wxNOT_FOUND ) m_oasNodeLbls.Add( os1 );

    if( cCpnt == wxT('Q') )
    {
      if( ostk1.CountTokens( ) > 3 )
      {
        os1 = ostk1.GetNextToken( ); // Third node
        if( m_oasNodeLbls.Index( os1 ) == wxNOT_FOUND ) m_oasNodeLbls.Add( os1 );
      }
      continue;
    }

    if( cCpnt == wxT('X') )
    {
      for( size_t sz2=3; sz2<ostk1.CountTokens( )-1; sz2++ )
      {
        os1 = ostk1.GetNextToken( ); // Nth node
        if( m_oasNodeLbls.Index( os1 ) == wxNOT_FOUND ) m_oasNodeLbls.Add( os1 );
      }
      continue;
    }
  }

  if( m_oasNodeLbls.IsEmpty( ) ) return( FALSE );

  m_oasNodeLbls.Remove( wxT("0") ); // Remove the earth/ground node ???
  m_oasNodeLbls.Remove( wxT("GND") );

  m_oasNodeLbls.Sort( &iCompare );

  return( TRUE );
}


//*****************************************************************************
// Extract all component labels from the component line string array.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractCpntLbls( void )
{
  // Check that there are some components
  if( m_oasCpnts.GetCount( ) <= 0 ) return( FALSE );

  wxStringTokenizer  ostk1;
  wxString           os1;
  size_t             sz1;

  for( sz1=0; sz1<m_oasCpnts.GetCount( ); sz1++ )
  {
    os1 = m_oasCpnts.Item( sz1 );
    ostk1.SetString( os1 );

    if( ostk1.CountTokens( ) < 1 ) return( FALSE );

    os1 = ostk1.GetNextToken( );

    // Only collect two terminal components
    switch( wxToupper( os1.GetChar(0) ) )
    {
      case 'C': break; // Capacitor
      case 'D': break; // Diode
      case 'I': break; // Current source
      case 'L': break; // Inductor
      case 'R': break; // Resistor
      case 'V': break; // Voltage source
      case 'Y': break; // Admittance
      default: continue;
    }
    m_oasCpntLbls.Add( os1 );
  }

  m_oasCpntLbls.Sort( &iCompare );

  return( TRUE );
}

//*****************************************************************************
// Extract the schematic file names from which the net list was derived.
// When gSpiceUI imports a net list from schematic file/s it places a reference
// to the schematic file/s near the top of the net list.
// Eg. * Schematics : circuit1.sch circuit2.sch
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtractSchems( void )
{
  wxString  os1, os2;
  size_t    sz1;

  // Scan circuit description for schematic file name/s
  for( sz1=0; sz1<GetCount( ); sz1++ )
  {
    os1 = Item( sz1 );

    if( os1.IsEmpty( )   )             continue;
    if( os1.GetChar( 0 ) != wxT('*') ) continue;
    if( os1.StartsWith( wxT("* Schematic : "), &os2 ) ) 
    {
      m_osSchems = os2.Strip( );
      break;
    }
  }

  return( ! os2.IsEmpty( ) );
}

//*****************************************************************************
// Extract the data that has been loaded into the string array this class is
// derived from.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bExtract( void )
{
  bool  bRtn=TRUE;

  if( ! IsEmpty( ) )
  {
    if( ! bExtractTitle   ( ) ) bRtn = FALSE;
    if( ! bExtractIncludes( ) ) bRtn = FALSE;
    if( ! bExtractCpnts   ( ) ) bRtn = FALSE;
    if( ! bExtractModels  ( ) ) bRtn = FALSE;
    if( ! bExtractSubCcts ( ) ) bRtn = FALSE;
    if( ! bExtractNodeLbls( ) ) bRtn = FALSE;
    if( ! bExtractCpntLbls( ) ) bRtn = FALSE;
    if( ! bExtractSchems  ( ) ) ;
  }
  else bRtn = FALSE;

  return( bRtn );
}

//*****************************************************************************
// Makes the object and its attributes empty and frees memory allocated to it.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bClear( void )
{
  wxArrayString::Clear( );

  m_oasTitle   .Clear( );
  m_oasIncludes.Clear( );
  m_oasCpnts   .Clear( );
  m_oasModels  .Clear( );
  m_oasSubCcts .Clear( );

  m_oasNodeLbls.Clear( );
  m_oasCpntLbls.Clear( );

  return( TRUE );
}

//*****************************************************************************
// Makes the object and its attributes empty, but doesn't free memory allocated
// to it. This function should be used when the object is going to be reused
// for storing other data.
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bEmpty( void )
{
  wxArrayString::Empty( );

  m_oasTitle   .Empty( );
  m_oasIncludes.Empty( );
  m_oasCpnts   .Empty( );
  m_oasModels  .Empty( );
  m_oasSubCcts .Empty( );

  m_oasNodeLbls.Empty( );
  m_oasCpntLbls.Empty( );

  return( TRUE );
}

//*****************************************************************************
// Load (or reload) the circuit from file.
//
// Arguments:
//   rosFName - The name of the file to be loaded
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bLoadFile( const wxString & rosFName )
{
  wxString  os1;

  // Is this a load or reload?
  if( ! rosFName.IsEmpty( ) )
       { if( ! bSetLoadFile( rosFName )          )        return( FALSE ); }
  else { if( m_ofnLoad.GetFullPath( ).IsEmpty( ) )        return( FALSE ); }

  bEmpty( );  // Clear the object attributes (without releasing memory)

  // Open the file
  wxTextFile  oFileCct( m_ofnLoad.GetFullPath( ) );
  oFileCct.Open( );
  if( ! oFileCct.IsOpened( )       ) { oFileCct.Close( ); return( FALSE ); }
  if( oFileCct.GetLineCount( ) < 3 ) { oFileCct.Close( ); return( FALSE ); }

  // Load the file contents
  for( os1=oFileCct.GetFirstLine(); !oFileCct.Eof(); os1=oFileCct.GetNextLine() )
    Add( os1 );

  // Need the following line in case the last line in the file is not empty
  // but has no line termination character
  if( os1.Length( ) > 0 ) Add( os1 );

  oFileCct.Close( ); // Close the file

  // Attempt to extract the circuit description from file
  if( ! bExtract( ) )                         { Empty( ); return( FALSE ); }

  return( TRUE );
}

//*****************************************************************************
// Save (or resave) the circuit to file.
//
// Arguments:
//   rosFName - The name of the file to be saved
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bSaveFile( const wxString & rosFName )
{
  wxString  os1;
  wxString  os2;
  size_t    sz1;

  // Is this a save or resave?
  if( ! rosFName.IsEmpty( ) )
       { if( ! bSetSaveFile( rosFName ) ) return( FALSE ); }
  else { if( ! m_ofnSave.IsOk( )        ) return( FALSE ); }

  // Open the file
  wxTextFile  oFileCct( m_ofnSave.GetFullPath( ) );
  if( oFileCct.Exists( ) )
       { if( ! oFileCct.Open( )   )       return( FALSE ); }
  else { if( ! oFileCct.Create( ) )       return( FALSE ); }

  // Clear the file if it contains lines
  for( sz1=oFileCct.GetLineCount( ); sz1>0; sz1-- )
    oFileCct.RemoveLine( 0 );

  // Save the circuit description title line/s
  os1.Empty( );
  os1 << wxT("*****************************************************");
  oFileCct.AddLine( os1 );
  os2.Empty( );
  os2 << wxT("*       SPICE file generated by ") << APP_NAME << wxT("       *");
  oFileCct.AddLine( os2 );
  os2.Empty( );
  os2 << wxT("*         ") << APP_VERSION << wxT("         *");
  oFileCct.AddLine( os2 );
  oFileCct.AddLine( os1 );

  // Save the schematic file name/s if specified
  if( ! m_osSchems.IsEmpty( ) )
  {
    os1.Empty( );
    os1 << wxT("* Schematic : ") << m_osSchems;
    oFileCct.AddLine( wxT("") );
    oFileCct.AddLine( os1 );
    oFileCct.AddLine( wxT("") );
  }

  // Save the title line/s
  if( ! m_oasTitle.IsEmpty( ) )
  {
    for( sz1=0; sz1<m_oasTitle.Count( ); sz1++ )
      oFileCct.AddLine( m_oasTitle[ sz1 ] );
    oFileCct.AddLine( wxT("") );
  }
  
  // Save the include directives
  if( ! m_oasIncludes.IsEmpty( ) )
  {
    oFileCct.AddLine( wxT("* Include Directives") );
    for( sz1=0; sz1<m_oasIncludes.Count( ); sz1++ )
      oFileCct.AddLine( m_oasIncludes[ sz1 ] );
    oFileCct.AddLine( wxT("") );
  }

  // Save the component definition lines
  oFileCct.AddLine( wxT("* Component Definitions") );
  for( sz1=0; sz1<m_oasCpnts.Count( ); sz1++ )
    oFileCct.AddLine( m_oasCpnts[ sz1 ] );
  oFileCct.AddLine( wxT("") );

  // Save the sub-circuit definitions
  if( ! m_oasSubCcts.IsEmpty( ) )
  {
    oFileCct.AddLine( wxT("* Sub-Circuit Definitions") );
    for( sz1=0; sz1<m_oasSubCcts.Count( ); sz1++ )
      oFileCct.AddLine( m_oasSubCcts[ sz1 ] );
    oFileCct.AddLine( wxT("") );
  }

  // Save the model definition lines
  if( ! m_oasModels.IsEmpty( ) )
  {
    oFileCct.AddLine( wxT("* Model Definitions") );
    for( sz1=0; sz1<m_oasModels.Count( ); sz1++ )
      oFileCct.AddLine( m_oasModels[ sz1 ] );
    oFileCct.AddLine( wxT("") );
  }

  // Save the circuit description terminator
  oFileCct.AddLine( wxT(".END") );

  // Save the changes to disk
  oFileCct.Write( );

  oFileCct.Close( );

  return( TRUE );
}

//*****************************************************************************
// Set the load file name.
//
// Arguments:
//   rosFName - The load file name
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bSetLoadFile( const wxString & rosFName )
{
  wxFileName  ofn1;

  ofn1 = rosFName;

  if( m_osSchems.IsEmpty( ) )
  {
    // Is the file name valid and does it exist?
    if( ! ofn1.IsOk( ) )       return( FALSE );
    if( ! ofn1.FileExists( ) ) return( FALSE );
  }

  m_ofnLoad = ofn1;

  return( TRUE );
}

//*****************************************************************************
// Set the save file name.
//
// Arguments:
//   rosFName - The save file name
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bSetSaveFile( const wxString & rosFName )
{
  wxFileName  ofn1;

  ofn1 = rosFName;

  // Is the file name valid?
  if( ! ofn1.IsOk( ) ) return( FALSE );

  m_ofnSave = ofn1;

  return( TRUE );
}

//*****************************************************************************
// Set the schematic file name/s from which the net list was derived.
//
// Arguments:
//   rosFName - The schematic file name/s
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  NetList::bSetSchems( const wxString & rosFNames )
{
  m_osSchems = rosFNames;

  return( TRUE );
}

//*****************************************************************************
// Get a component description associated with a component name.
//
// Arguments:
//   psName - The component name (label)
//
// Return Values:
//   Success - The complete component description
//   Failure - An empty string

const wxString & NetList::rosGetCpnt( const wxChar * psName )
{
  static  wxString  osEmpty;
  wxString  os1, os2;
  size_t  sz1;

  os1 = psName;
  if( os1.IsEmpty( ) )                    return( osEmpty );
  if( roasGetCpnts( ).IsEmpty( ) )        return( osEmpty );

  for( sz1=0; sz1<roasGetCpnts( ).GetCount( ); sz1++ )
  {
    const wxString & ros1 = roasGetCpnts( ).Item( sz1 );
    if( ros1.StartsWith( os1.c_str( ) ) ) return( ros1 );
  }
  
  return( osEmpty );
}

//*****************************************************************************
// Dump attributes contents to standard error.

void  NetList::PrintCpntLines( void )
{
  wxString  os1;
  size_t    sz1;

  std::cerr << "NetList::PrintCpntLines( )\n";

  for( sz1=0; sz1<m_oasCpnts.GetCount( ); sz1++ )
  {
    os1 = m_oasCpnts.Item( sz1 );
    std::cerr << "  " << os1.mb_str( ) << '\n';
  }

  std::cerr << '\n';
}

//*****************************************************************************
