//*****************************************************************************
//                                ConvertType.hpp                             *
//                               -----------------                            *
//  Started     : 21/09/2004                                                  *
//  Last Update : 28/08/2007                                                  *
//  Copyright   : (C) 2004 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "utility/ConvertType.hpp"

//*****************************************************************************
// Constructor.

ConvertType::ConvertType( void )
{
}

//*****************************************************************************
// Destructor.

ConvertType::~ConvertType( )
{
}

//*****************************************************************************
// Parse a float value into it's mantissa and exponent.
//
// Argument List:
//   dfNum - The double float value to be parsed
//   pfMan - A pointer to float   to hold the mantissa (1.000 to 9.999)
//   piExp - A pointer to integer to hold the exponent
//
// Return Values:
//   Success - TRUE
//   Failure - FALSE

bool  ConvertType::bParseFlt( double dfNum, float * pfMan, int * piExp )
{
  if( dfNum<-FLT_MAX || dfNum>FLT_MAX ) return( FALSE );

  if( dfNum != 0.0 )
  {
    *piExp = (int) floor( log10( fabs( dfNum ) ) );
    *pfMan = dfNum / pow( 10.0, *piExp );

    if( fabs(*pfMan) <   1.0 ) { (*pfMan) *= 10.0; (*piExp)--; }
    if( fabs(*pfMan) >= 10.0 ) { (*pfMan) /= 10.0; (*piExp)++; }
  }
  else
  {
    *pfMan = 0.0;
    *piExp = 0;
  }

  return( TRUE );
}

//*****************************************************************************
// Convert a string into a floating point number.
// This function will convert as much of the string as possible into a float
// but return an error if any characters in the string cannot be converted.
//
// Some examples of strings that can be converted to a float:
//   * 12.3
//   * 12.3m  ("m" could alternatively be T, Tera, G, Giga, M, Meg, Mega, K, k,
//                                        u, n, p, f)
//   * 12.3mF ("F" could alternatively be Ohm, H, V, A, Sec and is ignored)
//   * 12.3E-3
//
// Argument List:
//   rosNum - The string to be converted
//   pdfNum - The equivalent double float
//
// Return Values:
//   Success - TRUE
//   Failure - FALSE

bool  ConvertType::bStrToFlt( const wxString & rosNum, double * pdfNum )
{
  wxString  osNum;
  wxString  os1;
  wxChar    oc1;
  size_t    szt1;

  osNum = rosNum.Strip( wxString::both ); // Strip leading/trailing space chars.

  // Do initial argument validity checks
  if( osNum.IsEmpty( ) )         return( FALSE );

  // Extract the first part of the number
  os1.Empty( );
  for( szt1=0; szt1<osNum.Length( ); szt1++ )
  {
    oc1 = osNum.GetChar( szt1 );
    if( !wxIsdigit( oc1 ) && oc1!=wxT('+') && oc1!=wxT('-') &&
        oc1!=wxT('.') && oc1!=wxT('e') && oc1!=wxT('E') ) break;
    os1 << oc1;
  }
  if( ! os1.ToDouble( pdfNum ) ) return( FALSE );

  // Check for a units specifier
  if( szt1 < osNum.length( ) )
  {
    switch( oc1 )
    {
      case wxT('T') : *pdfNum *= 1.0E+12; break;
      case wxT('G') : *pdfNum *= 1.0E+09; break;
      case wxT('M') : *pdfNum *= 1.0E+06; break;
      case wxT('K') :
      case wxT('k') : *pdfNum *= 1.0E+03; break;
      case wxT('m') : *pdfNum *= 1.0E-03; break;
      case wxT('u') : *pdfNum *= 1.0E-06; break;
      case wxT('n') : *pdfNum *= 1.0E-09; break;
      case wxT('p') : *pdfNum *= 1.0E-12; break;
      case wxT('f') : *pdfNum *= 1.0E-15; break;
      default:                   return( FALSE );
    }
  }

  return( TRUE );
}

//*****************************************************************************
// Convert a string into a long integer number.
// This function will convert as much of the string as possible into a long
// integer but return an error if any characters in the string cannot be
// converted.
//
// Some examples of strings that can be converted to a float:
//   * 12
//   * 12k  ("k" could alternatively be T, Tera, G, Giga, M, Meg, Mega, K)
//   * 12kV ("V" could alternatively be Ohm, F, H, A, Sec and is ignored)
//
// Argument List:
//   rosNum - The string value
//   pliNum - The equivalent long integer
//
// Return Values:
//   Success - TRUE
//   Failure - FALSE

bool  ConvertType::bStrToInt( const wxString & rosNum, long * pliNum )
{
  wxString  osNum;
  wxString  os1;
  wxChar    oc1;
  size_t    szt1;

  osNum = rosNum.Strip( wxString::both ); // Strip leading/trailing space chars.

  // Do initial argument validity checks
  if( osNum.IsEmpty( )  )      return( FALSE );

  // Extract the first part of the number
  os1.Empty( );
  for( szt1=0; szt1<osNum.Length( ); szt1++ )
  {
    oc1 = osNum.GetChar( szt1 );
    if( !wxIsdigit( oc1 ) && oc1!=wxT('+') && oc1!=wxT('-') ) break;
    os1 << oc1;
  }
  if( ! os1.ToLong( pliNum ) ) return( FALSE );

  // Check for a units specifier
  if( szt1 < osNum.Length( ) )
  {
    switch( oc1 )
    {
      case wxT('T') : *pliNum *= 1000;
      case wxT('G') : *pliNum *= 1000;
      case wxT('M') : *pliNum *= 1000;
      case wxT('K') :
      case wxT('k') : *pliNum *= 1000; break;
      default:                 return( FALSE );
    }
  }

  return( TRUE );
}

//*****************************************************************************
// Convert a floating point number into a fixed format string.
// The format used by this function is as follows:
//   *  1.234e99
//   * -1.234e99
//   *  1.234e-99
//   * -1.234e-99
//
// Argument List:
//   dfNum  - The double float value
//   rosNum - The string to hold the result
//
// Return Values:
//   Success - TRUE
//   Failure - FALSE

bool  ConvertType::bFltToStr( double dfNum, wxString & rosNum )
{
  if( rosNum.Printf( wxT("%5.3e"), dfNum ) < 0 ) return( FALSE );

  if( rosNum.Length( ) < 10 ) rosNum.Prepend( wxT(" ") );

  if( rosNum.Length( ) != 10 )                   return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Convert a floating point value to a string using engineering format.
// (Eg. convert 100000 to 100k.)
//
// Argument List:
//   dfNum  - The double float value
//   rosNum - The string to hold the result
//
// Return Values:
//   Success - The string containing the engineering format
//   Failure - An empty string

bool  ConvertType::bFltToStrEng( double dfNum, wxString & rosNum )
{
  wxString  osUnits;
  float  fMan; // Mantissa
  int    iExp; // Exponent

  rosNum .Empty( );
  osUnits.Empty( );

  if( ! bParseFlt( dfNum, &fMan, &iExp ) ) return( FALSE );

  switch( iExp )
  {
    case  14: osUnits = wxT("Tera"); fMan *= 100.0; break;
    case  13: osUnits = wxT("Tera"); fMan *=  10.0; break;
    case  12: osUnits = wxT("Tera");                break;
    case  11: osUnits = wxT("Giga"); fMan *= 100.0; break;
    case  10: osUnits = wxT("Giga"); fMan *=  10.0; break;
    case   9: osUnits = wxT("Giga");                break;
    case   8: osUnits = wxT("Meg");  fMan *= 100.0; break;
    case   7: osUnits = wxT("Meg");  fMan *=  10.0; break;
    case   6: osUnits = wxT("Meg");                 break;
    case   5: osUnits = wxT("K");    fMan *= 100.0; break;
    case   4: osUnits = wxT("K");    fMan *=  10.0; break;
    case   3: osUnits = wxT("K");                   break;
    case   2:                        fMan *= 100.0; break;
    case   1:                        fMan *=  10.0; break;
    case   0:                                       break;
    case  -1: osUnits = wxT("m");    fMan *= 100.0; break;
    case  -2: osUnits = wxT("m");    fMan *=  10.0; break;
    case  -3: osUnits = wxT("m");                   break;
    case  -4: osUnits = wxT("u");    fMan *= 100.0; break;
    case  -5: osUnits = wxT("u");    fMan *=  10.0; break;
    case  -6: osUnits = wxT("u");                   break;
    case  -7: osUnits = wxT("n");    fMan *= 100.0; break;
    case  -8: osUnits = wxT("n");    fMan *=  10.0; break;
    case  -9: osUnits = wxT("n");                   break;
    case -10: osUnits = wxT("p");    fMan *= 100.0; break;
    case -11: osUnits = wxT("p");    fMan *=  10.0; break;
    case -12: osUnits = wxT("p");                   break;
    case -13: osUnits = wxT("f");    fMan *= 100.0; break;
    case -14: osUnits = wxT("f");    fMan *=  10.0; break;
    case -15: osUnits = wxT("f");                   break;
    default:                               return( FALSE );
  }

  // Probably need to round fMantissa here instead of truncating it ??? 07/10/2004

  if( rosNum.Printf( wxT("%#.2f%s"), fMan, osUnits.c_str( ) ) < 0 )
    return( FALSE );

  return( TRUE );
}

//*****************************************************************************
