/* gtd_rman.c - Request managment
 *
 * Copyright (C) 1997, 98 Free Software Foundation
 * Copyright (C) 1994, 1995, 1996 Eric M. Ludlam
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 * 
 *  Manages lists of request types.  For convenience, all requests are
 * converted into the most advanced talk message type, and linked
 * together.  The version of the original, is, of course, stored.
 * 
 * History:
 * $Log: gtd_rman.c,v $
 * Revision 1.13  1998/05/01 22:29:25  zappo
 * Made copy routines more robust to arbitrary shaped data.
 *
 * Revision 1.12  1998/05/01 16:42:43  zappo
 * Updated the control reader to handle the new extension format.
 *
 * Revision 1.11  1998/02/15 13:58:58  zappo
 * Fill in and use extended name buffers which prevent buffer overflows.
 *
 * Revision 1.10  1997/12/14 19:19:55  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.9  1997/07/27 20:40:27  zappo
 * Request object now uses the generic list
 *
 * Revision 1.8  1996/06/25 03:02:57  zappo
 * Update matching algorithm which had a slight bug in it.
 *
 * Revision 1.7  1996/03/02  03:33:21  zappo
 * Fixed some warnings
 *
 * Revision 1.6  1995/12/10  03:57:16  zappo
 * error messages now use DISP_message
 *
 * Revision 1.5  1995/09/22  13:50:25  zappo
 * Initialize the new _answer_ field in the requestobject
 *
 * Revision 1.4  1995/03/25  04:24:10  zappo
 * Updated copyright
 *
 * Revision 1.3  1995/03/04  14:48:26  zappo
 * Added use of syslog to report errors when daemon not run on tty
 *
 * Revision 1.2  1995/03/03  02:51:43  zappo
 * Extra extend parameter was added to alloc object.
 *
 * Revision 1.1  1995/02/01  03:50:49  zappo
 * Initial revision
 *
 * zappo   9/17/94         Created
 *
 * Tokens: ::Header:: gtalkd.h
 */
#include "gtalklib.h"
#include "gtalkd.h"

static MakeList(list);

/* give every control a unique id number when being stowed away
 * for types LEAVE_INVITE and ANNOUNCE
 */
static long my_ids = 0;


/*
 * Function: remove_control_characters
 *
 *   Removes non-printable characters from STR by replacing them with .
 * We do this destructively since the strings we are modifying are
 * should be non-critical.
 *
 * Returns:     Nothing
 * Parameters:  str - String to search for non-printable characters
 *
 * History:
 * zappo   2/20/97    Created
 */
static void remove_control_characters(str, length)
     char *str;
     int length;
{
  int count = 0;

  while(*str && (count < length))
    {
      if(*str < ' ') *str = '.';
      str++;
      count++;
    }
}


/*
 * Function: GCM_alloc
 *
 *   Allocates and initializes a new request object, and links it to
 * the local list of such objects.  It then returns the object.
 *
 * Returns:     struct RequestObject * - the new message
 * Parameters:  Ctxt   - Context
 *              dev    - Pointer to device
 *              source - Pointer to source request
 *              extend - Pointer to extensions
 * History:
 * zappo   9/17/94    Created
 * zappo   3/2/95     Added extend perameter
 */
struct RequestObject *GCM_alloc(Ctxt, dev, source, extend)
     struct DaemonContext *Ctxt;
     struct InputDevice   *dev;
     union ctl_msg        *source;
     char                 *extend;
{
  struct RequestObject *new;
  struct timezone       timezone;

  new = (struct RequestObject *)LIST_alloc(&list, 
					   sizeof(struct RequestObject));

  if(!new)
    {
      DISP_message(NULL, "GCM_malloc: malloc error in GCM_alloc!", LOG_ERR);
      return NULL;
    }

  /* stamp it for deletion later
   */
  gettimeofday(&new->stamp, &timezone);

  /* grab who sent it just in case there is some structureal error.
   */
  new->sender = UDP_byaddr(dev, &dev->lraddr);

  switch(Ctxt->type)
    {
    case OTALKD:
      new->request.vers = 0;
      new->request.type = source->otalk.type;
      new->request.answer = 0;
      new->request.extended = 0;
      new->request.id_num = source->otalk.id_num;
      new->request.addr = source->otalk.addr;
      new->request.ctl_addr = source->otalk.ctl_addr;
      new->retto = UDP_byaddr(dev, (struct sockaddr_in *)&
			      source->otalk.ctl_addr);
      strncpy(new->request.l_name, source->otalk.l_name, ONAME_SIZE);
      new->request.l_name[ONAME_SIZE] = 0; /* This is safe because 
					    * ONAME_SIZE < GNAME_SIZE
					    */
      strncpy(new->request.r_name, source->otalk.r_name, ONAME_SIZE);
      new->request.r_name[ONAME_SIZE] = 0;
      strncpy(new->request.r_tty, source->otalk.r_tty, OTTY_SIZE);
      /* Cannot terminate TTY.  May be the same size */
      break;
    case NTALKD:
    case GTALKD:
      new->request = source->gtalk;
      new->retto = UDP_byaddr(dev, (struct sockaddr_in *)&
			      source->gtalk.ctl_addr);
      break;
    default:
      if(verbose)
	fprintf(stderr, "Error in request type %d...", Ctxt->type);
    }

  /* Copy the names into our NULL terminated safe buffers
   * so we can use them with NULL knowledgable string functions.
   */
  strncpy(new->l_name, new->request.l_name, GNAME_SIZE);
  new->l_name[GNAME_SIZE] = 0;
  strncpy(new->r_name, new->request.r_name, GNAME_SIZE);
  new->r_name[GNAME_SIZE] = 0;
  strncpy(new->r_tty, new->request.r_tty, GTTY_SIZE);
  new->r_tty[GTTY_SIZE] = 0;

  /* give these new id's which will be used when sending back the
   * response to these messages.
   */
  if((new->request.type == LEAVE_INVITE) ||
     (new->request.type == ANNOUNCE))
    new->request.id_num = ++my_ids;

  /* Now stuff info about the extension characters into the object. */
  if(new->request.extended != 0)
    {
      new->extend = malloc(new->request.extended + 1);
      if(!new->extend)
	{
	  DISP_message(NULL, "GCM_malloc: malloc error in GCM_alloc!", LOG_ERR);
	  exit(0);
	}
      memcpy(new->extend, extend, new->request.extended);
    }
  else
    new->extend = NULL;

  /* Remove bomb-characters if applicable */
  remove_control_characters(new->l_name, NAME_SIZE);
  /* Not only do we remove bomb characters here, but add NULLs to our
   * strings if applicable */
  if(new->extend)
    {
      char *data;
      int   length;
      
      data = (char *)EXT_fetch_extension_data(EXTENSION_APPLICATION_NAME,
					      new->extend,
					      &length);
      if(data) {
	*(data + length - 1) = 0;
	remove_control_characters(data, length-1);
      }
      
      data = (char *)EXT_fetch_extension_data(EXTENSION_PERSONAL_NAME,
					      new->extend,
					      &length);
      if(data) {
	*(data + length - 1) = 0;
	remove_control_characters(data, length-1);
      }
    }
      
  new->answer = -1;		/* non-answer */

  return new;
}

/*
 * Function: GCM_free_request
 *
 *   Frees up space and removes request from the local list.
 *
 * Returns:     Nothing
 * Parameters:  request -  request to be freed
 *
 * History:
 * zappo   9/17/94         Created
 */
void GCM_free_request(request)
     struct RequestObject *request;
{
  if(verbose) {
    printf("Freeing request type %s for [%s] from [%s]...\n",
	   msg_types[request->request.type], 
	   request->r_name, request->l_name);
  }

  if(request->extend != NULL)
    {
      free(request->extend);
    }

  LIST_deallocate(&list, request);
}


/*
 * Function: GCM_timeout_messages
 *
 *   This functions is called periodically to see if any functions
 * need to be deleted due to age.  When there are no messages left,
 * exit when the daemon flag is set to do so.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *              dev  - Pointer to device
 * History:
 * zappo   9/18/94    Created
 */
void GCM_timeout_messages(Ctxt, dev)
     struct DaemonContext *Ctxt;
     struct InputDevice *dev;
{
  struct timeval        timev;     
  struct timezone       timezone;
  struct RequestObject *tmp = FirstElement(list);
  int                   numq = 0, numt = 0;

  /* stamp it for deletion later
   */
  gettimeofday(&timev, &timezone);

  while(tmp)
    {
      if((timev.tv_sec - tmp->stamp.tv_sec) > MAX_LIFE)
	{
	  struct RequestObject *t;

	  t = tmp;
	  tmp = NextElement(tmp);
	  if(verbose)
	    printf("timeout: ");

	  GCM_free_request(t);
	  numt++;
	}
      else
	{
	  tmp = NextElement(tmp);
	  numq++;
	}
    }
  if(!list.first)
    {
      /* When there is nothing left, exit completly unless the forever
       * flag is set
       */
      if(Ctxt->forever == FALSE)
	exit(0);

      if(verbose)
	printf("timeout: Nothing to do...\n");
    }
  else
    {
      /* Reset the checkout rate to make sure that things in the Q do
       * eventually get disposed of 
       */
      dev->timeout = Ctxt->checkrate;
    }
  if(verbose)
    {
      printf("Processed %d requests in Q with %d timeouts.\n", numq, numt);
    }
}


/*
 * Function: GCM_lookfor
 *
 *   Looks for a "LOOKUP" match in an "INVITE" message from RO.  The
 * r_name, and l_name must match in both directions.
 *
 * Returns:     struct RequestObject * - found object.
 * Parameters:  ro - Pointer to request object to look up.
 *
 * History:
 * zappo   9/17/94         Created
 */
static unsigned char MatchRequest(ro, criteria)
     struct RequestObject *ro;
     struct RequestObject *criteria;
{
  return ((ro->request.type == LEAVE_INVITE) &&
	  !strcmp(ro->r_name, criteria->l_name) &&
	  !strcmp(ro->l_name, criteria->r_name));
}

struct RequestObject *GCM_lookfor(ro)
     struct RequestObject *ro;
{
  struct RequestObject *tmp;

  if(verbose)
    printf("Looking for invite match to  %s for %s\n",
	   msg_types[ro->request.type], ro->request.r_name);

  tmp = (struct RequestObject *)LIST_find(&list, MatchRequest, ro);

  if(verbose) printf("Returning pointer %p\n", tmp);

  return tmp;
}


/*
 * Function: GCM_looksame
 *
 *   Looks for an identical object just like RO in our list.  (if it
 * has the same ptr as RO, then ignore, however.)
 *
 * Returns:     struct RequestObject * - the found objct
 * Parameters:  ro - Pointer to request object
 *
 * History:
 * zappo   9/17/94    Created
 */
static unsigned char MatchIdenticalRequest(ro, criteria)
     struct RequestObject *ro;
     struct RequestObject *criteria;
{
  return ((ro != criteria)                                      && 
	  (ro->request.type == criteria->request.type)          &&
	  !strcmp(ro->r_tty, criteria->r_tty)   &&
	  !strcmp(ro->r_name, criteria->r_name) &&
	  !strcmp(ro->l_name, criteria->l_name));
}
struct RequestObject *GCM_looksame(ro)
     struct RequestObject *ro;
{
  struct RequestObject *tmp;

  if(verbose)
    printf("Looking for same match to %s for %s\n",
	   msg_types[ro->request.type], ro->r_name);

  tmp = (struct RequestObject *)LIST_find(&list, MatchIdenticalRequest, ro);

  if(verbose) printf("Returning pointer %p\n", tmp);

  return tmp;
}


/*
 * Function: GCM_lookid
 *
 *   locates a saved request by the id number (for deletes and such).
 *
 * Returns:     struct RequestObject * - the request
 * Parameters:  ro - Pointer to request object
 *
 * History:
 * zappo   9/17/94    Created
 */
static unsigned char MatchId(ro, criteria)
     struct RequestObject *ro;
     struct RequestObject *criteria;
{
  return ((ro != criteria) && 
	  (ro->request.id_num == criteria->request.id_num));
}
     
struct RequestObject *GCM_lookid(ro)
     struct RequestObject *ro;
{
  struct RequestObject *tmp;

  if(verbose)
    printf("Looking for id match to %s for %s id %ld\nChecking ...",
	   msg_types[ro->request.type], ro->request.r_name,
	   ro->request.id_num);

  tmp = (struct RequestObject *)LIST_find(&list, MatchId, ro);

  if(verbose) printf("Returning pointer %p\n", tmp);

  return tmp;
}

/*
 * Function: GCM_findannounce
 *
 *   Finds a Request object created by an announcing process.  This is
 * used for looking up addresses of people who announced messages to
 * people.
 *
 * Returns:     struct RequestObject * - the found object
 * Parameters:  ro - Pointer to the request object to compare against
 *
 * History:
 * zappo   11/16/94   Created
 */
static unsigned char MatchAnnounce(ro, criteria)
     struct RequestObject *ro;
     struct RequestObject *criteria;
{
  return ((ro->request.type == ANNOUNCE) &&
	  !strcmp(ro->request.r_name, criteria->request.l_name));
}
struct RequestObject *GCM_findannounce(ro)
     struct RequestObject *ro;
{
  struct RequestObject *tmp;

  if(verbose)
    printf("Looking for announce match to %s for %s\n",
	   /* Here we display local name from QUERY request */
	   msg_types[ro->request.type], ro->request.l_name);

  tmp = (struct RequestObject *)LIST_find(&list, MatchAnnounce, ro);

  if(verbose) printf("Find user returning pointer %p\n", tmp);

  return tmp;
}


/*
 * Function: GCM_print
 *
 *   Prints out the list of all requests currently stored.
 *
 * Returns:     Nothing
 * Parameters:  None
 *
 * History:
 * zappo   9/17/94         Created
 */
void GCM_print()
{
  printf("Not supported.\n");
}






