/*  GTKtalog.
 *  Copyright (C) 1999-2000  Mathieu VILLEGAS & Yves METTIER
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#ifdef USE_GNOMEVFS
#include <libgnomevfs/gnome-vfs-mime.h>
#endif
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <unistd.h>
#include "folder.h"

GPtrArray *new_descriptions;

FILE_DATA *
get_file_data_from_gnode (GNode * gn)
{
  FILE_DATA *fd;
  FOLDER *racine;

  if (G_NODE_IS_ROOT (gn))
    {
      racine = gn->data;
      fd = g_ptr_array_index (racine->datas, 0);
    }
  else
    {
      fd = gn->data;
    }
  return (fd);
}

FOLDER *
get_catalog_from_file_data (FILE_DATA * fd)
{
  GNode *root_gn;

  root_gn = g_node_get_root (fd->node);

  return (root_gn->data);
}

FOLDER *
get_catalog_from_gnode (GNode * gn)
{
  GNode *root_gn;
  root_gn = g_node_get_root (gn);
  return (root_gn->data);
}

void
catalog_set_currently_displayed_gnode (GNode * gn)
{
  FOLDER *f;
  f = get_catalog_from_gnode (gn);
  f->currently_displayed_gnode = gn;
}

const gchar *
get_disk_from_node (GNode * gn)
{
  FILE_DATA *fd;
  GNode *tmp = gn;

  while (g_node_depth (tmp) > 2)
    tmp = tmp->parent;
  fd = get_file_data_from_gnode (tmp);

  return (fd->name->str);
}

GString *
get_real_path_from_node (GNode * gn)
{
  FILE_DATA *fd;
  GNode *tmp = gn;
  GString *rp;

  rp = g_string_new ("");
  while (g_node_depth (tmp) >= 3)
    {
      fd = get_file_data_from_gnode (tmp);
      rp = g_string_prepend (rp, fd->name->str);
      rp = g_string_prepend_c (rp, '/');
      tmp = tmp->parent;
    }
  fd = get_file_data_from_gnode (tmp);
  if (fd->type == IS_CD)
    {
      rp = g_string_prepend (rp, my_config->mount_point->str);
    }
  else
    {
      fd = get_file_data_from_gnode (tmp);
      rp = g_string_prepend (rp, fd->information->str);
    }
  rp = g_string_prepend_c (rp, '/');

  return (rp);
}

GString *
get_path_from_node (GNode * gn)
{
  FILE_DATA *fd;
  GNode *tmp = gn;
  GString *gs = g_string_new ("/");
  while (g_node_depth (tmp) > 3)
    {
      tmp = tmp->parent;
      fd = get_file_data_from_gnode (tmp);
      gs = g_string_prepend (gs, fd->name->str);
      gs = g_string_prepend_c (gs, '/');
    }
  return (gs);
}

FILE_DATA *
folder_make_data (char *name, char *information, guint type, gchar * mime,
		  guint32 taille, time_t date,
		  guint32 categorie, guint32 description)
{
  FILE_DATA *data;
  data = (FILE_DATA *) g_malloc (sizeof (FILE_DATA));

  data->name = g_string_new (name);
  data->taille = taille;
  data->type = type;
  data->mime = g_string_new (mime);
  data->date = date;
  data->categorie = categorie;
  data->description = description;
  data->a_parent_node_is_vfs = type;

  if ((information != NULL) && (strlen (information) != 0))
    data->information = g_string_new (information);
  else
    data->information = NULL;

#if defined(__DEBUG__)
  GML_malloc_forget (data);
  GML_malloc_forget (data->name);
  GML_malloc_forget (data->mime);
  if (data->information)
    GML_malloc_forget (data->information);
#endif
  return (data);

}

/* Add a file to the FOLDER structure */
GNode *
folder_add (FOLDER * fd, GNode * parent, char *name,
	    char *information, guint type, gchar * mime, guint32 taille,
	    time_t date, guint32 categorie, guint32 description)
{

  FILE_DATA *data;
  GNode *gn;

  data =
    folder_make_data (name, information, type, mime, taille, date,
		      categorie, description);

  g_ptr_array_add (fd->datas, data);
  data->id = fd->datas->len;
  gn = g_node_new (data);
  data->node = gn;
  data->ctree_node = NULL;
  g_node_insert (parent, -1, gn);


#if defined(__DEBUG__)
  GML_malloc_forget (data);
#endif
  return (gn);
}

GNode *
folder_add_link (FOLDER * fd, GNode * parent,	/* parent that contains the link */
		 char *linkname,
		 char *destname,
		 char *information, guint32 taille,
		 time_t date, guint32 categorie, guint32 description)
{
  GNode *gn;
  gn =
    folder_add (fd, parent, linkname, information, IS_LINK, NULL,
		0, date, categorie, description);
  folder_add (fd, gn, destname, NULL, 0, NULL, 0, date, categorie,
	      description);
  return (gn);
}

void
change_name (GNode * gn, char *name)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  g_string_free (fd->name, TRUE);
  fd->name = g_string_new (name);
#if defined(__DEBUG__)
  if (fd->name)
    GML_malloc_forget (fd->name);
#endif
}

void
folder_check_and_set_unique_filename (FOLDER * racine, FILE_DATA * fd)
{
  GNode *gn = fd->node;
  GNode *pgn = g_node_first_sibling (gn);
  GString *name = fd->name;
  int i = 1;
  FILE_DATA *pfd;

  while (pgn)
    {
      pfd = pgn->data;
      if (pfd != fd)
	{
	  if (!strcmp (pfd->name->str, name->str))
	    {
	      if ((is_link (pfd->node)) && (!is_link (gn)))
		{
		  folder_check_and_set_unique_filename (racine, pfd);
		  name = fd->name;
		}
	      else
		{

		  if (name == fd->name)
		    name = g_string_new (fd->name->str);
		  i++;
		  g_string_sprintf (name, "%s (%d)", fd->name->str, i);
		}
	      pgn = g_node_first_sibling (gn);
	    }
	}
      pgn = g_node_next_sibling (pgn);
    }
  if (i > 1)
    {
      g_string_free (fd->name, TRUE);
      fd->name = name;
    }
}

gboolean
is_disk_name_unique (FOLDER * racine, gchar * temp)
{
  GNode *gn = racine->tree;
  GNode *pgn = g_node_first_child (gn);
  FILE_DATA *pfd;

  while (pgn)
    {
      pfd = pgn->data;
      if (!strcmp (pfd->name->str, temp))
	{
	  return (FALSE);
	}

      pgn = g_node_next_sibling (pgn);
    }
  return (TRUE);
}

void
change_description (GNode * gn, guint i)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  fd->description = i;
}

void
change_category (GNode * gn, guint i)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  fd->categorie = i;
}

void
change_information (GNode * gn, char *information)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->information)
    g_string_free (fd->information, TRUE);
  if ((information != NULL) && (strlen (information) != 0))
    fd->information = g_string_new (information);
  else
    fd->information = NULL;
#if defined(__DEBUG__)
  if (fd->information)
    GML_malloc_forget (fd->information);
#endif
}

void
change_type (GNode * gn, guint type)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  fd->type = type;
}

/* traitement des infos sur la date, la taille et le type. */

const char *
folder_get_name (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  return (fd->name->str);
}

const char *
folder_get_mime (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->mime)
    return (fd->mime->str);
#ifdef USE_GNOMEVFS
  return (gnome_vfs_mime_type_from_name_or_default (fd->name->str, NULL));
#else
  return (gnome_mime_type_or_default (fd->name->str, NULL));
#endif
}

const char *
folder_get_informations (GNode * gn)
{
  FILE_DATA *fd;
  static gchar *cd_string = N_("Cdrom");

  if (is_cd (gn))
    return (cd_string);

  fd = get_file_data_from_gnode (gn);
  if (fd->information)
    return (fd->information->str);
  else
    return (NULL);
}

const char *
folder_get_information_from_file_data (FILE_DATA * fd)
{
  if (fd->information)
    return (fd->information->str);
  else
    return (NULL);
}

const char *
folder_get_category_from_file_data (FILE_DATA * fd)
{
  FOLDER *racine;

  racine = get_catalog_from_file_data (fd);
  return (get_categorie ((FOLDER *) racine, fd->categorie));
}

const char *
folder_get_description_from_file_data (FILE_DATA * fd)
{
  FOLDER *racine;

  racine = get_catalog_from_file_data (fd);
  return (get_description ((FOLDER *) racine, fd->description));
}

char *
get_time (time_t nb_sec, gint type, char *pattern)
{
  struct tm *time;
  time_t nb_sec2;
  char *mydate;
  if (nb_sec)
    {
      if ((pattern) && (type == DATE_FORMAT_IS_USERDEFINED))
	{
	  time = localtime (&nb_sec);
	  mydate = (char *) g_malloc (sizeof (char) * 256);
	  if (!strftime (mydate, 255, pattern, time))
	    {
	      g_free (mydate);
	      memcpy (&nb_sec2, &nb_sec, sizeof (time_t));
	      mydate = g_strdup (ctime (&nb_sec2));
	    }
	}
      else if (type == DATE_FORMAT_IS_LOCALE)
	{
	  memcpy (&nb_sec2, &nb_sec, sizeof (time_t));
	  mydate = g_strdup (ctime (&nb_sec2));
	}
      else if (type == DATE_FORMAT_IS_ISO8601)
	{
	  time = localtime (&nb_sec);
	  mydate = (char *) g_malloc (sizeof (char) * 256);
	  if (!strftime (mydate, 255, "%G-%m-%d %H:%M:%S", time))
	    {
	      g_free (mydate);
	      memcpy (&nb_sec2, &nb_sec, sizeof (time_t));
	      mydate = g_strdup (ctime (&nb_sec2));
	    }
	}
      else
	{
	  memcpy (&nb_sec2, &nb_sec, sizeof (time_t));
	  mydate = g_strdup (ctime (&nb_sec2));
	}
    }
  else
    {
      mydate = g_strdup ("-");
    }
  return (mydate);
}

static char *
get_size_unit (guint32 nb_octs)
{
  if (my_config->size_format == SIZE_FORMAT_GB ||
      (((my_config->size_format == SIZE_FORMAT_HUMAN) ||
	(my_config->size_format == SIZE_FORMAT_B_HUMAN)) && (nb_octs >> 30)))
    return g_strdup_printf ("%.2f %s", (gfloat) nb_octs / (1 << 30), _("Gb"));

  if (my_config->size_format == SIZE_FORMAT_MB ||
      ((my_config->size_format == SIZE_FORMAT_HUMAN ||
	(my_config->size_format == SIZE_FORMAT_B_HUMAN)) && (nb_octs >> 20)))
    return g_strdup_printf ("%.2f %s", (gfloat) nb_octs / (1 << 20), _("Mb"));

  if ((my_config->size_format == SIZE_FORMAT_KB) ||
      (((my_config->size_format == SIZE_FORMAT_HUMAN) ||
	my_config->size_format == SIZE_FORMAT_B_HUMAN) && (nb_octs >> 10)))
    return g_strdup_printf ("%.2f %s", (gfloat) nb_octs / (1 << 10), _("Kb"));

  else
    return g_strdup_printf ("%u", nb_octs);
}

char *
get_size (guint32 nb_octs)
{
  gchar *unit;
  gchar *ret;

  unit = get_size_unit (nb_octs);
  if ((my_config->size_format == SIZE_FORMAT_B_HUMAN) && (nb_octs >> 10))
    {
      ret = g_strdup_printf ("%u (%s)", nb_octs, unit);
      g_free (unit);
    }
  else
    ret = unit;

  return ret;
}

char *
get_size_mb (guint32 nb_mocts)
{
  if ((my_config->size_format == SIZE_FORMAT_B_HUMAN) && (nb_mocts >> 10))
    return g_strdup_printf ("%u %s (%.2f %s)", nb_mocts, _("Mb"),
			    (gfloat) nb_mocts / (1 << 10), _("Gb"));

  else if (((my_config->size_format == SIZE_FORMAT_HUMAN) && (nb_mocts >> 10))
	   || my_config->size_format == SIZE_FORMAT_GB)
    return g_strdup_printf ("%.2f %s", (gfloat) nb_mocts / (1 << 10),
			    _("Gb"));
  else
    return g_strdup_printf ("%u %s", nb_mocts, _("Mb"));
}

int
get_type (guint type_info)
{
  //int temp;

  // temp = type_info>>30;
  // return(temp);
  return (type_info);
}


const char *
get_categorie (FOLDER * racine, guint16 i)
{
  if ((i <= racine->categories->len) && (i > 0))
    {
      return (((GString *)
	       g_ptr_array_index (racine->categories, i - 1))->str);
    }
  else
    {
      return (NULL);
    }
}

gint
category_id (FOLDER * racine, GString * gs)
{
  gint i;
  gint id;
  i = 0;
  id = -1;
  while ((i < racine->categories->len) && (id == -1))
    {
      if (strcmp ((((GString *)
		    g_ptr_array_index (racine->categories, i))->str),
		  gs->str) == 0)
	id = i + 1;
      i++;
    }
  return (id);
}

const char *
get_description (FOLDER * racine, guint16 i)
{
  if ((i <= racine->descriptions->len) && (i > 0))
    {
      return (((GString *)
	       g_ptr_array_index (racine->descriptions, i - 1))->str);
    }
  else
    {
      return (NULL);
    }
}

gboolean
is_unreadable (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->type == IS_UNREADABLE)
    return (TRUE);
  return (FALSE);
}

gboolean
is_file (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->type == IS_FILE)
    return (TRUE);
  return (FALSE);
}

gboolean
is_dir (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->type == IS_DIR)
    return (TRUE);
  return (FALSE);
}

gboolean
is_link (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->type == IS_LINK)
    return (TRUE);
  return (FALSE);
}



gboolean
is_disk (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if ((fd->type == IS_DISK) || (fd->type == IS_CD))
    return (TRUE);
  return (FALSE);
}

gboolean
is_cd (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->type == IS_CD)
    return (TRUE);
  return (FALSE);
}

gboolean
is_a_parent_cd (GNode * gn)
{
  FILE_DATA *fd;

  GNode *gnp = gn;
  while (g_node_depth (gnp) > 1)
    gnp = gnp->parent;
  fd = get_file_data_from_gnode (gnp);
  if (fd->type == IS_CD)
    return (TRUE);
  return (FALSE);
}

gboolean
is_vfs (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->type == IS_VFS)
    return (TRUE);
  return (FALSE);
}

gboolean
is_vfs_extended (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->type == IS_VFS_EXTENDED)
    return (TRUE);
  return (FALSE);
}

gboolean
is_vfs_any (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if ((fd->type == IS_VFS) || (fd->type == IS_VFS_EXTENDED))
    return (TRUE);
  return (FALSE);
}

gboolean
is_in_vfs_any (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if ((fd->a_parent_node_is_vfs == IS_VFS)
      || (fd->a_parent_node_is_vfs == IS_VFS_EXTENDED))
    return (TRUE);
  return (FALSE);
}

gboolean
is_link_dest (GNode * gn)
{
  FILE_DATA *fd;

  fd = get_file_data_from_gnode (gn);
  if (fd->type == IS_LINK_DEST)
    return (TRUE);
  return (FALSE);
}

/* Find a FOLDER structure f with (f->name == name).
 * already_found sould be NULL
 * recursive is a flag to make a recursive search or only a one-level search
 */
GArray *
folder_search_name_in_array (GNode * gn, gchar * name)
{
  GPtrArray *a;
  FILE_DATA *fd;
  guint i;
  GArray *already_found;
  FOLDER *folder;

  folder = get_catalog_from_gnode (gn);

  already_found = g_array_new (FALSE, FALSE, sizeof (guint));

  a = folder->datas;
  for (i = 0; i < a->len; i++)
    {
      fd = g_ptr_array_index (a, i);
      if (strcmp ((fd->name)->str, name) == 0)
	{
	  already_found = g_array_append_val (already_found, i);
	}
    }

  return (already_found);
}


gboolean
_gnode_clear_data (GNode * gn, gpointer data)
{
  FOLDER *f = data;
  FILE_DATA *fd;

  if (G_NODE_IS_ROOT (gn) != TRUE)
    {
      fd = get_file_data_from_gnode (gn);
      if (fd->name)
	g_string_free (fd->name, TRUE);
      if (fd->mime)
	g_string_free (fd->mime, TRUE);
      if (fd->information)
	g_string_free (fd->information, TRUE);

/* warning: the FILE_DATA entry in the GPtrArray is just set to NULL. It is not suppressed here. */
      g_ptr_array_index (f->datas, fd->id) = NULL;

      g_free (fd);
    }
  return (FALSE);
}

gboolean
_gnode_clear_ctree_data (GNode * gn, gpointer data)
{
  FOLDER *racine = data;
  FILE_DATA *fd;

  if (((is_dir (gn) == TRUE) || (is_disk (gn) == TRUE)
       || (is_vfs_any (gn) == TRUE)) && (G_NODE_IS_ROOT (gn) != TRUE))
    {
      fd = get_file_data_from_gnode (gn);
      if (fd->ctree_node)
	{
	  gtk_ctree_node_set_row_data (GTK_CTREE (racine->ctree),
				       fd->ctree_node, NULL);
	}
    }
  return (FALSE);
}

void
suppress_dir (GNode * dir, gboolean suppress_gui)
{
  FOLDER *racine;
  GPtrArray *gpa;
  FILE_DATA *fd;
  gint i;

  racine = get_catalog_from_gnode (dir);
  gpa = racine->datas;

  for (i = 0; i < gpa->len; i++)
    {
      fd = g_ptr_array_index (gpa, i);
      fd->id = i;
    }


  fd = dir->data;

  if (suppress_gui)
    {
      if ((is_dir (dir) == TRUE) || (is_disk (dir) == TRUE)
	  || (is_vfs_any (dir) == TRUE) || (G_NODE_IS_ROOT (dir) == TRUE))
	{
	  g_node_traverse (dir, G_PRE_ORDER, G_TRAVERSE_ALL, -1,
			   _gnode_clear_ctree_data, racine);
	  if (!G_NODE_IS_ROOT (dir))
	    gtk_ctree_remove_node (GTK_CTREE (racine->ctree), fd->ctree_node);
	}
    }
  g_node_traverse (dir, G_PRE_ORDER, G_TRAVERSE_ALL, -1, _gnode_clear_data,
		   racine);

  if ((G_NODE_IS_ROOT (dir) == TRUE))
    {
      fd = get_file_data_from_gnode (dir);
      if (fd)
	{
	  if (fd->name)
	    g_string_free (fd->name, TRUE);
	  if (fd->information)
	    g_string_free (fd->information, TRUE);

	  g_ptr_array_index (gpa, 0) = NULL;
	  g_free (fd);
	}
      dir->data = NULL;
    }
  else
    {
      g_node_unlink (dir);
      g_node_destroy (dir);
    }

  /* Garbage collector for the FOLDER data structure */
  while (g_ptr_array_remove_fast (gpa, NULL));
  for (i = 0; i < gpa->len; i++)
    {
      fd = g_ptr_array_index (gpa, i);
      fd->id = i;
    }

  if ((G_NODE_IS_ROOT (dir) == TRUE))
    {
      g_node_destroy (dir);

    }
  else
    {
      racine->tree->data = racine;
    }
  racine->selected_folder = NULL;
  racine->is_modified = TRUE;
}

typedef struct
{
  GList *list;
  FOLDER *f;
}
_DISK_LIST;

void
_make_disk_list (GNode * gn, gpointer data)
{
  _DISK_LIST *disk_list = data;

  FILE_DATA *fd;

  fd = gn->data;

  disk_list->list = g_list_append (disk_list->list, fd->name->str);
  return;
}

gint
sort_filenames_with_numbers (gchar * text1, gchar * text2)
{
  char *p1, *p2;
  gint n1, n2;
  gint result;

  if (!text2)
    return (text1 != NULL);

  if (!text1)
    return -1;

  if (my_config->strict_alphabetic_sort)
    return (strcmp (text1, text2));

  result = 0;
  p1 = text1;
  p2 = text2;
  while ((p1[0] != 0) && (p2[0] != 0) && (result == 0))
    {
      if ((p1[0] >= '0') && (p1[0] <= '9') && (p2[0] >= '0')
	  && (p2[0] <= '9'))
	{
	  n1 = strtol (p1, NULL, 10);
	  n2 = strtol (p2, NULL, 10);
	  if (n1 > n2)
	    result = 1;
	  if (n1 < n2)
	    result = -1;
	  while ((p1[0] >= '0') && (p1[0] <= '9'))
	    p1++;
	  while ((p2[0] >= '0') && (p2[0] <= '9'))
	    p2++;
	}
      else
	{
	  result = strncmp (p1, p2, 1);
	  p1++;
	  p2++;
	}
    }
  if (result == 0)
    {

      if (p1[0] && (!p2[0]))
	result = 1;
      if (p2[0] && (!p1[0]))
	result = -1;
    }

  return (result);

}

gint
_sort_disk_list (gconstpointer a, gconstpointer b)
{

  return (sort_filenames_with_numbers ((gchar *) a, (gchar *) b));

}


GList *
make_disk_list (FOLDER * racine)
{
  static _DISK_LIST disk_list;
  disk_list.list = NULL;
  disk_list.f = racine;

  g_node_children_foreach (racine->tree, G_TRAVERSE_ALL,
			   _make_disk_list, &disk_list);
  disk_list.list = g_list_sort (disk_list.list, _sort_disk_list);
  return (disk_list.list);
}

gint
_cmp_intptrs (gconstpointer a, gconstpointer b)
{
  if (GPOINTER_TO_UINT (a) < GPOINTER_TO_UINT (b))
    return (-1);
  if (GPOINTER_TO_UINT (a) > GPOINTER_TO_UINT (b))
    return (+1);
  return (0);
}

gboolean
_test_description (GNode * gn, gpointer data)
{
  FILE_DATA *fd;
  guint u = 0;
  gpointer gp;

  if (G_NODE_IS_ROOT (gn) == TRUE)
    return (FALSE);
  fd = get_file_data_from_gnode (gn);

  u = fd->description;
  if (u > 0)
    {
      gp = g_ptr_array_index (my_config->racine->descriptions, u - 1);
      if (gp)
	{
	  if (g_tree_lookup ((GTree *) data, GUINT_TO_POINTER (u)) == NULL)
	    {
	      g_ptr_array_add (new_descriptions, (gp));
	      g_assert (gp != NULL);
	      g_ptr_array_index (my_config->racine->descriptions, u - 1) =
		NULL;
	      g_tree_insert ((GTree *) data, GUINT_TO_POINTER (u),
			     GUINT_TO_POINTER (new_descriptions->len));
	    }
	}
      else
	{
	  if (g_tree_lookup ((GTree *) data, GUINT_TO_POINTER (u)) == NULL)
	    fd->description = 0;
	}
    }
  return (FALSE);
}

gboolean
_adapt_description (GNode * gn, gpointer data)
{
  FILE_DATA *fd;
  guint u = 0, nu;
  if (G_NODE_IS_ROOT (gn) == TRUE)
    return (FALSE);
  fd = get_file_data_from_gnode (gn);
  u = fd->description;
  if (u > 0)
    {
      nu =
	GPOINTER_TO_UINT (g_tree_lookup
			  ((GTree *) data, GUINT_TO_POINTER (u)));
      g_assert ((nu > 0) && (nu < 65535));
      fd->description = nu;
    }
  return (FALSE);
}

void
description_garbage_collection ()
{
  FOLDER *racine = my_config->racine;
  GTree *corresp = g_tree_new (_cmp_intptrs);
  GString *gs;
  gint i;
  new_descriptions = g_ptr_array_new ();
  g_node_traverse (racine->tree,
		   G_PRE_ORDER, G_TRAVERSE_ALL, -1, _test_description,
		   corresp);
  g_node_traverse (racine->tree, G_PRE_ORDER, G_TRAVERSE_ALL, -1,
		   _adapt_description, corresp);
  for (i = 0; i < racine->descriptions->len; i++)
    {
      if ((gs = g_ptr_array_index (racine->descriptions, i)))
	{
	  g_string_free (gs, TRUE);
	}
    }
  g_ptr_array_free (racine->descriptions, FALSE);
  racine->descriptions = new_descriptions;
}

FOLDER *
init_folder ()
{
  FILE_DATA *fd;
  FOLDER *racine;

  racine = (FOLDER *) g_malloc (sizeof (FOLDER));
  racine->datas = g_ptr_array_new ();
  fd = (FILE_DATA *) g_malloc (sizeof (FILE_DATA));
  g_ptr_array_add (racine->datas, fd);
  racine->tree = g_node_new (racine);
  fd->name = NULL;
  fd->taille = 0;
  fd->type = 0;
  fd->mime = NULL;
  fd->date = 0;
  fd->categorie = 0;
  fd->description = 0;
  fd->information = NULL;
  fd->ctree_node = NULL;
  fd->a_parent_node_is_vfs = IS_DISK;


  racine->categories = g_ptr_array_new ();

  racine->descriptions = g_ptr_array_new ();
  racine->catalog_filename = g_string_new (_("No_name"));
  racine->catalog_filename_is_valid = FALSE;
  racine->selected_folder = NULL;
  racine->currently_displayed_gnode = NULL;
  racine->is_modified = FALSE;

  racine->ctree = NULL;


  fd->node = racine->tree;

#if defined(__DEBUG__)
  GML_malloc_forget (racine);
  GML_malloc_forget (fd);
  GML_malloc_forget (racine->catalog_filename);
#endif
  return (racine);
}

void
clear_folder (FOLDER * racine)
{
  gint i;

  if (racine->tree)
    {
      suppress_dir (racine->tree, TRUE);
    }

  if (racine->descriptions)
    {
      for (i = 0; i < racine->descriptions->len; i++)
	{
	  g_string_free (g_ptr_array_index (racine->descriptions, i), TRUE);
	}
      g_ptr_array_free (racine->descriptions, FALSE);
    }

  if (racine->categories)
    {
      for (i = 0; i < racine->categories->len; i++)
	{
	  g_string_free (g_ptr_array_index (racine->categories, i), TRUE);
	}
      g_ptr_array_free (racine->categories, FALSE);
    }

  if (racine->catalog_filename)
    g_string_free (racine->catalog_filename, TRUE);

  if (racine->ctree)
    gtk_widget_destroy (GTK_WIDGET (racine->ctree));

  g_free (racine);

}

gboolean
_scan_folders_for_stats (GNode * gn, gpointer data)
{
  guint32 *stats = data;
  FILE_DATA *fd;

  if (is_dir (gn))
    {
      stats[2]++;
      if (is_in_vfs_any (gn))
	stats[0]++;
    }
  else if (!is_disk (gn))
    {
      stats[3]++;
      if (is_in_vfs_any (gn))
	stats[1]++;
      else if ((is_file (gn)) || (is_vfs_any (gn)))
	{
	  fd = get_file_data_from_gnode (gn);
	  stats[5] += (fd->taille % 1048576);	/* 1024^2 = 1048576 */
	  stats[4] += (fd->taille / 1048576) + (stats[5] / 1048576);
	  stats[5] = (stats[5] % 1048576);

	}
      if (is_vfs_any (gn))
	stats[6]++;
    }
  return (FALSE);
}

void
folder_statistics (FOLDER * racine, GNode * gn, guint32 * nb_desc,
		   guint32 * nb_cat, guint32 * nb_disks, guint32 * nb_dirs,
		   guint32 * nb_dirs_with_vfs, guint32 * nb_files,
		   guint32 * nb_files_with_vfs, guint32 * total_size,
		   guint32 * total_size_2, guint32 * nb_vfs)
{
  guint32 stats[7] = { 0, 0, 0, 0, 0, 0, 0 };
  /* stats[0] = nb of dirs only if inside vfs
   * stats[1] = nb of files only if inside vfs
   * stats[2] = nb of dirs including those inside vfs
   * stats[3] = nb of files including those inside vfs
   * stats[4] = Total size (Mo)
   * stats[5] = Total size(Mo) - Total size
   * stats[6] = Nb of VFS
   */
  if (nb_desc)
    {
      *nb_desc += racine->descriptions->len;
    }
  if (nb_cat)
    {
      *nb_cat += racine->categories->len;
    }
  if (nb_disks)
    {
      *nb_disks += g_node_n_children (racine->tree);
    }
  if (nb_dirs || nb_dirs_with_vfs || nb_files || nb_files_with_vfs
      || total_size)
    {
      g_node_traverse (gn, G_PRE_ORDER, G_TRAVERSE_ALL, -1,
		       _scan_folders_for_stats, stats);
      if (nb_dirs)
	*nb_dirs += (stats[2] - stats[0]);
      if (nb_dirs_with_vfs)
	*nb_dirs_with_vfs += stats[2];
      if (nb_files)
	{
	  *nb_files += (stats[3] - stats[1]);
	  if (G_NODE_IS_ROOT (gn))
	    *nb_files -= 1;
	}
      if (nb_files_with_vfs)
	{
	  *nb_files_with_vfs += stats[3];
	  if (G_NODE_IS_ROOT (gn))
	    *nb_files_with_vfs -= 1;
	}
      if (total_size)
	*total_size += stats[4];
      if (total_size_2)
	*total_size_2 += stats[5];
      if (nb_vfs)
	*nb_vfs += stats[6];
      if (total_size_2)
	{
	  if (total_size)
	    {
	      *total_size += (*total_size_2 / 1048576);
	      *total_size_2 = (*total_size_2 % 1048576);
	    }
	}
    }
}
