/*
	guile database glue
	by C. Ray C.
*/

#include <libguile.h>
#include <db.h>

#define SCM_DB_P(x) SCM_SMOB_PREDICATE(db_tag, x)
#define SCM_BYTE_ARRAY_P(x) (SCM_TYP7(x) == scm_tc7_byvect)

static long db_tag;

static SCM scm_db_open(SCM filename);
void *db_malloc(size_t size);
static SCM scm_db_put_binary(SCM handle, SCM key, SCM data);
static SCM scm_db_get_binary(SCM handle, SCM key);
static SCM scm_db_put_text(SCM handle, SCM key, SCM data);
static SCM scm_db_get_text(SCM handle, SCM key);
static SCM scm_db_close(SCM handle);
static scm_sizet free_db(SCM db_smob);
void scm_init_db(void);

static SCM
scm_db_open(SCM filename)
{
	DB		*db;

	SCM_ASSERT_TYPE(SCM_STRINGP(filename), filename, 1,
		"db-open", "filename");

	if (db_create(&db, NULL, 0))
		scm_misc_error("db-open", "db_create() failed =^P", SCM_EOL);

	db->set_malloc(db, db_malloc);

	SCM_STRING_COERCE_0TERMINATION_X(filename);

	if (db->open(db, SCM_STRING_CHARS(filename), NULL,
		DB_BTREE, DB_CREATE | DB_THREAD, 0))
		scm_misc_error("db-open", "db->open() failed =^|", SCM_EOL);

	SCM_RETURN_NEWSMOB(db_tag, db);
}

void *
db_malloc(size_t size)
{
	return scm_must_malloc(size, "libdb3");
}

static SCM
scm_db_put_binary(SCM handle, SCM key, SCM data)
{
	DB	*db;
	DBT	keyt;
	DBT	datat;

	SCM_ASSERT_TYPE(SCM_DB_P(handle), handle, 1,
		"db-put-binary", "handle");
	SCM_ASSERT_TYPE(SCM_BYTE_ARRAY_P(key), key, 2,
		"db-put-binary", "key");
	SCM_ASSERT_TYPE(SCM_BYTE_ARRAY_P(data), data, 3,
		"db-put-binary", "data");

	db = (DB *)SCM_SMOB_DATA(handle);
	if (db == NULL)
		scm_misc_error("db-put-binary",
			"given handle has been closed", SCM_EOL);

	memset(&keyt, 0, sizeof(DBT));
	keyt.data = SCM_UVECTOR_BASE(key);
	keyt.size = SCM_UVECTOR_LENGTH(key);

	memset(&datat, 0, sizeof(DBT));
	datat.data = SCM_UVECTOR_BASE(data);
	datat.size = SCM_UVECTOR_LENGTH(data);

	if (db->put(db, NULL, &keyt, &datat, 0))
		scm_misc_error("db-put-binary", "db->put() failed =^|",
			SCM_EOL);

	return SCM_UNSPECIFIED;
}

static SCM
scm_db_get_binary(SCM handle, SCM key)
{
	DB	*db;
	DBT	keyt;
	DBT	datat;
	SCM	v;

	SCM_ASSERT_TYPE(SCM_DB_P(handle), handle, 1,
		"db-get-binary", "handle");
	SCM_ASSERT_TYPE(SCM_BYTE_ARRAY_P(key), key, 2,
		"db-get-binary", "key");

	db = (DB *)SCM_SMOB_DATA(handle);
	if (db == NULL)
		scm_misc_error("db-get-binary",
			"given handle has been closed", SCM_EOL);

	memset(&keyt, 0, sizeof(DBT));
	keyt.data = SCM_UVECTOR_BASE(key);
	keyt.size = SCM_UVECTOR_LENGTH(key);

	memset(&datat, 0, sizeof(DBT));
	datat.flags = DB_DBT_MALLOC;

	if (db->get(db, NULL, &keyt, &datat, 0))
		scm_misc_error("db-get-binary", "db->get() failed =^|",
			SCM_EOL);

	SCM_NEWCELL(v);
	SCM_DEFER_INTS;
	SCM_SET_UVECTOR_BASE(v, datat.data);
	SCM_SET_UVECTOR_LENGTH(v, datat.size, scm_tc7_byvect);
	SCM_ALLOW_INTS;

	return v;
}

static SCM
scm_db_put_text(SCM handle, SCM key, SCM data)
{
	DB	*db;
	DBT	keyt;
	DBT	datat;

	SCM_ASSERT_TYPE(SCM_DB_P(handle), handle, 1,
		"db-put-text", "handle");
	SCM_ASSERT_TYPE(SCM_STRINGP(key), key, 2,
		"db-put-text", "key");
	SCM_ASSERT_TYPE(SCM_STRINGP(data), data, 3,
		"db-put-text", "data");

	db = (DB *)SCM_SMOB_DATA(handle);
	if (db == NULL)
		scm_misc_error("db-put-text", "given handle has been closed",
			SCM_EOL);

	memset(&keyt, 0, sizeof(DBT));
	keyt.data = SCM_STRING_CHARS(key);
	keyt.size = SCM_STRING_LENGTH(key);

	memset(&datat, 0, sizeof(DBT));
	datat.data = SCM_STRING_CHARS(data);
	datat.size = SCM_STRING_LENGTH(data);

	if (db->put(db, NULL, &keyt, &datat, 0))
		scm_misc_error("db-put-text", "db->put() failed =^|",
			SCM_EOL);

	return SCM_UNSPECIFIED;
}

static SCM
scm_db_get_text(SCM handle, SCM key)
{
	DB	*db;
	DBT	keyt;
	DBT	datat;
	SCM	s;

	SCM_ASSERT_TYPE(SCM_DB_P(handle), handle, 1,
		"db-get-text", "handle");
	SCM_ASSERT_TYPE(SCM_STRINGP(key), key, 2,
		"db-get-text", "key");

	db = (DB *)SCM_SMOB_DATA(handle);
	if (db == NULL)
		scm_misc_error("db-get-text", "given handle has been closed",
			SCM_EOL);

	memset(&keyt, 0, sizeof(DBT));
	keyt.data = SCM_STRING_CHARS(key);
	keyt.size = SCM_STRING_LENGTH(key);

	memset(&datat, 0, sizeof(DBT));
	datat.flags = DB_DBT_MALLOC;

	if (db->get(db, NULL, &keyt, &datat, 0))
		scm_misc_error("db-get-text", "db->get() failed =^|",
			SCM_EOL);

	SCM_NEWCELL(s);
	SCM_DEFER_INTS;
	SCM_SET_STRING_CHARS(s, datat.data);
	SCM_SET_STRING_LENGTH(s, datat.size);
	SCM_ALLOW_INTS;

	return s;
}

static SCM
scm_db_close(SCM handle)
{
	DB	*db;

	db = (DB *)SCM_SMOB_DATA(handle);
	if (db == NULL)
		scm_misc_error("db-close",
			"given handle has already been closed", SCM_EOL);
	db->close(db, 0);
	SCM_SET_SMOB_DATA(handle, NULL);

	return SCM_UNSPECIFIED;
}

static scm_sizet
free_db(SCM db_smob)
{
	DB	*db;

	db = (DB *)SCM_SMOB_DATA(db_smob);
	db->close(db, 0);

	return 0;
}

void
scm_init_db3(void)
{
	struct function_s {
		char	*name;
		int	req;
		int	opt;
		int	rest;
		SCM	(*func)();
	};

	struct function_s functions[] = {
		{ "db-open", 1, 0, 0, scm_db_open },
		{ "db-put-binary", 3, 0, 0, scm_db_put_binary },
		{ "db-get-binary", 2, 0, 0, scm_db_get_binary },
		{ "db-put-text", 3, 0, 0, scm_db_put_text },
		{ "db-get-text", 2, 0, 0, scm_db_get_text },
		{ "db-close", 1, 0, 0, scm_db_close },
		{ NULL, 0, 0, 0, NULL }
	};

	struct function_s	*p;

	for (p = functions; p->name != NULL; p++)
		scm_make_gsubr(p->name, p->req, p->opt, p->rest, p->func);

	db_tag = scm_make_smob_type("db3 handle", 0);
	scm_set_smob_free(db_tag, free_db);
}

