/*****************************************************************************
*   Gnome Wave Cleaner Version 0.19
*   Copyright (C) 2001 Jeffrey J. Welty
*   
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation; either version 2
*   of the License, or (at your option) any later version.
*   
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*   
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*******************************************************************************/

/* audio_util.c */

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <memory.h>
#include <fcntl.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <sndfile.h>
#include "gwc.h"
#include "fmtheaders.h"
#include "encoding.h"
#include "audio_device.h"
#include "soundfile.h"

int audio_state = AUDIO_IS_IDLE ;
int wavefile_fd = -1 ;
long audio_bytes_written ;
int rate = 44100 ;
int stereo = 1 ;
int audio_bits = 16 ;
int BYTESPERSAMPLE = 2 ;
int MAXSAMPLEVALUE = 1 ;
int PLAYBACK_FRAMESIZE = 4 ;
int FRAMESIZE = 4 ;
/*  int dump_sample = 0 ;  */
long wavefile_data_start ;

SNDFILE      *sndfile = NULL ;
SF_INFO      sfinfo ;

extern struct view audio_view ;
extern struct sound_prefs prefs ;
extern struct encoding_prefs encoding_prefs;

int current_sample ;

void position_wavefile_pointer(long sample_number) ;

void audio_normalize(int flag)
{
    if(flag == 0)
	sf_command(sndfile, SFC_SET_NORM_DOUBLE, NULL, SF_FALSE) ;
    else
	sf_command(sndfile, SFC_SET_NORM_DOUBLE, NULL, SF_TRUE) ;
}

void write_wav_header(int thefd, int speed, long bcount, int bits, int stereo)
{
    /* Spit out header here... */
	    wavhead header;

	    char *riff = "RIFF";
	    char *wave = "WAVE";
	    char *fmt = "fmt ";
	    char *data = "data";

	    memcpy(&(header.main_chunk), riff, 4);
	    header.length = sizeof(wavhead) - 8 + bcount;
	    memcpy(&(header.chunk_type), wave, 4);

	    memcpy(&(header.sub_chunk), fmt, 4);
	    header.sc_len = 16;
	    header.format = 1;
	    header.modus = stereo + 1;
	    header.sample_fq = speed;
	    header.byte_p_sec = ((bits > 8)? 2:1)*(stereo+1)*speed;
/* Correction by J.A. Bezemer: */
	    header.byte_p_spl = ((bits > 8)? 2:1)*(stereo+1);
    /* was: header.byte_p_spl = (bits > 8)? 2:1; */

	    header.bit_p_spl = bits;

	    memcpy(&(header.data_chunk), data, 4);
	    header.data_length = bcount;
	    write(thefd, &header, sizeof(header));
}

void config_audio_device(int rate_set, int bits_set, int stereo_set)
{
    AUDIO_FORMAT format,format_set;
    int channels ;
/*      int fragset = 0x7FFF000F ;  */

    bits_set = 16 ;
    /* play everything as 16 bit, signed integers */
    format_set = GWC_S16_LE ;

    rate = rate_set ;
    audio_bits = bits_set ;
    stereo = stereo_set ;
    format = format_set ;

/*      if(ioctl(audio_fd, SNDCTL_DSP_SETFRAGMENT, &fragset) == -1) {  */
/*  	warning("error setting buffer size on audio device") ;  */
/*      }  */

    channels = stereo + 1 ;
    rate = rate_set ;

    if (audio_device_set_params(&format_set, &channels, &rate) == -1) {
	warning("unknown error setting device parameter") ;
    }

    if(format != format_set) {
	char *buf_fmt_str ;
	char buf[80] ;
	switch(format_set) {
	    case GWC_U8 : buf_fmt_str = "8 bit (unsigned)" ; bits_set = 8 ; break ;
	    case GWC_S8 : buf_fmt_str = "8 bit (signed)" ; bits_set = 8 ; break ;
	    case GWC_S16_BE :
	    case GWC_S16_LE : buf_fmt_str = "16 bit" ; bits_set = 16 ; break ;
	    default : buf_fmt_str = "unknown!" ; bits_set = 8 ; break ;
	}
        sprintf(buf, "Set bits to %s - does your soundcard support what you requested?\n", buf_fmt_str) ;
	warning(buf) ;
    }

    if(channels != stereo + 1) {
	char buf[80] ;
	if(stereo == 0)
	    sprintf(buf, "Failed to set mono mode\nYour sound card may not support mono\n") ;
	else
	    sprintf(buf, "Failed to set stereo mode\nYour sound card may not support stereo\n") ;
	warning(buf) ;
    }
    stereo_set = channels - 1 ;

    if(ABS(rate_set-rate) > 10) {
	char buf[80] ;
	sprintf(buf, "Rate set to %d instead of %d\nYour sound card may not support the desired rate\n",
	             rate_set, rate) ;
	warning(buf) ;
    }

    rate = rate_set ;
    audio_bits = bits_set ;
    stereo = stereo_set ;
}

long playback_samples_remaining = 0 ;
long playback_total_bytes ;
int playback_bytes_per_block ;
int looped_count ;

#define MAXBUFSIZE 32768
int BUFSIZE ;
unsigned char audio_buffer[MAXBUFSIZE] ;

long playback_start_position ;
long playback_end_position ;
long playback_position ;
long first_playback_sample ;

long set_playback_cursor_position(struct view *v, long millisec_per_visual_frame)
{
    long first, last ;

    if(audio_state == AUDIO_IS_PLAYBACK) {
	long bytes = audio_device_processed_bytes()-looped_count*playback_total_bytes ;
	get_region_of_interest(&first, &last, v) ;

	v->cursor_position = first_playback_sample+bytes/(PLAYBACK_FRAMESIZE) ;

	return playback_total_bytes - bytes ;
    }

    {
	long inc = rate*millisec_per_visual_frame/1000 ;
/*  	g_print("inc:%ld\n", inc) ;  */
	v->cursor_position += inc ;
	return 1 ;
    }
    
}

long start_playback(char *output_device, struct view *v, struct sound_prefs *p, double seconds_per_block, double seconds_to_preload)
{
    long first, last ;
    long playback_samples ;
    gfloat lv, rv ;

    if(sndfile == NULL) return 1 ;

    audio_device_close() ;

    if (audio_device_open(output_device) == -1) {
	char buf[80] ;
	sprintf(buf, "Failed to open output device %s", output_device) ;
	warning(buf) ;
	return 1 ;
    }

    get_region_of_interest(&first, &last, v) ;

/*      g_print("first is %ld\n", first) ;  */
/*      g_print("last is %ld\n", last) ;  */
/*      g_print("rate is %ld\n", (long)p->rate) ;  */

    first_playback_sample = first ;

    playback_start_position =  first ;
    playback_end_position = last+1;
    playback_position = playback_start_position ;
    playback_samples = p->rate*seconds_per_block ;
    playback_bytes_per_block = playback_samples*PLAYBACK_FRAMESIZE ;

    //  This was moved down 8 lines to make it work in OS X.  Rob
    config_audio_device(p->rate, p->playback_bits, p->stereo);	//Set up the audio device.
    //stereo is 1 if it is stereo
    //playback_bits is the number of bits per sample
    //rate is the number of samples per second

    BUFSIZE = audio_device_best_buffer_size(playback_bytes_per_block);

    playback_bytes_per_block = BUFSIZE ;

    if(playback_bytes_per_block > MAXBUFSIZE) {
	playback_bytes_per_block = MAXBUFSIZE ;
    }

    playback_samples = playback_bytes_per_block/PLAYBACK_FRAMESIZE ;

    BUFSIZE = playback_bytes_per_block ;

    playback_samples_remaining = (last-first+1) ;
    playback_total_bytes = playback_samples_remaining*PLAYBACK_FRAMESIZE ;

    audio_state = AUDIO_IS_PLAYBACK ;

    sf_seek(sndfile, playback_position, SEEK_SET) ;

    /* put some data in the buffer queues, to avoid underflows */
    if(0) {
	int n = (int)(seconds_to_preload / seconds_per_block+0.5) ;
	int old_playback_bytes = playback_bytes_per_block ;

	playback_bytes_per_block *= n ;
	if(playback_bytes_per_block > MAXBUFSIZE) playback_bytes_per_block = MAXBUFSIZE ;
	process_audio(&lv, &rv) ;
	v->cursor_position = first+playback_bytes_per_block/(PLAYBACK_FRAMESIZE) ;
	playback_bytes_per_block = old_playback_bytes ;
    }

/*      g_print("playback_start_position is %ld\n", playback_start_position) ;  */
/*      g_print("playback_samples is %ld\n", playback_samples) ;  */
/*      g_print("BUFSIZE %ld (%lg fragments)\n", (long)BUFSIZE, (double)BUFSIZE/(double)oss_info.fragsize) ;  */

    v->prev_cursor_position = -1 ;
    looped_count = 0 ;

    return playback_samples ;
}

void *wavefile_data ;
#ifdef TRUNCATE_OLD
void truncate_wavfile(struct view *v)
{
#define REALLY_TRUNCATE
#ifndef REALLY_TRUNCATE
    warning("Truncation temporailty disabled, while incorporating libsndfile...") ;
#else
    /* we must do 3 things:
	1. Shift all samples forward by v->truncate_head
	2. Rescan the sample blocks (along the way)
	3. Physically truncate the size of the file on 
	   the filesystem by (v->truncate_head + (n_samples-1)-v->truncate_tail) samples
    */

    long prev ;
    long new ;
    int n_in_buf ;
    long first, last ;

#define TMPBUFSIZE     (SBW*1000)
    long left[TMPBUFSIZE], right[TMPBUFSIZE] ;

    push_status_text("Truncating audio data") ;
    update_status_bar(0.0, STATUS_UPDATE_INTERVAL, TRUE) ;

    /* something like this, gotta buffer this or the disk head will
       burn a hole in the platter */

    if(v->truncate_head > 0) {

	for(prev = v->truncate_head ; prev <= v->truncate_tail ; prev += TMPBUFSIZE) {
	    update_status_bar((gfloat)(prev-v->truncate_head)/(gfloat)(v->truncate_tail-v->truncate_head), STATUS_UPDATE_INTERVAL, FALSE) ;
	    last = MIN((prev+TMPBUFSIZE-1), v->truncate_tail) ;
	    n_in_buf = read_wavefile_data(left, right, prev, last) ;
	    new = prev - v->truncate_head ;
	    first = new ;
	    last = new + n_in_buf - 1 ;
	    write_wavefile_data(left, right, first, last) ;
	    resample_audio_data(&prefs, first, last) ;
	}

    }

    prefs.n_samples = v->truncate_tail - v->truncate_head + 1 ;

    if(1) save_sample_block_data(&prefs) ;

    if(1) {
	sf_count_t total_samples =  prefs.n_samples ;
	if(sf_command(sndfile, SFC_FILE_TRUNCATE, &total_samples, sizeof(total_samples)))
	    warning("Libsndfile reports truncation of audio file failed") ;
    }

    pop_status_text() ;

#endif
}
#endif /* !TRUNCATE_OLD */

int close_wavefile(struct view *v)
{
#ifdef TRUNCATE_OLD
    int r ;

    if(v->truncate_head > 0 || v->truncate_tail < v->n_samples -1) {
	r = yesnocancel("Part of the waveform is selected for truncation, do you really want to truncate?") ;
	if(r == 2) return 0 ;
	if(r == 0) truncate_wavfile(v) ;
    }
#endif /* TRUNCATE_OLD */
    if(sndfile != NULL) {
	sf_close(sndfile) ;
    }
    audio_device_close() ;
    sndfile = NULL ;

    return 1 ;
}

void save_selection_as_wavfile(char *filename_new, struct view *v)
{
    int fd_new ;
    SNDFILE *sndfile_new ;
    SF_INFO sfinfo_new ;

    long total_samples ;
    long total_bytes ;

    total_samples =  v->selected_last_sample-v->selected_first_sample+1 ;

    if(total_samples < 0 || total_samples > v->n_samples) {
	warning("Invalid selection") ;
	return ;
    }

    total_bytes = total_samples*FRAMESIZE ;

    {
	char buf[1000] ;
	sprintf(buf, "Save %ld bytes to %s", total_bytes, filename_new) ;
	if(yesno(buf))  {
	    return ;
	}
    }

    fd_new = open(filename_new, O_RDONLY) ;

    if(fd_new > -1) {
	char buf[1000] ;
	close(fd_new) ;
	sprintf(buf, "%s exists, overwrite ?", filename_new) ;
	if(yesno(buf))  {
	    return ;
	}
    }


/*      fd_new = open(filename_new, O_WRONLY|O_CREAT|O_TRUNC,S_IRGRP|S_IROTH|S_IRUSR|S_IWUSR) ;  */

    sfinfo_new = sfinfo ;
    sfinfo_new.frames = total_samples ;

    if (! (sndfile_new = sf_open (filename_new, SFM_WRITE, &sfinfo_new))) {
	/* Open failed so print an error message. */

	char buf[80] ;
	sprintf(buf, "Failed to save selection %s", filename_new) ;
	warning(buf) ;
	return ;

        /* Print the error message from libsndfile. */
/*          sf_perror (NULL) ;  */
/*          return  1 ;  */
    } ;

    push_status_text("Saving selection") ;


    /* something like this, gotta buffer this or the disk head will
       burn a hole in the platter */

    position_wavefile_pointer(v->selected_first_sample) ;

    {
	long n_copied  ;

#define TMPBUFSIZE     (SBW*1000)
	unsigned char buf[TMPBUFSIZE] ;
	long framebufsize = (TMPBUFSIZE/FRAMESIZE) * FRAMESIZE ;

	update_status_bar(0.0,STATUS_UPDATE_INTERVAL,TRUE) ;

	for(n_copied = 0 ; n_copied < total_bytes ; n_copied += framebufsize) {
	    long n_to_copy = framebufsize ;

#ifdef MAC_OS_X /* MacOSX */
	    usleep(2) ; // prevents segfault on OSX, who knows, something to do with status bar update...
#endif

	    update_status_bar((gfloat)(n_copied)/(gfloat)(total_bytes),STATUS_UPDATE_INTERVAL,FALSE) ;

	    if(n_copied + n_to_copy > total_bytes) n_to_copy = total_bytes - n_copied ;

	    n_to_copy = sf_read_raw(sndfile, buf, n_to_copy) ;
	    sf_write_raw(sndfile_new, buf, n_to_copy) ;
	}

    }

    update_status_bar((gfloat)0.0,STATUS_UPDATE_INTERVAL,TRUE) ;

    sf_close(sndfile_new) ;

    pop_status_text() ;

}


int is_valid_audio_file(char *filename)
{
    SNDFILE *sndfile ;
    SF_INFO sfinfo ;

    sfinfo.format = 0 ;

    if((sndfile = sf_open(filename, SFM_READ, &sfinfo)) != NULL) {
	sf_close(sndfile) ;
	return 1 ;
    }

    return 0 ;
}


struct sound_prefs open_wavefile(char *filename, struct view *v)
{
    struct sound_prefs wfh ;

    if(close_wavefile(v)) {
	wfh.successful_open = TRUE ;
    } else {
	wfh.successful_open = FALSE ;
	return wfh ;
    }


    wfh.wavefile_fd = wavefile_fd ;
    wfh.sample_buffer_exists = FALSE ;

    if (! (sndfile = sf_open (filename, SFM_RDWR, &sfinfo))) {
	/* Open failed so print an error message. */

	char buf[80] ;
	sprintf(buf, "Failed to open  %s, no permissions or unknown audio format", filename) ;
	warning(buf) ;
	wfh.successful_open = FALSE ;
	return wfh ;

        /* Print the error message from libsndfile. */
/*          sf_perror (NULL) ;  */
/*          return  1 ;  */
    } ;

    wfh.wavefile_fd = 1 ;


    /* determine soundfile properties */
    {
	wfh.rate = sfinfo.samplerate ;
	wfh.n_channels = sfinfo.channels ;
	wfh.stereo = stereo = sfinfo.channels-1 ;

	wfh.n_samples = sfinfo.frames ;
	
	switch(sfinfo.format & 0x00000F) {
	    case SF_FORMAT_PCM_U8 : BYTESPERSAMPLE=1 ; MAXSAMPLEVALUE = 1 << 8 ; break ;
	    case SF_FORMAT_PCM_S8 : BYTESPERSAMPLE=1 ; MAXSAMPLEVALUE = 1 << 7 ; break ;
	    case SF_FORMAT_PCM_16 : BYTESPERSAMPLE=2 ; MAXSAMPLEVALUE = 1 << 15 ; break ;
	    case SF_FORMAT_PCM_24 : BYTESPERSAMPLE=3 ; MAXSAMPLEVALUE = 1 << 23 ; break ;
	    case SF_FORMAT_PCM_32 : BYTESPERSAMPLE=4 ; MAXSAMPLEVALUE = 1 << 31 ; break ;
	    default : warning("Soundfile format not allowed") ; break ;
	}

	FRAMESIZE = BYTESPERSAMPLE*wfh.n_channels ;
	PLAYBACK_FRAMESIZE = 2*wfh.n_channels ;

	wfh.playback_bits = audio_bits = wfh.bits = BYTESPERSAMPLE*8 ;

	wfh.max_allowed = MAXSAMPLEVALUE-1 ;

#ifdef WAVHEADER_FIX
	/* do some simple error checking on the wavfile header , so we don't seek data where it isn't */
	if(!fstat(wavefile_fd, &wav_stat)) {
	    if(wav_stat.st_size != wfh.data_offset+wfh.data_length) {
		if(!yesno("Wave file header is damaged, attempt temporary fix?")) {
		    header.data_length = wav_stat.st_size - wfh.data_offset ;

		    wfh.n_samples = header.data_length / BYTESPERSAMPLE / wfh.n_channels ;
		    wfh.data_length = header.data_length ;
		    
		}
	    }
	}
#endif
    }

    gwc_window_set_title(filename) ;

    return wfh ;
}

void position_wavefile_pointer(long sample_number)
{
    sf_seek(sndfile, sample_number, SEEK_SET) ;
}

int read_raw_wavefile_data(char buf[], long first, long last)
{
    long n = last - first + 1 ;
    int n_read ;

    position_wavefile_pointer(first) ;

    n_read = sf_read_raw(sndfile, buf, n*FRAMESIZE) ;

    return n_read/FRAMESIZE ;
}

int write_raw_wavefile_data(char buf[], long first, long last)
{
    long n = last - first + 1 ;
    int n_read ;

    position_wavefile_pointer(first) ;

    n_read = sf_write_raw(sndfile, buf, n*FRAMESIZE) ;

    return n_read/FRAMESIZE ;
}

int read_wavefile_data(long left[], long right[], long first, long last)
{
    long n = last - first + 1 ;
    long s_i = 0 ;
    long bufsize_long ;
    long j ;
    int *p_int ;

    position_wavefile_pointer(first) ;
    p_int = (int *)audio_buffer ;
    bufsize_long = sizeof(audio_buffer) / sizeof(long) ;

    while(s_i < n) {
	long n_this = MIN((n-s_i)*(stereo+1), bufsize_long) ;
	long n_read = sf_read_int(sndfile, p_int, n_this) ;

	for(j = 0  ; j < n_read ; ) {

	    left[s_i] = p_int[j] ;
	    j++ ;

	    if(stereo) {
		right[s_i] = p_int[j] ;
		j++ ;
	    } else {
		right[s_i] = left[s_i] ;
	    }

	    s_i++ ;
	}

	if(n_read == 0) {
	    char tmp[100] ;
	    sprintf(tmp, "Attempted to read past end of audio, first=%ld, last=%ld", first, last) ;
	    warning(tmp) ;
	    exit(1) ;
	}
    }

    return s_i ;
}

int read_fft_real_wavefile_data(fftw_real left[], fftw_real right[], long first, long last)
{
    long n = last - first + 1 ;
    long s_i = 0 ;
    long j ;

    double *buffer = (double *)audio_buffer ;

    int bufsize_double = sizeof(audio_buffer) / sizeof(double) ;

    position_wavefile_pointer(first) ;

    while(s_i < n) {
	long n_this = MIN((n-s_i)*(stereo+1), bufsize_double) ;
	long n_read = sf_read_double(sndfile, buffer, n_this) ;

	for(j = 0  ; j < n_read ; j++) {
	    left[s_i] = buffer[j] ;
	    if(stereo) {
		j++ ;
		right[s_i] = buffer[j] ;
	    } else {
		right[s_i] = left[s_i] ;
	    }
	    s_i++ ;
	}

	if(n_read == 0) {
	    char tmp[100] ;
	    sprintf(tmp, "Attempted to read past end of audio, first=%ld, last=%ld", first, last) ;
	    warning(tmp) ;
	    exit(1) ;
	}

    }

    return s_i ;
}

int sf_write_values(void *ptr, int n_samples)
{
    int n = 0 ;

    if(n_samples > 0) {
	n = sf_write_int(sndfile, ptr, n_samples) ;
    }

    return n * BYTESPERSAMPLE ;
}

long n_in_buf = 0 ;

int WRITE_VALUE_TO_AUDIO_BUF(char *ivalue)
{
    int i ;
    int n_written = 0 ;


    if(BYTESPERSAMPLE+n_in_buf > MAXBUFSIZE) {
	n_written = sf_write_values(audio_buffer, n_in_buf/BYTESPERSAMPLE) ;
	n_in_buf = 0 ;
    }

    for(i = 0 ; i < BYTESPERSAMPLE ; i++, n_in_buf++)
	audio_buffer[n_in_buf] = ivalue[i] ;

    return n_written ;
}

#define FLUSH_AUDIO_BUF(n_written) {\
	n_written += sf_write_values(audio_buffer, n_in_buf/BYTESPERSAMPLE) ;\
	n_in_buf = 0 ;\
    }

    

int write_fft_real_wavefile_data(fftw_real left[], fftw_real right[], long first, long last)
{
    long n = last - first + 1 ;
    long j ;
    long n_written = 0 ;

/* make SURE MAXBUF is a multiple of 2 */
#define MAXBUF 500
    double buf[MAXBUF] ;

    n_in_buf = 0 ;

    position_wavefile_pointer(first) ;

    for(j = 0  ; j < n ; j++) {
	buf[n_in_buf] = left[j] ;
	n_in_buf++ ;
	if(stereo) {
	    buf[n_in_buf] = right[j] ;
	    n_in_buf++ ;
	}

	if(n_in_buf == MAXBUF) {
	    n_written += sf_write_double(sndfile, buf, n_in_buf) ;
	    n_in_buf = 0 ;
	}
    }

    if(n_in_buf > 0) {
	n_written += sf_write_double(sndfile, buf, n_in_buf) ;
	n_in_buf = 0 ;
    }

    return n_written/2 ;
#undef MAXBUF
}

int write_wavefile_data(long left[], long right[], long first, long last)
{
    long n = last - first + 1 ;
    long j ;
    long n_written = 0 ;
/* make SURE MAXBUF is a multiple of 2 */
#define MAXBUF 500
    int buf[MAXBUF] ;

    n_in_buf = 0 ;

    position_wavefile_pointer(first) ;

    for(j = 0  ; j < n ; j++) {
	buf[n_in_buf] = left[j] ;
	n_in_buf++ ;
	if(stereo) {
	    buf[n_in_buf] = right[j] ;
	    n_in_buf++ ;
	}

	if(n_in_buf == MAXBUF) {
	    n_written += sf_write_int(sndfile, buf, n_in_buf) ;
	    n_in_buf = 0 ;
	}
    }

    if(n_in_buf > 0) {
	n_written += sf_write_int(sndfile, buf, n_in_buf) ;
	n_in_buf = 0 ;
    }

    return n_written/2 ;
#undef MAXBUF
}

void flush_wavefile_data(void)
{
    fsync(wavefile_fd) ;
}

/* process_audio for mac_os_x is found in the audio_osx.c */
#ifndef MAC_OS_X
int process_audio(gfloat *pL, gfloat *pR)
{
    int len = 0 ;
    int i, frame ;
    short *p_short ;
    int *p_int ;
    unsigned char  *p_char ;
    short maxl = 0, maxr = 0 ;
    extern int audio_playback ;
    long n_samples_to_read, n_read ;
    double maxpossible ;
    double feather_out_N ;
    int feather_out = 0 ;

    *pL = 0.0 ;
    *pR = 0.0 ;
	

    if(audio_state == AUDIO_IS_IDLE) {
	d_print("process_audio says NOTHING is going on.\n") ;
	return 1 ;
    }

    if(audio_state == AUDIO_IS_RECORDING) {
	if((len = audio_device_read(audio_buffer, BUFSIZE)) == -1) {
	    warning("Error on audio read...") ;
	}
    } else if(audio_state == AUDIO_IS_PLAYBACK) {
        len = audio_device_nonblocking_write_buffer_size(
            MAXBUFSIZE, playback_samples_remaining*PLAYBACK_FRAMESIZE);

	if (len <= 0) {
	    return 0 ;
	}
    }

    n_samples_to_read = len/PLAYBACK_FRAMESIZE ;

    if(n_samples_to_read*PLAYBACK_FRAMESIZE != len)
	g_print("ACK!!\n") ;

    p_char = (unsigned char *)audio_buffer ;
    p_short = (short *)audio_buffer ;
    p_int = (int *)audio_buffer ;

    /* for now force playback to 16 bit... */
#define BYTESPERSAMPLE 2

    if(BYTESPERSAMPLE < 3) {
	maxpossible = 1 << 15 ;
	n_read = sf_readf_short(sndfile, p_short, n_samples_to_read) ;
    } else {
	maxpossible = 1 << 23 ;
	n_read = sf_readf_int(sndfile, p_int, n_samples_to_read) ;
    }

#define FEATHER_WIDTH 30000
    if(playback_samples_remaining - n_read < 1) {
	feather_out = 1 ;
	feather_out_N = MIN(n_read, FEATHER_WIDTH) ;
	printf("Feather out n_read=%d, playback_samples_remaining=%d, N=%lf\n", n_read, playback_samples_remaining, feather_out_N) ;
    }

    for(frame = 0  ; frame < n_read ; frame++) {
	int vl, vr ;
	i = frame*2 ;

	if(BYTESPERSAMPLE < 3) {
	    if(feather_out == 1 && n_read-(frame+1) < FEATHER_WIDTH) {
		int j = ((n_read-(frame))-1) ;
		double p = (double)(j)/feather_out_N ;

		if(i > n_read - 100) {
		    //printf("j:%d %lf %hd %hd ", j, p, p_short[i], p_short[i+1]) ;
		}

		p_short[i] *= p ;
		p_short[i+1] *= p ;

		//if(i > n_read - 100) {
		//    printf("%hd %hd\n", p_short[i], p_short[i+1]) ;
		//}

		if(frame == n_read-1) printf("Feather out final %lf, n_read=%d", p, n_read) ;
	    }

	    vl = p_short[i] ;
	    vr = p_short[i+1] ;

	} else {
	    if(feather_out == 1 && n_read-(i+1) < 10000) {
		double p = 1.0 - (double)(n_read-(i+1))/9999.0 ;
		printf(".") ;
		p_int[i] *= p ;
		p_int[i+1] *= p ;
	    }

	    vl = p_int[i] ;
	    vr = p_int[i+1] ;
	}

	if(vl > maxl) maxl = vl ;
	if(-vl > maxl) maxl = -vl ;

	if(stereo) {
	    if(vr > maxr) maxr = vr ;
	    if(-vr > maxr) maxr = -vr ;
	} else {
	    maxr = maxl ;
	}
    }
#undef BYTESPERSAMPLE
    if(feather_out == 1) printf("\n") ;

    *pL = (gfloat) maxl / maxpossible ;
    *pR = (gfloat) maxr / maxpossible ;

    if(audio_state == AUDIO_IS_RECORDING) {
	len = write(wavefile_fd, audio_buffer, len) ;
	audio_bytes_written += len ;
    } else if(audio_state == AUDIO_IS_PLAYBACK) {
	len = audio_device_write(p_char, len) ;
	playback_position += n_read ;
	playback_samples_remaining -= n_read ;

	if(playback_samples_remaining < 1) {
	    extern int audio_is_looping ;

	    if(audio_is_looping == FALSE) {
		char zeros[1024] ;
		long zeros_needed ;
		memset(zeros,0,sizeof(zeros)) ;
		audio_state = AUDIO_IS_PLAYBACK ;
		audio_playback = FALSE ;

		zeros_needed = playback_bytes_per_block - (playback_total_bytes % playback_bytes_per_block) ;
		do {
		    len = audio_device_write(zeros, MIN(zeros_needed, sizeof(zeros))) ;
		    zeros_needed -= len ;
		} while (len >= 0 && zeros_needed > 0) ;

		g_print("Stop playback with playback_samples_remaining:%ld\n", playback_samples_remaining) ;
		return 1 ;
	    } else {
		playback_position = playback_start_position ;
		playback_samples_remaining = (playback_end_position-playback_start_position) ;
		sf_seek(sndfile, playback_position, SEEK_SET) ;
		g_print("Loop with playback_samples_remaining:%ld\n", playback_samples_remaining) ;
		looped_count++ ;
	    }
	}
    }
    return 0 ;
}
#endif

void stop_playback(int force)
{
    if(!force) {
	/* Robert altered */
	int new_playback = audio_device_processed_bytes();
	int  old_playback;

	while(new_playback < playback_total_bytes) {
	    /* Robert altered */
	    usleep(100) ;
	    old_playback = new_playback;
	    new_playback=audio_device_processed_bytes();

	    /* check if more samples have been processed, if not,quit */
	    if (old_playback==new_playback){
		fprintf(stderr,"Playback appears frozen\n Breaking\n");
		break;
	    }
	}

	usleep(100) ;
    }

/*      fprintf(stderr, "Usleeping 300000\n") ;  */
/*      usleep(300000) ;  */
/*      fprintf(stderr, "Done usleeping 300000\n") ;  */

    audio_state = AUDIO_IS_IDLE ;
    audio_device_close() ;
}
