{-# OPTIONS -fglasgow-exts #-}
-- for deriving Typeable

--Copyright (C) 2005 HAppS.org. All Rights Reserved.
module HAppS.Protocols.SMTP
    (-- * Handlers
     smtpServer, smtpServerIO, SMTPConf(..),
     -- * Types
     Envelope(..), ezEnvelope, Address(..), Domain, Email, EMail,
     -- * Sending messages
     send, autoSend, envSend, hostSend, handleSend,
     -- * Errors
     smtpErrorMajorCode, smtpTry
    ) where

import Control.Concurrent(forkIO)
import Control.Exception as E
import Data.Char(isAlphaNum)
import Data.Dynamic(fromDynamic)
import Data.List(intersperse)
import Data.Typeable(Typeable)
import Network
import System.IO(Handle,hClose)
import System.Environment(getEnv)

import HAppS.MACID
import HAppS.MACID.Logger
import HAppS.Protocols.DNS.MXClient
import HAppS.Util.Common
import HAppS.Util.StdMain.Config
import HAppS.Util.TimeOut

type Domain = String
type EMail = Address
data Address = Address {mailBox::String,domain::Domain} deriving (Ord,Eq)
type Email = Address

data Envelope msg = Envelope
    { relay      :: Domain     -- ^ Name of server in HELO
    , sender     :: Address    -- ^ Sender address
    , recipients :: [Address]  -- ^ Recipients
    , contents   :: msg
    } deriving (Read,Show,Typeable)

-- | Simple envelope with dummy relay.
ezEnvelope fromAddr toAddr contents = 
	Envelope "host" fromAddr [toAddr] contents

instance Show Address where
	show (Address u d)=u++'@':d

instance Read Address where
	readsPrec _ s = if isAddr then [(Address userId domain,rest)] else []
		where
		(userId,dr)=split (=='@') $ trim s
		(domain,rest) = break (\x-> not $ isAlphaNum x || (x `elem` "-_.")) dr
		isAddr = not $ null userId || null domain
		--(text,rest) = span (\x->not $ x `elem` ",|;()\"'#$^&*<>?/\\`~= \r\n\t") s

data SMTPState = RSET
	       | HELO Domain
	       | RCPTTO Domain Address [Address]

data SMTPError = SMTPError String deriving(Show,Typeable)

-- | The first number of the SMTP error code.
smtpErrorMajorCode :: SMTPError -> Int
smtpErrorMajorCode (SMTPError (e:_)) = fromEnum e - fromEnum '0'
smtpErrorMajorCode _                 = -1

-- | Try to do an SMTP action.
smtpTry :: IO a                  -- ^ Action
        -> (Exception -> IO b)   -- ^ Generic errors
        -> (SMTPError -> IO b)   -- ^ SMTP protocol errors
        -> (a -> IO b)           -- ^ Success
        -> IO b                  -- ^ Final result
smtpTry x a b c = do
    v <- try x
    case v of
      Right v' -> c v'
      Left (DynException e) | Just se <- fromDynamic e -> b se
      Left e                                           -> a e

smtpProtocol q handle = do
	hPutLine handle "220 thishost SMTP BLah"
        smtpProtocol' q RSET handle

smtpProtocol' queueReq session handle =
	do
	line <- hGetLn handle 
	command <- return $ take 4 line
	case command of
				 "QUIT" -> quit
				 "RSET" -> rset
				 "HELO" -> helo $ toDomain $ drop 5 line
				 "EHLO" -> helo $ toDomain $ drop 5 line
				 "MAIL" -> (mail session) $ read.unBracket $ drop 10 line
				 "RCPT" -> (rcpt session) $ read.unBracket $ drop 8 line
				 "DATA" -> message session
				 "NOOP" -> smtp "250 2.0.0 OK" session
				 _ -> quit	
	where
	ok addr entity = 
		"250 2.1.2 <"++(show addr)++">... "++entity++" ok"
	toDomain = id -- words.map (\x->if x=='.' then ' ' else x) 
--	toAddr text = if isAddr then Just $ Address userId (tail domain) else Nothing
--		where
--		(userId,domain)=span (/='@') $ unBracket text
--		isAddr = not $ null userId || null domain
	smtp resp sess = 
		hPutLine handle resp >> 
		smtpProtocol' queueReq sess handle
	quit = hPutLine handle "221 2.0.0 mail.i2x closing connection" >> hClose handle
	badOrder = quit
	rset = smtp "250 2.0.0 Reset state" RSET
	helo relay = smtp "250 blah SMTP" (HELO relay)
	mail (HELO relay) sender = smtp (ok sender "Sender") $ RCPTTO relay sender []
	rcpt (RCPTTO r s as) addr = smtp (ok addr "Recipient") (RCPTTO r s (addr:as))
	rcpt _ _ = badOrder -- out of order error? why bother?
	beginDataText = "354 Enter mail, end with \".\" on a line by itself" 
	message      (RCPTTO _ _ []) = 	badOrder
	message sess@(RCPTTO _ _ _)  = hPutLine handle beginDataText >>  message' sess []
	message _ = badOrder
	message' session@(RCPTTO relay sender addrs) mlines =
		do
		line <- hGetLn handle 
		if line/="." then message' session (undot line:mlines)
		   else queueReq msg >> smtp "250 2.0.0 Message accepted for delivery" (HELO relay)
		where
		undot line = if length line>1 && head line == '.' then tail line else line 
		msg = Envelope relay sender (reverse addrs) 
			  (concat $ intersperse "\r\n" $ reverse mlines)
	
-- | Try sending a message first with "envSend" and then with "autoSend".
send :: Envelope String -> IO ()
send msg = withSafeTimeOut (60*second) $ do
    res <- E.try $ envSend msg
    case res of
      Left _ -> autoSend msg
      _      -> return ()

-- | Send automatically to each recipient directly.
-- Resolves MX for the domains and opens a TCP connection to them.
-- Does not work from *DSL connection because they
-- are usually blacklisted from sending mail.
autoSend :: Envelope String -> IO ()
autoSend msg = sequence_ [ withMXConnection domain 25 (flip handleSend msg) | Address _ domain <- recipients msg ]

-- | Send through the given host.
hostSend :: HostName -> Int -> Envelope String -> IO ()
hostSend host portNum msg = bracket (connectTo host (PortNumber $ fromIntegral portNum))
                                    (\h -> hClose h `E.catch` (\_ -> return ()))
                                    (flip handleSend msg)

-- | Send using the given Handle.
handleSend :: Handle -> Envelope String -> IO ()
handleSend smtpH msg = do
	hGetLn smtpH >>= isReady
	mapM_ (isAccept=<<) $ 
		  doLine smtpH ("HELO "++(relay msg)) :
		  doLine smtpH ("MAIL FROM: <"++(show $ sender msg)++">") :
		  map (\addr->doLine smtpH $ "RCPT TO: <"++(show addr)++">") (recipients msg)
	isData =<< doLine smtpH "DATA" 
	hPutLine smtpH (contentData msg)
	isAccept =<<  doLine smtpH "."
	isClose =<< doLine smtpH "QUIT" 
	where
	isReady ('2':'2':'0':' ':_) = print "ISREADY" >> return ()
	isReady msg = failure msg
	isAccept ('2':'5':'0':' ':_) = return ()
	isAccept ('2':'5':'1':' ':_) = return ()
	isAccept msg = failure msg
	isData ('3':'5':'4':' ':_) = return ()
	isData msg = failure msg
	isClose ('2':'2':'1':' ':_) = return ()
	isClose msg = failure msg
--        readM str = case reads str of [(x,"")] -> return x; _ -> fail "No parse"
        failure s = do hPutLine smtpH "QUIT"
                       throwDyn $ SMTPError s

contentData msg = unlines $ 
                  map (\x->if x/=[] && head x=='.' then ('.':x) else x) $ 
		  lines $ contents msg
doLine h line = hPutLine h line >>	hGetLn h
	
-- | Send a message using the environment variable @SMTP_RELAY@ as the
--   mailserver to contact. If it is not defined envSend will fail.
envSend :: Envelope String -> IO ()
envSend message = 
	do
	(host,portText) <- getEnv "SMTP_RELAY" >>= return.split (==':')
	port <- return $ if null portText then 25 else read portText
	hostSend host port message


data SMTPConf = SMTPConf { smtpPort :: Int
                         }


smtpServerIO :: Ev st (Envelope String) (IO ()) -> SMTPConf -> Handler st
smtpServerIO ev conf = SyncH $ return (work, serv)
    where work etf = etf ev
          fork_ c  = forkIO c >> return ()
          serv  h  = fork_ $ do
                       let aloop s = do (ha,_,_) <- accept s
                                        fork_ $ smtpProtocol h ha
                                        aloop s
                       aloop =<< listenOn (PortNumber (fromIntegral $ smtpPort conf))

smtpServer :: Ev st (Envelope String) () -> SMTPConf -> Handler st
smtpServer = smtpServerIO . fmap return

-- Instances

instance (Typeable a, Show a, Read a) => Serialize (Envelope a) where
    typeString x  = "HAppS.Protocols.SMTP.Envelope ("++defaultTypeString (y x)++")"
        where y :: Proxy (Envelope a) -> Proxy a
              y _ = undefined
    decodeStringM = defaultDecodeStringM
    encodeStringM = defaultEncodeStringM
instance Show a => LogFormat (Envelope a) where
    logFormat _ = show

ho = [Option [] ["smtp-port"] (ReqArg (\h c -> do x <- readM h; return c { smtpPort = x }) "port") "port to bind mail server"]
instance ConfHandler SMTPConf where
    confUsage _ = copt ho
    confHandler fun = getOptM ho >>= foldl (>>=) (return $ SMTPConf 0) >>= return . fun
