/*
 *  IHU -- I Hear U, easy VoIP application using Speex and Qt
 *
 *  Copyright (C) 2003-2006 Matteo Trotta - <mrotta@users.sourceforge.net>
 *
 *  http://ihu.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
 */

#include <stdlib.h>
#include <time.h>

#include <qvariant.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qslider.h>
#include <qtoolbutton.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qfiledialog.h>
#include <qwhatsthis.h>
#include <qaction.h>
#include <qmenubar.h>
#include <qpopupmenu.h>
#include <qtoolbar.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qmessagebox.h>
#include <qstatusbar.h>
#include <qinputdialog.h>
#include <qcombobox.h>
#include <qlistbox.h>

#include "Error.h"
#include "Ihu.hpp"
#include "Settings.hpp"
#include "LogViewer.hpp"

// Timeout (ms) for statistics
#define STAT_TIME 1000
#define RESTART_TIME 3000
#define MAX_COUNT 10

#define VDATE "2006/Oct/23"

/*
 *  Constructs a Ihu as a child of 'parent', with the
 *  name 'name' and widget flags set to 'f'.
 *
 */
Ihu::Ihu( QWidget* parent, const char* name, WFlags fl )
    : QMainWindow( parent, name, fl ), config(Config::instance())
{
	setName( "IHU" );
	setIcon( QPixmap::fromMimeSource( "ihu.png" ) );
	setCentralWidget( new QWidget( this, "qt_central_widget" ) );
	setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)1, (QSizePolicy::SizeType)1, 0, 0, sizePolicy().hasHeightForWidth() ) );
	setMinimumSize(QSize(300, 400));
	resize( QSize(300, 400).expandedTo(minimumSizeHint()) );
	
	statusbar = statusBar();
	QWidget *trafficWidget = new QWidget( this, "trafficWidget");
	QBoxLayout *tLayout = new QHBoxLayout(trafficWidget);
	trafficLabel = new QLabel( trafficWidget, "trafficLabel" );
	tLayout->addWidget(trafficLabel);
	statusbar->addWidget( trafficWidget, 0, TRUE);
	
	lockWidget = new QWidget( this, "lockWidget");
	QBoxLayout *lLayout = new QHBoxLayout(lockWidget);
	lockPixmap = new QLabel( lockWidget, "txPixmap" );
	lockPixmap->setPixmap( QPixmap::fromMimeSource( "lock.png" ) );
	lockPixmap->setEnabled( FALSE );
	lLayout->addWidget(lockPixmap);
	statusbar->addWidget( lockWidget, 0, TRUE);
	
	QBoxLayout *ihuLayout = new QVBoxLayout( centralWidget(), 15, 15 );
	
	mainFrame = new QFrame( centralWidget(), "mainFrame" );
	mainFrame->setFrameStyle( QFrame::Panel | QFrame::Raised );
	ihuLayout->addWidget(mainFrame);
	
	QBoxLayout *mLayout = new QVBoxLayout( mainFrame, 15, 10);
	
	QBoxLayout *mhLayout = new QHBoxLayout(mLayout);
	
	hostname = new QLabel( mainFrame, "hostname" );
	QFont hostname_font(  hostname->font() );
	hostname_font.setPointSize( 12 );
	hostname_font.setBold( TRUE );
	hostname->setFont( hostname_font );
	
	mhLayout->addStretch();
	mhLayout->addWidget(hostname);
	mhLayout->addStretch();
	
	mLayout->addStretch();
	
	hostEdit = new QComboBox( TRUE, mainFrame, "hostEdit" );
	hostEdit->setMinimumSize( QSize(100, 30) );
	hostEdit->setAutoCompletion(TRUE);
	hostEdit->setDuplicatesEnabled(FALSE);
	hostEdit->setInsertionPolicy(QComboBox::NoInsertion);
	
	hostEdit->clear();
	for (int i=0; i<MAX_COUNT; i++)
	{
		QString host_name = config.readEntry(QString("/ihu/host/%1").arg(i));
		if (!host_name.isEmpty())
			hostEdit->insertItem(host_name);
	}
	hostEdit->setEditText(config.readEntry("/ihu/host/default"));
	mLayout->addWidget(hostEdit);
	
	mLayout->addStretch();
	
	QBoxLayout *mhLayout1 = new QHBoxLayout(mLayout, 25);
	
	callButton = new QPushButton( mainFrame, "callButton" );
	callButton->setMinimumSize( QSize(130, 35) );
	callButton->setIconSet( QIconSet( QPixmap::fromMimeSource( "start.png" ) ) );
	mhLayout1->addWidget(callButton, 2);
	
	ringButton = new QToolButton( mainFrame, "ringButton" );
	ringButton->setFixedSize( QSize(40, 35) );
	ringButton->setToggleButton(TRUE);
	ringButton->setIconSet( QIconSet( QPixmap::fromMimeSource( "bell.png" ) ) );
	ringButton->setEnabled(FALSE);
	mhLayout1->addWidget(ringButton, 1);
	
	stopButton = new QPushButton( mainFrame, "stopButton" );
	stopButton->setMinimumSize( QSize(130, 35) );
	stopButton->setEnabled( FALSE );
	stopButton->setIconSet( QIconSet( QPixmap::fromMimeSource( "stop.png" ) ) );
	mhLayout1->addWidget(stopButton, 2);
	
	mLayout->addStretch();
	
	QBoxLayout *mhLayout2 = new QHBoxLayout(mLayout);
	
	mhLayout2->addSpacing(25);
	
	txPixmap = new QLabel( mainFrame, "txPixmap" );
	txPixmap->setPixmap( QPixmap::fromMimeSource( "red.png" ) );
	txPixmap->setEnabled(FALSE);
	mhLayout2->addWidget(txPixmap);
	
	tx = new QLabel( mainFrame, "tx" );
	tx->setEnabled(FALSE);
	mhLayout2->addWidget(tx);
	
	mhLayout2->addSpacing(5);
	
	waitButton = new QPushButton( mainFrame, "waitButton" );
	waitButton->setMinimumSize( QSize(160, 40) );
	waitButton->setToggleButton(TRUE);
	waitButton->setIconSet( QIconSet( QPixmap::fromMimeSource( "receive_no.png" ) ) );
	mhLayout2->addWidget(waitButton, 1);
	
	mhLayout2->addSpacing(5);
	
	rx = new QLabel( mainFrame, "rx" );
	rx->setEnabled(FALSE);
	mhLayout2->addWidget(rx);
	
	rxPixmap = new QLabel( mainFrame, "rxPixmap" );
	rxPixmap->setPixmap( QPixmap::fromMimeSource( "green.png" ) );
	rxPixmap->setEnabled(FALSE);
	mhLayout2->addWidget(rxPixmap);
	
	mhLayout2->addSpacing(25);
	
	otherFrame = new QFrame( centralWidget(), "otherFrame" );
	otherFrame->setFrameStyle( QFrame::Box | QFrame::Sunken );
	ihuLayout->addWidget(otherFrame);
	
	QBoxLayout *oLayout = new QVBoxLayout( otherFrame, 15, 15);
	
	QGridLayout *gLayout = new QGridLayout( oLayout, 2, 2, 10);
	
	threshold = new QLabel( otherFrame, "threshold" );
	
	thSlider = new QSlider( otherFrame, "thSlider" );
	thSlider->setTracking( TRUE );
	thSlider->setOrientation( QSlider::Horizontal );
	thSlider->setMinValue(0);
	thSlider->setMaxValue(100);
	thSlider->setValue(config.readNumEntry("/ihu/sound/threshold"));
	
	agcLabel = new QLabel( otherFrame, "agcLabel" );
	agcLabel->setEnabled(FALSE);
	
	agcSlider = new QSlider( otherFrame, "agcSlider" );
	agcSlider->setTracking(TRUE);
	agcSlider->setOrientation( QSlider::Horizontal );
	agcSlider->setMinValue(0);
	agcSlider->setMaxValue(100);
	agcSlider->setValue(config.readNumEntry("/ihu/sound/agclevel"));
	agcSlider->setEnabled(FALSE);
	
	gLayout->addWidget(threshold, 0, 0, Qt::AlignBottom | Qt::AlignHCenter);
	gLayout->addWidget(thSlider, 1, 0, Qt::AlignTop);
	gLayout->addWidget(agcLabel, 0, 1, Qt::AlignBottom | Qt::AlignHCenter);
	gLayout->addWidget(agcSlider, 1, 1, Qt::AlignTop);
	
	oLayout->addStretch();
	
	QBoxLayout *ohLayout = new QHBoxLayout( oLayout, 30 );
	
	ohLayout->addSpacing(20);
	
	muteMicButton = new QToolButton( otherFrame, "muteMicButton" );
	muteMicButton->setMaximumSize( QSize(50, 40) );
	muteMicButton->setToggleButton(TRUE);
	muteMicButton->setIconSet( QIconSet( QPixmap::fromMimeSource( "mic.png" ) ) );
	ohLayout->addWidget(muteMicButton);
	
	QBoxLayout *ovLayout = new QVBoxLayout( ohLayout, 10 );
	
	settingsButton = new QPushButton( otherFrame, "settingsButton" );
	settingsButton->setIconSet( QIconSet( QPixmap::fromMimeSource( "configure.png" ) ) );
	settingsButton->setFlat(TRUE);
	ovLayout->addWidget(settingsButton);
	
	logButton = new QPushButton( otherFrame, "logButton" );
	logButton->setFlat(TRUE);
	ovLayout->addWidget(logButton);
	
	muteSpkButton = new QToolButton( otherFrame, "muteSpkButton" );
	muteSpkButton->setMaximumSize( QSize(50, 40) );
	muteSpkButton->setToggleButton(TRUE);
	muteSpkButton->setIconSet( QIconSet( QPixmap::fromMimeSource( "speaker.png" ) ) );
	ohLayout->addWidget(muteSpkButton);
	
	ohLayout->addSpacing(20);
	
	// actions
	fileQuitAction = new QAction( this, "fileQuitAction" );
	helpContentsAction = new QAction( this, "helpContentsAction" );
	helpAboutAction = new QAction( this, "helpAboutAction" );
	settingsAction = new QAction( this, "settingsAction" );
	settingsAction->setIconSet( QIconSet( QPixmap::fromMimeSource( "configure.png" ) ) );
	filePlayFileAction = new QAction( this, "filePlayFileAction" );
	filePlayFileAction->setIconSet( QIconSet( QPixmap::fromMimeSource( "open.png" ) ) );
	fileConvertFileAction = new QAction( this, "fileConvertFileAction" );
	fileConvertFileAction->setIconSet( QIconSet( QPixmap::fromMimeSource( "open.png" ) ) );
	cryptAction = new QAction( this, "cryptAction" );
	cryptAction->setToggleAction(TRUE);
	changeKeyAction = new QAction( this, "changeKeyAction" );
	setDecKeyAction = new QAction( this, "setDecKeyAction" );
	adrAction = new QAction( this, "adrAction" );
	adrAction->setToggleAction(TRUE);
	agcAction = new QAction( this, "agcAction" );
	agcAction->setToggleAction(TRUE);
	changeKeyAction->setEnabled(FALSE);
	
	// menubar
	menubar = new QMenuBar( this, "menubar" );
	
	fileMenu = new QPopupMenu( this );
	
	filePlayFileAction->addTo( fileMenu );
	fileConvertFileAction->addTo( fileMenu );
	fileMenu->insertSeparator();
	fileQuitAction->addTo( fileMenu );
	menubar->insertItem( "", fileMenu, 0 );
	
	optionsMenu = new QPopupMenu( this );
	cryptAction->addTo( optionsMenu );
	changeKeyAction->addTo( optionsMenu );
	setDecKeyAction->addTo( optionsMenu );
	optionsMenu->insertSeparator();
	adrAction->addTo( optionsMenu );
	agcAction->addTo( optionsMenu );
	optionsMenu->insertSeparator();
	settingsAction->addTo( optionsMenu );
	menubar->insertItem( "", optionsMenu, 1 );
	
	helpMenu = new QPopupMenu( this );
	
	helpContentsAction->addTo( helpMenu );
	helpMenu->insertSeparator();
	helpAboutAction->addTo( helpMenu );
	menubar->insertItem( "", helpMenu, 2 );
	
	logViewer = new LogViewer(this);
	timer = new QTimer(this);
	restartTimer = new QTimer(this);
	
	languageChange();
	
	if (config.readBoolEntry("/ihu/sound/adr"))
		adrAction->toggle();
	
	if (config.readBoolEntry("/ihu/sound/agc"))
		agcAction->toggle();
	
	//signals and slots connections
	connect( timer, SIGNAL(timeout()), this, SLOT(statistics()) );
	connect( restartTimer, SIGNAL(timeout()), this, SLOT(listen()));
	connect( fileQuitAction, SIGNAL( activated() ), this, SLOT( fileQuit() ) );
	connect( helpContentsAction, SIGNAL( activated() ), this, SLOT( helpContents() ) );
	connect( helpAboutAction, SIGNAL( activated() ), this, SLOT( helpAbout() ) );
	connect( settingsAction, SIGNAL( activated() ), this, SLOT( settings() ) );
	connect( cryptAction, SIGNAL( toggled(bool) ), this, SLOT( crypt(bool) ) );
	connect( changeKeyAction, SIGNAL( activated() ), this, SLOT( changeKey() ) );
	connect( setDecKeyAction, SIGNAL( activated() ), this, SLOT( setDecryptionKey() ) );
	connect( logButton, SIGNAL( clicked() ), this, SLOT( log() ) );
	connect( settingsButton, SIGNAL( clicked() ), this, SLOT( settings() ) );
	connect( ringButton, SIGNAL( toggled(bool) ), this, SLOT( ringOn(bool) ) );
	connect( callButton, SIGNAL( clicked() ), this, SLOT( startAll() ) );
	connect( stopButton, SIGNAL( clicked() ), this, SLOT( stop() ) );
	connect( waitButton, SIGNAL( toggled(bool) ), this, SLOT( waitCalls(bool) ) );
	connect( adrAction, SIGNAL( toggled(bool) ), this, SLOT( adrOn(bool) ) );
	connect( agcAction, SIGNAL( toggled(bool) ), this, SLOT( agcOn(bool) ) );
	connect( agcSlider, SIGNAL( valueChanged(int) ), this, SLOT( agcSliderChanged(int) ) );
	connect( thSlider, SIGNAL( valueChanged(int) ), this, SLOT( sliderChanged(int) ) );
	connect( thSlider, SIGNAL( sliderPressed() ), this, SLOT( sliderPress() ) );
	connect( thSlider, SIGNAL( sliderReleased() ), this, SLOT( sliderRelease() ) );
	connect( filePlayFileAction, SIGNAL( activated() ), this, SLOT( play() ) );
	connect( fileConvertFileAction, SIGNAL( activated() ), this, SLOT( convert() ) );
	connect( muteMicButton, SIGNAL( toggled(bool) ), this, SLOT( transmitterStatus(bool) ) );
	connect( muteSpkButton, SIGNAL( toggled(bool) ), this, SLOT( receiverStatus(bool) ) );
	connect( hostEdit->lineEdit(), SIGNAL( returnPressed() ), this, SLOT( startAll() ) );
}

/*
 *  Destroys the object and frees any allocated resources
 */
Ihu::~Ihu()
{
	// no need to delete child widgets, Qt does it all for us
	delete rsa;
	delete transmitter;
	delete receiver;
	delete player;
	delete recorder;
	delete logger;
}

/*
 *  Sets the strings of the subwidgets using the current
 *  language.
 */
void Ihu::languageChange()
{
	setCaption( tr( "IHU" ) );
	trafficLabel->setText( tr( "0.0 KB/s" ) );
	rx->setText( tr( "RX" ) );
	tx->setText( tr( "TX" ) );
	threshold->setText( QString( "TX threshold: %1 \%" ).arg( thSlider->value() ) );
	agcLabel->setText( QString( "AGC volume: %1 \%" ).arg( agcSlider->value() ) );
	hostname->setText( tr( "Receiver address" ) );
	callButton->setText( tr( "&Call" ) );
	stopButton->setText( tr( "&Hang up" ) );
	waitButton->setText( tr( "&Wait for calls" ) );
	ringButton->setText( QString::null );
	settingsButton->setText( tr( "&Settings" ) );
	logButton->setText( tr( "View &Log" ) );
	fileQuitAction->setText( tr( "Quit" ) );
	fileQuitAction->setMenuText( tr( "&Quit" ) );
	helpContentsAction->setText( tr( "Contents" ) );
	helpContentsAction->setMenuText( tr( "&Contents..." ) );
	helpContentsAction->setAccel( QString::null );
	helpContentsAction->setStatusTip( tr( "Show help contents" ) );
	helpAboutAction->setText( tr( "About" ) );
	helpAboutAction->setMenuText( tr( "&About" ) );
	helpAboutAction->setAccel( QString::null );
	helpAboutAction->setStatusTip( tr( "Show info on IHU" ) );
	filePlayFileAction->setText( tr( "Play File"  ) );
	filePlayFileAction->setMenuText( tr( "&Play file..." ) );
	filePlayFileAction->setStatusTip( tr( "Play IHU File" ) );
	fileConvertFileAction->setText( tr( "Convert .ihu file to .spx"  ) );
	fileConvertFileAction->setMenuText( tr( "&Convert .ihu file to .spx..." ) );
	fileConvertFileAction->setStatusTip( tr( "Converts .ihu files to standard Ogg speex" ) );
	settingsAction->setText( tr( "Settings" ) );
	settingsAction->setMenuText( tr( "&Settings..." ) );
	settingsAction->setStatusTip( tr( "Adjust settings" ) );
	adrAction->setText( tr( "Audio Delay Reduction" ) );
	adrAction->setMenuText( tr( "Audio Delay &Reduction (ADR)" ) );
	adrAction->setStatusTip( tr( "Activates the automatic Audio Delay Reduction (ADR)" ) );
	agcAction->setText( tr( "Automatic Gain Control" ) );
	agcAction->setMenuText( tr( "Automatic &Gain Control (AGC)" ) );
	agcAction->setStatusTip( tr( "Activates the Automatic Gain Control (AGC)" ) );
	changeKeyAction->setText( tr( "Change encryption key..." ) );
	changeKeyAction->setMenuText( tr( "&Change encryption key" ) );
	changeKeyAction->setStatusTip( tr( "Change the encryption key" ) );
	setDecKeyAction->setText( tr( "Set/reset decryption passphrase..." ) );
	setDecKeyAction->setMenuText( tr( "Set/reset &decryption key" ) );
	setDecKeyAction->setStatusTip( tr( "Set/reset decryption key" ) );
	cryptAction->setText( tr( "Encrypt outgoing stream" ) );
	cryptAction->setMenuText( tr( "&Encrypt outgoing stream" ) );
	cryptAction->setStatusTip( tr( "Encrypt the outgoing stream" ) );
	
	muteMicButton->setAccel( QKeySequence( tr( "Alt+I" ) ) );
	muteSpkButton->setAccel( QKeySequence( tr( "Alt+O" ) ) );
	ringButton->setAccel( QKeySequence( tr( "Alt+B" ) ) );
	fileQuitAction->setAccel( QKeySequence( tr( "Ctrl+Q" ) ) );
	settingsAction->setAccel( QKeySequence( tr( "Ctrl+S" ) ) );
	filePlayFileAction->setAccel( QKeySequence( tr( "Ctrl+F" ) ) );
	fileConvertFileAction->setAccel( QKeySequence( tr( "Ctrl+T" ) ) );
	adrAction->setAccel( QKeySequence( tr( "Ctrl+R" ) ) );
	agcAction->setAccel( QKeySequence( tr( "Ctrl+G" ) ) );
	setDecKeyAction->setAccel( QKeySequence( tr( "Ctrl+D" ) ) );
	changeKeyAction->setAccel( QKeySequence( tr( "Ctrl+K" ) ) );
	cryptAction->setAccel( QKeySequence( tr( "Ctrl+E" ) ) );
	
	QToolTip::add( hostEdit, tr( "Write the address (IP or DNS hostname) of the computer that you want to call.\nThen press Enter or click the Call button to start the communication." ) );
	QToolTip::add( agcSlider, tr( "AGC volume level" ) );
	QToolTip::add( thSlider, tr( "Threshold value: set up to a value that\nTX led is off when you are not speaking" ) );
	QToolTip::add( tx, tr( "This led is on when transmitting data" ) );
	QToolTip::add( rx, tr( "This led is on when receiving data" ) );
	QToolTip::add( trafficLabel, tr( "Total network traffic" ) );
	QToolTip::add( lockPixmap, tr( "Encryption status" ) );
	QToolTip::add( muteMicButton, tr( "Disable audio input" ) );
	QToolTip::add( muteSpkButton, tr( "Disable audio output" ) );
	QToolTip::add( callButton, tr( "Start the communication" ) );
	QToolTip::add( stopButton, tr( "Hang up" ) );
	QToolTip::add( waitButton, tr( "Enable this button to receive calls" ) );
	QToolTip::add( ringButton, tr( "Ring!" ) );
	QToolTip::add( logButton, tr( "Open the Log Viewer" ) );
	QToolTip::add( settingsButton, tr( "Open the Settings dialog to configure IHU" ) );
	
	menubar->findItem( 0 )->setText( tr( "&File" ) );
	menubar->findItem( 1 )->setText( tr( "O&ptions" ) );
	menubar->findItem( 2 )->setText( tr( "H&elp" ) );
}

void Ihu::initIhu(bool file)
{
	show();
	
	try
	{
		fromFile = file;
		ihu2spx = false;
		sliderFree = true;
		listening = false;
		received = false;
		tocrypt = false;
		closing = false;
		
		trayIcon = NULL;
		
		rsa = new Rsa(RSA_STRENGTH);
	
		player = new Player();
		receiver = new Receiver(player, rsa);
	
		recorder = new Recorder();
		transmitter = new Transmitter(recorder, rsa);
	
		logger = new Logger();
	
		connect( receiver, SIGNAL(newSocket(int,int,struct sockaddr_in)), this, SLOT(newConnection(int,int,struct sockaddr_in)) );
		connect( receiver, SIGNAL(keyRequest()), this, SLOT(sendKeyRequest()) );
		connect( receiver, SIGNAL(sendNewKey()), this, SLOT(sendKey()) );
		connect( receiver, SIGNAL(newKey(QString)), this, SLOT(receivedNewKey(QString)) );
		connect( receiver, SIGNAL(finish()), this, SLOT(stopSignal()) );
		connect( receiver, SIGNAL(message(QString)), this, SLOT(message(QString)) );
		connect( receiver, SIGNAL(warning(QString)), this, SLOT(warning(QString)) );
		connect( receiver, SIGNAL(error(QString)), this, SLOT(abortAll(QString)) );
		connect( receiver, SIGNAL(fileProgress(int)), this, SLOT(changeProgress(int)) );
		connect( receiver, SIGNAL(ledEnable(bool)), this, SLOT(rxLedEnable(bool)) );
		connect( receiver, SIGNAL(ringReply()), transmitter, SLOT(sendRingReplyPacket()) );
		connect( transmitter, SIGNAL(finish()), this, SLOT(stopSignal()) );
		connect( transmitter, SIGNAL(message(QString)), this, SLOT(message(QString)) );
		connect( transmitter, SIGNAL(ringMessage()), this, SLOT(ringMessage()) );
		connect( transmitter, SIGNAL(ledEnable(bool)), this, SLOT(txLedEnable(bool)) );
		connect( transmitter, SIGNAL(error(QString)), this, SLOT(abortAll(QString)) );	
		connect( player, SIGNAL(warning(QString)), this, SLOT(showWarning(QString)) );	
		connect( recorder, SIGNAL(warning(QString)), this, SLOT(showWarning(QString)) );	
	
		applySettings();
	}
	catch (Error e)
	{
		showMessageCritical(e.getText());
		qWarning("Program abort due to a critical error: " + e.getText());
		exit(1);
	}
	
	if (config.readBoolEntry("/ihu/general/autohide"))
		hide();
	
	if (config.readBoolEntry("/ihu/security/crypt"))
		cryptAction->toggle();

	if (!fromFile && config.readBoolEntry("/ihu/general/autowait"))
		waitForCalls();
}

void Ihu::applySettings()
{
	int srate, quality, vbr, abr, complexity, vad, dtx, txstop;
	float vbrquality;

	transmitter->setName(config.readEntry("/ihu/general/name"));

	if (config.readBoolEntry("/ihu/general/tray"))
	{
		if (trayIcon == NULL)
		{
			trayIcon = new TrayIcon( this, "trayIcon", QPixmap::fromMimeSource( "ihu_tray.png" ), "IHU" );
			trayIcon->show();
			connect( trayIcon, SIGNAL( clicked() ), this, SLOT( toggleVisibility() ) );
			connect( trayIcon, SIGNAL( contextMenuRequested( const QPoint& ) ), this, SLOT( trayMenuRequested( const QPoint& ) ) );
		}
	}
	else
	{
		if (trayIcon)
		{
			trayIcon->close();
			delete trayIcon;
			trayIcon = NULL;
		}
	}
	
	switch (config.readNumEntry("/ihu/speex/mode"))
	{
		case 0:
				srate = 8000;
				break;
		case 1:
				srate = 16000;
				break;
		case 2:
				srate = 32000;
				break;
		default:
				srate = 8000;
				break;
	}
	quality = config.readNumEntry("/ihu/speex/quality");
	abr = config.readNumEntry("/ihu/speex/abr")*1000;
	vbr = config.readNumEntry("/ihu/speex/bitratemode");
	switch (vbr)
	{
		case 0:
				abr = 0;
				break;
		case 1:
				abr = 0;
				break;
		case 2:
				vbr = 0;
				break;
	}
	vbrquality = (float) config.readNumEntry("/ihu/speex/vbrquality");
	complexity = config.readNumEntry("/ihu/speex/complexity");
	vad = config.readBoolEntry("/ihu/speex/vad");
	dtx = config.readBoolEntry("/ihu/speex/dtx");
	txstop = config.readNumEntry("/ihu/sound/txstop");
	
	int th = 0;
	if (dtx)
	{
		thSlider->setEnabled(TRUE);
		th = config.readNumEntry("/ihu/sound/threshold");
	}
	else 
		thSlider->setEnabled(FALSE);
	transmitter->setThreshold(th);
	transmitter->setup(srate, quality, abr, vbr, vbrquality, complexity, vad, dtx, txstop);

	recorder->setup(config.readNumEntry("/ihu/sound/inputdriver"), config.readEntry("/ihu/sound/inputinterface"));
	player->setup(config.readNumEntry("/ihu/sound/outputdriver"), config.readEntry("/ihu/sound/outputinterface"), 
			config.readNumEntry("/ihu/sound/ringvolume"), config.readNumEntry("/ihu/sound/prepackets"));
	
	if (fromFile || config.readNumEntry("/ihu/sound/outputdriver"))
		adrAction->setEnabled(FALSE);
	else
	{
		adrAction->setEnabled(TRUE);
		adrRefresh(config.readBoolEntry("/ihu/sound/adr"));
	}
	
	agcRefresh(config.readBoolEntry("/ihu/sound/agc"));
	
	QString nname;
	
	try
	{
		nname = QString("/ihu/security/logfile");
		logger->enable(config.readEntry(nname));
		nname = QString("/ihu/general/dumpout");
		transmitter->dump(config.readEntry(nname));
		nname = QString("/ihu/general/dumpin");
		receiver->dump(config.readEntry(nname));
	}
	catch (Error e)
	{
		showMessageCritical(e.getText());
		config.writeEntry(nname, "");
	}

}

void Ihu::txLedEnable(bool on)
{
	txPixmap->setEnabled(on);
}

void Ihu::rxLedEnable(bool on)
{
	rxPixmap->setEnabled(on);
}

void Ihu::startAll()
{
	if (fromFile)
		receiver->swap();
	else
	{
		skipStat = 0;
		seconds = 0;
		if (received)
			answer();
		else
			call(hostEdit->currentText());
	}
}

void Ihu::answer()
{
	try
	{
		callButton->setEnabled(FALSE);
		stopButton->setEnabled(TRUE);
		stopButton->setText("&Hang up");
		hostEdit->setEnabled(FALSE);
		hostname->setEnabled(FALSE);
		filePlayFileAction->setEnabled(FALSE);
		fileConvertFileAction->setEnabled(FALSE);
		ringButton->setEnabled(TRUE);
		waitButton->setEnabled(FALSE);
		received = false;
		warning(QString("Connected with %1 (%2).").arg(receiver->getCallerName()).arg(receiver->getIp()));
		receiver->resetCalls();
		transmitter->answer();
		timer->start(STAT_TIME, false);
		changeTrayIcon(IHU_ICON_TALK);
	}
	catch (Error e)
	{
		abortAll(e.getText());
	}
}

void Ihu::call(QString host)
{
	try
	{
		if (host.isEmpty())
			throw Error(tr("No host specified!"));
		callButton->setEnabled(FALSE);
		stopButton->setEnabled(TRUE);
		stopButton->setText("&Hang up");
		hostEdit->setEnabled(FALSE);
		hostname->setEnabled(FALSE);
		filePlayFileAction->setEnabled(FALSE);
		fileConvertFileAction->setEnabled(FALSE);
		ringButton->setEnabled(TRUE);
		waitButton->setEnabled(FALSE);
		tx->setEnabled(TRUE);
		rx->setEnabled(TRUE);
		restartTimer->stop();
		if (listening)
		{
			listening = false;
			receiver->end();
		}
		int callPort = config.readNumEntry("/ihu/net/outport");
		QString callHost = host;
		int tmpInd = host.findRev(':');
		if (tmpInd > 0)
		{
			callHost = host.left(tmpInd);
			callPort = host.right(host.length() - tmpInd - 1).toInt();
		}
		int sd = transmitter->call(callHost, callPort, config.readNumEntry("/ihu/net/protocol"));
		receiver->start(sd, config.readNumEntry("/ihu/net/protocol"));
		logViewer->addLog(logger->logOutgoingCall(host, transmitter->getIp()));
		ringButton->toggle();
		receiver->waitConnection();
		if (hostEdit->listBox()->findItem(host) == 0)
			hostEdit->insertItem(host, 0);
		changeTrayIcon(IHU_ICON_TALK);
	}
	catch (Error e)
	{
		abortAll(e.getText());
	}
}

void Ihu::waitCalls(bool on)
{
	if (on)
	{
		waitButton->setIconSet( QIconSet( QPixmap::fromMimeSource( "receive.png" ) ) );
		waitButton->setText( tr( "&Waiting for calls" ) );
		receiver->resetCalls();
		listen();
	}
	else
	{
		restartTimer->stop();
		filePlayFileAction->setEnabled(TRUE);
		fileConvertFileAction->setEnabled(TRUE);
		rxLedEnable(FALSE);
		waitButton->setIconSet( QIconSet( QPixmap::fromMimeSource( "receive_no.png" ) ) );
		waitButton->setText( tr( "&Wait for calls" ) );
		if (trayIcon)
		{
			changeTrayIcon(IHU_ICON_NORMAL);
			QToolTip::add( trayIcon, QString("IHU"));
		}
		logViewer->addLog(logger->logStopReceiver(receiver->getCalls(), receiver->getConnections()));
		message(tr("Receiver stopped."));
		receiver->end();
		listening = false;
	}
}

void Ihu::listen()
{
	try {
		filePlayFileAction->setEnabled(FALSE);
		fileConvertFileAction->setEnabled(FALSE);
		if (trayIcon)
		{
			changeTrayIcon(IHU_ICON_WAIT);
			QToolTip::add( trayIcon, QString("IHU - %1 calls").arg(receiver->getConnections()));
		}
		message("Starting Receiver...");
		listening = true;
		if (receiver->getConnections() > 0)
		{
			changeTrayIcon(IHU_ICON_MISSED);
			rxLedEnable(TRUE);
		}
		receiver->listen(config.readNumEntry("/ihu/net/inport"), config.readBoolEntry("/ihu/net/udp"), config.readBoolEntry("/ihu/net/tcp"));
	}
	catch (Error e)
	{
		abortAll(e.getText());
	}
}

void Ihu::newConnection(int sd, int protocol, struct sockaddr_in sa)
{
	try
	{
		if (listening)
		{
			callButton->setText( tr("&Answer") );
			callButton->setEnabled(TRUE);
			stopButton->setText( tr("&Refuse") );
			stopButton->setEnabled(TRUE);
			waitButton->setEnabled(FALSE);
			tx->setEnabled(TRUE);
			rx->setEnabled(TRUE);
			received = true;
			transmitter->newConnection(sd, sa, protocol);
			changeTrayIcon(IHU_ICON_ALARM);
			if (config.readBoolEntry("/ihu/general/answer"))
				answer();
		}
		else
		{
			if (ringButton->isOn())
				ringButton->toggle();
			timer->start(STAT_TIME, false);
		}
	}
	catch (Error e)
	{
		abortAll(e.getText());
	}
}

void Ihu::stopSignal()
{
	if (fromFile)
	{
		statusbar->message(tr("End of file"));
	}
	else
	{
		QString text = QString("%1 ").arg(receiver->getCallerName());
		if (receiver->refused())
			text += tr("refused the call.");
		else
		if (receiver->getTotal() > 0)
			text += tr("closed the call.");
		else
			text = QString("%1 rejected the call.").arg(receiver->getIp());
		logViewer->addLog(logger->logStop(receiver->getTotal(), transmitter->getTotal()));
		warning(text);
	}
	stopAll();
}

void Ihu::stop()
{
	if (received)
		transmitter->sendRefusePacket();
	else
		transmitter->sendClosePacket();
	if (fromFile)
	{
		message("Stopped.");
	}
	else
	{
		message("Communication closed.");
		logViewer->addLog(logger->logStop(receiver->getTotal(), transmitter->getTotal()));
	}
	stopAll();
}

void Ihu::stopAll()
{
	raise();
	try
	{
		callButton->setEnabled(TRUE);
		callButton->setText("&Call");
		stopButton->setEnabled(FALSE);
		stopButton->setText("&Hang up");
		hostEdit->setEnabled(TRUE);
		hostname->setEnabled(TRUE);
		ringButton->setEnabled(FALSE);
		waitButton->setEnabled(TRUE);
		tx->setEnabled(FALSE);
		rx->setEnabled(FALSE);
		trafficLabel->setText( tr( "0.0 KB/s" ) );
		QToolTip::add( txPixmap, QString::null);
		QToolTip::add( rxPixmap, QString::null);
		if (ringButton->isOn())
			ringButton->toggle();
		timer->stop();
		fileName = QString::null;
		receiver->end();
		transmitter->end();
		if (fromFile)
		{
			fromFile = false;
			adrAction->setEnabled(TRUE);
			adrRefresh(config.readBoolEntry("/ihu/sound/adr"));
			thSlider->setValue(config.readNumEntry("/ihu/sound/threshold"));
			threshold->setText( QString( "TX threshold: %1 \%" ).arg( thSlider->value() ) );
		}
		ihu2spx = false;
		received = false;
		sliderFree = true;
		rxLedEnable(FALSE);
		txLedEnable(FALSE);
		changeTrayIcon(IHU_ICON_NORMAL);
		if (trayIcon)
			QToolTip::add( trayIcon, "IHU" );
		if (waitButton->isOn())
			restartTimer->start(RESTART_TIME, true);
		else
		{
			filePlayFileAction->setEnabled(TRUE);
			fileConvertFileAction->setEnabled(TRUE);
		}
	}
	catch (Error e)
	{
		QMessageBox::critical(0, "IHU Error", tr("Error: ") + e.getText());
	}
}

void Ihu::play()
{
        QString name=QFileDialog::getOpenFileName("","*"IHU_EXT,this,0,"Open IHU file to play...");
	playFile(name);
}

void Ihu::convert()
{
        QString name=QFileDialog::getOpenFileName("","*"IHU_EXT,this,0,"Open IHU file to convert...");
	ihu2spx = true;
	playFile(name);
}

void Ihu::playFile(QString name)
{
	if (!name.isEmpty())
	{
		try
		{
			sliderFree = true;
			stopButton->setEnabled(TRUE);
			waitButton->setEnabled(FALSE);
			filePlayFileAction->setEnabled(FALSE);
			fileConvertFileAction->setEnabled(FALSE);
			adrAction->setEnabled(FALSE);
			hostEdit->setEnabled(FALSE);
			hostname->setEnabled(FALSE);
			callButton->setText( tr("&Pause") );
			stopButton->setText( tr("S&top") );
			fileName = name;
			if (ihu2spx)
				message("Converting " + fileName + "...") ;
			else
				message("Playing " + fileName + "...") ;
			threshold->setText(tr("File progress: 0 \%"));
			fromFile = true;
			if (ihu2spx)
				receiver->convertFile(fileName);
			else
				receiver->playFile(fileName);
			timer->start(STAT_TIME, false);
		}
		catch (Error e)
		{
			abortAll(e.getText());
		}
	}
}

void Ihu::abortAll(QString message)
{
	if (waitButton->isOn())
		waitButton->toggle();
	stopAll();
	showMessageCritical(message);
}

void Ihu::showMessageCritical(QString text)
{
	message("Aborted");
	QMessageBox::critical(0, "IHU Error", tr("Error: ") + text);
}

void Ihu::showWarning(QString text)
{
	message("WARNING!");
	QMessageBox::warning(0, "IHU Warning", tr("Warning: ") + text);
}

void Ihu::closeEvent(QCloseEvent *e)
{
	if (!trayIcon || closing)
	{
		e->accept();
	}
	else
	{
		e->ignore();
		hide();
	}
}

void Ihu::fileQuit()
{
	stopAll();
	closing = true;
	config.writeEntry("/ihu/host/default", hostEdit->currentText());
	for (int i=0; (i<hostEdit->count()) && (i<MAX_COUNT); i++)
	{
		config.writeEntry(QString("/ihu/host/%1").arg(i), hostEdit->text(i));
	}
	close();
}

void Ihu::message(QString text)
{
	skipStat = 1;
	statusbar->message(text, STAT_TIME*3);
}

void Ihu::warning(QString text)
{
	skipStat = 2;
	statusbar->message(text);
	if (trayIcon)
		QToolTip::add( trayIcon, text );
	logViewer->addLog(logger->log(text));
	raise();
}

void Ihu::log()
{
	logViewer->show();
}

void Ihu::settings()
{
	Settings *settings = new Settings(this);
	int ret = settings->exec();
	if (ret == QDialog::Accepted)
	{
		try
		{
			applySettings();
		}
		catch (Error e)
		{
			abortAll(e.getText());
		}
	}
	delete settings;
}

void Ihu::ringOn(bool on)
{
	transmitter->ring(on);
}

void Ihu::cryptOn()
{
	if (!cryptAction->isOn())
		cryptAction->toggle();
}

void Ihu::crypt(bool on)
{
	config.writeEntry("/ihu/security/crypt", on);
	if (on)
	{
		srand(time(NULL));
		if (changeKey())
		{
			changeKeyAction->setEnabled(TRUE);
			lockPixmap->setEnabled(TRUE);
			message(tr("Encryption enabled."));
		}
		else
			cryptAction->toggle();
	}
	else
	{
		lockPixmap->setEnabled(FALSE);
		message(tr("Encryption disabled."));
		transmitter->disableCrypt();
		changeKeyAction->setEnabled(FALSE);
	}
}

bool Ihu::changeKey()
{
	bool ok = false;
	try
	{
		if (config.readBoolEntry("/ihu/security/random"))
		{
			int i, len = config.readNumEntry("/ihu/security/keylen")/8;
			char key[len];
			for (i=0; i<len; i++)
				key[i] = (char)((rand()%256)-128);
			transmitter->enableCrypt(key, len);
			ok = true;
		}
		else
		{
			QString text = QInputDialog::getText("Encryption passphrase change", "Enter the encryption passphrase (max 56 chars, please use at least 30 chars):", QLineEdit::Password, QString::null, &ok, this );
			if (ok)
			{
				ok = false;
				if (!text.isEmpty())
				{
					transmitter->enableCrypt((char *) text.ascii(), text.length());
					ok = true;
				}
			}
		}
		if (ok)
			message(tr("Encryption key changed successfully."));
	}
	catch (Error e)
	{
		abortAll(e.getText());
	}
	return ok;
}

void Ihu::setDecryptionKey()
{
	bool ok;
	QString text = QInputDialog::getText("Decryption Passphrase", "Enter the decryption passphrase, leave blank to reset.", QLineEdit::Password, QString::null, &ok, this );
	if (ok)
	{
		receiver->disableDecrypt();
		if (!text.isEmpty())
			receiver->enableDecrypt((char *) text.ascii(), text.length());
		message(tr("Decryption key changed successfully."));
	}
}

void Ihu::sendKey()
{
	if (transmitter->isWorking())
		transmitter->sendKeyPacket();
}

void Ihu::sendKeyRequest()
{
	if (transmitter->isWorking())
		transmitter->sendKeyRequestPacket();
	else
	{
		receiver->stop();
		switch (QMessageBox::warning(0, "IHU Error", "Warning: stream is crypted but the decryption key is not available.\n\nOK if you want to enter the decryption passphrase, IGNORE if you\nwant to continue anyway, CANCEL to stop.", QMessageBox::Ok, QMessageBox::Ignore, QMessageBox::Cancel))
		{
			case QMessageBox::Ok:
				setDecryptionKey();
				receiver->go();
				break;
			case QMessageBox::Ignore:
				receiver->noDecrypt();
				receiver->go();
				break;
			default:
				stopSignal();
				break;
		}
	}
}

void Ihu::agcRefresh(bool on)
{
	float step = (float) (config.readNumEntry("/ihu/sound/agcstep")/1000.f);
	float level = (float) (config.readNumEntry("/ihu/sound/agclevel")/100.f);
	float min = (float) config.readNumEntry("/ihu/sound/agcmin");
	float max = (float) config.readNumEntry("/ihu/sound/agcmax");
	player->setAgc(on, step, level, min, max);
	config.writeEntry("/ihu/sound/agc", on);
	if (on)
	{
		agcSlider->setEnabled(TRUE);
		agcLabel->setEnabled(TRUE);
	}
	else
	{
		agcSlider->setEnabled(FALSE);
		agcLabel->setEnabled(FALSE);
	}
}

void Ihu::agcOn(bool on)
{
	agcRefresh(on);
	if (on)
		message(tr("Automatic Gain Control (AGC) enabled."));
	else
		message(tr("Automatic Gain Control (AGC) disabled."));
}

void Ihu::adrRefresh(bool on)
{
	float mindelay = (float) (config.readNumEntry("/ihu/sound/adrmindelay")/1000.f);
	float maxdelay = (float) (config.readNumEntry("/ihu/sound/adrmaxdelay")/1000.f);
	float stretch = (float) config.readNumEntry("/ihu/sound/adrstretch");
	player->setAdr(on, mindelay, maxdelay, stretch);
	config.writeEntry("/ihu/sound/adr", on);
}

void Ihu::adrOn(bool on)
{
	adrRefresh(on);
	if (on)
		message(tr("Audio Delay Reduction (ADR) enabled."));
	else
		message(tr("Audio Delay Reduction (ADR) disabled."));
}

void Ihu::transmitterStatus(bool on)
{
	try
	{
		if (on)
		{
			message("Audio input mute.");
			transmitter->changeStatus(Transmitter::TRANSMITTER_STATUS_MUTE);
			muteMicButton->setIconSet( QIconSet( QPixmap::fromMimeSource( "mic_mute.png" ) ) );
		}
		else 
		{
			message("Audio input enabled.");
			transmitter->changeStatus(Transmitter::TRANSMITTER_STATUS_WAITING);
			muteMicButton->setIconSet( QIconSet( QPixmap::fromMimeSource( "mic.png" ) ) );
		}
	}
	catch (Error e)
	{
		abortAll(e.getText());
	}
}

void Ihu::receiverStatus(bool on)
{
	try
	{
		if (on)
		{
			message("Audio output mute.");
			receiver->changeStatus(Receiver::RECEIVER_STATUS_MUTE);
			muteSpkButton->setIconSet( QIconSet( QPixmap::fromMimeSource( "speaker_mute.png" ) ) );
		}
		else 
		{
			message("Audio output enabled.");
			receiver->changeStatus(Receiver::RECEIVER_STATUS_NORMAL);
			muteSpkButton->setIconSet( QIconSet( QPixmap::fromMimeSource( "speaker.png" ) ) );
		}
	}
	catch (Error e)
	{
		abortAll(e.getText());
	}
}

void Ihu::changeProgress(int v)
{
	if (sliderFree)
		thSlider->setValue(v);
}

void Ihu::sliderChanged(int v)
{
	if (fromFile)
	{
		threshold->setText(QString("File progress: %1 \%").arg( v ));
	}
	else
	{
		transmitter->setThreshold(v);
		threshold->setText(QString("TX threshold: %1 \%").arg( v ));
		config.writeEntry("/ihu/sound/threshold", v);
	}
}

void Ihu::sliderPress()
{
	if (fromFile)
	{
		receiver->stop();
		sliderFree = false;
	}
}

void Ihu::sliderRelease()
{
	if (fromFile)
	{
		receiver->seekFile(thSlider->value());
		receiver->go();
		sliderFree = true;
	}
}

void Ihu::agcSliderChanged(int v)
{
	bool agc = config.readBoolEntry("/ihu/sound/agc");
	config.writeEntry("/ihu/sound/agclevel", v);
	agcLabel->setText(QString("AGC volume: %1 \%").arg( v ));
	agcRefresh(agc);
}

void Ihu::statistics()
{
	if (fromFile)
	{
		if (ihu2spx)
			message("Converting " + fileName + "...") ;
		else
			message(QString("Playing %1 (%2)").arg(fileName).arg(receiver->getCallerName()));
	}
	else
	{
		seconds++;
		float tot = (float) STAT_TIME;
		float tx_kb = (float) transmitter->getBytes()/tot;
		float rx_kb = (float) receiver->getBytes()/tot;
		QToolTip::add( txPixmap, QString( "%1 KB/s" ).arg(tx_kb, 2, 'f', 1 ) );
		QToolTip::add( rxPixmap, QString( "%1 KB/s" ).arg(rx_kb, 2, 'f', 1 ) );
		tot = rx_kb + tx_kb;
		QString statName = QString("%1").arg(receiver->getCallerName());
		QString statTime;
		statTime.sprintf("%02d:%02d", (int)(seconds/60), (int)(seconds%60));
		QString statTraffic = QString("%1 KB/s").arg(tot, 2, 'f', 1 );
		trafficLabel->setText(statTraffic);
		if (skipStat > 0)
			skipStat--;
		else
			statusbar->message(QString ("%1 - %2").arg(statName).arg(statTime));
		if (trayIcon)
			QToolTip::add( trayIcon, QString ("%1 - %2 - %3").arg(statName).arg(statTime).arg(statTraffic) );
		
		if (tocrypt)
		{
			cryptOn();
			tocrypt = false;
		}
	}
}

void Ihu::helpContents()
{
	QMessageBox::information( this, "Help on IHU",
	"Please read the manual inside the IHU package, or visit:\n\n"
	"http://ihu.sourceforge.net/doc/manual.html");
}

void Ihu::disableIn()
{
	muteMicButton->toggle();
}

void Ihu::disableOut()
{
	muteSpkButton->toggle();
}

void Ihu::waitForCalls()
{
	if (!waitButton->isOn())
		waitButton->toggle();
}

void Ihu::ringMessage()
{
	QString text = QString("Ringing %1").arg(transmitter->getIp());
	if (receiver->replied())
		text += QString(" (%1)").arg(receiver->getCallerName());
	else
		text += QString(" (waiting for reply...)");
	warning(text);
}

void Ihu::toggleVisibility()
{
	if (isVisible())
		hide();
	else
		show();
}

void Ihu::trayMenuRequested( const QPoint& pos )
{
	QPopupMenu* trayMenu = new QPopupMenu(this);
	trayMenu->setCheckable(TRUE);
	trayMenu->insertItem( isVisible() ? "Hide IHU" : "Show IHU" , this, SLOT( toggleVisibility() ), 0, 0 );
	trayMenu->insertSeparator();
	trayMenu->insertItem( QIconSet( QPixmap::fromMimeSource( "start.png" ) ), callButton->text() , this, SLOT( startAll() ), 0, 1 );
	trayMenu->setItemEnabled(1, callButton->isEnabled());
	trayMenu->insertItem( QIconSet( QPixmap::fromMimeSource( "stop.png" ) ), stopButton->text() , this, SLOT( stop() ), 0, 2 );
	trayMenu->setItemEnabled(2, stopButton->isEnabled());
	if (waitButton->isOn())
		trayMenu->insertItem( QIconSet( QPixmap::fromMimeSource( "receive.png" ) ), waitButton->text() , waitButton, SLOT( toggle() ), 0, 3);
	else
		trayMenu->insertItem( QIconSet( QPixmap::fromMimeSource( "receive_no.png" ) ), waitButton->text() , waitButton, SLOT( toggle() ), 0, 3);
	trayMenu->setItemChecked(3, waitButton->isOn());
	trayMenu->setItemEnabled(3, waitButton->isEnabled());
	trayMenu->insertSeparator();
	trayMenu->insertItem( tr("&Quit"), this, SLOT( fileQuit() ) );
	trayMenu->exec(pos);
	delete trayMenu;
}

void Ihu::changeTrayIcon(icon_type icn)
{
	if (trayIcon)
	{
		trayIcon->erase();
		switch(icn)
		{
			case IHU_ICON_NORMAL:
				trayIcon->setPixmap( QPixmap::fromMimeSource( "ihu_tray.png" ) );
				break;
			case IHU_ICON_WAIT:
				trayIcon->setPixmap( QPixmap::fromMimeSource( "ihu_wait.png" ) );
				break;
			case IHU_ICON_ALARM:
				trayIcon->setPixmap( QPixmap::fromMimeSource( "ihu_alarm.png" ) );
				break;
			case IHU_ICON_TALK:
				trayIcon->setPixmap( QPixmap::fromMimeSource( "ihu_talk.png" ) );
				break;
			case IHU_ICON_MISSED:
				trayIcon->setPixmap( QPixmap::fromMimeSource( "receive.png" ) );
				break;
		}
	}
}

void Ihu::receivedNewKey(QString text)
{
	if (config.readBoolEntry("/ihu/security/showkey"))
	{
		warning(QString("New decryption key: %1").arg(text));
	}
	tocrypt = true;
}

void Ihu::helpAbout()
{
	QMessageBox::about( this, "About IHU", "I  Hear  U  v. "VERSION" ("VDATE") written by Matteo Trotta\n\n"
	"Copyright (C) 2003-2006 Matteo Trotta. All rights reserved.\n"
	"Distributed under the terms of GNU General Public License.\n\n"
	"Using Speex - Copyright (C) Jean-Marc Valin\n"
	"Using SoundTouch - Copyright (C) Olli Parviainen\n\n"
	"Visit home page at http://ihu.sourceforge.net\n"
	);
}
