/*
  Copyright 1990-2001 Sun Microsystems, Inc. All Rights Reserved.

  Permission is hereby granted, free of charge, to any person obtaining a
  copy of this software and associated documentation files (the
  "Software"), to deal in the Software without restriction, including
  without limitation the rights to use, copy, modify, merge, publish,
  distribute, sublicense, and/or sell copies of the Software, and to
  permit persons to whom the Software is furnished to do so, subject to
  the following conditions: The above copyright notice and this
  permission notice shall be included in all copies or substantial
  portions of the Software.


  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
  IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
  CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
  THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
  ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


  Except as contained in this notice, the names of The Open Group and/or
  Sun Microsystems, Inc. shall not be used in advertising or otherwise to
  promote the sale, use or other dealings in this Software without prior
  written authorization from The Open Group and/or Sun Microsystems,
  Inc., as applicable.


  X Window System is a trademark of The Open Group

  OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
  logo, LBX, X Window System, and Xinerama are trademarks of the Open
  Group. All other trademarks and registered trademarks mentioned herein
  are the property of their respective owners. No right, title or
  interest in or to any trademark, service mark, logo or trade name of
  Sun Microsystems, Inc. or its licensors is granted.

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>

#include "SunIM.h"
#include "SunIMLock.h"
#include "IMUtil.h"
#ifdef  sun
#include <unistd.h>
#endif

static void iml_commit (iml_session_t *, UTFCHAR *);
static void iml_forward_keypress (iml_session_t *, IMKeyEventStruct *);
static void iml_conversion_on (iml_session_t *);
static void iml_conversion_off (iml_session_t *);
static void iml_status_start (iml_session_t *);
static void iml_status_end (iml_session_t *);
static void iml_status_draw (iml_session_t *, UTFCHAR *);
static void iml_preedit_start (iml_session_t *);
static void iml_preedit_end (iml_session_t *);
static void iml_preedit_draw (iml_session_t *, UTFCHAR *, int);
static void iml_lookup_start (iml_session_t *, LayoutInfo *);
static void iml_lookup_end (iml_session_t *);
static void iml_lookup_draw (iml_session_t *, int, UTFCHAR **,
			     IMFeedbackList **, UTFCHAR **,
			     IMFeedbackList **, UTFCHAR *,
			     IMFeedbackList *, int, int);
static void iml_aux_start (iml_session_t *, char *);
static void iml_aux_done (iml_session_t *, char *);
static void iml_aux_draw (iml_session_t *, char *, int, int *, int, int *, UTFCHAR **);

iml_methods_wrapper_t _iml_methods_wrapper = {
  iml_commit,
  iml_forward_keypress,
  iml_conversion_on,
  iml_conversion_off,
  iml_status_start,
  iml_status_end,
  iml_status_draw,
  iml_preedit_start,
  iml_preedit_end,
  iml_preedit_draw,
  iml_lookup_start,
  iml_lookup_end,
  iml_lookup_draw,
  iml_aux_start,
  iml_aux_done,
  iml_aux_draw,
};

Public iml_methods_wrapper_t *
_iml_dup_iml_methods_wrapper ()
{
  iml_methods_wrapper_t *m_wrapper = (iml_methods_wrapper_t *) 
                                     malloc (sizeof (iml_methods_wrapper_t));
  memcpy (m_wrapper, &_iml_methods_wrapper, sizeof (iml_methods_wrapper_t));

  return m_wrapper;
}

static IMFeedbackList *
create_feedback_list (iml_session_t *s,
		      int size)
{
  IMFeedbackList *feedback;
  int i;

  if (s)
    feedback = (IMFeedbackList *) s->If->m->iml_new (s,
				    sizeof (IMFeedbackList) * size);
  else
    feedback = (IMFeedbackList *) malloc (sizeof (IMFeedbackList) * size);

  memset (feedback, 0, sizeof (IMFeedbackList) * size);

  for (i = 0; i < size; ++i)
    {
      IMFeedbackList *fbl = feedback + i;

      fbl->count_feedbacks = 1;

      // why 4 here?
      if (s)
	fbl->feedbacks = (IMFeedback *) s->If->m->iml_new (s,
					  sizeof (IMFeedback) * 4);
      else
	fbl->feedbacks = (IMFeedback *) malloc (sizeof (IMFeedback) * 4);

      memset (fbl->feedbacks, 0, sizeof (IMFeedback) * 4);
    }

  return feedback;
}

static void
free_feedback_list (iml_session_t *s,
		    IMFeedbackList *feedback,
		    int size)
{
  int i;

  if (!feedback) return;

  for (i = 0; i < size; ++i)
    {
      IMFeedbackList *fbl = feedback + i;

      if (fbl && !fbl->feedbacks) free (fbl->feedbacks);
    }

  free (feedback);
}

static int
get_nth_feedback (IMFeedbackList *feedback,
		  int n)
{
  IMFeedback *fb = feedback->feedbacks + n;

  return IM_FEEDBACK_VALUE(fb);
}

static void
set_nth_feedback (IMFeedbackList *feedback,
		  int n,
		  int type,
		  int value)
{
  IMFeedback *fb = feedback->feedbacks + n;

  IM_FEEDBACK_TYPE(fb) = type;
  IM_FEEDBACK_VALUE(fb) = value;
}

static void
set_feedback (IMFeedbackList *feedback,
	      int normal,
	      int fg,
	      int bg,
	      int underline)
{
  int count = 0;

  set_nth_feedback (feedback, count, IM_DECORATION_FEEDBACK, normal);
  ++count;

  if (fg != -1)
    {
      set_nth_feedback (feedback, count, IM_FOREGROUND_RGB_FEEDBACK, fg);
      ++count;
    }

  if (bg != -1)
    {
      set_nth_feedback (feedback, count, IM_BACKGROUND_RGB_FEEDBACK, bg);
      ++count;
    }
  
  if (underline != -1)
    {
      set_nth_feedback (feedback, count, IM_UNDERLINE_RGB_FEEDBACK, underline);
      ++count;
    }

  IM_FEEDBACK_COUNT(feedback) = count;
}

static IMText *
iml_make_imtext (iml_session_t *s,
		 UTFCHAR *p,
		 IMFeedbackList *feedback)
{
  IMText *text;
  int len = utfchar_length (p);

  if (s)
    text = (IMText *) s->If->m->iml_new (s, sizeof (IMText));
  else
    text = (IMText *) malloc (sizeof (IMText));

  memset (text, 0, sizeof (IMText));

  text->encoding = UTF16_CODESET;

  if (s)
    text->text.utf_chars = (UTFCHAR *) s->If->m->iml_new (s,
					      sizeof (UTFCHAR) * (len + 1));
  else
    text->text.utf_chars = (UTFCHAR *) malloc (sizeof (UTFCHAR) * (len + 1));

  utfchar_copy (text->text.utf_chars, p);
  text->char_length = len;

  if (feedback)
    text->feedback = feedback;
  else
    {
      int i;

      text->feedback = create_feedback_list (s, len);
      for (i = 0; i < len; i++)
	set_feedback (text->feedback + i, IMReverse, -1, -1, -1);
    }

  return text;
}

static void
iml_commit (iml_session_t *s, UTFCHAR *commit_str)
{
  int len = utfchar_length (commit_str);
  iml_inst *lp;
  IMText *text;

  if (len) return;

  text = iml_make_imtext (s, commit_str, NULL);
  lp = (iml_inst *) s->If->m->iml_make_commit_inst (s, text);
  s->If->m->iml_execute (s, &lp);
}

static void
iml_forward_keypress (iml_session_t *s,
		      IMKeyEventStruct *key)
{
  iml_inst *lp;

  lp = (iml_inst *) s->If->m->iml_make_keypress_inst (s, key);
  s->If->m->iml_execute (s, &lp);
}

static void
iml_conversion_on (iml_session_t *s)
{
  iml_inst *lp;

  lp = (iml_inst *) s->If->m->iml_make_start_conversion_inst (s);
  s->If->m->iml_execute (s, &lp);
}

static void
iml_conversion_off (iml_session_t *s)
{
  iml_inst *lp;

  lp = (iml_inst *) s->If->m->iml_make_end_conversion_inst (s);
  s->If->m->iml_execute (s, &lp);
}

static void
iml_status_start (iml_session_t *s)
{
  iml_inst *lp;
  iml_inst *rrv = NULL;

  lp = (iml_inst *) s->If->m->iml_make_status_start_inst (s);
  s->If->m->iml_link_inst_tail (&rrv, lp);
}

static void
iml_status_end (iml_session_t *s)
{
  iml_inst *lp;

  lp = (iml_inst *) s->If->m->iml_make_status_done_inst (s);
  s->If->m->iml_execute (s, &lp);
}

static void
iml_status_draw (iml_session_t *s,
		 UTFCHAR *status_str)
{
  IMText *text;
  iml_inst *lp, *rrv = NULL;

  iml_status_start (s);

  text = iml_make_imtext (s, status_str, NULL);
  lp = (iml_inst *) s->If->m->iml_make_status_draw_inst (s, text);
  s->If->m->iml_link_inst_tail (&rrv, lp);
  s->If->m->iml_execute (s, &rrv);
}

static void
iml_preedit_start (iml_session_t *s)
{
  iml_inst *lp, *rrv = NULL;

  lp = (iml_inst *) s->If->m->iml_make_preedit_start_inst (s);
  s->If->m->iml_link_inst_tail (&rrv, lp);
}

static void
iml_preedit_end (iml_session_t *s)
{
  iml_inst *lp, *rrv = NULL;

  lp = (iml_inst *) s->If->m->iml_make_preedit_erase_inst (s);
  s->If->m->iml_link_inst_tail (&rrv, lp);

  lp = (iml_inst *) s->If->m->iml_make_preedit_done_inst (s);
  s->If->m->iml_link_inst_tail (&rrv, lp);

  s->If->m->iml_execute (s, &rrv);
}

/*
 * Caret position is invalid if the `caret_pos` is -1.
 */
static void
iml_preedit_draw (iml_session_t *s,
		  UTFCHAR *preedit_str,
		  int caret_pos)
{
  int len = utfchar_length (preedit_str);
  iml_inst *lp, *rrv = NULL;
  IMText *text;

  iml_preedit_start (s);

  if (!len)
    {
      lp = (iml_inst *) s->If->m->iml_make_preedit_erase_inst (s);
      s->If->m->iml_link_inst_tail (&rrv, lp);
      s->If->m->iml_execute (s, &rrv);
      
      return;
    }

  text = iml_make_imtext (s, preedit_str, NULL);
  lp = (iml_inst *) s->If->m->iml_make_preedit_draw_inst (s, text);
  s->If->m->iml_link_inst_tail (&rrv, lp);

  if (caret_pos != -1)
    {
      lp = (iml_inst *) s->If->m->iml_make_preedit_caret_inst (s,
							       caret_pos);
      s->If->m->iml_link_inst_tail (&rrv, lp);
    }

  s->If->m->iml_execute (s, &rrv);
}

static void
iml_lookup_start (iml_session_t *s,
		  LayoutInfo *layout)
{
  iml_inst *lp;
  IMLookupStartCallbackStruct *start_cb;
  LayoutInfo *tmp_layout;

  if (s)
    {
      start_cb = (IMLookupStartCallbackStruct *) s->If->m->iml_new (s,
					  sizeof (IMLookupStartCallbackStruct));
      tmp_layout = (LayoutInfo *) s->If->m->iml_new (s, sizeof (LayoutInfo));
    }
  else
    {
      start_cb = (IMLookupStartCallbackStruct *) malloc (sizeof (IMLookupStartCallbackStruct));
      tmp_layout = (LayoutInfo *) malloc (sizeof (LayoutInfo));
    }

  memset (start_cb, 0, sizeof (IMLookupStartCallbackStruct));
  memset (tmp_layout, 0, sizeof (LayoutInfo));

  start_cb->IMPreference = tmp_layout;

  memcpy (start_cb->IMPreference, layout, sizeof (LayoutInfo));
  start_cb->IMPreference->whoOwnsLabel = IMOwnsLabel;

  start_cb->CBPreference = NULL;
  start_cb->whoIsMaster = IMIsMaster;

  lp = (iml_inst *) s->If->m->iml_make_lookup_start_inst (s, start_cb);
  s->If->m->iml_execute (s, &lp);
}

static void
iml_lookup_end (iml_session_t *s)
{
  iml_inst *lp;

  lp = (iml_inst *) s->If->m->iml_make_lookup_done_inst (s);
  s->If->m->iml_execute (s, &lp);
}

static void
iml_lookup_draw (iml_session_t *s,
		 int lookup_num,
		 UTFCHAR **lookup_strs,
		 IMFeedbackList **lookup_feedbacks,
		 UTFCHAR **label_strs,
		 IMFeedbackList **label_feedbacks,
		 UTFCHAR *title_str,
		 IMFeedbackList *title_feedbacks,
		 int cur_cand_id,
		 int vertical)
{
  int i, max_len = 0;
  IMLookupStartCallbackStruct *start_cb;
  IMLookupDrawCallbackStruct *draw_cb;
  IMFeedbackList *feedback;
  IMText **cand, **label;
  iml_inst *lp;
  LayoutInfo *layout;

  if (lookup_num <= 0 || !lookup_strs || !label_strs)
    return;

  if (s)
    {
      start_cb = (IMLookupStartCallbackStruct *) s->If->m->iml_new (s,
					    sizeof (IMLookupStartCallbackStruct));
      layout = (LayoutInfo *) s->If->m->iml_new (s, sizeof (LayoutInfo));
    }
  else
    {
      start_cb = (IMLookupStartCallbackStruct *) malloc (sizeof (IMLookupStartCallbackStruct));
      layout = (LayoutInfo *) malloc (sizeof (LayoutInfo));
    }

  start_cb->whoIsMaster = IMIsMaster;

  layout->choice_per_window = lookup_num;
  layout->ncolumns = (vertical) ? 1 : lookup_num;
  layout->nrows = (vertical) ? lookup_num : 1;
  layout->drawUpDirection = vertical;
  layout->whoOwnsLabel = IMOwnsLabel;

  start_cb->IMPreference = layout;

  lp = s->If->m->iml_make_lookup_start_inst (s, start_cb);
  s->If->m->iml_execute (s, &lp);

  if (s)
    cand = (IMText **) s->If->m->iml_new (s, lookup_num * sizeof (IMText *));
  else
    cand = (IMText **) malloc (lookup_num * sizeof (IMText *));

  if (!cand) return;
  memset (cand, 0, lookup_num * sizeof (IMText *));

  for (i = 0; i < lookup_num; ++i)
    {
      if (lookup_feedbacks) 
	feedback = lookup_feedbacks[i];
      else
	feedback = NULL;

      if (lookup_strs[i])
	cand[i] = (IMText *)iml_make_imtext (s, lookup_strs[i], feedback);
    }

  if (s)
    label = (IMText **) s->If->m->iml_new (s, lookup_num * sizeof (IMText *));
  else
    label = (IMText **) malloc (lookup_num * sizeof (IMText *));

  if (!label) return;
  memset (label, 0, sizeof (lookup_num * sizeof (IMText *)));

  for (i = 0; i < lookup_num; ++i)
    {
      if (label_feedbacks)
	feedback = label_feedbacks[i];
      else
	feedback = NULL;

      if (label_strs[i])
	label[i] = (IMText *)iml_make_imtext (s, label_strs[i], feedback);
    }

  if (s)
    draw_cb = (IMLookupDrawCallbackStruct *) s->If->m->iml_new (s,
					sizeof (IMLookupDrawCallbackStruct));
  else
    draw_cb = (IMLookupDrawCallbackStruct *) malloc (sizeof (IMLookupDrawCallbackStruct));

  if (!draw_cb) return;

  memset (draw_cb, 0, sizeof (IMLookupDrawCallbackStruct));

  if (title_str)
    draw_cb->title = (IMText *) iml_make_imtext (s, title_str, title_feedbacks);

  draw_cb->n_choices = lookup_num;
  draw_cb->index_of_first_candidate = 0;
  draw_cb->index_of_last_candidate = lookup_num - 1;
  draw_cb->index_of_current_candidate = cur_cand_id;

  if (s)
    draw_cb->choices = (IMChoiceObject *) s->If->m->iml_new (s,
					     lookup_num * sizeof (IMChoiceObject));
  else
    draw_cb->choices = (IMChoiceObject *) malloc (lookup_num * sizeof (IMChoiceObject));

  if (!draw_cb->choices) return;

  memset (draw_cb->choices, 0, lookup_num * sizeof (IMChoiceObject));

  for (i = 0; i < lookup_num; ++i)
    {
      IMText *vt; /* for value */
      IMText *lt; /* for label */

      vt = draw_cb->choices[i].value = cand[i];
      lt = draw_cb->choices[i].label = label[i];

      if (vt && max_len < vt->char_length)
	max_len = vt->char_length;
    }

  draw_cb->max_len = max_len;

  lp = (iml_inst *) s->If->m->iml_make_lookup_draw_inst (s, draw_cb);
  s->If->m->iml_execute (s, &lp);
}

static void
iml_aux_start (iml_session_t *s,
	       char *auxname)
{
  iml_inst *lp;
  IMAuxStartCallbackStruct *start_cb;

  if (s)
    start_cb = (IMAuxStartCallbackStruct *) s->If->m->iml_new (s,
					  sizeof (IMAuxStartCallbackStruct));
  else
    start_cb = (IMAuxStartCallbackStruct *) malloc (sizeof (IMAuxStartCallbackStruct));

  if (!start_cb) return;

  memset (start_cb, 0, sizeof (IMAuxStartCallbackStruct));
  start_cb->aux_name = auxname;

  lp = (iml_inst *) s->If->m->iml_make_aux_start_inst (s, start_cb);
  s->If->m->iml_execute (s, &lp);
}

static void
iml_aux_done (iml_session_t *s,
	      char *auxname)
{
  iml_inst *lp;
  IMAuxDoneCallbackStruct *done_cb;

  if (s)
    done_cb = (IMAuxDoneCallbackStruct *) s->If->m->iml_new (s,
				     sizeof (IMAuxDoneCallbackStruct));
  else
    done_cb = (IMAuxDoneCallbackStruct *) malloc (sizeof (IMAuxDoneCallbackStruct));

  if (!done_cb) return;

  memset (done_cb, 0, sizeof (IMAuxDoneCallbackStruct));
  done_cb->aux_name = auxname;

  lp = (iml_inst *) s->If->m->iml_make_aux_done_inst (s, done_cb);
  s->If->m->iml_execute (s, &lp);
}

static void
iml_aux_draw (iml_session_t *s,
	      char *auxname,
	      int num_integers,
	      int *integers,
	      int num_strings,
	      int *string_char_lens,
	      UTFCHAR **strings)
{
  iml_inst *lp;
  IMText *text;
  IMAuxDrawCallbackStruct *draw_cb;
  int i, len;

  if (s)
    draw_cb = (IMAuxDrawCallbackStruct *) s->If->m->iml_new (s,
						     sizeof (IMAuxDrawCallbackStruct));
  else
    draw_cb = (IMAuxDrawCallbackStruct *) malloc (sizeof (IMAuxDrawCallbackStruct));

  if (!draw_cb) return;

  memset (draw_cb, 0, sizeof (IMAuxDrawCallbackStruct));
  draw_cb->aux_name = auxname;

  draw_cb->count_integer_values = num_integers;
  if (num_integers)
    {
      draw_cb->integer_values = (int *) s->If->m->iml_new (s,
					    sizeof (int) * num_integers);
      memset (draw_cb->integer_values, 0, sizeof (int) * num_integers);
      memset (draw_cb->integer_values, 0, sizeof (int) * num_integers);

      for (i = 0; i < num_integers; ++i)
	draw_cb->integer_values[i] = integers[i];
    }

  draw_cb->count_string_values = num_strings;
  if (num_strings)
    {
      draw_cb->string_values = (IMText *) s->If->m->iml_new (s,
					     sizeof (IMText) * num_strings);
      memset (draw_cb->string_values, 0, sizeof (IMText) * num_strings);

      draw_cb->string_values->encoding = UTF16_CODESET;
      for (i = 0, text = draw_cb->string_values; i < num_strings; ++i, ++text)
	{
	  len = strings[i] ? utfchar_length (strings[i]) : 0;
	  text->text.utf_chars = (UTFCHAR *) s->If->m->iml_new (s,
						sizeof (UTFCHAR) * (len + 1));
	  memset (text->text.utf_chars, 0, sizeof (UTFCHAR) * (len + 1));
	  text->char_length = len + 1;
	  if (strings[i])
	    memcpy (text->text.utf_chars, strings[i], len * sizeof (UTFCHAR));
	}
    }

  lp = (iml_inst *) s->If->m->iml_make_aux_draw_inst (s, draw_cb);
  s->If->m->iml_execute (s, &lp);
}

/* Local Variables: */
/* c-file-style: "iiim-project" */
/* End: */
