/*
Copyright 1990-2003 Sun Microsystems, Inc. All Rights Reserved.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions: The above copyright notice and this
permission notice shall be included in all copies or substantial
portions of the Software.


THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE OPEN GROUP OR SUN MICROSYSTEMS, INC. BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF
CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH
THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE EVEN IF
ADVISED IN ADVANCE OF THE POSSIBILITY OF SUCH DAMAGES.


Except as contained in this notice, the names of The Open Group and/or
Sun Microsystems, Inc. shall not be used in advertising or otherwise to
promote the sale, use or other dealings in this Software without prior
written authorization from The Open Group and/or Sun Microsystems,
Inc., as applicable.


X Window System is a trademark of The Open Group

OSF/1, OSF/Motif and Motif are registered trademarks, and OSF, the OSF
logo, LBX, X Window System, and Xinerama are trademarks of the Open
Group. All other trademarks and registered trademarks mentioned herein
are the property of their respective owners. No right, title or
interest in or to any trademark, service mark, logo or trade name of
Sun Microsystems, Inc. or its licensors is granted.

*/

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

enum {
	OK = 0,
	ERROR = -1,
	ERROR_HZCODE = -2,
	ERROR_CANGJIECODE = -3,
};

/* define struct for file with text format */
typedef struct {
	char cangjie_code[6];
	unsigned char *hz_str;
} TTextItemStruct, *PTextItemStruct;

typedef struct {
	long num_items;
	PTextItemStruct *items;
} TTextItemListStruct;

#define LEN_BUFFER_ALLOC  5000


char *skip_space(char *s)
{
	while (*s && (*s==' ' || *s=='\t')) s++;
	return s;
}

char *to_space(char *s)
{
	while (*s && !(isspace(*s) || *s == '\n')) s++;
	return s;
}

/*
parse the line into gbk_code, cangjie_code and frequence:

return OK if success;
return ERROR_HZCODE  if gbk_code error;
return ERROR_CANGJIECODE if cangjie_code error;
*/

int get_hex_code(char high_code, char low_code)
{
	unsigned char high, low;

	high = toupper(high_code);
	if (high >= '0' && high <= '9') {
		high = high - '0' + 0x8;
	} else if (high >= 'A' && high <= 'F') {
		high = high - 'A' + 0xa;
	}
	high <<= 4;

	low = toupper(low_code);
	if (low >= '0' && low <= '9') {
		low = low - '0' + 0x0;
	} else if (low >= 'A' && low <= 'F') {
		low = low - 'A' + 0xa;
	}
	
	return((high | low) & 0x00ff);
}

int parse_line_for_cangjie(char *line_buf, TTextItemStruct *item)
{
	unsigned char hz_code[6], hanzi_str[5], qu_code;
	char *ptr, *frequence_ptr;
	int i, cur_pos;

	ptr = skip_space(line_buf);

	if (*ptr == '\n' || *ptr == '#') 
		return(ERROR);

	/* get gbk code information */
	memset(hz_code, 0, 6);
	memcpy(hz_code, ptr, 5);

	cur_pos = 0;
	qu_code = toupper(hz_code[0]);
	if (qu_code >= '2' && qu_code <= '7') {
		hanzi_str[cur_pos++] = 0x8e;
		hanzi_str[cur_pos++] = 0xa1 + (qu_code - '1');
	} else if (qu_code >= 'A' && qu_code <= 'F') {
		hanzi_str[cur_pos++] = 0x8e;
		hanzi_str[cur_pos++] = 0xaa + (qu_code - 'A');
	}
	hanzi_str[cur_pos++] = get_hex_code(hz_code[1], hz_code[2]);
	hanzi_str[cur_pos++] = get_hex_code(hz_code[3], hz_code[4]);
	hanzi_str[cur_pos] = 0;

	item->hz_str = (unsigned char *)strdup(hanzi_str);

	/* get cangjie code information */
	ptr += 5;
	ptr = skip_space(ptr);

	i = 0;
	while(*ptr && *ptr != '\n' && !isspace(*ptr)) {
		if (i < 5) {
			if (*ptr & 0x80) 
				return(ERROR_CANGJIECODE);
		
			item->cangjie_code[i] = tolower(*ptr);
		}
		*ptr ++;
		i++;
	} 

	return(OK);
}

int TextItemList_Alloc(TTextItemListStruct *ItemList, int num_alloced)
{
	int i;

	ItemList->num_items = 0;
	ItemList->items = malloc(num_alloced * sizeof(PTextItemStruct*));

	if (ItemList->items == NULL) {
		fprintf(stderr, "Error: TextItemList malloc\n");
		return(ERROR);
	}

	for (i=0; i<num_alloced; i++)
		ItemList->items[i] = NULL;

	ItemList->num_items = num_alloced;
	
	return(OK);
}

int TextItemList_ReAlloc(TTextItemListStruct *ItemList, int num_alloced)
{
	int i;

	ItemList->items = realloc((void *)ItemList->items, num_alloced * sizeof(PTextItemStruct*));

	if (ItemList->items == NULL) {
		fprintf(stderr, "Error: TextItemList realloc\n");
		ItemList->num_items = 0;
		return(ERROR);
	}

	for (i=ItemList->num_items; i<num_alloced; i++)
		ItemList->items[i] = NULL;

	ItemList->num_items = num_alloced;
	
	return(OK);
}

int TextItemList_Free(TTextItemListStruct *ItemList)
{
	int i;

	for (i=0; i<ItemList->num_items; i++) {
		if (ItemList->items[i]->hz_str != NULL) 
			free((char *)ItemList->items[i]->hz_str);

		if (ItemList->items[i] != NULL)
			free((char *)ItemList->items[i]);
	}

	if (ItemList->items != NULL)
		free((char *)ItemList->items);

	ItemList->num_items = 0;

	return(OK);
}

int TextItem_Compare_By_CangJieCode(const void *p1, const void *p2)
{
	TTextItemStruct *item1 = *((TTextItemStruct **)p1);
	TTextItemStruct *item2 = *((TTextItemStruct **)p2);
	int ret;

	ret = strncmp(item1->cangjie_code, item2->cangjie_code, 4);

	return(ret);
}

int TextItemList_Sort(TTextItemListStruct *ItemList, int(*compare)(const void *, const void *))
{
	qsort((void *)ItemList->items, ItemList->num_items, sizeof(PTextItemStruct), compare);
}

int TextItemList_Print_For_Single(TTextItemListStruct *ItemList)
{
	int i;
	char *cangjie_code;
	unsigned char *hz_str;
	long frequence;

	for (i=0; i<ItemList->num_items; i++) {
		if (ItemList->items[i] == NULL) {
			continue;
		}

		cangjie_code = ItemList->items[i]->cangjie_code;
		hz_str = ItemList->items[i]->hz_str;

		printf("%s\t%s\n",  cangjie_code, (char *)hz_str);
/*
		printf("%s%s\n",  (char *)hz_str, cangjie_code);
*/
	}
}

int Read_TextItemList_From_File(char *file_name, TTextItemListStruct *ItemList, int(*parse_line)(char *, TTextItemStruct *))
{
	FILE *ifile;
	char line_buf[256];

	int num_malloc_items = LEN_BUFFER_ALLOC;
	int num_items = 0;
	int line_no = 0;
	int ret;

	ifile = fopen(file_name, "r");
	if (ifile == NULL) {
		fprintf(stderr, "Error: open file %s\n", file_name);
		return(ERROR);
	}

	TextItemList_Alloc(ItemList, num_malloc_items);

	while(fgets(line_buf, 256, ifile) != NULL) {

		line_no++;

		/* realloc buffer for items */
		if (num_items >= num_malloc_items) {
			num_malloc_items += LEN_BUFFER_ALLOC;
			ret = TextItemList_ReAlloc(ItemList, num_malloc_items);
			if (ret == ERROR) 
				break;
		}
		
		if (ItemList->items[num_items] == NULL) {
			ItemList->items[num_items] = (TTextItemStruct *)calloc(1, sizeof(TTextItemStruct));
			if (ItemList->items[num_items] == NULL) {
				fprintf(stderr, "Error:  No Memory for TextItemList\n");
				break;
			}
		}

		ret = parse_line(line_buf, ItemList->items[num_items]);
		if (ret != 0) {
			if (ret == ERROR_HZCODE) {
				fprintf(stderr, "Error: hz_code error in line %d\n", line_no);
			} else if (ret == ERROR_CANGJIECODE) {
				fprintf(stderr, "Error: cangjie_code error in line %d\n", line_no);
			}
			continue;
		} 

		num_items++;
	}

	TextItemList_ReAlloc(ItemList, num_items);

	fclose(ifile);
	return(OK);
}

int main(int argc, char **argv)
{
	int ret = OK;
	char *cangjie_file;

	TTextItemListStruct CangJie_ItemList;

	if (argc != 2) {
		printf("Usage:  %s cangjie_file\n", argv[0]);
		exit(-1);
	}

	cangjie_file = argv[1];

	ret = Read_TextItemList_From_File(cangjie_file, 
					  &CangJie_ItemList, 
					  parse_line_for_cangjie);
	if (ret == ERROR) {
		TextItemList_Free(&CangJie_ItemList);
		exit(-1);
	}

/*
	TextItemList_Sort(&CangJie_ItemList, TextItem_Compare_By_CangJieCode);
*/
	TextItemList_Print_For_Single(&CangJie_ItemList);
	TextItemList_Free(&CangJie_ItemList);

	return(OK);
}
