// -*- C++ -*-
// --------------------------------------------------------------------
// IpeCanvasFonts maintains the Freetype fonts for the canvas
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef IPEFONTS_H
#define IPEFONTS_H

#include "ipebase.h"
#include "ipegeo.h"
#include "ipefontpool.h"

#include <qpixmap.h>

//------------------------------------------------------------------------

class IpeFontPool;
class IpeFaceSize;
class IpeFace;
class IpeFont;
struct FT_FaceRec_;

/*! \class IpeFontsServices
  \brief Interface for IpeFonts callbacks.
*/

class IpeFontsServices {
public:
  virtual ~IpeFontsServices() { /* nothing */ } // placate compiler
  //! Request data for a font that is not embedded in Pdflatex output.
  virtual IpeBuffer StandardFont(IpeString fontName) = 0;
};

class IpeFace {
public:
  IpeFace(int id, const IpeFont &font, bool antiAlias,
	  IpeFontsServices *services);
  ~IpeFace();
  inline bool Matches(int id) const { return id == iId; }
  inline IpeFont::TType Type() const { return iType; }
  inline int Width(int ch) const { return iWidth[ch]; }

private:
  int iId;
  bool iAntiAlias;
  IpeFont::TType iType;
  int iGlyphIndex[0x100];
  int iWidth[0x100];

  IpeBuffer iData;
  FT_FaceRec_ *iFace;

  friend class IpeFaceSize;
};

class IpeFaceSize {
public:
  IpeFaceSize(IpeFace *face, const IpeLinear &m);
  ~IpeFaceSize();

  inline bool Matches(int id, const IpeLinear &m) const {
    return (id == iFace->iId && m == iMatrix); }
  inline IpeFace *Face() const { return iFace; }
  uchar *GetGlyph(int c, int &x, int &y, int &w, int &h);
  bool DrawChar(int ch, QRgb rgb, QPixmap *pixmap, int x, int y);

private:
  IpeFace *iFace;
  IpeLinear iMatrix;
  int iGlyphW, iGlyphH;		// size of glyph pixmaps
  int iGlyphSize;		// size of glyph pixmaps, in bytes
  uchar *iData;
};

class IpeCanvasFonts {
public:
  static IpeCanvasFonts *New(bool antiAlias, const IpeFontPool *fontPool,
			     IpeFontsServices *services);
  ~IpeCanvasFonts();

  IpeFaceSize *GetSize(int id, const IpeLinear &matrix);
  IpeFace *GetFace(int id);
  inline bool AntiAlias() const { return iAntiAlias; }

private:
  IpeCanvasFonts(bool antiAlias, const IpeFontPool *fontPool,
		 IpeFontsServices *services);

private:
  bool iAntiAlias;
  const IpeFontPool *iFontPool;
  IpeFontsServices *iServices;

  typedef std::list<IpeFace *> FaceSeq;
  typedef std::list<IpeFaceSize *> FontSeq;
  FaceSeq iFaces;
  FontSeq iFonts;
};

//------------------------------------------------------------------------
#endif
