/*******************************************************\
* irmp3-ncurses - An ncurses frontend for irmp3 using   *
* the Network Control Module                            *
* (C) 2003 Ross Axe                                     *
*                                                       *
* main.c - Miscellaneous                                *
\*******************************************************/

#if HAVE_CONFIG_H
#  include "config.h"
#endif

#if HAVE_LOCALE_H
#  include <locale.h>
#endif
#if HAVE_UNISTD_H
#  include <unistd.h>
#endif
#include <time.h>
#if HAVE_GETOPT_H
#  include <getopt.h>
#endif
#include <signal.h>

#include "irmp3-ncurses.h"

vcid("$Id: main.c,v 1.70 2005/12/30 23:01:59 ross Exp $");

#ifdef LOGGING
int loglevel = LOGGING;
#endif
static int pf = 0;	/*  Protocol Family: 4 for IPv4 or 6 for IPv6 */

#define eprintf(...) fprintf(stderr, __VA_ARGS__)

#if !HAVE_GETOPT_LONG
#  define getopt_long(argc, argv, optstring, longopts, longindex) \
    getopt(argc, argv, optstring)
#endif


/* Sleep for 2 seconds to allow the user to see why we're quitting */
void error_sleep(void)
{
#define ERROR_DELAY 2	/* seconds */
#if HAVE_NANOSLEEP
    nanosleep(&(const struct timespec){ERROR_DELAY, 0}, NULL);
#elif HAVE_SELECT && !USE_WINSOCK
    select(0, NULL, NULL, NULL, &(const struct timeval){ERROR_DELAY, 0});
#elif HAVE__SLEEP
    _sleep(ERROR_DELAY);
#elif HAVE_USLEEP
#  if HAVE_ALARM
#warning Mixing calls to alarm() and usleep() is a bad idea
#  endif
    usleep(ERROR_DELAY * 1000);
#elif HAVE_SLEEP
#  if HAVE_ALARM
#warning Mixing calls to alarm() and sleep() is a bad idea
#  endif
    sleep(ERROR_DELAY);
#else
#warning Cannot find any way to sleep - critical error messages will be missed
#endif
}


/*
 * usage
 * print quick help message
 */
static void usage(int argc, char *argv[])
{
    (void)argc;
    eprintf("%s\n", PACKAGE_STRING);
    eprintf("%s\n", COPYING);
    eprintf(_("Usage: %s [-h] [-p[0|1]] [-l[loglevel]] [-4 | -6] "
	      "[host [port]]\n"), argv[0]);
    eprintf(_("Default target is %s:%s\n"), IRMP3_HOST, IRMP3_PORT);
#if ENABLE_NLS
    eprintf(_("Reading locale data from %s\n"), LOCALEDIR);
#endif
}


/*
 * parseoptions
 * Check all command line options here
 * Return false to cause immediate exit
 */
static bool parseoptions(int argc, char *argv[])
{
    int opt_char;
    int tmp_playlist = show_playlist, tmp_loglevel = loglevel;
#if HAVE_GETOPT_LONG
    int long_index;
    const struct option long_opts[] = {{
	.name = "help",
	.has_arg = no_argument,
	.val = 'h',
    }, {
	.name = "hide-playlist",
	.has_arg = no_argument,
	.val = 'P',
    }, {
	.name = "show-playlist",
	.has_arg = optional_argument,
	.val = 'p',
    }, {
	.name = "loglevel",
	.has_arg = optional_argument,
	.val = 'l',
    }, {0,0,0,0}};
#endif
    int i;

    while(-1 != (opt_char = getopt_long(argc, argv, "hPp::l::46",
					long_opts, &long_index))) {
	switch(opt_char) {
	case 'h':
	    assert(!optarg);
	    usage(argc, argv);
	    return false;

	case 'p':
	    if(optarg) {
		if(strcmp(optarg, "0") == 0) {
		case 'P':
		    tmp_playlist = false;
		} else if(strcmp(optarg, "1") == 0)
		    tmp_playlist = true;
		else {
		    eprintf(_("%s is an invalid option for -p\n"), optarg);
		    usage(argc, argv);
		    return false;
		}
	    } else
		tmp_playlist = true;
	    break;

	case 'l':
	    tmp_loglevel = optarg ? atoi(optarg) : 1;
	    break;

#if HAVE_GETOPT_LONG
	case 0:
	    eprintf("%d - %s\n", long_index, long_opts[long_index].name);
	    abort();
#endif

	case '4':
	    pf = 4;
	    break;
	case '6':
	    pf = 6;
	    break;

	case '?':
	    usage(argc, argv);
	    return false;

	default:
	    eprintf("%d - %c\n", opt_char, opt_char);
	    abort();
	}
    }

#if PLAYLIST
    show_playlist = tmp_playlist;
#else
    if(tmp_playlist)
	eprintf(_("Playlist is disabled\n"));
#endif
#ifdef LOGGING
    loglevel = tmp_loglevel;
#else
    if(tmp_loglevel)
	eprintf(_("Logging is disabled\n"));
#endif

    assert(optind <= argc);

    switch(argc - optind) {
    case 0:
	return set_address(IRMP3_HOST, IRMP3_PORT);
    case 1:
	return set_address(argv[optind], IRMP3_PORT);
    case 2:
	return set_address(argv[optind], argv[optind + 1]);
    default:
	optind += 2;
	eprintf(ngettext("Surplus argument:", "Surplus arguments:",
			 argc - optind));
	for(i = optind; i < argc; i++)
	    eprintf(" %s", argv[i]);
	eprintf("\n");
	usage(argc, argv);
	return false;
    }
}

int main(int argc, char *argv[])
{
    int sock;

    setlocale(LC_ALL, "");
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);

    if(!start_network())
	return EXIT_FAILURE;
    if(!parseoptions(argc, argv))
	return EXIT_FAILURE;

    sock = connect_irmp3(pf);
    if(sock == -1) {
	stop_network();
	return EXIT_FAILURE;
    }

#if XCURSES
    Xinitscr(argc, argv);
#else
    initscr();
#endif
#if USE_COLOUR
    if(has_colors()) {
	start_color();
	init_pair(COLOURPAIR_TITLE, COLOR_WHITE, COLOR_RED);
	init_pair(COLOURPAIR_BORDER, COLOR_RED, COLOR_BLACK);
	init_pair(COLOURPAIR_CURSOR, COLOR_WHITE, COLOR_BLUE);
    }
#endif
    raw();
    noecho();
    nonl();
    intrflush(stdscr, FALSE);
    keypad(stdscr, TRUE);
    curs_set(0);
#if USE_MOUSE
    mousemask(BUTTON1_CLICKED | BUTTON1_DOUBLE_CLICKED, NULL);
#endif

    init_ui();

    sbar_printf(_("Welcome to %s %s"), PACKAGE_STRING, COPYING);

    init_queries(sock);
    sbar_printf(_("Press ? for help"));

    /* now we enter main program loop */
    main_poll_loop(sock);

    disconnect_irmp3(sock);
    sbar_printf(NULL);		/* Close any logfile */
    destroy_ui();
    curs_set(1);
    endwin();
#if XCURSES
    XCursesExit();
#endif
    if(!stop_network())
	return EXIT_FAILURE;
    return EXIT_SUCCESS;
}
