/*******************************************************\
* irmp3-ncurses - An ncurses frontend for irmp3 using   *
* the Network Control Module                            *
* (C) 2005 Ross Axe                                     *
*                                                       *
* netio.c - low-level network I/O                       *
\*******************************************************/

#if HAVE_CONFIG_H
#  include <config.h>
#endif

#include "autosocket.h"

#include "irmp3-ncurses.h"

vcid("$Id: netio.c,v 1.25 2006/01/04 14:30:59 ross Exp $");

static const char *irmp3_hostname;
static const char *irmp3_portname;

#if USE_WINSOCK
bool start_network(void)
{
    WSADATA wd;
#if USE_WINSOCK >= 2
#define WS_MAJOR 2
#define WS_MINOR 2
#else
#define WS_MAJOR 1
#define WS_MINOR 1
#endif
    int ret = WSAStartup(MAKEWORD(WS_MAJOR, WS_MINOR), &wd);

    if(ret != 0) {
	fprintf(stderr, "WSAStartup failed with error %d\n", ret);
	return false;
    }

    if(LOBYTE(wd.wVersion) != WS_MAJOR || HIBYTE(wd.wVersion) != WS_MINOR) {
	WSACleanup();
	fprintf(stderr, _("Couldn't find suitable Winsock DLL.  "
			  "Requested version %d.%d, got %d.%d\n"),
		WS_MAJOR, WS_MINOR, LOBYTE(wd.wVersion), HIBYTE(wd.wVersion));
	return false;
    }

    return true;
}

bool stop_network(void)
{
    return WSACleanup();
}

#else

bool start_network(void)
{
    return true;
}

bool stop_network(void)
{
    return true;
}
#endif


bool set_address(const char *hostname, const char *port)
{
    assert(hostname && port);
    irmp3_hostname = hostname;
    irmp3_portname = port;
    return true;
}


bool handle_net(int sock)
{
    static char buf[1024];
    static unsigned long len = 0;
    long recvd;
    char *ptr;

    recvd = recv(sock, buf + len, sizeof buf - len - 1, 0);
    switch(recvd) {
    case 0:
	sbar_printf(_("Quit"));
	return false;
    case -1:
	switch(sock_errno) {
	case SOCK_EINTR:
	    return true;
	default:
	    sbar_printf(_("Recieve error: %d %s"),
			sock_errno, sock_strerror(sock_errno));
	    error_sleep();
	    return false;
	}
    }

    len += recvd;
    buf[len] = '\0';
    dbg_printf(2, "%ld bytes were recieved. buf (%ld chars) now contains %s",
	       recvd, len, buf);
    if(memchr(buf, '\0', len)) {
	sbar_printf("Error: irmp3d sent us a NUL");
	error_sleep();
	return false;
    }
    assert(len == strlen(buf));

    /* strip any blank lines from start of input
       (I thought that this couldn't happen, but in the case where
       a CRLF is cut in half by the previous read, it can...) */
    for(ptr = buf; ptr[0] == '\r' || ptr[0] == '\n'; ptr++) ;
    if(ptr != buf) {		/* blank lines *were* found */
	memmove(buf, ptr, strlen(ptr) + 1);
	len -= (ptr - buf);
    }

    do {
	ptr = strpbrk(buf, "\r\n");	/* look for EOL */
	if(!ptr) {
	    if(len < sizeof buf)
		break;		       /* no EOL, exit & wait for more input */
	    ptr = strchr(buf, '\0');   /* give up, the input line's just too
					  long */
	    sbar_printf(_("Line too long!"));
	    /* drop through and try and parse what we have */
	}
	while(ptr[0] == '\r' || ptr[0] == '\n') {
	    ptr[0] = '\0';
	    ptr++;
	}
	/* now buf is a NUL terminated line, and ptr is the next line */
	dbg_printf(1, "Recieved line: %s", buf);
	if(buf[0] == '\0')
	    dbg_printf(1, "WTF! buf is zero length. ptr (at offset %d) "
		       "holds %s", ptr - buf, ptr);

	if(!(brwnd ? browser_net_callback : main_net_callback)(sock, buf))
	    return false;

	/* shunt next lines back to the start of buf */
	memmove(buf, ptr, strlen(ptr) + 1);
	len -= (ptr - buf);

	if(buf[0] != '\0' && len == 0)
	    sbar_printf("WTF! len==0 but buf holds %s", buf);
	if(buf[0] == '\r' || buf[0] == '\n')
	    sbar_printf("WTF! buf starts with EOL. but holds %s", buf);
    } while(buf[0]);

    return true;		/* ok, no errors */
}

int connect_irmp3(int pf)
{
    int sock, ret;
    const struct addrinfo hints = {
	.ai_flags = 0,
	.ai_family = (pf == 4 ? PF_INET :
#ifdef PF_INET6
		      pf == 6 ? PF_INET6 :
#endif
		      PF_UNSPEC),
	.ai_socktype = SOCK_STREAM,
	.ai_protocol = IPPROTO_TCP,
    };
    struct addrinfo *addrs, *i;

#ifdef SIGPIPE
    signal(SIGPIPE, SIG_IGN);
#endif

    ret = getaddrinfo(irmp3_hostname, irmp3_portname, &hints, &addrs);
    switch(ret) {
    case 0:
	break;
#ifdef EAI_SYSTEM
    case EAI_SYSTEM:
	ret = sock_errno;
	fprintf(stderr, _("Couldn't find address for %s:%s: %s\n"),
		irmp3_hostname, irmp3_portname, sock_strerror(ret));
	return -1;
#endif
    default:
	fprintf(stderr, _("Couldn't find address for %s:%s: %s\n"),
		irmp3_hostname, irmp3_portname, gai_strerror(ret));
	return -1;
    }

    assert(addrs);
    for(i = addrs; i; i = i->ai_next) {
	assert(i->ai_socktype == SOCK_STREAM);
	sock = socket(i->ai_family, i->ai_socktype, i->ai_protocol);
	if(sock == -1) {
	    sock_perror("socket");
	    continue;
	}
	ret = connect(sock, i->ai_addr, i->ai_addrlen);
	if(ret) {
	    const int tmp = sock_errno;
	    char host[NI_MAXHOST], port[NI_MAXSERV];

	    ret = getnameinfo(i->ai_addr, i->ai_addrlen, host, sizeof host,
			      port, sizeof port,
			      NI_NUMERICHOST | NI_NUMERICSERV);
	    assert(ret == 0);
	    fprintf(stderr, _("Connection to %s:%s failed: %s\n"),
		    host, port, sock_strerror(tmp));
	    disconnect_irmp3(sock);
	    sock = -1;
	    continue;
	} else {
	    char host[NI_MAXHOST], port[NI_MAXSERV];

	    ret = getnameinfo(i->ai_addr, i->ai_addrlen, host, sizeof host,
			      port, sizeof port,
			      NI_NUMERICHOST | NI_NUMERICSERV);
	    assert(ret == 0);
	    fprintf(stderr, _("Connected to %s:%s\n"), host, port);
	}
	break;
    }

    freeaddrinfo(addrs);
    return sock;
}

int disconnect_irmp3(int sock)
{
#if USE_WINSOCK
    return closesocket(sock);
#else
    return close(sock);
#endif
}

int netputs(const char *msg, int sock)
{
#ifdef MSG_NOSIGNAL
#  define SENDFLAGS MSG_NOSIGNAL
#else
#  define SENDFLAGS 0
#endif
    int rval = send(sock, msg, strlen(msg), SENDFLAGS);

    send(sock, "\r\n", 2, SENDFLAGS);
    dbg_printf(1, "sent: \"%s\"", msg);
    return rval;
}

int netprintf(int sock, const char *fmt, ...)
{
    va_list ap;
    int rval;
    char buf[1024];

    va_start(ap, fmt);
    rval = vsnprintf(buf, sizeof buf, fmt, ap);
    va_end(ap);
    netputs(buf, sock);
    return rval;
}
