#ifndef K3DSDK_IPROPERTY_H
#define K3DSDK_IPROPERTY_H

// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares iproperty, an abstract interface for a name-value pair with fixed type
		\author Tim Shead (tshead@k-3d.com)
*/

#include "iunknown.h"
#include "signal_system.h"

#include <boost/any.hpp>

#include <string>

namespace k3d
{

class inode;

/// Abstract interface for a name-value pair with fixed type (note - if you want to write to a property, query for iwritable_property and be prepared for it to fail for read-only properties)
class iproperty :
	public virtual iunknown
{
public:
	/// Returns the property name, which will be used as a unique identifier for programmatic access to the property (i.e. through scripting)
	virtual const std::string property_name() = 0;
	/// Returns a human-readable lable for the property, which should be localized for display in UI code
	virtual const std::string property_label() = 0;
	/// Returns a human-readable "one-liner" description of the property, which should be localized for display in the UI (e.g. as a tooltip)
	virtual const std::string property_description() = 0;
	/// Returns the property type
	virtual const std::type_info& property_type() = 0;
	/// Returns the property value
	virtual const boost::any property_value() = 0;
	/// Returns a reference to the object that owns the property (if any)
	virtual inode* property_node() = 0;
	/// Defines a signal that will be emitted if the property value changes.  The signal includes a pointer to an optional "hint" object that may provide additional information about what changed.
	typedef sigc::signal1<void, iunknown*> changed_signal_t;
	virtual changed_signal_t& property_changed_signal() = 0;
	/// Defines a signal that will be emitted when the property is destroyed
	typedef sigc::signal0<void> deleted_signal_t;
	virtual deleted_signal_t& property_deleted_signal() = 0;

protected:
	iproperty() {}
	iproperty(const iproperty&) {}
	iproperty& operator = (const iproperty&) { return *this; }
	virtual ~iproperty() {}
};

} // namespace k3d

#endif // K3DSDK_IPROPERTY_H


