/***************************************************************************
                          carchiveoperation.h  -  description
                             -------------------
    begin                : Tue Nov 15 17:03:53 CET 2005
    copyright            : (C) 2005 by Eric Coquelle
    email                : coquelle@caramail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef ARCHIVEOPERATION_H
#define ARCHIVEOPERATION_H
 

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qstring.h>
#include <qobject.h>

#include <kprocess.h>
#include <qthread.h>

#include "carchive.h"
#include "cextraction.h"
#include "caddfilestoarchive.h"


/** Performs various operation on archives, such as extraction,
  * conversion of archive format...
  * @author Eric Coquelle
  */
class CArchiveOperation : public QObject
{
Q_OBJECT
  public:
    CArchiveOperation( QObject * parent = 0, const char * name = 0 );
    CArchiveOperation( CArchive* archive_obj, QProgressBar* progress_bar, QString temp_dir="");
    ~CArchiveOperation();
  /**Erases karchiver's temp dir content*/
  void eraseTempDir();
  QString getArchiveName();
  CArchive* getArchiveObject() { return archiveobj; };
  int getKindofCompressor() {return  kindofCompressor; }
  virtual void convertCurrentArchive(QString) {};
  virtual void doArchiveConversion(int , const QString & , QString =NULL) {};
  virtual QString canDisplayArchive() { return NULL;};
  virtual void displayArchiveContent(){};
  virtual void extractCurrentArchive(){};
  virtual void extractMultiFiles( QStringList , bool , QCString , bool  ){};
  virtual void makeSfxArchive(QString=QString::null, int=0){};
  virtual void displayFile(QListViewItem* ){};
  virtual void displaySelectedFiles(){};
  virtual void createNewArchive(){};
  virtual void addFilesToCurrentArchive(){};
  virtual void addFilesToCurrentArchive(QStringList , bool =false, int =ADD_AND_REPLACE_FILES, QString =NULL){};
  virtual void testArchive(){};
  virtual void scanArchive(){};
  virtual void createArchiveFromCommandLine( QStringList , QString  ){};
  virtual void processPastedFiles( QStringList , bool) {};
  virtual void mailCurrentArchive() {};
  virtual void launchWizard(QListViewItem*) {};
  virtual void doCuttUnCutt(QString, int =1450000 ) {};
  
  signals:
    void operationEnded(int, QString);

  protected:
    /** Check if @param archivename is a valid archive. Returns either an error message,
      * or QString::NULL if everything is OK. Set also kindofCompressor*/
    QString checkCanIdentifyArchive(QString&);
    /** Creates the adequate CArchive-child object */
    CArchive* createsArchiveObject(CArchive*, int);
       
  protected:
    QString tempdir; //temp directory where files are extracted
    CArchive* archiveobj;
    QProgressBar* progressbar;
    QString newarchivename;
    int kindofCompressor;
    /* the extension of current compressor */
    QString extension;
    /* error messages returned by checkCanIdentifyArchive - here required compressor not installed */
    QString errormsgnocompressor;
};




/**The CArchiveOperationConversion class
  * convert the current archive in @param newname
  * First check if everything is OK with convertCurrentArchive()
  */
class CArchiveOperationConversion : public CArchiveOperation
{
  Q_OBJECT
  public:
    CArchiveOperationConversion(QString, CArchive*, QProgressBar*);
    ~CArchiveOperationConversion();
    /** convert current archive type into another one */
    void convertCurrentArchive(QString newname);
    /** Convert an archive to type, and store it in the directory repconv. If @param  previousarchivename is provided, the new archivename will be reconstructed from it, with the proper extension (@param newarchivetype)*/
    void doArchiveConversion(int newarchivetype, const QString & targetdirectory, QString previousarchivename=NULL);

  protected slots:
    /** Convert an archive to a new one*/
    void doArchiveConversion();
    void conversionDone();
};




/**The CArchiveOperationDisplay class
  * displays the content of @param archivename
  * First check if the required compressor is available with canDisplayArchive()
  * Then create a CArchive child and display the archive with displayArchiveContent()
  */
class CArchiveOperationDisplay : public CArchiveOperation
{
  Q_OBJECT
  public:
    CArchiveOperationDisplay(QString archivename, bool readwithstream, bool viewbydirs, CArchive* archive_obj, QProgressBar* progress_bar );
    QString canDisplayArchive();

    void displayArchiveContent();
  
  protected slots:
    /** Creates the adequate CArchive object, and launches the display process */
    void slotDisplayProcessEnded();
    
  protected:
    bool readarchivewithstream;
    bool viewbydirectories;
    bool processrunning;
};




//////////////////////////////////////////////////////////////////////////////////////
/**The CArchiveOperationExtract class
  * extracts the content of current archive
  * using extractCurrentArchive();
  */
class CArchiveOperationExtract: public CArchiveOperation
{
  Q_OBJECT
  public:
    CArchiveOperationExtract( CArchive* archive_obj, QProgressBar* progress_bar );
    /** Extract current archive. Promt for Extract dir and other options...*/
    void extractCurrentArchive();
    
  protected:
    /** Extract current archive. Extract in @param extractpath all or some files depending
      * on @param kindofextraction */
    void extractCurrentArchive(int kindofextraction, QString extractpath);
    
  protected slots:
    virtual void slotExtractProcessEnded();
  
  protected:
    CExtraction extractdlg;
};



//////////////////////////////////////////////////////////////////////////////////////
/**The CArchiveOperationMail class
  * mail all or anly some files from current archive
  */
class CArchiveOperationMail: public CArchiveOperationExtract
{
  Q_OBJECT
  public:
    CArchiveOperationMail( CArchive* archive_obj, QProgressBar* progress_bar, QString tmpdir );
    /** Mail some or all files fromcurrent archive. Promt which operation to perform in a dialog...*/
    void mailCurrentArchive();
    
  protected slots:
    void slotExtractProcessEnded();
    
  private:
    QStringList filestomail;
    QString adress;
};



//////////////////////////////////////////////////////////////////////////////////////
/**The CArchiveOperationExtractMultiFiles class
  * extract a list of archives
  */
class CArchiveOperationExtractMultiFiles: public CArchiveOperationExtract
{
  Q_OBJECT
  public:
    CArchiveOperationExtractMultiFiles( CArchive* archive_obj, QProgressBar* progress_bar );
    
    void extractMultiFiles( QStringList archives, bool promptdir, QCString extractdir, bool chooseExtractDirFromArchive );
    
  protected:
    void extractMultiFiles();
    
  protected slots:
    /**The archive is displayed (usefull to perform CArchive::checkFiles(). Check is everything 
      * is OK, and extract the archive. @param value = display erros/success message*/
    void slotArchiveDisplayed(int, QString);
    void slotArchiveExtracted();
    
  protected:
    QStringList archivesqueue;
    CArchiveOperationDisplay* displayobj;
    bool m_chooseExtractDirFromArchive;
    bool m_promptdir;
    QCString m_extractdir;
};




//////////////////////////////////////////////////////////////////////////////////////
/**The CArchiveOperationView class
  * displays the selected file(s) with the adequate software
  */
class CArchiveOperationViewFiles: public CArchiveOperation
{
  Q_OBJECT
  public:
    CArchiveOperationViewFiles( CArchive* archive_obj, QProgressBar* progress_bar, QString temp_dir );

    /** display the file designed by the listentry @param currentItem using its associated program */
    void displayFile(QListViewItem* currentItem);
    /** Get recursively all files contained in the directory @param d */
    void displaySelectedFiles();
    
  protected slots:  
    void slotFilesExtracted();
    
  private:
    /** Get recursively all files contained in the directory @param d */
    QStringList getAllFiles(QDir);
};




//////////////////////////////////////////////////////////////////////////////////////
/**The CArchiveOperationSfx class
  * extracts the content of current archive to @param temp_dir + "Data/"
  * and create the sfx module using makeSfxArchive();
  */
class CArchiveOperationSfx : public CArchiveOperation
{
  Q_OBJECT
  public:
    CArchiveOperationSfx(CArchive* archive_obj, QProgressBar* progress_bar, QString temp_dir );
    /** Creates an sfx archive */
    void makeSfxArchive(QString, int);

  protected:
    /** Catenates an archive to its sfx module */
    void cat();
    /** Make a Sfx executable file from files extracted in tempdirsfx*/
    void makeUniversalSfx();
    void make7zSfx();
    /** Extract current archive in tempdirsfx */
    void extractAllFiles();
    
  protected slots:
    /** Create the Sfx archive from the files extracted in tempdirsfx */
    virtual void slotExtractProcessEnded();
    virtual void slotFilesAdded();
    virtual void slot7zSfxReady();
    
  private:
    CArchive* sfx_archiveobj;
    int sfxtype;
  protected:
    QString tempdirsfx;
    QString exearchivename;
  public:
    enum possiblesfxtypes {UNIVERSAL_SFX=0, SEVENZ_SFX};
};




//////////////////////////////////////////////////////////////////////////////////////
/**The CArchiveOperationTest class
  * extracts the content of current archive to @param temp_dir
  * and look for any errors using testArchive();
  */
class CArchiveOperationTest : public CArchiveOperationSfx
{
  Q_OBJECT
  public:
    CArchiveOperationTest(CArchive* archive_obj, QProgressBar* progress_bar, QString temp_dir );
    void testArchive();

  protected slots:
    void slotExtractProcessEnded();
    void slotIntegrityProcessEnded();
    void archiveRepaired();
};



//////////////////////////////////////////////////////////////////////////////////////
/**The CArchiveOperationScanForVirus class
 * extracts the content of current archive to @param temp_dir
 * and scans for any viruses using testArchive();
 */
class CArchiveOperationScanForVirus : public CArchiveOperationSfx
{
  Q_OBJECT
  public:
    CArchiveOperationScanForVirus(CArchive* archive_obj, QProgressBar* progress_bar, QString temp_dir );
    void scanArchive();

  protected slots:
    void slotExtractProcessEnded();
    void slotScanProcessEnded(KProcess*);
    
  protected:
    KProcess processav;
    QString scanprogramm;
    QString scanprogrammargument;
};



//////////////////////////////////////////////////////////////////////////////////////
/**The CArchiveOperationWizard class
 * extracts the content of current archive to @param temp_dir
 * and launches KArchiver's wizards;
 */
class CArchiveOperationWizard : public CArchiveOperationSfx
{
  Q_OBJECT
  public:
    CArchiveOperationWizard(CArchive* archive_obj, QProgressBar* progress_bar, QString temp_dir );
    void launchWizard(QListViewItem* current);

  protected:
    QString getCommonDirectory(QDir d);
    
  protected slots:
    void slotExtractProcessEnded();
    void slotMessageArchiveOperation(int, QString);
    
  protected:
    QListViewItem* currentItem;
    CArchiveOperation* archiveoperation;
};



//////////////////////////////////////////////////////////////////////////////////////
/**The CArchiveOperationSplit class
  * Split current archive in a series of files whose maximum size is blocksize
  * Also performs the opposite oeration to reconstruct the archive
  */
class CArchiveOperationSplit : public CArchiveOperation
{
  Q_OBJECT
  public:
    CArchiveOperationSplit(CArchive* archive_obj, QProgressBar* progress_bar );
    
    /** unsplit the archive @param name
      * or split current archive into the directory @param name in block of size @param blocksize*/
    void doCuttUnCutt(QString name, int blocksize=1450000);

};



//////////////////////////////////////////////////////////////////////////////////////
/**The CArchiveOperationAdd class
  * add some files to an existing archive
  */
class CArchiveOperationAdd:  public CArchiveOperation
{
  Q_OBJECT
  public:
    CArchiveOperationAdd( CArchive* archive_obj, QProgressBar* progress_bar );
    /**Add files to current archives. Launches a dialog box to get the files to add*/
    void addFilesToCurrentArchive();
    /** Add files @param files to the archive @param archivename.*/
    void addFilesToThisArchive(QString& archivename, QStringList files, bool removefiles=false, int actionfiles=ADD_AND_REPLACE_FILES, QString relativepath=NULL);
    /** Add files @param files to the current archive */
    void addFilesToCurrentArchive(QStringList files, bool removefiles=false, int actionfiles=ADD_AND_REPLACE_FILES, QString relativepath=NULL);

  protected slots:
    void slotFilesAdded();
    virtual void slotFilesAddedToAnotherArchive();
    
  protected:  
    CAddFilesToArchive addfilestoarchive;
};



//////////////////////////////////////////////////////////////////////////////////////
/** The CArchiveOperationCreate class
  * creates a new archive and add some files to it
  */
class CArchiveOperationCreate: public CArchiveOperationAdd
{
  Q_OBJECT
  public:
    CArchiveOperationCreate( CArchive* archive_obj, QProgressBar* progress_bar );
    void createNewArchive();
    
  protected:
    QString checkCanCreateNewArchive(QString&);
    void addFilesToNewArchive(QString& , QStringList, QString relativepath=NULL );
    
  protected slots:
    virtual void slotArchiveCreated();
};



class CArchiveOperationProcessInputFiles : public CArchiveOperationCreate
{
Q_OBJECT
public:
  CArchiveOperationProcessInputFiles( CArchive* , QProgressBar* , QString, bool, int  );
  void createArchiveFromCommandLine( QStringList files, QString preferedtype );
  void processPastedFiles( QStringList , bool );

protected:
  QString openRemoteFile(const KURL&);
  void processInputFiles(QStringList files, QString preferedtype=NULL );
  /** User has selected to open @param archivename. Try to lauch the adequate process. Otherwise, return false*/
  bool canProcessDisplayArchive(QString& );
  
protected slots:
  void slotFilesAddedToAnotherArchive();
  void slotArchiveCreated();
  void slotArchiveDisplayed(int , QString );
  void slotFilesAdded();
  
protected:
  QString preferedArchiveType;
  bool includewholepath;
  int idmessageok;
  int idmessageerror;
  CArchiveOperationDisplay *archiveoperationdisplay;
  bool viewbydirs;
  int actionaddfiles;
};

#endif
