/***************************************************************************
 *   Copyright (C) 2005 by Roberto Cappuccio and the Kat team              *
 *   Roberto Cappuccio : roberto.cappuccio@gmail.com                       *
 *   Praveen Kandikuppa : praveen9@gmail.com                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           *
 ***************************************************************************/

#include <kdebug.h>
#include <kglobal.h>
#include <qapplication.h>
#include <qtimer.h>
#include <qdir.h>
#include <kio/global.h>
#include <kio/jobclasses.h>
#include <kio/metainfojob.h>
#include <kio/previewjob.h>
#include <qbuffer.h>
#include <qregexp.h>

#include <katengine.h>

#include <kathelpers.h>
#include <katfulltextjob.h>
#include <katpreviewjob.h>
#include <katlanguagemanager.h>

#include <katinfoextractor.h>

using namespace KIO;

KatInfoExtractor::KatInfoExtractor( KatCatalog* cat, CppSQLite3DB* db )
    : QObject (0, "katinfoextractor"),
                m_db( db ), m_cat( cat ), m_bAsync( true ), m_totalSize( 0L ), m_totalFiles( 0L ),
                m_totalSubdirs( 0L ), m_totalPreviews( 0L ), m_totalMetaData( 0L ),
                m_totalFullTexts( 0L ), m_totalExtractions( 0L )
{
    m_lp = KatLanguageManager::loadAllLanguageProfiles();

    // Restore the number of files,previews,metadat etc. from catalog
    m_totalExtractions = m_cat->extractions();
    m_totalPreviews = m_cat->thumbnails();
    m_totalFullTexts = m_cat->fullTexts();
    m_totalMetaData = m_cat->metaData();

    m_subJobs.setAutoDelete (false);

    m_thumbnail.setAutoDelete( true );
    m_metadata.setAutoDelete( true );
    m_fulltext.setAutoDelete( true );

    m_save = false;
}

KatInfoExtractor::~KatInfoExtractor()
{
    for (KIO::Job *job = m_subJobs.first(); job; job = m_subJobs.next())
        job->kill();

    m_subJobs.clear();

    // Delete any saved info
    slotDiscardInfo();

    delete m_lp; // TODO: delete also the profiles in the list
}

void KatInfoExtractor::updateFiles( const QStringList& files )
{
    QMap<QString, long> rFiles = readFiles( m_cat->catalogId(), files);
    updateFiles( rFiles );
}

void KatInfoExtractor::updateFiles( const QMap<QString, long>& files )
{
    deleteFiles (files);

    if ( files.empty() )
    {
        // no more items in the database
        emit completed();
        if ( !m_bAsync )
            qApp->exit_loop();

        return;
    }

    // create the 3 subjobs for the LIMIT files
    KFileItemList kfiles_preview;
    KFileItemList kfiles_metainfo;
    KFileItemList kfiles_fulltext;

    QMap<QString,long>::ConstIterator it;
    QMap<QString,long>::ConstIterator end( files.end() );
    for ( it = files.begin(); it != end; ++it )
    {
        QString fullName = it.key();
        long fileId = it.data();

        kdDebug() << fullName << " : " << endl;

        KFileItem* kfi_preview = new KFileItem( KFileItem::Unknown, // _mode
                                                KFileItem::Unknown, // _permissions
                                                KURL::fromPathOrURL( fullName ), // url
                                                true // determineMimeTypeOnDemand
                                              );
        KFileItem* kfi_metainfo = new KFileItem( *kfi_preview );
        KFileItem* kfi_fulltext = new KFileItem( *kfi_preview );

        if ( kfi_preview->mimetype() != "inode/directory" )
        {
            extraInfo* ei_preview = new extraInfo();
            extraInfo* ei_metainfo = new extraInfo();
            extraInfo* ei_fulltext = new extraInfo();

            ei_preview->fileId = fileId;
            ei_metainfo->fileId = fileId;
            ei_fulltext->fileId = fileId;

            kfi_preview->setExtraData( "extra", ei_preview );
            kfi_metainfo->setExtraData( "extra", ei_metainfo );
            kfi_fulltext->setExtraData( "extra", ei_fulltext );

            kfiles_preview.append( kfi_preview );
            kfiles_metainfo.append( kfi_metainfo );
            kfiles_fulltext.append( kfi_fulltext );
        }
    }

    if ( !kfiles_preview.isEmpty() )
    {
        KatPreviewJob* pj = new KatPreviewJob( kfiles_preview, // items
                                               m_cat->thumbnailSize(), // width
                                               m_cat->thumbnailSize(), // height
                                               0, // iconSize
                                               0, // iconAlpha
                                               false, // scale
                                               NULL, // enabledPlugins TODO: implement
                                               true // deleteItems
                                              );
        m_subJobs.append (pj);

        connect( pj, SIGNAL( gotPreview( const KFileItem*, const QPixmap& ) ),
                 this, SLOT( gotThumbnail( const KFileItem*, const QPixmap& ) ) );
        connect( pj, SIGNAL( failed( const KFileItem* ) ),
                 this, SLOT( noThumbnail( const KFileItem* ) ) );
        connect( pj, SIGNAL( result( KIO::Job* ) ),
                 this, SLOT( resultThumbnail( KIO::Job* ) ) );
    }

    if ( !kfiles_metainfo.isEmpty() )
    {
        KIO::MetaInfoJob* mij = new KIO::MetaInfoJob( kfiles_metainfo, // items
                                                      true // deleteItems
                                                     );
        m_subJobs.append (mij);

        connect( mij, SIGNAL( gotMetaInfo( const KFileItem* ) ),
                 this, SLOT( gotMetaInfo( const KFileItem* ) ) );
        connect( mij, SIGNAL( failed( const KFileItem* ) ),
                 this, SLOT( noMetaInfo( const KFileItem* ) ) );
        connect( mij, SIGNAL( result( KIO::Job* ) ),
                 this, SLOT( resultMetaInfo( KIO::Job* ) ) );
    }

    if ( !kfiles_fulltext.isEmpty() )
    {
        KatFullTextJob* ftj = new KatFullTextJob( kfiles_fulltext, // items
                                                  NULL, // enabledPlugins TODO: implement
                                                  true // deleteItems
                                                 );
        m_subJobs.append ( ftj );

        connect( ftj, SIGNAL( gotFullText( KIO::Job*, const KFileItem*, const QString& ) ),
                 this, SLOT( gotFullText( KIO::Job*, const KFileItem*, const QString& ) ) );
        connect( ftj, SIGNAL( failed( const KFileItem* ) ),
                 this, SLOT( noFullText( const KFileItem* ) ) );
        connect( ftj, SIGNAL( result( KIO::Job* ) ),
                 this, SLOT( resultFullText( KIO::Job* ) ) );
    }

    if ( m_subJobs.isEmpty() )
    {
        // no items to dispatch
        emit completed();
        if ( !m_bAsync )
            qApp->exit_loop();

        return;
    }
}

KatInfoExtractor* KatInfoExtractor::extractInfoJob( KatCatalog* cat, CppSQLite3DB* db )
{
    return new KatInfoExtractor( cat, db );
}

void KatInfoExtractor::slotDiscardInfo()
{
    m_thumbnail.clear();
    m_metadata.clear();
    m_fulltext.clear();
}

void KatInfoExtractor::slotSaveInfo ()
{
    // make all operations in one single transaction
    openTransaction();

    m_save = true;

    int totalPreviews = m_totalPreviews;
    int totalMetaData = m_totalMetaData;
    int totalFullTexts = m_totalFullTexts;
    int totalExtractions = m_totalExtractions;

    QIntDictIterator<QByteArray> tn_it( m_thumbnail );
    for ( ; tn_it.current() && m_save ; ++tn_it )
    {
        saveThumbnailRecord( tn_it.currentKey(), tn_it.current(), tn_it.current()->size() );

        totalPreviews++;
        totalExtractions++;
    }
    m_thumbnail.clear();

    if ( !m_save )
    {
        rollbackTransaction();
        return;
    }

    // Now save metadata records
    QIntDictIterator<QString> md_it( m_metadata );
    for ( ; md_it.current() && m_save ; ++md_it )
    {
        if ( !md_it.current()->isEmpty() )
        {
            saveMetaDataRecord( md_it.currentKey(), md_it.current() );

            totalMetaData++;
        }
    }
    m_metadata.clear();

    if ( !m_save )
    {
        rollbackTransaction();
        return;
    }

    int fileId = 0;
    QString* text;

    QIntDictIterator<QString> ft_it( m_fulltext );
    while  ( ft_it.current() && m_save )
    {
        fileId = ft_it.currentKey();
        text = ft_it.current();

        // Put this
        if ( *text == "<fulltext></fulltext>" )
        {
            totalFullTexts++;
            totalExtractions++;

            m_fulltext.remove( fileId );
            continue;
        }

        *text = text->replace( "<fulltext>", QString::null ); // strip the <fulltext></fulltext> tags
        *text = text->replace( "</fulltext>", QString::null );
        *text = text->replace( QRegExp( "<page[^>]*>" ), " " ); // strip all <page x> tags
        *text = text->replace( QRegExp( "</page>" ), " " ); // strip all </page> tags
        *text = text->simplifyWhiteSpace(); // clean up white spaces

        if ( text->isEmpty() )
        {
            totalFullTexts++;
            totalExtractions++;

            m_fulltext.remove( fileId );
            continue;
        }

        // save fulltext
        QByteArray fullTextData = QCString( *text );
        fullTextData = qCompress( fullTextData );
        long fullTextDataLength = fullTextData.size();
        saveFullTextRecord( fileId, fullTextData, fullTextDataLength );

        // clean up the text a bit more
        *text = text->lower();
        *text = text->replace( QRegExp( "[\\W]" ), " " );
        *text = text->replace( QRegExp( "[0-9]" ), " " );
        *text = text->simplifyWhiteSpace();

        if ( text->isEmpty() )
        {
            m_totalFullTexts++;
            m_cat->setFullTexts( m_totalFullTexts );
            m_totalExtractions++;
            m_cat->setExtractions( m_totalExtractions );

            m_fulltext.remove( fileId );
            continue;
        }
        ++ft_it;
    }

    if ( !m_save )
    {
        rollbackTransaction();
        return;
    }

    commitTransaction();

    m_totalPreviews = totalPreviews;
    m_totalMetaData = totalMetaData;
    m_totalExtractions = totalExtractions;
    m_totalFullTexts = totalFullTexts;

    m_cat->setThumbnails( m_totalPreviews );
    m_cat->setExtractions( m_totalExtractions );
    m_cat->setMetaData( m_totalMetaData );
    m_cat->setExtractions( m_totalExtractions );
    m_cat->setFullTexts( m_totalFullTexts );

    // Now save Word records
    QIntDictIterator<QString> wd_it( m_fulltext );
    for ( ; wd_it.current() && m_save ; ++wd_it )
    {
        fileId = wd_it.currentKey();
        text = wd_it.current();

        // create a list of words out of fulltext
        QStringList words = QStringList::split( " ", *text, false );
        if ( !words.empty() )
        {
            words.sort();

            long wordId;
            long o;

            int totalWords = m_cat->words();

            // make all operations in one single transaction
            openTransaction();

            // cycle the words, count occurrences and check if they are
            // already present in database
            // long wordCount = 0L;
            // long wordsTotal = words.size();
            QStringList::Iterator it = words.begin();
            while ( ( it != words.end() ) && m_save )
            {
                QString currentWord = *it;
                it++;

                o = 1;
                while ( *it == currentWord )
                {
                    o++;
                    it++;
                }

                // check if the word is already present in the database
                wordId = getWordId( currentWord );
                if ( wordId == -1 )
                {
                    // if the word is not present, save it with a new wordid
                    wordId = saveWordRecord( currentWord );
                    saveWordfileRecord( wordId, fileId, o );

                    totalWords++;
                }
                else
                {
                    // if it is present, save only the wordfile record
                    saveWordfileRecord( wordId, fileId, o );
                }
            }

            if ( !m_save )
            {
                rollbackTransaction();
                return;
            }

            commitTransaction();

            m_cat->setWords( totalWords );
        }

        if ( text->length() < 20 )
        {
            // If it is shorter, we don't try to identify language
            m_totalFullTexts++;
            m_cat->setFullTexts( m_totalFullTexts );
            m_totalExtractions++;
            m_cat->setExtractions( m_totalExtractions );
            continue;
        }

        // identify the language of the document
        QString language = KatLanguageManager::identifyLanguage( *text, *m_lp );

        // update the file record with the information about the language
        updateLanguageOfFile( fileId, language );

        m_totalFullTexts++;
        m_cat->setFullTexts( m_totalFullTexts );
        m_totalExtractions++;
        m_cat->setExtractions( m_totalExtractions );

    }

    m_fulltext.clear();
}

void KatInfoExtractor::gotThumbnail( const KFileItem* kfileitem, const QPixmap& qpixmap )
{
    extraInfo* extra = (extraInfo*)kfileitem->extraData( "extra" );
    long fileid = extra->fileId;

    QByteArray* ba = new QByteArray();
    QBuffer buffer( *ba );
    buffer.open( IO_WriteOnly );
    qpixmap.save( &buffer, "PNG" );
    buffer.close();

    m_thumbnail.insert ( fileid, ba );

    delete extra;
    extra = 0L;
}

void KatInfoExtractor::noThumbnail( const KFileItem* kfileitem )
{
    extraInfo* extra = (extraInfo*)kfileitem->extraData( "extra" );
    m_totalExtractions++;
    m_cat->setExtractions( m_totalExtractions );

    delete extra;
    extra = 0L;
}

void KatInfoExtractor::gotMetaInfo( const KFileItem* kfileitem )
{
    extraInfo* extra = (extraInfo*)kfileitem->extraData( "extra" );
    long fileid = extra->fileId;

    if ( kfileitem->metaInfo().isValid() && !kfileitem->metaInfo().isEmpty() )
    {
        QStringList sk = kfileitem->metaInfo().supportedKeys();
        QString* metaData = new QString("");
        QStringList::Iterator end( sk.end() );
        for ( QStringList::Iterator it = sk.begin(); it != end; ++it )
        {
            QString key = *it;

            if ( kfileitem->metaInfo().contains( key ) )
            {
                // PRESENT
                KFileMetaInfoItem mii = kfileitem->metaInfo().item( key );

                if ( mii.isValid() )
                {
                    QString mKey( mii.key() );

                    QVariant::Type type( mii.type() );
                    QString sType;

                    switch ( type )
                    {
                        case QVariant::BitArray:
                            sType = "bitarray";
                            break;
                        case QVariant::ByteArray:
                            sType = "bytearray";
                            break;
                        case QVariant::Bitmap:
                            sType = "bitmap";
                            break;
                        case QVariant::Bool:
                            sType = "bool";
                            break;
                        case QVariant::Brush:
                            sType = "brush";
                            break;
                        case QVariant::Color:
                            sType = "color";
                            break;
                        case QVariant::ColorGroup:
                            sType = "colorgroup";
                            break;
                        case QVariant::Cursor:
                            sType = "cursor";
                            break;
                        case QVariant::Date:
                            sType = "date";
                            break;
                        case QVariant::DateTime:
                            sType = "datetime";
                            break;
                        case QVariant::Double:
                            sType = "double";
                            break;
                        case QVariant::Font:
                            sType = "font";
                            break;
                        case QVariant::IconSet:
                            sType = "iconset";
                            break;
                        case QVariant::Image:
                            sType = "image";
                            break;
                        case QVariant::Int:
                            sType = "int";
                            break;
                        case QVariant::KeySequence:
                            sType = "keysequence";
                            break;
                        case QVariant::List:
                            sType = "list";
                            break;
                        case QVariant::LongLong:
                            sType = "longlong";
                            break;
                        case QVariant::ULongLong:
                            sType = "unsignedlonglong";
                            break;
                        case QVariant::Map:
                            sType = "map";
                            break;
                        case QVariant::Palette:
                            sType = "palette";
                            break;
                        case QVariant::Pen:
                            sType = "pen";
                            break;
                        case QVariant::Pixmap:
                            sType = "pixmap";
                            break;
                        case QVariant::Point:
                            sType = "point";
                            break;
                        case QVariant::PointArray:
                            sType = "pointarray";
                            break;
                        case QVariant::Rect:
                            sType = "rect";
                            break;
                        case QVariant::Region:
                            sType = "region";
                            break;
                        case QVariant::Size:
                            sType = "size";
                            break;
                        case QVariant::SizePolicy:
                            sType = "sizepolicy";
                            break;
                        case QVariant::String:
                            sType = "string";
                            break;
                        case QVariant::CString:
                            sType = "cstring";
                            break;
                        case QVariant::StringList:
                            sType = "stringlist";
                            break;
                        case QVariant::Time:
                            sType = "time";
                            break;
                        case QVariant::UInt:
                            sType = "unsignedint";
                            break;
                        default:
                            sType = "unknown";
                    }

                    if ( QString( mii.string() ).isEmpty() )
                    {
                        // INVALID VALUE
                    }
                    else
                    {
                        QString mValue( mii.string() );
                        mValue = mValue.simplifyWhiteSpace();

                        if ( !mValue.isEmpty() )
                            *metaData = *metaData + mKey + "|" + sType + "|" + mValue + "|";
                    }
                }
                else
                {
                    // INVALID ITEM
                }
            }
            else
            {
                // ABSENT
            }
        }

        if ( !metaData->isEmpty() )
            m_metadata.insert ( fileid, metaData );
        else
        {
            delete metaData;
            metaData = 0;
        }

    }
    else
    {
        // INVALID OR EMPTY METAINFO
    }

    m_totalExtractions++;
    m_cat->setExtractions( m_totalExtractions );

    delete extra;
    extra = 0L;
}

void KatInfoExtractor::noMetaInfo( const KFileItem* kfileitem )
{
    extraInfo* extra = (extraInfo*)kfileitem->extraData( "extra" );
    m_totalExtractions++;
    m_cat->setExtractions( m_totalExtractions );

    delete extra;
    extra = 0L;
}

void KatInfoExtractor::gotFullText( KIO::Job*, const KFileItem* kfileitem, const QString& fullText )
{
    extraInfo* extra = (extraInfo*)kfileitem->extraData( "extra" );
    long fileId = extra->fileId;

    m_fulltext.insert ( fileId, new QString(fullText) );

    delete extra;
    extra = 0L;
}

void KatInfoExtractor::noFullText( const KFileItem* kfileitem )
{
    extraInfo* extra = (extraInfo*)kfileitem->extraData( "extra" );
    m_totalExtractions++;
    m_cat->setExtractions( m_totalExtractions );

    delete extra;
    extra = 0L;
}

void KatInfoExtractor::resultThumbnail( KIO::Job* job )
{
    m_subJobs.remove (job);
    if ( m_subJobs.isEmpty() )
        emit completed();
}

void KatInfoExtractor::resultMetaInfo( KIO::Job* job )
{
    m_subJobs.remove (job);
    if ( m_subJobs.isEmpty() )
        emit completed();
}

void KatInfoExtractor::resultFullText( KIO::Job* job )
{
    m_subJobs.remove (job);
    if ( m_subJobs.isEmpty() )
        emit completed();
}

void KatInfoExtractor::slotAbortExtraction()
{
    for (KIO::Job *job = m_subJobs.first(); job; job = m_subJobs.next())
        job->kill();

    m_subJobs.clear();

    emit completed();
}

void KatInfoExtractor::slotAbortSaveInfo()
{
    m_save = false;
}

// Database access functions

int KatInfoExtractor::openTransaction()
{
    try
    {
        m_db->execDML( "begin transaction;" );
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << e.errorMessage() << endl;
        return e.errorCode();
    }
    return 0;
}

int KatInfoExtractor::commitTransaction()
{
    try
    {
        m_db->execDML( "commit transaction;" );
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << e.errorMessage() << endl;
        return e.errorCode();
    }
    return 0;
}

int KatInfoExtractor::rollbackTransaction()
{
    try
    {
        m_db->execDML( "rollback transaction;" );
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << e.errorMessage() << endl;
        return e.errorCode();
    }
    return 0;
}

long KatInfoExtractor::getWordId( QString& word )
{
    QString DML;
    long wordId;

    // TODO: Find a regular expression to eliminate all possible problems with words in SQL
    QString sword( word );
    sword = sword.replace( QRegExp( "'" ), "''" );

    try
    {
        DML = "select wordid from words where word = '" + sword + "';";
        CppSQLite3Query q = m_db->execQuery( DML );
        if ( q.eof() )
        {
            // word absent
            wordId = -1L;
        }
        else
        {
            // word already present
            wordId = q.getIntField( "wordid" );
        }
        q.finalize();

    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << e.errorMessage() << endl;
        return -1L;
    }

    return wordId;
}

long KatInfoExtractor::saveWordRecord( QString &word )
{
    QString DML;

    // TODO: Find a regular expression to eliminate all possible problems with words in SQL
    QString sword( word );
    sword = sword.replace( QRegExp("'"), "''" );

    try
    {
        // insert the word into the words table
        CppSQLite3Statement stmt = m_db->compileStatement( "insert into words (word) values(?);" );
        stmt.bind( 1, word );

        stmt.execDML();
        stmt.finalize();
    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << e.errorMessage() << " WORD WAS: " << sword << " WORDID: " << m_db->lastRowId() << endl;
        return -1;
    }

    return m_db->lastRowId();
}

int KatInfoExtractor::saveWordfileRecord( long wordId, long fileId, long wordCount )
{
    QString DML;

    try
    {
        // insert the wordfile information into the wordfile table
        CppSQLite3Statement stmt = m_db->compileStatement( "insert into wordfile values(?,?,?);" );
        stmt.bind( 1, wordId );
        stmt.bind( 2, fileId );
        stmt.bind( 3, wordCount );

        stmt.execDML();
        stmt.finalize();

    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << e.errorMessage() << endl;
        return e.errorCode();
    }

    return 0;
}

int KatInfoExtractor::saveThumbnailRecord( int fileId, QByteArray* thumbnailData, long thumbnailDataLength )
{
    // Save the thumbnail into the thumbnails table
    try
    {
        CppSQLite3Statement stmt = m_db->compileStatement( "insert into thumbnails values(?,?,?);" );
        stmt.bind(  1, fileId );
        stmt.bind(  2, (unsigned char*)thumbnailData->data(), thumbnailDataLength );
        stmt.bind(  3, thumbnailDataLength );

        stmt.execDML();
        stmt.finalize();

    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << e.errorMessage() << endl;
        return e.errorCode();
    }

    return 0;
}

int KatInfoExtractor::saveMetaDataRecord( int fileId, QString* metaData )
{
    // Tokenize the metaData string and insert it into the metadata table
    if ( !metaData->isNull() )
    {
        QString name;
        QString type;
        QString value;

        QStringList meta = QStringList::split( "|", *metaData, false );
        QStringList::Iterator end( meta.end() );
        for ( QStringList::Iterator it = meta.begin(); it != end; ++it )
        {
            name = *it;
            it++;
            type = *it;
            it++;
            value = *it;

            try
            {
                CppSQLite3Statement stmt = m_db->compileStatement( "insert into metadata values(?,?,?,?);" );
                stmt.bind(  1, fileId );
                stmt.bind(  2, name );
                stmt.bind(  3, type );
                stmt.bind(  4, value );

                stmt.execDML();
                stmt.finalize();

            }
            catch ( CppSQLite3Exception& e )
            {
                kdDebug() << e.errorMessage() << endl;
                return e.errorCode();
            }
        }
    }

    return 0;
}

int KatInfoExtractor::saveFullTextRecord( int fileId, QByteArray& fullTextData, long fullTextDataLength )
{
    // Save the fulltext into the fulltexts table
    try
    {
        CppSQLite3Statement stmt = m_db->compileStatement( "insert into fulltexts values(?,?,?);" );
        stmt.bind(  1, fileId );
        stmt.bind(  2, (unsigned char*)fullTextData.data(), fullTextDataLength );
        stmt.bind(  3, fullTextDataLength );

        stmt.execDML();
        stmt.finalize();

    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << e.errorMessage() << endl;
        return e.errorCode();
    }

    return 0;
}

int KatInfoExtractor::updateLanguageOfFile( int fileId, QString& language )
{
    // Update the language information in the file record
    try
    {
        m_db->execDML( "update files set language = '" + language + "' where fileid = " + QString::number( fileId ) + ";" );

    }
    catch ( CppSQLite3Exception& e )
    {
        kdDebug() << e.errorMessage() << endl;
        return e.errorCode();
    }

    return 0;
}

QMap<QString, long> KatInfoExtractor::readFiles( int catalogId, QStringList nfiles )
{
    QString DML;
    QMap<QString, long> files;

    QStringList::Iterator it = nfiles.begin();
    while ( it != nfiles.end() )
    {
        try
        {
            DML = "select catalogid, fileid, fullname from files where catalogid = " + QString::number( catalogId ) +
                    " and fullname = '" + *it + "';";
            CppSQLite3Query q = m_db->execQuery( DML );
            if ( !q.eof() )
            {
                files[ q.getStringField( "fullname" ) ] = q.getIntField( "fileid" );
            }
            q.finalize();

        }
        catch ( CppSQLite3Exception& e )
        {
            kdDebug() << k_funcinfo << " " << e.errorMessage() << endl;
        }
        it++;
    }

    return files;
}

void KatInfoExtractor::deleteFiles( const QMap<QString, long>& files )
{
    QString DML;

    openTransaction();

    QMap<QString, long>::ConstIterator it = files.begin();
    QMap<QString, long>::ConstIterator end( files.end() );
    for (; it != end; it++)
    {
        try
        {
            if ( it.data() ) {
                if (m_db->execDML( "delete from fulltexts where fileid = " + QString::number( it.data() ) + ";" )) {
                    m_totalFullTexts --;
                    m_cat->setFullTexts(m_totalFullTexts);
                }
                if (m_db->execDML( "delete from thumbnails where fileid = " + QString::number( it.data() ) + ";" )) {
                    m_totalPreviews --;
                    m_cat->setThumbnails(m_totalPreviews);
                }
                if (m_db->execDML( "delete from metadata where fileid = " + QString::number( it.data() ) + ";" )) {
                    m_totalMetaData --;
                    m_cat->setMetaData(m_totalMetaData);
                }
                //FIXME -- the corresponding words must be deleted from the 'words' table
                //NOTE - It might be better not to delete from the words file since they may crop up again
                //      when new files are added to the database
                if (m_db->execDML( "delete from wordfile where fileid = " + QString::number( it.data() ) + ";" ))
                    m_cat->setWords( m_cat->words() - 1 );
            }
            else
            {
                kdDebug () << "No metainfo exist in the database for file " << it.key() << endl;
            }
        }
        catch ( CppSQLite3Exception& e )
        {
            kdDebug() << k_funcinfo << " " << e.errorMessage() << endl;
        }
    }

    // delete the WORDS which do not have a record in WORDFILE
    m_db->execDML( "delete from words where wordid not in (select distinct wordid from wordfile);" );

    commitTransaction();
}

void KatInfoExtractor::deleteFiles( const QStringList& files )
{
    QMap<QString, long> rFiles = readFiles( m_cat->catalogId(), files);

    deleteFiles( rFiles );
}

#include "katinfoextractor.moc"
