/*
 * cups.cc
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <cassert>
#include "cups.hh"
#include <map>
#include "utils.hh"
#include "macros.h"

// TODO: Handle non English encodings.

Cups::Cups()
{
  http = NULL;
  cups_dest_t *dests;
  int y = cupsGetDests(&dests);
  for (int x = 0; x < y; x++)
    {
      CupsPrinter printer;
      printer.can_custom = false;
      printer.name = dests[x].name;
      if (dests[x].instance)
	printer.instance = dests[x].instance;
      if (dests[x].is_default == 1)
	printer.is_default = true;
      else
	printer.is_default = false;
      for (int i = 0; i < dests[x].num_options; i++)
	{
	  CupsOption option;
	  option.name = dests[x].options[i].name;
	  option.value = dests[x].options[i].value;
	  printer.options.push_back(option);
	}

      // Now for the ppd.
      const char *ppd = cupsGetPPD(dests[x].name);
      ppd_file_t *ppd_t = ppdOpenFile(ppd);

      // Let's generate our dictionary.
      std::map<std::string, std::string> dict;
      for (int g = 0; g < ppd_t->num_groups; g++)
	{
	  for (int o = 0; o < ppd_t->groups[g].num_options; o++)
	    {
	      if (!strcmp(ppd_t->groups[g].options[o].keyword, "PageSize"))
		{
		  printer.default_paper = ppd_t->groups[g].options[o].defchoice;
		  for (int c = 0; c < ppd_t->groups[g].options[o].num_choices; c++)
		    {
		      // Let's populate our dictionary.
dict[ppd_t->groups[g].options[o].choices[c].choice] = ppd_t->groups[g].options[o].choices[c].text;
		    }
		  break;
		}
	    }
	}

      for (int z = 0; z < ppd_t->num_sizes; z++)
	{
	  // We can also use variable_sizes. It'll tell us whether we can have
	  // Custom sizes or not.
	  if (!strcmp(ppd_t->sizes[z].name, "Custom"))
	    {
	      printer.can_custom = true;
	      printer.custom.id = ppd_t->sizes[z].name;
	      printer.custom.name = dict[ppd_t->sizes[z].name];
	    }
	  else
	    {
	      CupsSize size;
	      size.id = ppd_t->sizes[z].name;
	      size.name = dict[ppd_t->sizes[z].name];
	      size.width = ppd_t->sizes[z].width;
	      size.length = ppd_t->sizes[z].length;
	      size.left = ppd_t->sizes[z].left;
	      size.bottom = ppd_t->sizes[z].bottom;
	      size.right = ppd_t->sizes[z].right;
	      size.top = ppd_t->sizes[z].top;
	      printer.sizes.push_back(size);
	    }
	}
      if (printer.can_custom)
	{
	  printer.custom.min1 = ppd_t->custom_min[0];
	  printer.custom.min2 = ppd_t->custom_min[1];
	  printer.custom.max1 = ppd_t->custom_max[0];
	  printer.custom.max2 = ppd_t->custom_max[1];
	  printer.custom.margin1 = ppd_t->custom_margins[0];
	  printer.custom.margin2 = ppd_t->custom_margins[1];
	  printer.custom.margin3 = ppd_t->custom_margins[2];
	  printer.custom.margin4 = ppd_t->custom_margins[3];
	}
      ppdClose(ppd_t);
      set_status(printer);
      printers.push_back(printer);
    }
  _has_default = false;
  cupsFreeDests(y, dests);
  const char *def = cupsGetDefault();
  if (def)
    {
      for (unsigned j = 0; j < printers.size(); j++)
	{
	  if (!strcmp(printers[j].name.c_str(), def))
	    {
	      _default = &printers[j];
	      _has_default = true;
	      break;
	    }
	}
    }
  if (http)
    {
      httpClose(http);
      http = NULL;
    }
}

Cups::~Cups()
{
}

std::vector<std::string> Cups::sizes(unsigned x)
{
  std::vector<std::string> ret;
  assert(x < printers.size());
  for (unsigned i = 0; i < printers[x].sizes.size(); i++)
    {
      ret.push_back(printers[x].sizes[i].name);
    }
  return ret;
}

std::vector<std::string> Cups::get_all()
{
  std::vector<std::string> all;
  for (unsigned j = 0; j < printers.size(); j++)
    all.push_back(printers[j].name);
  return all;
}

std::string Cups::get_default()
{
  if (_default)
    return _default->name;
  else if (printers.size() > 0)
    return printers[0].name;
  else
    return "";
}

bool Cups::print(std::string& file, std::string& err)
{
  int jid=cupsPrintFile(printers[printer].name.c_str(), file.c_str(), file.c_str(), 0, NULL);
  if (jid > 0)
    return true;
  err = _("Couldn't print the file.");
  // TODO: How to gt the error from cups ??
  //  err = ippErrorString(cupsLastError());
  //  err = cupsLastErrorString();
  return false;
}

// TODO: Any memory leaks ??
void Cups::set_status(CupsPrinter& printer)
{
  printer.state = _("Unknown");
  if (!http)
    {
      http = httpConnectEncrypt(cupsServer(), ippPort(),
				cupsEncryption());
      if (http == NULL)
	  return;
    }

  std::string url = Utils::substitute("ipp://localhost/printers/%s", printer.name);
  ipp_t *request = ippNewRequest(CUPS_GET_PRINTERS);
  //  ipp_t *request = ippNew();
  ipp_t *response;
  ipp_attribute_t *attr;

  static const char *pattrs[] = {
    "printer-state",
    "printer-state-message",
    "printer-state-reasons",
  };

  ippAddStrings(request, IPP_TAG_OPERATION, IPP_TAG_KEYWORD,
                "requested-attributes", sizeof(pattrs) / sizeof(pattrs[0]),
                NULL, pattrs);
  ippAddString(request, IPP_TAG_OPERATION, IPP_TAG_NAME, "requesting-user-name",
               NULL, cupsUser());


  ippAddString (request, IPP_TAG_OPERATION, IPP_TAG_URI,
		"printer-uri", NULL, url.c_str());
  ippAddString (request,  IPP_TAG_PRINTER, IPP_TAG_NAME,
		"printer-name", NULL, printer.name.c_str());
  ippAddBoolean (request, IPP_TAG_PRINTER, "printer-is-accepting-jobs", 1);
  ippAddInteger(request, IPP_TAG_PRINTER, IPP_TAG_ENUM, "printer-state",
		IPP_PRINTER_IDLE);

  if ((response = cupsDoRequest(http, request, "/")) != NULL)
    {
      if (response->request.status.status_code > IPP_OK_CONFLICT)
	{
	  ippDelete(response);
	  return;
	}
      else {
	for (attr = response->attrs; attr != NULL; attr = attr->next)
	  {
	    if ((!strcmp(attr->name, "printer-state")) && attr->value_tag == IPP_TAG_ENUM)
	      {
		switch ((ipp_pstate_t)attr->values[0].integer)
		  {
		  case IPP_PRINTER_IDLE:
		    printer.state = _("Ready");
		    break;
		  case IPP_PRINTER_PROCESSING:
		    printer.state = _("Printing");
		    break;
		  case IPP_PRINTER_STOPPED:
		    printer.state = _("Disabled");
		    break;
		  }
	      }
	    attr = attr->next;
	    if (attr == NULL)
	      break;
	  }
      }
    }
}
