/***************************************************************************
 *   Copyright (C) 2004-2005 by Thomas Fischer                             *
 *   fischer@unix-ag.uni-kl.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <qregexp.h>

#include <encoderlatex.h>
#include <file.h>
#include <entry.h>
#include "idsuggestions.h"

namespace KBibTeX
{
    const QRegExp IdSuggestions::unwantedChars( "\\s|['\"$\\\\{}:-,]+" );

    IdSuggestions::IdSuggestions()
    {
// nothing
    }


    IdSuggestions::~IdSuggestions()
    {
// nothing
    }

    /**
    * Determine list of authors for a given entry
    */
    QStringList IdSuggestions::authorsLastName( BibTeX::Entry *entry )
    {
        QStringList result;

        /** retrieve field holding authors information for entry */
        BibTeX::EntryField *field = entry->getField( BibTeX::EntryField::ftAuthor );
        if ( field == NULL )
            return result; /** no author field available */

        /** fetch container holding list of author names */
        BibTeX::PersonContainer *personContainer = field != NULL ? dynamic_cast<BibTeX::PersonContainer*>( field->value()->items.isEmpty() ? NULL : field->value()->items.first() ) : NULL;
        if ( personContainer == NULL || personContainer->persons.isEmpty() )
            return result; /** container not found or is empty */

        /** iterate through container and fetch each author's last name */
        for ( QValueList<BibTeX::Person*>::ConstIterator it = personContainer->persons.begin(); it != personContainer->persons.end(); ++it )
            result.append(( *it )->lastName() );

        return result;
    }

    /**
    * Normalize a given text by removing spaces or other unwanted characters
    */
    QString IdSuggestions::normalizeText( const QString& text )
    {
        BibTeX::EncoderLaTeX encoder;
        QString result = text;
        result = encoder.encode( result );
        result = result.remove( unwantedChars );

        return result;
    }

    /**
    * Determine year for a given entry
    */
    int IdSuggestions::extractYear( BibTeX::Entry *entry )
    {
        /** retrieve field holding year information for entry */
        BibTeX::EntryField *field = entry->getField( BibTeX::EntryField::ftYear );
        if ( field == NULL )
            return -1; /** no year field available */

        /** *fetch value item holding year */
        BibTeX::ValueItem *valueItem = field != NULL ? ( field->value()->items.isEmpty() ? NULL : field->value()->items.first() ) : NULL;
        if ( valueItem == NULL )
            return -1; /** no value item found or is empty */

        /** parse value item's text */
        bool ok = FALSE;
        int year = QString( valueItem->text() ).toInt( &ok );
        if ( !ok ) year = -1;

        return year;
    }

    /**
    * Determine title for a given entry
    */
    QString IdSuggestions::extractTitle( BibTeX::Entry *entry )
    {
        /** retrieve field holding title information for entry */
        BibTeX::EntryField *field = entry->getField( BibTeX::EntryField::ftTitle );
        if ( field == NULL )
            return QString::null; /** no title field available */

        /** *fetch value item holding title */
        BibTeX::ValueItem *valueItem = field->value()->items.isEmpty() ? NULL : field->value()->items.first();
        if ( valueItem == NULL )
            return QString::null; /** no value item found or is empty */

        return valueItem->text(); // TODO: Perform some postprocessing?
    }

    QStringList IdSuggestions::createSuggestions( BibTeX::File *file, BibTeX::Entry *entry, const QStringList& formatStrList )
    {
        QStringList result;
        QStringList allKeys = file->allKeys();
        entry = new BibTeX::Entry( entry );
        file->completeReferencedFields( entry );

        for ( QStringList::ConstIterator it = formatStrList.begin(); it != formatStrList.end(); ++it )
        {
            QString id;
            QStringList tokenList = QStringList::split( '|', *it );
            for ( QStringList::ConstIterator tit = tokenList.begin(); tit != tokenList.end(); ++tit )
                id.append( translateToken( entry, *tit ) );

            if ( id.isEmpty() || result.contains( id ) )
                continue;

            if ( !allKeys.contains( id ) )
                result.append( id );
            else
                for ( int i = 0; i < 26; ++i )
                {
                    QString id2 = id.append( QChar( 'a' + i ) );
                    if ( !allKeys.contains( id2 ) )
                    {
                        result.append( id2 );
                        break;
                    }
                }
        }

        delete entry;

        return result;
    }

    QString IdSuggestions::translateToken( BibTeX::Entry *entry, const QString& token )
    {
        switch ( token[0] )
        {
        case 'a': return translateAuthorsToken( entry, token.mid( 1 ), TRUE );
        case 'A': return translateAuthorsToken( entry, token.mid( 1 ), FALSE );
        case 'y':
            {
                int year = extractYear( entry );
                if ( year > -1 )
                    return QString::number( year % 100 + 100 ).mid( 1 );
                else
                    return QString::null;
            }
        case 'Y':
            {
                int year = extractYear( entry );
                if ( year > -1 )
                    return QString::number( year % 10000 + 10000 ).mid( 1 );
                else
                    return QString::null;
            }
        case 't': return translateTitleToken( entry, token.mid( 1 ), FALSE );
        case 'T': return translateTitleToken( entry, token.mid( 1 ), TRUE );
        case '"': return token.mid( 1 );
        default: return QString::null;
        }
    }

    QString IdSuggestions::translateAuthorsToken( BibTeX::Entry *entry, const QString& token, bool onlyFirst )
    {
        unsigned int len = 0x00ffffff;
        unsigned int pos = 0;
        bool toLower = FALSE;
        bool toUpper = FALSE;
        QString inBetween = QString::null;

        if ( token.length() > pos )
        {
            int dv = token[pos].digitValue();
            if ( dv > -1 )
            {
                len = dv;
                ++pos;
            }
        }

        if ( token.length() > pos )
        {
            toLower = token[pos] == 'l';
            toUpper = token[pos] == 'u';
            if ( toUpper || toLower )
                ++pos;
        }

        if ( token.length() > pos + 1 && token[pos] == '"' )
            inBetween = token.mid( pos + 1 );

        QString result;
        bool first = TRUE;
        QStringList authors = authorsLastName( entry );
        for ( QStringList::ConstIterator it = authors.begin(); it != authors.end(); ++it )
        {
            if ( first )
                first = FALSE;
            else
                result.append( inBetween );

            QString author =  normalizeText( *it ).left( len );
            result.append( author );
            if ( onlyFirst )
                break;
        }

        if ( toUpper )
            result = result.upper();
        else if ( toLower )
            result = result.lower();

        return result;
    }

    QString IdSuggestions::translateTitleToken( BibTeX::Entry *entry, const QString& token, bool removeSmallWords )
    {
        unsigned int len = 0x00ffffff;
        unsigned int pos = 0;
        bool toLower = FALSE;
        bool toUpper = FALSE;
        QString inBetween = QString::null;

        if ( token.length() > pos )
        {
            int dv = token[pos].digitValue();
            if ( dv > -1 )
            {
                len = dv;
                ++pos;
            }
        }

        if ( token.length() > pos )
        {
            toLower = token[pos] == 'l';
            toUpper = token[pos] == 'u';
            if ( toUpper || toLower )
                ++pos;
        }

        if ( token.length() > pos + 1 && token[pos] == '"' )
            inBetween = token.mid( pos + 1 );

        QString result;
        bool first = TRUE;
        QStringList titleWords = QStringList::split( QRegExp( "\\s+" ), extractTitle( entry ) );
        for ( QStringList::ConstIterator it = titleWords.begin(); it != titleWords.end(); ++it )
        {
            if ( first )
                first = FALSE;
            else
                result.append( inBetween );

            QString lowerText = ( *it ).lower();
            if ( !removeSmallWords || ( lowerText != "and" && lowerText != "on" && lowerText != "in"  && lowerText != "the" ) )
                result.append( normalizeText( *it ).left( len ) );
        }

        if ( toUpper )
            result = result.upper();
        else if ( toLower )
            result = result.lower();

        return result;
    }

}
