/***************************************************************************
                         ftpsession.h  -  pure-ftpd FTP session data
                         -------------------
    begin                : Wed Feb 6 2002
    copyright            : (C) 2002-2003 by Claudiu Costin
    email                : claudiuc@kde.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef FTPSESSION_H
#define FTPSESSION_H

#include <sys/types.h>

#include <qstring.h>
#include <qvaluelist.h>

/**
  * 
  * @short Store pure-ftpd session data grabbed from an XML structure
  * @author Claudiu Costin <claudiuc@kde.org>
  * @version $Id: ftpsession.h,v 1.7 2003/07/23 23:11:50 claudiuc Exp $
  */

class FTPSessionItem {
  public:
    /** Posible states for an FTP session, as is reported by pure-ftpwho
      *
      */
    enum  Status {
      /** User connected to pure-ftpd but doesn't authenticated yet
        */
      Idle, 
      /** User download a file from server
        */
      Download, 
      /** User upload a file to server
        */
      Upload, 
      /** Error condition in FTP session. Session will close soon.
        */
      Error, 
      /** Unknow FTP state from pure-ftpwho. Normaly should not happen at all.
        */
      Unknown
    };
    
    /** Build an empty FTP session item with attributes set to 0 
      * or null string
      */
    FTPSessionItem();
    /** Copy constructor FTP session item.
      */
    FTPSessionItem(const FTPSessionItem &item);
    /** Make a new FTP session item from parameters values
      * @param percentage upload or download session completion as percent
      * @param pid pure-ftpd server process PID which manage this session
      * @param state type of current session 
      * @param time elapsed time from the moment when user connected
      * @param bandwidth transfer rate for current upload or download operation
      * @param resume byte position from which user restarted the transfer. It's 0 if not restarted.
      * @param current_size amount of bytes transfered so far
      * @param total_size size for uploaded or downloaded file
      * @param account user account is set only after authentication
      * @param file current transfered file name
      * @param host host name from which FTP connection was made
      * @param localhost where the pure-ftpd server is bind to serve current session
      * @param port where the pure-ftpd server listen to serve current session
      */
    FTPSessionItem(int percentage, pid_t pid, Status state, unsigned long time,
               unsigned long bandwidth,
               unsigned long long resume, unsigned long long current_size,
               unsigned long long total_size,
               const QString &account,
               const QString &file, const QString &host,
               const QString &localhost, const QString &port);
    /** Destroy an FTP session object.
      */
    ~FTPSessionItem();
    
    /** Return the completion percentage for current download or upload operation.
      */
    int percentage()          { return _percentage; }
    pid_t pid()               { return _pid; }
    unsigned long time()      { return _time; }
    unsigned long bandwidth() { return _bandwidth; }
    QString account()         { return _account; }
    Status state()            { return _state; }
    QString file()            { return _file; }
    QString host()            { return _host; }
    QString localhost()       { return _localhost; }
    QString port()            { return _port; }
    unsigned long long resume()       { return _resume; }
    unsigned long long current_size() { return _current_size; }
    unsigned long long total_size()   { return _total_size; }
    

  private:  
    /** Percentage of current download or upload operation
      */
    int _percentage;
    /** Pure-ftpd server process PID which manage this session
      */
    pid_t _pid;
    /** State type of current session 
      */
    Status _state;
    unsigned long 
    /** Elapsed time from the moment when user connected. Represented as seconds.
      */
    _time, 
    /** Transfer rate for current upload or download operation. Represented as bytes/s.
      */
    _bandwidth;
    unsigned long long 
    /** Byte position from which user restarted the transfer; 0 if not restarted
      */
    _resume, 
    /** Amount of bytes transfered so far
      */
    _current_size, 
    /** Size for uploaded or downloaded file
      */
    _total_size;
    QString 
    /** User account for which FTP connection is authorized. It is null before
      * authentication is made.
      */
    _account, 
    /** Current transfered file name
      */
    _file, 
    /** Host name or IP from which FTP connection come
      */
    _host, 
    /** Host name or IP where the pure-ftpd server is bind to serve current session
      */
    _localhost, 
    /** Port where the pure-ftpd server listen to serve current session
      */
    _port;
};


#endif
