/***************************************************************************
 *                                                                         *
 *   copyright (C) 2004 by Michael Buesch                                  *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#include "pwminit.h"
#include "configuration.h"
#include "randomizer.h"
#include "pwm.h"
#include "pwmexception.h"
#include "pwmtray.h"
#include "pwmdoc.h"
#include "pwgenwndimpl.h"

#ifdef CONFIG_KWALLETIF
# include "kwalletemu.h"
#endif // CONFIG_KWALLETIF

#include <qmessagebox.h>
#include <qclipboard.h>

#include <kmessagebox.h>
#include <kcmdlineargs.h>
#include <kapplication.h>
#include <kiconloader.h>
#include <kwin.h>
#include <kinputdialog.h>
#include <dcopclient.h>

#include <signal.h>

static PwMInit *sig_init_pointer;
static NOREGPARM void sig_handler(int signum)
{
	switch (signum) {
		case SIGINT:
		case SIGTERM:
			sig_init_pointer->shutdownApp(20 + signum);
			break;
		default:
			printDebug(string("unhandled signal ")
				   + tostr(signum));
	}
}

PwMInit::PwMInit(KApplication *_app)
 : runStatus (unknown)
 , _curWidget (0)
 , _dcopClient (0)
 , _kwalletEmu (0)
 , _tray (0)
{
	sig_init_pointer = this;
	app = _app;
}

PwMInit::~PwMInit()
{
	// close all open mainwnds
	QValueList<PwM *>::iterator i = _mainWndList.begin(),
				    end = _mainWndList.end();
	while (i != end) {
		disconnect(*i, SIGNAL(closed(PwM *)),
			   this, SLOT(mainWndClosed(PwM *)));
		delete *i;
		++i;
	}
	_mainWndList.clear();
	// close all remaining open documents
	PwMDocList *_dl = PwMDoc::getOpenDocList();
	vector<PwMDocList::listItem> dl = *(_dl->getList());
	vector<PwMDocList::listItem>::iterator i2 = dl.begin(),
					       end2 = dl.end();
	while (i2 != end2) {
		delete (*i2).doc;
		++i2;
	}

#ifdef CONFIG_KWALLETIF
	delete_ifnot_null(_kwalletEmu);
#endif // CONFIG_KWALLETIF
	delete_ifnot_null(_tray);

	Configuration::cleanup();
}

void PwMInit::copyToClipboard(const QString &text)
{
	QClipboard *cb = QApplication::clipboard();
	if (cb->supportsSelection())
		cb->setText(text, QClipboard::Selection);
	cb->setText(text, QClipboard::Clipboard);
}

void PwMInit::genPassword(QWidget *parent)
{
	if (!parent)
		parent = sig_init_pointer->curWidget();
	PwGenWndImpl wnd(parent);
	if (wnd.exec() == 0)
		return;
	bool ok = false;
	QString pw(wnd.getPassword());
	pw = KInputDialog::getText(i18n("Generated Password"),
				   i18n("This is your new Password.\n"
					"Click on OK to copy it to the clipboard."),
				   pw, &ok, parent);
	if (ok)
		copyToClipboard(pw);
}

void PwMInit::initializeApp()
{
	PWM_ASSERT(runStatus == unknown);
	runStatus = init;
	initPosixSignalHandler();
	Configuration::init();
	if (conf()->confGlobBase64Storage()) {
		printWarn("You are using backward incompatible base64-storage. "
			  "(base64Storage=true in [GLOBAL] section of config file)\n"
			  "You will not be able to read the .pwm files with "
			  "PwManager versions before 1.2.4.");
	}
	initDCOP();
	initKWalletEmu();
	initTray();
	handleCmdLineArgs();

	if (!PwMDoc::getOpenDocList()->getList()->size()) {
		bool openDeeplocked = false;
		if (conf()->confGlobAutostartDeepLocked() ||
		    savedCmd.open_deeplocked)
			openDeeplocked = true;
		if (conf()->confWndAutoMinimizeOnStart() ||
		    savedCmd.minToTray) {
			PwMDoc *newDoc = createDoc();
			if (!newDoc->openDocUi(newDoc,
					       conf()->confGlobAutoStart(),
					       openDeeplocked)) {
				delete newDoc;
			}
		} else {
			createMainWnd(conf()->confGlobAutoStart(),
				      openDeeplocked,
				      true,
				      0,
				      savedCmd.minimized);
		}
	}

	runStatus = running;
}

void PwMInit::shutdownApp(int exitStatus)
{
	printDebug(string("PwMInit::shutdownApp(")
		   + tostr(exitStatus) + ") called.");
	PWM_ASSERT((runStatus == running) || (runStatus == init));
	runStatus = shutdown;
	QApplication::exit(exitStatus);
	/* The destructor of PwMInit is called when control
	 * leaves main()
	 */
}

QWidget * PwMInit::curWidget()
{
	if (!sig_init_pointer)
		return 0;
	return sig_init_pointer->_curWidget;
}

void PwMInit::initPosixSignalHandler()
{
	signal(SIGINT, sig_handler);
	signal(SIGTERM, sig_handler);
}

void PwMInit::initDCOP()
{
	_dcopClient = app->dcopClient();
	_dcopClient->setNotifications(true);
}

void PwMInit::initKWalletEmu(bool forceDisable, bool forceReload)
{
#ifdef CONFIG_KWALLETIF
	if (!conf()->confGlobKwalletEmu() ||
	    forceDisable) {
		delete_ifnot_null(_kwalletEmu);
		return;
	}
	try {
		if (_kwalletEmu && forceReload)
			delete_and_null(_kwalletEmu);
		if (!_kwalletEmu)
			_kwalletEmu = new KWalletEmu(this);
	} catch (PwMException e) {
		string errMsg("initializing KWallet emulation failed.  ID: ");
		errMsg += tostr(static_cast<int>(e.getId()));
		errMsg += "  err-message: ";
		errMsg += e.getMessage();
		printWarn(errMsg);
		return;
	}
#else // CONFIG_KWALLETIF
	PARAM_UNUSED(forceDisable);
	PARAM_UNUSED(forceReload);
#endif // CONFIG_KWALLETIF
}

void PwMInit::initTray()
{
	if (!conf()->confGlobTray()) {
		if (!_tray)
			return;
		_tray->hide();
		delete_and_null(_tray);
		return;
	}
	if (_tray)
		return;
	_tray = new PwMTray(this);
	connect(_tray, SIGNAL(quitSelected()),
		this, SLOT(removeTrayAndQuit()));
	connect(_tray, SIGNAL(closed(PwMTray *)),
		this, SLOT(trayIconClosed(PwMTray *)));
	KIconLoader icons;
	_tray->setPixmap(icons.loadIcon(PACKAGE_NAME, KIcon::Small));
	_tray->show();
	// connect the signals of all open documents.
	const vector<PwMDocList::listItem> *dl = PwMDoc::getOpenDocList()->getList();
	vector<PwMDocList::listItem>::const_iterator i = dl->begin(),
						     end = dl->end();
	while (i != end) {
		_tray->connectDocToTray((*i).doc);
		++i;
	}
}

void PwMInit::removeTrayAndQuit()
{
	PWM_ASSERT(_tray);
	// _tray is deleted in ~PwMInit
	shutdownApp(0);
}

PwM * PwMInit::createMainWnd(const QString &loadFile,
			     bool loadFileDeepLocked,
			     bool virginity,
			     PwMDoc *doc,
			     bool minimized)
{
	PwM *newWnd;
	if (!doc)
		doc = createDoc();
	newWnd = new PwM(this, doc, virginity);
	_mainWndList.push_back(newWnd);
	connect(newWnd, SIGNAL(closed(PwM *)),
		this, SLOT(mainWndClosed(PwM *)));
	connect(newWnd, SIGNAL(gotFocus(PwM *)),
		this, SLOT(setCurWidget(PwM *)));
	connect(newWnd, SIGNAL(lostFocus(PwM *)),
		this, SLOT(resetCurWidget()));
	if (minimized)
		newWnd->showMinimized();
	else
		newWnd->show();
	if (loadFile != QString::null &&
	    loadFile != "") {
		newWnd->openDoc(loadFile, loadFileDeepLocked);
	}
	return newWnd;
}

PwMDoc * PwMInit::createDoc()
{
	PwMDoc *doc = new PwMDoc(this);
#ifdef CONFIG_KWALLETIF
	if (kwalletEmu())
		kwalletEmu()->connectDocSignals(doc);
#endif
	if (_tray)
		_tray->connectDocToTray(doc);
	return doc;
}

void PwMInit::mainWndClosed(PwM *wnd)
{
	bool doMinimizeToTray = false;
	bool doDeleteDoc = false;
	dcopClient()->suspend();
	dcopClient()->setAcceptCalls(false);
again:
	if (wnd->isForceMinimizeToTray()) {
		if (unlikely(!_tray)) {
			/* This should not happen! If we set forceMinimizeToTray ,
			 * we must be sure that _tray exists.
			 */
			BUG();
			wnd->setForceMinimizeToTray(false);
			goto again;
		}
		doMinimizeToTray = true;
	} else {
		if (_tray &&
		    runStatus != shutdown &&
		    !wnd->isForceQuit() &&
		    !wnd->curDoc()->isDeleted()) {
		    	if (conf()->confWndClose())
				doDeleteDoc = true;
			else
				doMinimizeToTray = true;
		} else {
			doDeleteDoc = true;
		}
	}
	if (doMinimizeToTray) {
		PWM_ASSERT(_tray);
		int mmlock = conf()->confGlobMinimizeLock();
		switch (mmlock) {
		case 0:		// don't lock anything
			break;
		case 1:		// normal lock
			wnd->curDoc()->lockAll(true);
			break;
		case 2:		// deep-lock
			wnd->curDoc()->deepLock();
			break;
		default:
			WARN();
		}
	} else if (doDeleteDoc) {
		if (!wnd->curDoc()->tryDelete()) {
			/* We failed deleting the doc,
			 * so open a new window with it, again.
			 */
			createMainWnd(QString::null, false,
				      false, wnd->curDoc());
		}
	}
	// find the closed window in the "mainWndList" and delete it.
	QValueList<PwM *>::iterator i = _mainWndList.begin(),
				    end = _mainWndList.end();
	while (i != end) {
		if (*i == wnd) {
			_mainWndList.erase(i);
			goto out_success;
		}
		++i;
	}
	BUG();
out_success:
	if (!_mainWndList.size()) {
		/* If there's no main window and no tray icon
		 * left, we have no user interface, so we can
		 * shut down the application.
		 */
		if (!_tray) {
			dcopClient()->setAcceptCalls(true);
			dcopClient()->resume();
			shutdownApp(0);
			return;
		}
		/* There is no widget left, so set
		 * _curWidget to 0
		 */
		resetCurWidget();
	}
	dcopClient()->setAcceptCalls(true);
	dcopClient()->resume();
}

void PwMInit::trayIconClosed(PwMTray *tray)
{
	if (runStatus != running)
		return;
	PARAM_UNUSED(tray);
	PWM_ASSERT(tray == _tray);
	/* If there's no main wnd left we have to
	 * shutdown the app (same as in mainWndClosed())
	 */
	if (!_mainWndList.size())
		shutdownApp(0);
}

void PwMInit::handleCmdLineArgs(bool /*initial*/)
{
	KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
	PWM_ASSERT(args);
	int i, numArgs = args->count();
	const char *curArg;

	// read all cmdline options
	savedCmd.open_deeplocked = args->isSet("open-deeplocked");
	savedCmd.minimized = args->isSet("minimized");
	savedCmd.minToTray = args->isSet("mintray");
	if (savedCmd.minimized &&
	    savedCmd.minToTray) {
		printInfo(i18n("Commandline option \"--minimized\" and "
			       "\"--mintray\" selected. These are incompatible. "
			       "\"--mintray\" will be selected.").latin1());
	}
	if (savedCmd.minToTray && !_tray) {
		printInfo(i18n("No tray Icon available. "
			       "Commandline option \"--mintray\" is replaced "
			       "by \"--minimized\".").latin1());
	}
	/* Iterate through all non-option arguments.
	 * Every non-option arg is a filename to open.
	 */
	for (i = 0; i < numArgs; ++i) {
		curArg = args->arg(i);
		PWM_ASSERT(curArg);
		if (savedCmd.minToTray && _tray) {
			PwMDoc *newDoc = createDoc();
			if (!newDoc->openDocUi(newDoc,
					       curArg,
					       savedCmd.open_deeplocked)) {
				delete newDoc;
			}
		} else {
			PwM *newInstance = createMainWnd(QString::null,
							 false,
							 true,
							 0,
							 savedCmd.minimized);
			PwMDoc *newDoc = newInstance->openDoc(curArg,
							      savedCmd.open_deeplocked);
			if (!newDoc) {
				newInstance->setForceQuit(true);
				delete_and_null(newInstance);
			}
		}
	}

	if (savedCmd.minToTray) {
		minimizeAllMainWnd(true);
	} else if (savedCmd.minimized) {
		minimizeAllMainWnd(false);
	}
	args->clear();
}

void PwMInit::minimizeAllMainWnd(bool toTray)
{
	if (!_mainWndList.size())
		return;
	const QValueList<PwM *> *ml = mainWndList();
	QValueList<PwM *>::const_iterator it = ml->begin(),
					  end = ml->end();
	PwM *wnd;
	if (toTray && _tray) {
		/* minimize to tray.
		 * close all mainWnd.
		 */
		while (it != end) {
			wnd = *it;
			wnd->setForceMinimizeToTray(true);
			wnd->close_slot();
			++it;
		}
	} else {
		// normal minimize
		while (it != end) {
			wnd = *it;
			wnd->hide();
			wnd->showMinimized();
			++it;
		}
	}
}

#include "pwminit.moc"
