/***************************************************************************
 *                                                                         *
 *   copyright (C) 2003, 2004, 2005  by Michael Buesch                     *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   Original implementation of the tray-tree                              *
 *   (c) by Matt Scifo <mscifo@o1.com>                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#include "pwmtray.h"
#include "pwmexception.h"
#include "pwm.h"
#include "pwmdoc.h"
#include "pwminit.h"
#include "configuration.h"

#include <klocale.h>

#include <qpainter.h>
#include <qpair.h>
#include <qvaluelist.h>
#include <qtooltip.h>


ActiveMenu::ActiveMenu(PwMTray *_tray, QWidget *parent, const char *name)
 : KPopupMenu (parent, name)
 , tray (_tray)
{
	connect(this, SIGNAL(aboutToShow()),
		this, SLOT(constructContent()));
}

ActiveMenu::~ActiveMenu()
{
	destructContent();
}

void ActiveMenu::hideEvent(QHideEvent *e)
{
	/* We need some magic to handle ActiveMenu hiding correctly.
	 * There are two points to remember:
	 *  - If an ActiveMenu hides, we have to delete the instance.
	 *  - But the ActiveMenu can hide because the user clicked
	 *    on a child.
	 * In the second case, we must not delete this, because
	 * we still need the child ActiveMenuItem. You remember? The
	 * SIGNAL for a click on an ActiveMenuItem is connected
	 * to SLOT(ActiveMenuItem::execIt()).
	 * So queue a QTimer. That's processed in the event loop after
	 * we have handled the click. hopefully...
	 */
	QTimer::singleShot(0, this, SLOT(destructContent()));
	QWidget::hideEvent(e);
}

void ActiveMenu::constructContent()
{
	switch (type) {
	case type_documentRoot: {
		insertTitle(i18n("Categories"));
		if (docInfo.doc->isDeepLocked()) {
			ActiveMenuItem *activeItem;
			int id;
			activeItem = new ActiveMenuItem(i18n("Deep-Locked. Please Unlock."),
							font(), ActiveMenuItem::unlock,
							tray, docInfo.doc, 0, 0);
			id = insertItem(activeItem);
			connectItem(id, activeItem, SLOT(execIt()));
		} else
			appendCategoryList();
		insertTitle(i18n("Manager"));
		appendMainWndCtrl();
		break;
	} case type_category: {
		appendEntryList();
		break;
	} case type_entry: {
		appendEntryMenu();
		break;
	} case type_unknown:
	default:
		BUG();
	}
}

void ActiveMenu::destructContent()
{
	clear();
}

void ActiveMenu::sortList(const QStringList &in,
			  QValueList< QPair<QString, unsigned int> > *out)
{
	PWM_ASSERT(out->isEmpty());
	QStringList::const_iterator i = in.begin(),
				    end = in.end();
	unsigned int cnt = 0;
	for ( ; i != end; ++i)
		out->append(qMakePair(*i, cnt++));
	QValueList< QPair<QString, unsigned int> >::iterator out_begin = out->begin(),
							     out_end = out->end();
	qHeapSortHelper(out_begin, out_end, *out_begin, out->size());
}

void ActiveMenu::appendCategoryList()
{
	QStringList cl;
	docInfo.doc->getCategoryList(&cl);
	QValueList< QPair<QString, unsigned int> > catList;
	sortList(cl, &catList);

	ActiveMenu *newMenu;
	QValueList< QPair<QString, unsigned int> >::const_iterator i = catList.begin(),
								   end = catList.end();
	for ( ; i != end; ++i) {
		newMenu = new ActiveMenu(tray);
		newMenu->setTypeCategory(docInfo.doc, (*i).second);
		insertItem((*i).first, newMenu);
	}
}

void ActiveMenu::appendEntryList()
{
	QStringList el;
	docInfo.doc->getEntryList(docInfo.category, &el);
	QValueList< QPair<QString, unsigned int> > entrList;
	sortList(el, &entrList);

	ActiveMenu *newMenu;
	QValueList< QPair<QString, unsigned int> >::const_iterator i = entrList.begin(),
								   end = entrList.end();
	for ( ; i != end; ++i) {
		newMenu = new ActiveMenu(tray);
		newMenu->setTypeEntry(docInfo.doc, docInfo.category, (*i).second);
		insertItem((*i).first, newMenu);
	}
}

void ActiveMenu::appendEntryMenu()
{
	ActiveMenuItem *newItem;
	int id;

	if (docInfo.doc->isBinEntry(docInfo.category, docInfo.entry)) {
		/* This is a binary entry. Don't insert the usual
		 * menu items.
		 */
		insertTitle(i18n("This is a binary entry.\n"
			"It is not a normal password-entry, as it contains "
			"binary data, which PwManager can't display here."));
		return;
	}

	newItem = new ActiveMenuItem(i18n("Copy Password to Clipboard"),
				     font(), ActiveMenuItem::pwToClipboard,
				     tray, docInfo.doc, docInfo.category, docInfo.entry);
	id = insertItem(newItem);
	connectItem(id, newItem, SLOT(execIt()));

	newItem = new ActiveMenuItem(i18n("Copy Username to Clipboard"),
				     font(), ActiveMenuItem::nameToClipboard,
				     tray, docInfo.doc, docInfo.category, docInfo.entry);
	id = insertItem(newItem);
	connectItem(id, newItem, SLOT(execIt()));

	newItem = new ActiveMenuItem(i18n("Copy Description to Clipboard"),
				     font(), ActiveMenuItem::descToClipboard,
				     tray, docInfo.doc, docInfo.category, docInfo.entry);
	id = insertItem(newItem);
	connectItem(id, newItem, SLOT(execIt()));

	newItem = new ActiveMenuItem(i18n("Copy URL to Clipboard"),
				     font(), ActiveMenuItem::urlToClipboard,
				     tray, docInfo.doc, docInfo.category, docInfo.entry);
	id = insertItem(newItem);
	connectItem(id, newItem, SLOT(execIt()));

	newItem = new ActiveMenuItem(i18n("Copy Launcher to Clipboard"),
				     font(), ActiveMenuItem::launcherToClipboard,
				     tray, docInfo.doc, docInfo.category, docInfo.entry);
	id = insertItem(newItem);
	connectItem(id, newItem, SLOT(execIt()));

	newItem = new ActiveMenuItem(i18n("Copy Comment to Clipboard"),
				     font(), ActiveMenuItem::commentToClipboard,
				     tray, docInfo.doc, docInfo.category, docInfo.entry);
	id = insertItem(newItem);
	connectItem(id, newItem, SLOT(execIt()));

	insertSeparator();

	newItem = new ActiveMenuItem(i18n("Execute \"Launcher\""),
				     font(), ActiveMenuItem::execLauncher,
				     tray, docInfo.doc, docInfo.category, docInfo.entry);
	id = insertItem(newItem);
	connectItem(id, newItem, SLOT(execIt()));

	newItem = new ActiveMenuItem(i18n("Execute \"Launcher\" and copy password"),
				     font(), ActiveMenuItem::execLauncherAndCopy,
				     tray, docInfo.doc, docInfo.category, docInfo.entry);
	id = insertItem(newItem);
	connectItem(id, newItem, SLOT(execIt()));

	newItem = new ActiveMenuItem(i18n("Go to \"URL\""),
				     font(), ActiveMenuItem::goToURL,
				     tray, docInfo.doc, docInfo.category, docInfo.entry);
	id = insertItem(newItem);
	connectItem(id, newItem, SLOT(execIt()));
}

void ActiveMenu::appendMainWndCtrl()
{
	ActiveMenuItem *activeItem;
	int id;

	activeItem = new ActiveMenuItem(i18n("&Open Main Manager Window"),
					font(), ActiveMenuItem::openMainWnd,
					tray, docInfo.doc, 0, 0);
	id = insertItem(activeItem);
	connectItem(id, activeItem, SLOT(execIt()));

	activeItem = new ActiveMenuItem(i18n("&Close This Document"),
					font(), ActiveMenuItem::closeDoc,
					tray, docInfo.doc, 0, 0);
	id = insertItem(activeItem);
	connectItem(id, activeItem, SLOT(execIt()));

	insertSeparator();

	activeItem = new ActiveMenuItem(i18n("&Lock All Entries"),
					font(), ActiveMenuItem::lock,
					tray, docInfo.doc, 0, 0);
	id = insertItem(activeItem);
	connectItem(id, activeItem, SLOT(execIt()));

	activeItem = new ActiveMenuItem(i18n("&Deep Lock All Entries"),
					font(), ActiveMenuItem::deepLock,
					tray, docInfo.doc, 0, 0);
	id = insertItem(activeItem);
	connectItem(id, activeItem, SLOT(execIt()));

	activeItem = new ActiveMenuItem(i18n("&Unlock All Entries"),
					font(), ActiveMenuItem::unlock,
					tray, docInfo.doc, 0, 0);
	id = insertItem(activeItem);
	connectItem(id, activeItem, SLOT(execIt()));
}



ActiveMenuItem::ActiveMenuItem(const QString &_text,
			       const QFont &_font,
			       Task _task,
			       PwMTray *_tray,
			       PwMDoc * _doc,
			       int _category,
			       int _entry)
 : tray (_tray)
 , text (_text)
 , font (_font)
 , task (_task)
 , docInfo (MenuItemDocInfo(_doc, _category, _entry))
{
}

ActiveMenuItem::~ActiveMenuItem()
{
}

void ActiveMenuItem::paint(QPainter *p, const QColorGroup &, bool,
			   bool, int x, int y, int w, int h)
{
	p->setFont(font);
	p->drawText(x, y, w, h, AlignLeft | AlignVCenter |
		    DontClip | ShowPrefix, text);
}

QSize ActiveMenuItem::sizeHint()
{
	return QFontMetrics(font).size(AlignLeft | AlignVCenter |
				       ShowPrefix | DontClip,  text);
}

void ActiveMenuItem::execIt()
{
	unsigned int entr = static_cast<unsigned int>(docInfo.entry);
	unsigned int cat = static_cast<unsigned int>(docInfo.category);
	PwMDoc *doc = docInfo.doc;

	switch (task) {
		case execLauncherAndCopy:
		case pwToClipboard: {
			PwMDataItem d;
			doc->getDataChangedLock();
			bool wasLocked = doc->isLocked(cat, entr);
			doc->getEntry(cat, entr, &d, true);
			if (wasLocked)
				doc->lockAt(cat, entr, true);
			doc->putDataChangedLock();
			PwM::copyToClipboard(d.pw.c_str());
			if (task == execLauncherAndCopy)
				doc->execLauncher(cat, entr);
			break;
		} case nameToClipboard: {
			PwMDataItem d;
			doc->getEntry(cat, entr, &d);
			PwM::copyToClipboard(d.name.c_str());
			break;
		} case descToClipboard: {
			PwMDataItem d;
			doc->getEntry(cat, entr, &d);
			PwM::copyToClipboard(d.desc.c_str());
			break;
		} case urlToClipboard: {
			PwMDataItem d;
			doc->getEntry(cat, entr, &d);
			PwM::copyToClipboard(d.url.c_str());
			break;
		} case launcherToClipboard: {
			PwMDataItem d;
			doc->getEntry(cat, entr, &d);
			PwM::copyToClipboard(d.launcher.c_str());
			break;
		} case commentToClipboard: {
			PwMDataItem d;
			doc->getEntry(cat, entr, &d);
			PwM::copyToClipboard(d.comment.c_str());
			break;
		} case execLauncher: {
			doc->execLauncher(cat, entr);
			break;
		} case goToURL: {
			doc->goToURL(cat, entr);
			break;
		} case openMainWnd: {
			// search if there is already an open window.
			const QValueList<PwM *> *wl = tray->init->mainWndList();
			QValueList<PwM *>::const_iterator i = wl->begin(),
							  end = wl->end();
			PwM *wnd;
			while (i != end) {
				wnd = *i;
				if (wnd->curDoc() == doc) {
					// now bring this window to the foreground.
					if (!wnd->hasFocus()) {
						wnd->hide();
						wnd->showNormal();
						wnd->setFocus();
					}
					return;
				}
				++i;
			}
			// there is no open window, so open a new one.
			tray->init->createMainWnd(QString::null, false, false, doc);
			break;
		} case closeDoc: {
			doc->tryDelete();
			break;
		} case lock: {
			doc->lockAll(true);
			break;
		} case deepLock: {
			doc->deepLock();
			break;
		} case unlock: {
			doc->lockAll(false);
			break;
		} default: {
			BUG();
		}
	}
}


PwMTray::PwMTray(PwMInit *_init, QWidget * parent, const char *name)
 : KSystemTray(parent, name)
 , init (_init)
{
	QToolTip::add(this, PROG_NAME " " PACKAGE_VER " '" PROG_NICKNAME "'");
	ctxMenu = contextMenu();
	docMenuList.setAutoDelete(true);
	buildMain();
}

PwMTray::~PwMTray()
{
	emit closed(this);
}

void PwMTray::buildMain()
{
	ctxMenu->insertSeparator();
	ctxMenu->insertItem(i18n("&New Main Window"), this,
			    SLOT(newMainWnd()));
	ctxMenu->insertItem(i18n("&Open File..."), this,
			    SLOT(openDoc()));
	ctxMenu->insertItem(i18n("&Generate Password..."), this,
			    SLOT(genPassword()));
	ctxMenu->insertSeparator();
	ctxMenu->insertItem(i18n("&Remove From Tray"), this,
			    SLOT(undock()));

	connect(ctxMenu, SIGNAL(aboutToShow()),
		this, SLOT(aboutToShowCtxMenu()));
	connect(ctxMenu, SIGNAL(aboutToHide()),
		this, SLOT(aboutToHideCtxMenu()));
}

void PwMTray::updateTree(PwMDoc *document)
{
	//FIXME: what to do here?
	PARAM_UNUSED(document);
}

void PwMTray::closeTreeEntry(PwMDoc *document)
{
	//FIXME: what to do here?
	PARAM_UNUSED(document);
}

void PwMTray::mouseReleaseEvent(QMouseEvent *e)
{
	QWidget *curWnd = init->curWidget();
	KPopupMenu *ctxMenu = contextMenu();
	// give focus to init->curWidget()
	if (curWnd && !curWnd->hasFocus()) {
		curWnd->hide();
		curWnd->showNormal();
		curWnd->setFocus();
	}
	// popup the context menu
	ctxMenu->popup(e->globalPos());
	emit clickedIcon(this);
}

void PwMTray::aboutToShowCtxMenu()
{
	buildDocMenus();
}

void PwMTray::aboutToHideCtxMenu()
{
	// Please read the comment in ActiveMenu::hideEvent()
	QTimer::singleShot(0, this, SLOT(destroyDocMenus()));
}

void PwMTray::appendDocMenu(PwMDoc *doc)
{
	ActiveMenu *newMenu = new ActiveMenu(this);
	newMenu->setTypeDocRoot(doc);
	ctxMenu->insertItem(doc->getTitle(), newMenu, -1, 1);
	docMenuList.append(newMenu);
}

void PwMTray::buildDocMenus()
{
	PwMDocList *docList = PwMDoc::getOpenDocList();
	const vector<PwMDocList::listItem> *dl = docList->getList();
	vector<PwMDocList::listItem>::const_iterator i = dl->begin(),
						     end = dl->end();
	for ( ; i < end; ++i)
		appendDocMenu((*i).doc);
}

void PwMTray::destroyDocMenus()
{
	/* This also deletes the ActiveMenu instances.
	 * docMenuList is set to auto-delete.
	 */
	docMenuList.clear();
}

void PwMTray::connectDocToTray(PwMDoc *doc)
{
	connect(doc, SIGNAL(dataChanged(PwMDoc *)),
		this, SLOT(updateTree(PwMDoc *)));
	connect(doc, SIGNAL(docClosed(PwMDoc *)),
		this, SLOT(closeTreeEntry(PwMDoc *)));
	// fake a dataChanged signal for this doc to update the tray tree.
	updateTree(doc);
}

void PwMTray::openDoc()
{
	// open the document in a new window.
	PwM *newInstance = init->createMainWnd();
	PwMDoc *newDoc = newInstance->openDoc("");
	if (!newDoc) {
		newInstance->setForceQuit(true);
		delete_and_null(newInstance);
	}
}

void PwMTray::newMainWnd()
{
	init->createMainWnd();
}

void PwMTray::genPassword()
{
	PwMInit::genPassword(init->curWidget());
}

void PwMTray::undock()
{
	conf()->confGlobTray(false);
	init->initTray();
	// Attention! "this" has just been deleted in initTray().
}

#include "pwmtray.moc"
