# -*- coding: utf-8 -*-
# KeyJnoteGUI: A frontend to KeyJnote, an eye-candy presentation programm
# Copyright (C) 2006  Sebastian Wiesner <basti.wiesner@gmx.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

"""This module provides wrapper classes for controlling KeyJnote.

:mvar KeyJnote: A wrapper class for KeyJnote"""

__revision__ = '$Id: kjnwrapper.py 119 2007-04-12 10:01:30Z lunar $'


import sys
import os
import subprocess
import gettext
import re

from distutils import spawn
from distutils.version import StrictVersion


MIN_KEYJNOTE_VERSION = StrictVersion('0.9.4')


_ = gettext.translation('keyjnotegui', fallback=True).ugettext


def encode_filename(filename):
    """Encodes `filename` according to file system encoding"""
    return unicode(filename).encode(sys.getfilesystemencoding())


def _search_kjn_executable():
    """Searches for KeyJnote executable.
    :returns: The path of the KeyJnote executable or None, if it wasn't
    found"""
    def_names = ['keyjnote', 'keyjnote.py']
    found = filter(None, map(spawn.find_executable, def_names))
    if found:
        return found[0]
    else:
        return None


def _assert_have_executable(settings):
    """Makes sure, that `settings` contains a valid KeyJnote executable.
    :raises KJnException: If now executable was found"""
    if settings.kjn_executable and os.path.exists(settings.kjn_executable):
        _exec = settings.kjn_executable
    else:
        _exec = _search_kjn_executable()
        if not _exec:
            msg = _('Couldn\'t find the KeyJnote executable.')
            raise KJnException(msg)
    # check if this is really keyjnote
    # NOTE: this doesn't prevent manipulation of evil hackers, it just
    # avoids, that the user accidently specifies the wrong thing
    process = subprocess.Popen([_exec, '--help'], stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE)
    # version is printed to stderr
    output = process.communicate()[1]
    if (not output) or ('KeyJnote' not in output.splitlines()[0]):
        msg = _('The file %s is no KeyJnote executable!')
        raise KJnException(msg % _exec)
    settings.kjn_executable = _exec


def _assert_correct_version(version):
    """Make sure, that we have the correct version of KeyJnote here."""
    if version < MIN_KEYJNOTE_VERSION:
        msg = _('This version of KeyJnote is not longer supported. '
                'A version later than %s is required.')
        raise KJnException(msg % MIN_KEYJNOTE_VERSION, version)


def check_settings(settings):
    """Checks, if the settings are correct.
    :raises KJnException: If settings are incorrect"""
    if not settings.selected:
        msg = _('No transitions specified...')
        raise KJnException(msg)


class KJnException(Exception):
    """An class for exception occuring during KeyJnote execution."""
    def __init__(self, message, version=None):
        """Creates a new instance.

        :param message: a descriptive, localized error message
        :param version: KeyJnote version. None, if error was raised before
        version could be determinated"""
        self.version = version
        self.message = message

    def __str__(self):
        if self.version:
            return 'KeyJnote %s: %s' % (self.version, self.message)
        else:
            return self.message

    def __unicode__(self):
        if self.version:
            return u'KeyJnote %s: %s' % (self.version, self.message)
        else:
            return self.message


class ModuleWrapper(object):
    """This class imports KeyJnote as a module and controls it directly.

    :ivar transitions: A list of all available transitions
    :ivar settings: The Settings object"""

    def __init__(self, settings):
        """Creates a new instance."""
        self.transitions = [t.__name__ for t in keyjnote.AllTransitions]
        self.transitions.sort()
        self.settings = settings
        _assert_correct_version(self.version)

    @property
    def version(self):
        """:returns: The version of KeyJnote
        :rtype: distutils.version.StrictVersion"""
        return StrictVersion(keyjnote.__version__)

    def transition_doc(self, transition):
        """:returns: the documentation for `transitition`
        :type transition: str
        :raises ValueError: if `transition` is unknown"""
        if not transition in self.transitions:
            raise ValueError("Transition %s is not known" % transition)
        return getattr(keyjnote, transition).__doc__
        
    def execute(self):
        """Executes KeyJnote"""
        check_settings(self.settings)
        keyjnote.FileName = self.settings.source
        keyjnote.ScreenWidth = self.settings.screen_geometry.width
        keyjnote.ScreenHeight = self.settings.screen_geometry.height
        keyjnote.Fullscreen = self.settings.fullscreen
        keyjnote.Scaling = self.settings.scale
        keyjnote.Supersample = self.settings.supersample
        keyjnote.Wrap = self.settings.wrap
        if self.settings.auto:
            # this is the way, KeyJnote itself calculates the value from cl
            # paramters
            keyjnote.AutoAdvance = self.settings.auto_time * 1000
        keyjnote.UseCache = self.settings.cache
        keyjnote.BackgroundRendering = self.settings.background_render
        keyjnote.CacheFile = not self.settings.memcache
        keyjnote.AllowExtensions = self.settings.useext
        keyjnote.AvailableTransitions = []
        keyjnote.Rotation = self.settings.page_rotation
        if self.settings.cursor_file:
            keyjnote.CursorImage = self.settings.cursor_file
        keyjnote.PollInterval = self.settings.poll
        keyjnote.EstimatedDuration = self.settings.duration or None
        for transition in self.settings.selected:
            trans = getattr(keyjnote, str(transition))
            keyjnote.AvailableTransitions.append(trans)
        try:
            keyjnote.main()
        # interlaced png files
        except IOError:
            raise KJnException(_('Interlaced PNG files were found inside '
                                 'the specified directory.\n'
                                 'The underlying Python Imaging Library '
                                 'does not support such files.'))


class CmdLineWrapper(object):
    """Uses the command line interface to control KeyJnote.

    :ivar transitions: A list of all available transitions"""
    
    transition_pattern = re.compile(r'\*\s*(?P<name>\w*)\s*'
                                    r'-\s*(?P<desc>.*)')

    def __init__(self, settings):
        """Creates a new instance.
        :raise KJnException: If no KeyJnote executable was found."""
        self._version = None
        self.__transition_dict = None
        self.settings = settings
        # this function raises an exception, if the settings object doesn't
        # provide any executable path
        _assert_have_executable(self.settings)
        _assert_correct_version(self.version)

    @property
    def version(self):
        """:returns: The version of KeyJnote
        :rtype: (int, int, int)"""
        # version is printed in the first line
        if not self._version:
            _assert_have_executable(self.settings)
            exec_ = self.settings.kjn_executable
            process = subprocess.Popen([exec_, '-l'],
                                       stdout=subprocess.PIPE,
                                       stderr=subprocess.PIPE)
            headline = process.communicate()[1].splitlines()[0]
            version_str = headline.split()[-1]
            self._version = StrictVersion(version_str)
        return self._version

    def _read_transitions(self):
        """Reads all transitions"""
        transitions = {}
        _assert_have_executable(self.settings)
        exec_ = self.settings.kjn_executable
        process = subprocess.Popen([exec_, '-l'],
                                   stdout=subprocess.PIPE)
        # get lines printed on stdout
        lines = process.communicate()[0].splitlines()
        for line in lines:
            match = self.transition_pattern.match(line)
            if match:
                transitions[match.group('name')] = match.group('desc')
        return transitions

    # This property should be used to access transition documentation
    @property
    def _transitions(self):
        """:returns: a dictionary, which maps transitions to their
        documentation"""
        if not self.__transition_dict:
            self.__transition_dict = self._read_transitions()
        return self.__transition_dict

    @property
    # returns a list of transition names
    def transitions(self):
        """Uses the output of ``keyjnote -l`` to read all known transitions
        :returns: A list of all transitions
        :rtype: [str]"""
        return sorted(self._transitions.keys())

    def transition_doc(self, transition):
        """:returns: the documentation for `transitition`
        :type transition: str
        :raises ValueError: if `transition` is unknown"""
        try:
            return self._transitions[transition]
        except KeyError:
            raise ValueError("Transition %s is not known" % transition)

    def _analyze(self, output):
        """Analyzes KeyJnote `output` and searches for error messages
        :returns: An error message or None, if execution was successful"""
        for line in output.splitlines():
            if line.startswith('Cannot analyze'):
                # keyjnote could not find any pictures inside a directory
                    return _('KeyJnote could not load the specified '
                             'source. Maybe it doesn\'t exist or is '
                             'corrupted.')
            if line == 'Error: Failed to open PDF file:':
                return _('KeyJnote could not load the specified PDF file. '
                         'It is potentially corrupted.')
            elif line == 'IOError: cannot read interlaced PNG files':
                # this error message comes from the PIL package, which seems
                # to be unable to handle interlaced png files (a quite nasty
                # bug, imho).
                return _('Interlaced PNG files were found inside the '
                         'specified directory.\n'
                         'The underlying Python Imaging Library does not '
                         'support such files.')
        # no errors
        return None

    def execute(self):
        """Executes KeyJnote.
        It uses the subprocess module to create the new process."""
        # build the command
        _assert_have_executable(self.settings)
        check_settings(self.settings)
        cmd = [self.settings.kjn_executable, '-g',
               str(self.settings.screen_geometry)]
        if not self.settings.useext:
            cmd.append('-e')
        if not self.settings.fullscreen:
            cmd.append('-f')
        if (self.settings.source_type == 'directory'
            and self.settings.scale):
            cmd.append('-s')
        if (self.settings.source_type == 'pdf'
            and self.settings.supersample):
            cmd.append('-s')
        if self.settings.wrap:
            cmd.append('-w')
        if self.settings.auto:
            cmd.append('-a')
            cmd.append(str(self.settings.auto_time))
        if not self.settings.cache:
            cmd.append('-c')
        if not self.settings.background_render:
            cmd.append('-b')
        if self.settings.memcache:
            cmd.append('-m')
        if self.settings.page_rotation is not None:
            cmd.extend(['-r', str(self.settings.page_rotation)])
        if self.settings.poll:
            cmd.extend(['-u', str(self.settings.poll)])
        if self.settings.cursor_file:
            cmd.extend(['-C', encode_filename(self.settings.cursor_file)])
        if self.settings.duration:
            cmd.extend(['-d', self.settings.duration])
        # transitions
        cmd.append('-t')
        selected_transitions = ','.join(map(str, self.settings.selected))
        cmd.append(selected_transitions)
        cmd.append(encode_filename(self.settings.source))
        try:
            process = subprocess.Popen(cmd, stderr=subprocess.PIPE)
            # get the error output and analyze it
            output = process.communicate()[1]
            msg = self._analyze(output)
            if msg:
                raise KJnException(msg)
        except OSError, e:
            msg = _('Couldn\'t execute KeyJnote: %s')
            raise KJnException(msg % e)


# this code determines the preferred wrapper class
try:
    import keyjnote
    KJnWrapper = ModuleWrapper
except ImportError:
    print >> sys.stderr, ('Module keyjnote not found. Using command '
                          'line interface.')
    KJnWrapper = CmdLineWrapper


# factory function
def create_wrapper(settings):
    """Create a wrapper with `settings`"""
    if settings.use_cmdline_wrapper:
        return CmdLineWrapper(settings)
    else:
        return KJnWrapper(settings)
