# -*- coding: utf-8 -*-
# KeyJnoteGUI: A frontend to KeyJnote, an eye-candy presentation programm
# Copyright (C) 2006  Sebastian Wiesner <basti.wiesner@gmx.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

"""Contains classes to store KeyJnoteGUI settings.

:mvar standard_resolutions: A list of standard geometries"""

__revision__ = '$Id: settings.py 108 2007-04-10 17:28:47Z lunar $'

from qt import QRect, QString
from kdecore import KApplication, KGlobalSettings


class Geometry: 
    """This class stores geometry values as used by KeyJnote.

    :ivar width: The width
    :ivar height: The height
    """
    
    def __init__(self, *args):
        """Creates a new instance. If you pass two numbers, they are
        interpreted as width and height. If you pass a single QString_ or
        basestring, it is interpreted as \"widthxheight\"
        (e.g. 1280x1024). Alternatively you can pass a QRect_ instance."""
        if len(args) == 1:
            string = args[0]
            if isinstance(string, QString):
                # convert QStrings
                string = str(string)
            if isinstance(string, basestring):
                # which is parsed
                parts = string.split('x')
                self.width = int(parts[0])
                self.height = int(parts[1])
            # or a QRect
            elif isinstance(args[0], QRect):
                self.width = args[0].width()
                self.height = args[0].height()
        elif len(args) == 2:
            # height and width were specified separatly
            self.width, self.height = args

    def __str__(self):
        return '%sx%s' % (self.width, self.height)

    def __cmp__(self, other):
        if self.width > other.width:
            # definitly larger
            return 1
        elif self.width == other.width:
            if self.height > other.height:
                return 1
            elif self.height == other.height:
                return 0
            else:
                # definitly smaller
                return -1
        else:
            # our width is definitly smaller
            return -1


class Settings:
    """Stores settings for KeyJnoteGUI."""

    def __init__(self):
        """Creates a new settings instance and invokes the load_ method to
        initially load all settings."""
        self._kconfig = KApplication.kApplication().config()
        self.load()

    def save(self):
        """Save settings into KConfig and write the changes to the disk."""
        cfg = self._kconfig
        cfg.setGroup('Executable')
        cfg.writePathEntry('path', self.kjn_executable)
        cfg.writeEntry('use_cmdline_wrapper', self.use_cmdline_wrapper)
        cfg.setGroup('Source')
        cfg.writePathEntry('source', self.source)
        cfg.writeEntry('source_type', self.source_type)
        cfg.writeEntry('poll', self.poll)
        cfg.setGroup('Appearance')
        cfg.writeEntry('fullscreen', self.fullscreen)
        cfg.writeEntry('screen_geometry', str(self.screen_geometry))
        cfg.writeEntry('scale', self.scale)
        cfg.writeEntry('supersample', self.supersample)
        cfg.writePathEntry('cursor_file', self.cursor_file)
        cfg.writeEntry('page_rotation', self.page_rotation)
        cfg.setGroup('Presentation')
        cfg.writeEntry('wrap', self.wrap)
        cfg.writeEntry('auto', self.auto)
        cfg.writeEntry('auto_time', self.auto_time)
        cfg.setGroup('Cache')
        cfg.writeEntry('cache', self.cache)
        cfg.writeEntry('memcache', self.memcache)
        cfg.writeEntry('background_render', self.background_render)
        cfg.setGroup('Misc')
        cfg.writeEntry('useext', self.useext)
        cfg.writeEntry('duration', self.duration)
        cfg.setGroup('Transitions')
        cfg.writeEntry('selected', self.selected)
        cfg.setGroup('Window')
        cfg.writeEntry('geometry', self.window_geometry)
        cfg.sync()

    def load(self):
        """Loads settings from configuration file."""
        cfg = self._kconfig
        cfg.setGroup('Executable')
        self.kjn_executable = str(cfg.readPathEntry('path'))
        self.use_cmdline_wrapper = cfg.readBoolEntry('use_cmdline_wrapper',
                                                     False)
        cfg.setGroup('Source')
        self.source = str(cfg.readPathEntry('source', '$HOME'))
        self.source_type = cfg.readEntry('source_type', 'pdf')
        # make sure we have a valid source type
        if self.source_type not in ('pdf', 'directory'):
            self.source_type = 'pdf'
        self.poll = cfg.readNumEntry('poll', 0)
        cfg.setGroup('Appearance')
        self.fullscreen = cfg.readBoolEntry('fullscreen', True)
        desktop_geometry = KGlobalSettings.splashScreenDesktopGeometry()
        desktop_geometry = str(Geometry(desktop_geometry))
        self.screen_geometry = Geometry(cfg.readEntry('screen_geometry',
                                                      desktop_geometry))
        self.page_rotation = cfg.readNumEntry('page_rotation', 0)
        self.scale = cfg.readBoolEntry('scale', False)
        self.supersample = cfg.readBoolEntry('supersample', False)
        self.cursor_file = str(cfg.readPathEntry('cursor_file'))
        cfg.setGroup('Presentation')
        self.wrap = cfg.readBoolEntry('wrap', False)
        self.auto = cfg.readBoolEntry('auto', False)
        self.auto_time = cfg.readNumEntry('auto_time', 5)
        cfg.setGroup('Cache')
        self.cache = cfg.readBoolEntry('cache', True)
        self.memcache = cfg.readBoolEntry('memcache', False)
        self.background_render = cfg.readBoolEntry('background_render',
                                                   True)
        cfg.setGroup('Misc')
        self.useext = cfg.readBoolEntry('useext', True)
        self.duration = str(cfg.readEntry('duration', ''))
        cfg.setGroup('Transitions')
        self.selected = cfg.readListEntry('selected')
        cfg.setGroup('Window')
        self.window_geometry = cfg.readRectEntry('geometry')


standard_resolutions = [
    Geometry(640, 480),
    Geometry(800, 600),
    Geometry(1024, 768),
    Geometry(1280, 960),
    Geometry(1280, 1024),
    Geometry(1600, 1200)
]
