/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : Dialog to allow photo properties to be edited
 *
 * Copyright 2005 by M. Asselstine

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */
#ifndef PHOTOPROPERTIES_H
#define PHOTOPROPERTIES_H

#include "photopropertiesui.h"

#include <qmap.h>
#include <qstring.h>
#include <qpixmap.h>
#include <qptrlist.h>
#include <qstringlist.h>

class Photo;
class QRegExpValidator;


/**
 * Provides the main widget used to edit photograph properties for either
 * a single photo or several photo in batch mode.
 * @short Photo properties edit widget.
 * @author M. Asselstine <asselsm@gmail.com>
 */
class PhotoProperties: public PhotoPropertiesUI
{
Q_OBJECT

public:
    /**
     * Constructor.
     * @param parent Pointer to parent widget
     * @param name Widget name
     */
    PhotoProperties(QWidget *parent = 0, const char *name = 0);

    /** Destructor. */
    ~PhotoProperties();

    /**
     * Clears the selection of both private and public radio buttons. This is
     * required for batch mode since if neither public or private are set the
     * individual public/private setting for each photo remains unchanged.
     */
    void setNeitherPublicOrPrivate();

    /**
     * Set the upload size to use for the photograph(s). In batch mode
     * it must be possible to set a blank size such that if the size is not
     * set the individual sizes for the photos in the batch remain unchanged.
     * @param size The new upload size for the photo(s)
     */
    void setPhotoSize(const QString &size);

    /**
     * Retrieve the list of tags for the photograph(s).
     * @return List of tags for the photo(s)
     */
    QStringList tags() const;

    /**
     * Set the list of tags for this photograph(s).
     * @lst List of tags for the photo(s)
     */
    void setTags(const QStringList &lst);

    /**
     * Clear all values, remove the preview image and disable buttons.
     * Basically make so that we are editing no photograph.
     */
    void clearAndDisable();

    /**
     * Edit a single photographs properties.
     * @param photo Reference to the photograph that is to be edited
     */
    void editSinglePhoto(Photo& photo);

    /**
     * Edit a batch of photographs in "batch mode".
     * @param photos A list of photographs that are included in the batch
     */
    void editPhotoBatch(QPtrList<Photo> photos);

public slots:
    /**
     * Update the title of the photo or photos being edited.
     * @param txt The new title string
     */
    void updateTitle(const QString& txt);

    /**
     * Update the description of the photo or photos being edited
     * using the string in the description text edit box.
     */
    void updateDescription();

    /**
     * Update the upload size of the photo or photos being edited.
     * @param newsize The string representing the new upload size
     */
    void updateSize(const QString& newsize);

    /**
     * Update the "public" setting of the photo or photos being edited.
     * @param b True to make the photo public
     */
    void updatePublic(bool b);

    /**
     * Update the "private" setting of the photo or photos being edited.
     * @param b True to make the photo private
     */
    void updatePrivate(bool b);

    /**
     * Update the "family" setting of the photo or photos being edited.
     * @param b True to mae the photo available to friends
     */
    void updateFamily(bool b);

    /**
     * Update the "friends" setting of the photo or photos being edited.
     * @param b True to make the photo available to family
     */
    void updateFriends(bool b);

    /**
     * Update the tags for the photo or photos being edited
     * using the tags currently in the tags list.
     */
    void updateTags();

    /**
     * Update the photoset for the photo or photos being edited.
     * @param set The new photoset name
     */
    void updatePhotoset(const QString& set);

    /**
     * Update the license for the photo or photos being edited.
     * @param lic The new license string to use
     */
    void updateLicense(const QString& lic);

    /**
     * Populate the list of the combo box found in the tags section of the widget.
     * This list should be retrieved from the user's flickr account such that
     * tags s/he has already used are easily available.
     * @param lst List of tags
     */
    void setAvailableTags(const QStringList& lst);

    /**
     * Set the list of available photoset for the photosets combobox as well as which
     * of them should be selected to start with.
     * @param titles String list of the titles of the photosets (retrieved from flickr)
     * @param sel The initial photoset selection
     */
    void setPhotosets(const QStringList& titles, const QString& sel);

    /**
     * Set the list of available licenses for the licenses combobox.
     * @param licenses String list of all available licenses (retrieved from flickr)
     */
    void setLicenses(const QStringList& licenses);

private slots:
    /**
     * Rotate the preview and the actual photograph being edited. Should only
     * be called if editing a single photograph as this is not valid in batch mode.
     * The rotation is by 90degrees in the counter-clockwise direction.
     */
    void rotatePhoto();

    /**
     * Add a tag from the tag section's combo box to the list of tags for the photo.
     */
    void addSelectedTag();

    /**
     * Update the state of remove button in the tag section of the dialog. The decision
     * to enable or disable the button will depend on the selection in the tag list.
     */
    void updateRemoveBtnState();

    /**
     * Update the state of add button in the tag section of the dialog. The decision
     * to enable or disable the button will depend on the contents of the combo box
     * in the tag section of the editor widget.
     */
    void updateAddBtnState(const QString &str);

    /**
     * Remove all tags that are selected in the tag list from the tag list.
     */
    void removeSelectedTags();

    /**
     * Display a large preview of the photo in a dialog.
     */
    void showLargerPreview();

    /**
     * Add a tag to the tag list for the photograph. Clears the combo box found in
     * the tag section of the widget so it is ready for the user to input another tag.
     */
    void insertNewTag(const QString&);

    /**
     * Set the photograph to be public or not. A photo graph can be
     * either private or public so we if we toggle one of the radio
     * buttons the other must toggle as well.
     * @param pub True if the photo(s) is to be public, False otherwise.
     */
    void setPublic(bool pub = true);

    /**
     * Set the photograph to be private or not. A photo graph can be
     * either private or public so we if we toggle one of the radio
     * buttons the other must toggle as well.
     * @param priv True if the photo(s) is to be private, False otherwise.
     */
    void setPrivate(bool priv = true);

    /**
     * Enable or disable the signaling of updates from widgets contained in the
     * PhotoProperties.
     * @param enable True if signals should be enabled, false otherwise.
     */
    void enableUpdates(bool enable);

    /**
     * Set the photo size selection in the photosize combobox.
     * @param sel The size selection to make the active selection in the combobox.
     * @param update If true calls updateSize() to update the size for the photo or photos being edited.
     */
    void setSizeSelection(const QString& sel, bool update = TRUE);

    /**
     * Set the width value for the photo upload size.
     * @param width The value to place in the width spinbox.
     */
    void setWidth(int width);

    /**
     * Set the height value for the photo upload size.
     * @param height The value to place in the height spinbox.
     */
    void setHeight(int height);

private:
    /**
     * Match the width and height values to the photo layout, portrait vs landscape. This is done since flickr
     * offers different download sizes depending on if the photograph is portrait or landscape.
     */
    void matchSizingsToPhotoLayout();

private:
    int m_customWidth;                  /// The last value used as a custom photo width value
    int m_customHeight;                 /// The last value used as a custom photo height value
    bool m_batchMode;                   /// True if we are editing a batch of photos
    Photo *m_activePhoto;               /// Pointer to the photograph we are editing, in single photo edit mode
    bool m_portraitSizings;             /// True if the sizings used in our SizeMap are set for a portrait layout
    QPtrList<Photo> m_batchPhotos;      /// List of pointer to photographs being edited in batch mode

    typedef QMap<QString,QString> SizeMap;  /// Maps a size type string to a width x height string
    SizeMap m_sizings;                      /// Maps a size string to a width and height
};

#endif // PHOTOPROPERTIES_H
