/***************************************************************************
 *   Copyright (C) 2004 by Emil Stoyanov                                   *
 *   emosto@users.sourceforge.net                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "kiaxprefswindow.h"


KiaxPrefsWindow::KiaxPrefsWindow(KiaxPrefs * iaxSettings, QWidget* parent, const char* name, bool modal, WFlags fl)
    : KiaxPrefsWindowUi(parent,name, modal,fl)
{
  edittedAccNumber =-1;
  settings = iaxSettings;
  settings->detectDevices();
  // prepare widgets
  initWidgets();
  setAudioFields();
  // insert account names in accountsListBox;
  loadAccountsList();
  setMiscFields();

}

KiaxPrefsWindow::~KiaxPrefsWindow() {}

void KiaxPrefsWindow::initWidgets()
{
  accountsListBox->clear();
  accountNameEdit->setText("");
  serverEdit->setText("");
  usernameEdit->setText("");
  passwordEdit->setText("");
  callerIdEdit->setText("");
  callerIdNumberEdit->setText("");
  registerAccountCheckBox->setChecked(false);
  enableControls(false);
}

void KiaxPrefsWindow::enableControls(bool en) {
  accountNameEdit->setEnabled(en);
  serverEdit->setEnabled(en);
  usernameEdit->setEnabled(en);
  passwordEdit->setEnabled(en);
  callerIdEdit->setEnabled(en);
  callerIdNumberEdit->setEnabled(en);
  deleteAccountButton->setEnabled(en);
  makeDefaultCheckBox->setEnabled(en);
  codecsGroup->setEnabled(en);
  registerAccountCheckBox->setEnabled(en);
}

void KiaxPrefsWindow::buttonOk_clicked()
{
  // set the values of the currently selected account in gui
  //    otherwise the changes will be lost
  int accNumber = accountsListBox->currentItem();
  setAccountValues(accNumber);
  setAudioValues();
  setMiscValues();
  // then save everything
  settings->saveSettings();
  accept();
}

void KiaxPrefsWindow::loadAccountsList()
{
  int accCount = settings->getAccounts().count();
  for (int i=0;i<accCount; i++)
  {
    KiaxAccount * account = settings->getAccount(i);
    accountsListBox->insertItem(account->accAlias,i);
  }
  if (accCount>0)
  {
    accountsListBox->setSelected(0, true);
    enableControls(true);
  }
  else
  {
    initWidgets();
  }
}

void KiaxPrefsWindow::setFieldsValues(int accountNumber)
{
  KiaxAccount * account = settings->getAccount(accountNumber);
  if (account)
  {
    accountNameEdit->setText(account->accAlias);
    serverEdit->setText(account->iaxServer);
    usernameEdit->setText(account->username);
    passwordEdit->setText(account->password);
    callerIdEdit->setText(account->callerId);
    callerIdNumberEdit->setText(account->callerIdNumber);
    setCodecByName(account->codec);
    makeDefaultCheckBox->setChecked(settings->getDefaultAccountId()==account->accId);
    registerAccountCheckBox->setChecked(!account->registerAccount);
  }
}

void KiaxPrefsWindow::setAccountValues(int accNumber)
{
  if (accNumber<0)
    return;
  KiaxAccount * account = settings->getAccount(accNumber);
  if (account)
  {
    account->accAlias = accountNameEdit->text();
    account->iaxServer = serverEdit->text();
    account->username = usernameEdit->text();
    account->password = passwordEdit->text();
    account->callerId = callerIdEdit->text();
    account->callerIdNumber = callerIdNumberEdit->text();
    account->codec = getSelectedCodecName();
    account->registerAccount = !registerAccountCheckBox->isChecked();
  }
}

void KiaxPrefsWindow::setMiscValues() {
    settings->setDocked(dockedCheckBox->isChecked());
    settings->setHiddenOnStartup(hiddenOnStartupCheckBox->isChecked());
    settings->setConnectedOnStartup(connectedOnStartupCheckBox->isChecked());
    settings->setRaiseWindow(raiseWindowCheckBox->isChecked());
    settings->setMaxCalls(maxCallsSpinBox->value());
    settings->setExecCommandOnRing(executeCommandCheckbox->isChecked());
    settings->setCommandOnRing(executeCommandLineEdit->text());
}

void KiaxPrefsWindow::setMiscFields() {
    dockedCheckBox->setChecked(settings->isDocked());
    hiddenOnStartupCheckBox->setChecked(settings->isHiddenOnStartup());
    connectedOnStartupCheckBox->setChecked(settings->isConnectedOnStartup());
    raiseWindowCheckBox->setChecked(settings->getRaiseWindow());
    maxCallsSpinBox->setValue(settings->getMaxCalls());
    executeCommandCheckbox->setChecked(settings->getExecCommandOnRing());
    executeCommandLineEdit->setEnabled(settings->getExecCommandOnRing());
    executeCommandLabel->setEnabled(settings->getExecCommandOnRing());
    executeCommandLineEdit->setText(settings->getCommandOnRing());
}

void KiaxPrefsWindow::setAudioValues()
{
  // set the values of the selected input/output devices
  settings->setInputDevice(inputDeviceComboBox->currentItem());
  settings->setOutputDevice(outputDeviceComboBox->currentItem());
  settings->setRingDevice(ringDeviceComboBox->currentItem());
  // set filters
  int aFilterFlag = 0;
  if (noiseReductionCheckBox->isChecked()) aFilterFlag |= IAXC_FILTER_DENOISE;
  if (aagcCheckBox->isChecked()) aFilterFlag |= IAXC_FILTER_AAGC;
  if (agcCheckBox->isChecked()) aFilterFlag |= IAXC_FILTER_AGC;
  if (echoCancelationCheckBox->isChecked()) aFilterFlag |= IAXC_FILTER_ECHO;
  if (comfortNoiseCheckBox->isChecked()) aFilterFlag |= IAXC_FILTER_CN;
  settings->setFilterFlag(aFilterFlag);
  if (silenceSuppressionCheckBox->isChecked()) settings->setSilenceThreshold(1);//auto
  else settings->setSilenceThreshold(-99);
  settings->setIncomingRingMute(incomingRingMuteCheckBox->isChecked());
  settings->setOutgoingRingMute(outgoingRingMuteCheckBox->isChecked());
  
}

void KiaxPrefsWindow::setAudioFields() {
  // audio devices
  for (uint i=0;i<settings->getInputDeviceNames().count(); i++)
    inputDeviceComboBox->insertItem(*settings->getInputDeviceNames().at(i),i);
  for (uint i=0;i<settings->getOutputDeviceNames().count(); i++)
    outputDeviceComboBox->insertItem(*settings->getOutputDeviceNames().at(i),i);
  for (uint i=0;i<settings->getRingDeviceNames().count(); i++)
    ringDeviceComboBox->insertItem(*settings->getRingDeviceNames().at(i),i);
  inputDeviceComboBox->setCurrentItem(settings->getInputDevice());
  outputDeviceComboBox->setCurrentItem(settings->getOutputDevice());
  ringDeviceComboBox->setCurrentItem(settings->getRingDevice());
  // filters
  noiseReductionCheckBox->setChecked(settings->getFilterFlag()&IAXC_FILTER_DENOISE);
  aagcCheckBox->setChecked(settings->getFilterFlag()&IAXC_FILTER_AAGC);
  agcCheckBox->setChecked(settings->getFilterFlag()&IAXC_FILTER_AGC);
  echoCancelationCheckBox->setChecked(settings->getFilterFlag()&IAXC_FILTER_ECHO);
  comfortNoiseCheckBox->setChecked(settings->getFilterFlag()&IAXC_FILTER_CN);
  if (settings->getSilenceThreshold()==1) silenceSuppressionCheckBox->setChecked(true);
  incomingRingMuteCheckBox->setChecked(settings->getIncomingRingMute());
  outgoingRingMuteCheckBox->setChecked(settings->getOutgoingRingMute());
}

void KiaxPrefsWindow::accountSelected(int accNumber)
{
  // save the entered in the form values
  if (edittedAccNumber>-1)
    setAccountValues(edittedAccNumber);
  edittedAccNumber = accNumber;
  if (accNumber > -1)
    setFieldsValues(accNumber);
}

void KiaxPrefsWindow::makeDefaultClicked()
{
  settings->setDefaultAccountId(settings->getAccount(accountsListBox->currentItem())->accId);
}

void KiaxPrefsWindow::deleteAccountClicked()
{
  // remove from account settings file
  uint accIndex = accountsListBox->currentItem();
  settings->removeAccount(accIndex);
  edittedAccNumber =-1; // reset the previous editted account number
  accountsListBox->removeItem(accIndex);
  if (accountsListBox->count()>0)
    if (accIndex<accountsListBox->count())accountsListBox->setSelected(accIndex, true);
    else accountsListBox->setSelected(accountsListBox->count()-1, true);
  else initWidgets();
}

QString KiaxPrefsWindow::getSelectedCodecName()
{
  QString result = "gsm";
  if (ilbcRadioButton->isChecked())
    result = "ilbc";
  if (gsmRadioButton->isChecked())
    result = "gsm";
  if (speexRadioButton->isChecked())
    result = "speex";
  if (ulawRadioButton->isChecked())
    result = "ulaw";
  return result;
}

void KiaxPrefsWindow::setCodecByName(QString codecName)
{
  ilbcRadioButton->setChecked(codecName == "ilbc");
  gsmRadioButton->setChecked(codecName == "gsm");
  speexRadioButton->setChecked(codecName == "speex");
  ulawRadioButton->setChecked(codecName == "ulaw");
}

void KiaxPrefsWindow::newAccountClicked()
{
  // remember the values of the currently selected account
  setAccountValues(edittedAccNumber);
  // create a new account
  int accIndex = settings->createNewAccount(0); // 0 = let the account generate its id
  settings->getAccount(accIndex)->accAlias = "Account#" + QString::number(settings->getAccounts().count());
  // mark the new as being editted
  edittedAccNumber = accIndex; // avoid overwriting of former selected account
  // show the default values in the fields
  setFieldsValues(accIndex);
  // insert the name of the account in the list
  accountsListBox->insertItem(settings->getAccount(accIndex)->accAlias,accIndex);
  accountsListBox->setSelected(accIndex,true);
  // and save the account in settings
  setAccountValues(accIndex);
  settings->saveAccount(accIndex);

  enableControls(true);

}

void KiaxPrefsWindow::accountNameEdit_lostFocus() // we want to update the accountname in the listbox
{
  QString tt = accountNameEdit->text();
  accountsListBox->changeItem(QString(accountNameEdit->text()), accountsListBox->currentItem());
}

void	KiaxPrefsWindow::executeCommandCheckbox_toggled(bool checked) {
	executeCommandLabel->setEnabled(checked);
	executeCommandLineEdit->setEnabled(checked);
}

