#include <string.h>
#include <stdio.h>
#include <time.h>

#include <gconf/gconf-client.h>
#include <gdk/gdkkeysyms.h>
#include <glib-object.h>
#include <gdk/gdk.h>
#include <glib.h>

#include "../kipina-i18n.h"
#include "../kparray2d.h"
#include "../kputil.h"
#include "../kpsplitworkout.h"
#include "kpentries.h"

#include "kpnewsplitworkoutdialog.h"
#include "kpnewcommentdialog.h"
#include "kpviewpopupmodel.h"
#include "kpentryviewmodel.h"
#include "kpworkouteditor.h"
#include "kpcalendarview.h"
#include "kpviewmodel.h"
#include "kpguiutils.h"
#include "kpcontexts.h"

/* KPViewModel interface stuff */

static void       kp_calendar_view_model_init     (KPViewModelIface *iface);
static void       kp_calendar_view_set_dmy        (KPViewModel *view,
                                                   guint d,
                                                   guint m,
                                                   guint y);
static void       kp_calendar_view_get_dmy        (KPViewModel *view,
                                                   guint *d,
                                                   guint *m,
                                                   guint *y);
static void       kp_calendar_view_set_view_type  (KPViewModel *view,
                                                   KPViewModelType type);
static
KPViewModelType   kp_calendar_view_get_view_type  (KPViewModel *view);
static void       kp_calendar_view_set_log        (KPViewModel *view,
                                                   KPTrainingLog *log);
static void       kp_calendar_view_unset_log      (KPViewModel *view);
static gchar     *kp_calendar_view_get_icon_name  (KPViewModel *view);
static void       kp_calendar_view_activate       (KPViewModel *cv);
static void       kp_calendar_view_deactivate     (KPViewModel *cv);

/* KPViewPopupModel */
static void       kp_calendar_view_popup_model_init 
                                                  (KPViewPopupModelIface *iface);
/* Some function typedefs */
typedef void    (*DayNumberFunc)                  (KPCalendarView *cv);
typedef void    (*MarkTextFunc)                   (KPCalendarView *cv,
                                                   GString *buffer,
                                                   guint col, guint row);

/* Functions to calculate correct day numbers */
static void       day_num                         (KPCalendarView *cv);
static void       week_num                        (KPCalendarView *cv);
static void       month_num                       (KPCalendarView *cv);
static void       year_num                        (KPCalendarView *cv);
static void       all_time_num                    (KPCalendarView *cv);

/* Functions to get marks */
static void       basic_mark                      (KPCalendarView *cv,
                                                   GString *buffer,
                                                   guint col, guint row);
static void       year_mark                       (KPCalendarView *cv,
                                                   GString *buffer,
                                                   guint col, guint row);
/* GUI-related functions */
static void       update                          (KPCalendarView *cv);
static void       update_marks                    (KPCalendarView *cv);
static gboolean   show_popup_menu                 (KPCalendarView *cv,
                                                   GdkEventButton *event);

static guint      get_hash_key_dmy                (guint d,
                                                   guint m,
                                                   guint y);
static GdkColor  *get_color                       (const gchar *color,
                                                   GtkWidget *widget);
static void       kp_calendar_view_set_title_date (KPCalendarView *cv);

/* Helper functions for keyboard navigation */
static void       find_index                      (KPCalendarView *cv,
                                                   gint *x, gint *y);
static void       find_first                      (KPCalendarView *cv,
                                                   guint *x, guint *y);
static void       find_last                       (KPCalendarView *cv,
                                                   guint *x, guint *y);

/* Helper functions for mouse selections */
static gint       get_col_from_x                  (KPCalendarView *cv,
                                                   gdouble x);
static gint       get_row_from_y                  (KPCalendarView *cv,
                                                   gdouble y);
static gdouble    row_height                      (KPCalendarView *cv);
static gdouble    col_width                       (KPCalendarView *cv);

/* Callbacks (GtkWidget & GObject stuff) */
static void       kp_calendar_view_destroy        (GtkObject *object);
static gboolean   kp_calendar_view_expose         (GtkWidget *widget,
                                                   GdkEventExpose *event);

static void       kp_calendar_view_size_request   (GtkWidget *widget,
                                                   GtkRequisition *requisition);

static void       kp_calendar_view_size_allocate  (GtkWidget *widget,
                                                   GtkAllocation *allocation);

static void       kp_calendar_view_init           (KPCalendarView *view);
static void       kp_calendar_view_realize        (GtkWidget *widget);
static void       kp_calendar_view_unrealize      (GtkWidget *widget);

static gint       kp_calendar_view_button_press   (GtkWidget *widget,
                                                   GdkEventButton *event);

static gint       kp_calendar_view_key_press      (GtkWidget *widget,
                                                   GdkEventKey *event);
static void       kp_calendar_view_class_init     (KPCalendarViewClass *class);
static void       kp_calendar_view_get_property   (GObject *object,
                                                   guint prop_id,
                                                   GValue *value,
                                                   GParamSpec *pspec);
static void       kp_calendar_view_set_property   (GObject *object,
                                                   guint prop_id,
                                                   const GValue *value,
                                                   GParamSpec *pspec);
static void       log_connect_signals             (KPTrainingLog *log,
                                                   KPCalendarView *cv);
static void       log_disconnect_signals          (KPTrainingLog *log,
                                                   KPCalendarView *cv);
static void       log_entry_removed               (KPTrainingLog *log,
                                                   guint d, guint m, guint y,
                                                   const gchar * mark_str,
                                                   KPCalendarView *cv);
static void       log_entry_added                 (KPTrainingLog *log,
                                                   KPCalendarEntry *entry,
                                                   KPCalendarView *cv);
static void       log_entry_changed               (KPTrainingLog *log,
                                                   guint d, guint m, guint y,
                                                   const gchar *mark,
                                                   const gchar *old_mark,
                                                   KPCalendarView *cv);




static struct KPCalendarViewTypeData_
{
  KPViewModelType         type;

  guint                   rows;
  guint                   cols;

  guint                   margin_top;
  guint                   margin_right;
  guint                   margin_bottom;
  guint                   margin_left;

  guint                   padding_x;
  guint                   padding_y;

  guint                   skip_padding_n_x;
  guint                   skip_padding_n_y;

  DayNumberFunc           number_func;
  MarkTextFunc            mark_func;
  guint                 **days;
  guint                 **months;
  guint                 **years;
  gchar                ***marks;

  const gchar            *str;

  gboolean                show_all_months_as_active;
  gdouble                 default_font_size;
}
cvt_data[] = {
  {
  KP_VIEW_MODEL_TYPE_DAY, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
      day_num,
      basic_mark,
      NULL,
      NULL,
      NULL,
      NULL,
      N_("day"),
      TRUE, PANGO_SCALE_LARGE,},
  {
  /* Week calendar */
  KP_VIEW_MODEL_TYPE_WEEK, 2, 4, 0, 0, 0, 0, 0, 0, 0, 0,
      week_num,
      basic_mark,
      NULL,
      NULL,
      NULL,
      NULL,
      N_("week"),
      TRUE, PANGO_SCALE_LARGE,},
  {
  /* Month calendar */
  KP_VIEW_MODEL_TYPE_MONTH, 6, 7, 0, 0, 0, 0, 0, 0, 0, 0,
      month_num,
      basic_mark,
      NULL,
      NULL,
      NULL,
      NULL,
      N_("month"),
      FALSE, PANGO_SCALE_MEDIUM,},
  {
  /* Year calendar */
  KP_VIEW_MODEL_TYPE_YEAR, 18, 28, 0, 0, 0, 0, 6, 6, 7, 6,
      year_num,
      year_mark,
      NULL,
      NULL,
      NULL,
      NULL,
      N_("year"),
      TRUE, PANGO_SCALE_XX_SMALL,
  },
  {
  /* All time calendar, not suppored */
  KP_VIEW_MODEL_TYPE_ALL_TIME, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0,
      all_time_num,
      NULL,
      NULL,
      NULL,
      NULL,
      NULL,
      N_("all time"),
      TRUE, PANGO_SCALE_XX_SMALL,
  },

};

typedef struct KPCalendarViewPrivateData_ KPCalendarViewPrivateData;
typedef struct KPCalendarViewTypeData_ KPCalendarViewTypeData;

struct KPCalendarViewPrivateData_
{
  KPViewModelType     type;
  KPCalendarViewTypeData *cvt;

  GdkWindow          *header_win;
  GdkWindow          *main_win;

  gchar               title_str[255];

  GtkWidget          *title_label;
  GtkWidget          *navi_buttons[4];

  guint               year;
  guint               week;
  guint               month;
  guint               day;

  gboolean            week_start_monday;
  gboolean            marks_need_update;
  gboolean            view_type_updated;
  
  gint                active_x;
  gint                active_y;

  guint               header_height;

  GdkColor            day_num_color;
  GdkColor            grid_color;

  KPTrainingLog      *log;
  gboolean            signals_connected;
  gboolean            view_is_active;

  guint               line_spacing;

  KPEntryPopup       *popup_data;
  
  GtkWidget          *popup_mi_properties;
  GtkWidget          *popup_mi_delete;
  GtkWidget          *popup_mi_edit;
  GtkWidget          *popup_menu_title;
  GtkWidget          *popup_menu;
  GtkWidget          *area;
};

static GHashTable  *marks_table = NULL;
static GHashTable  *entries_table = NULL;
static GHashTable  *allocated_colors = NULL;


#define SELECTED_BG_COLOR(widget) (&widget->style->base[ \
    GTK_WIDGET_HAS_FOCUS (widget) ? GTK_STATE_SELECTED : GTK_STATE_ACTIVE])

#define KP_CALENDAR_VIEW_PRIVATE_DATA(widget) (((KPCalendarViewPrivateData*) \
      (KP_CALENDAR_VIEW (widget)->private_data)))

#define CVT(object) ((object)->cvt[(object)->type])

static gchar *weekdays[7] = {
  N_("Mon"),
  N_("Tue"),
  N_("Wed"),
  N_("Thu"),
  N_("Fri"),
  N_("Sat"),
  N_("Sun")
};

static gchar *months[12] = {
  N_("January"),
  N_("February"),
  N_("March"),
  N_("April"),
  N_("May"),
  N_("June"),
  N_("July"),
  N_("August"),
  N_("September"),
  N_("October"),
  N_("November"),
  N_("December")
};

enum {
  PROP_0,
  PROP_LINE_SPACING,
};

enum {
  DAY_SELECTED_SIGNAL,
  LAST_SIGNAL
};

static guint kp_calendar_view_signals[LAST_SIGNAL] = { 0 };

GType
kp_calendar_view_get_type (void)
{
  static GType kp_calendar_view_type = 0;

  if (!kp_calendar_view_type) {
    static const GTypeInfo kp_calendar_view_info = {
      sizeof (KPCalendarViewClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_calendar_view_class_init,
      NULL,
      NULL,
      sizeof (KPCalendarView),
      0,
      (GInstanceInitFunc) kp_calendar_view_init,
      NULL
    };
    static const GInterfaceInfo view_model_info = {
      (GInterfaceInitFunc) kp_calendar_view_model_init,
      NULL,
      NULL
    };
    static const GInterfaceInfo view_popup_model_info = {
      (GInterfaceInitFunc) kp_calendar_view_popup_model_init,
      NULL,
      NULL
    };
    kp_calendar_view_type = g_type_register_static (GTK_TYPE_WIDGET,
                                                   "KPCalendarView",
                                                   &kp_calendar_view_info,
                                                    0);
    g_type_add_interface_static (kp_calendar_view_type,
                                 KP_TYPE_VIEW_MODEL,
                                &view_model_info);
    g_type_add_interface_static (kp_calendar_view_type,
                                 KP_TYPE_VIEW_POPUP_MODEL,
                                &view_popup_model_info);
  }
  return kp_calendar_view_type;
}


static void
kp_calendar_view_model_init (KPViewModelIface *iface)
{
  iface->set_dmy = kp_calendar_view_set_dmy;
  iface->get_dmy = kp_calendar_view_get_dmy;
  iface->set_log = kp_calendar_view_set_log;
  iface->unset_log = kp_calendar_view_unset_log;
  iface->set_view_type = kp_calendar_view_set_view_type;
  iface->get_view_type = kp_calendar_view_get_view_type;
  iface->get_icon_name = kp_calendar_view_get_icon_name;
  iface->activate = kp_calendar_view_activate;
  iface->deactivate = kp_calendar_view_deactivate;
}


  
KPCalendarEntry *
kp_calendar_view_get_active_entry (KPViewPopupModel *model)
{
  KPCalendarViewPrivateData *p_data;
  const gchar *str;
  guint32 key;
  GSList *list, *node;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (model);

  g_return_val_if_fail (p_data != NULL, NULL);
  g_return_val_if_fail ((guint)p_data->active_x <= CVT (p_data).cols, NULL);
  g_return_val_if_fail ((guint)p_data->active_y <= CVT (p_data).rows, NULL);

  if (!kp_calendar_view_get_active (KP_CALENDAR_VIEW (model)))
    return NULL;

  key = get_hash_key_dmy (CVT (p_data).days[p_data->active_y][p_data->active_x],
                          CVT (p_data).months[p_data->active_y][p_data->active_x], 
                          p_data->year);
  
  list = g_hash_table_lookup (entries_table, GUINT_TO_POINTER (key));

  for (node = list; node; node = node->next) {
    str = (KP_IS_CALENDAR_ENTRY (node->data)) 
        ? kp_calendar_entry_to_string (KP_CALENDAR_ENTRY (node->data)) 
        : NULL;

    if (strcmp (str, CVT (p_data).marks[p_data->active_y][p_data->active_x]) == 0)
      return KP_CALENDAR_ENTRY (node->data); 
  }

  return NULL;
}

static void
kp_calendar_view_popup_model_init (KPViewPopupModelIface *iface)
{
  iface->get_active_entry = kp_calendar_view_get_active_entry;
}


#define KEY_GRID_COLOR     "/apps/kipina/preferences/calendar_grid_color"
#define KEY_DAY_NUM_COLOR  "/apps/kipina/preferences/day_number_color"
 
static void
kp_calendar_view_init (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  GConfClient *client;
  GError *err = NULL;
  gchar *color_str;
  
  client = gconf_client_get_default ();
  
  cv->private_data = g_malloc (sizeof *p_data);
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  p_data->popup_menu = NULL;
  p_data->line_spacing = 0;
  p_data->signals_connected = FALSE;
  p_data->view_is_active = TRUE;
  
  if (marks_table == NULL)
    marks_table = g_hash_table_new_full (g_direct_hash, NULL, NULL, NULL);

  if (entries_table == NULL)
    entries_table = g_hash_table_new_full (g_direct_hash, NULL, NULL, NULL);
  
  p_data->marks_need_update = TRUE;
  p_data->view_type_updated = TRUE;
  p_data->cvt = &cvt_data[0];
  p_data->active_x = -1;
  p_data->active_y = -1;
  p_data->header_height = 20;
  p_data->log = NULL;
  p_data->popup_data = kp_entries_get_popup (KP_VIEW_POPUP_MODEL (cv));
  
  color_str = gconf_client_get_string (client, KEY_GRID_COLOR, &err);
 
  gdk_color_parse ((color_str) ? color_str : "black", &p_data->grid_color);
  gdk_colormap_alloc_color (gtk_widget_get_colormap (GTK_WIDGET (cv)),
                           &p_data->grid_color, FALSE, TRUE);

  if (err) {
    g_warning ("Can't get setting %s: %s!", KEY_GRID_COLOR, err->message);
    g_error_free (err);
  }
  
  err = NULL;  
  color_str = gconf_client_get_string (client, KEY_DAY_NUM_COLOR, NULL);

  gdk_color_parse ((color_str) ? color_str : "black", &p_data->day_num_color);
  gdk_colormap_alloc_color (gtk_widget_get_colormap (GTK_WIDGET (cv)),
                            &p_data->day_num_color, FALSE, TRUE);
  if (err) {
    g_warning ("Can't get setting %s: %s!", KEY_DAY_NUM_COLOR, err->message);
    g_error_free (err);
  }

  GTK_WIDGET_SET_FLAGS (GTK_WIDGET (cv), GTK_CAN_FOCUS);

  g_assert (KP_IS_CALENDAR_VIEW (cv));
}

static void
kp_calendar_view_set_property (GObject *object, guint prop_id,
                               const GValue *value, GParamSpec *pspec)
{
  KPCalendarViewPrivateData *p_data;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (object);
  
  switch (prop_id)
  {
    case PROP_LINE_SPACING:
      p_data->line_spacing = g_value_get_uint (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
kp_calendar_view_get_property (GObject *object, guint prop_id,
                               GValue *value, GParamSpec *pspec)
{
  KPCalendarViewPrivateData *p_data;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (object);
  
  switch (prop_id)
  {
    case PROP_LINE_SPACING:
      g_value_set_uint (value, p_data->line_spacing);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
kp_calendar_view_class_init (KPCalendarViewClass *class)
{
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (class);
  GtkObjectClass *gtk_object_class = GTK_OBJECT_CLASS (class);
  GObjectClass *object_class = G_OBJECT_CLASS (class);

  gtk_object_class->destroy = kp_calendar_view_destroy;
  widget_class->realize = kp_calendar_view_realize;
  widget_class->unrealize = kp_calendar_view_unrealize;
  widget_class->expose_event = kp_calendar_view_expose;
  widget_class->size_request = kp_calendar_view_size_request;
  widget_class->size_allocate = kp_calendar_view_size_allocate;
  widget_class->button_press_event = kp_calendar_view_button_press;
  widget_class->key_press_event = kp_calendar_view_key_press;
  object_class->set_property = kp_calendar_view_set_property;
  object_class->get_property = kp_calendar_view_get_property;

  g_object_class_install_property (object_class,
                                   PROP_LINE_SPACING,
                                   g_param_spec_uint ("line-spacing",
                                  "Line spacing",
                                  "Space in pixels between the lines of text "
                                  "in the calendar.",
                                   0,
                                   32,
                                   3,
                                   G_PARAM_READABLE|G_PARAM_WRITABLE));
  
  kp_calendar_view_signals[DAY_SELECTED_SIGNAL]
    = g_signal_new ("day-selected",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPCalendarViewClass, day_selected),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__POINTER,
                    G_TYPE_NONE,
                    1,
                    G_TYPE_POINTER);
}

static void
kp_calendar_view_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
  requisition->width = 300;
  requisition->height = 200;
}

static void
kp_calendar_view_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
  KPCalendarViewPrivateData *p_data;
  KPCalendarView       *cal = NULL;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (KP_IS_CALENDAR_VIEW (widget));
  g_return_if_fail (allocation != NULL);

  widget->allocation = *allocation;


  if (GTK_WIDGET_REALIZED (widget)) {
    cal = KP_CALENDAR_VIEW (widget);
    p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cal);

    gdk_window_move_resize (widget->window,
                            allocation->x,
                            allocation->y,
                            allocation->width,
                            allocation->height);
  }
}


static void
kp_calendar_view_realize (GtkWidget *widget)
{
  KPCalendarView       *cal;
  GdkWindowAttr       attributes;
  gint                attributes_mask;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (widget));

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  cal = KP_CALENDAR_VIEW (widget);

  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.event_mask = gtk_widget_get_events (widget)
    | GDK_BUTTON_PRESS_MASK| GDK_EXPOSURE_MASK | GDK_KEY_PRESS_MASK;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  widget->window =
    gdk_window_new (widget->parent->window, &attributes, attributes_mask);
  widget->style = gtk_style_attach (widget->style, widget->window);

  gdk_window_set_user_data (widget->window, widget);
  gtk_style_set_background (widget->style, widget->window, GTK_STATE_ACTIVE);

  kp_debug ("KPCalendarView realized.\n");

  update (KP_CALENDAR_VIEW (widget));
}

static void
kp_calendar_view_unrealize (GtkWidget *widget)
{
  KPCalendarViewPrivateData *p_data;
  GtkWidgetClass *parent_class;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (widget);

  kp_debug ("KPCalendarView unrealized\n");
  
  parent_class = g_type_class_peek_parent (
      KP_CALENDAR_VIEW_GET_CLASS (widget));
        
  if (GTK_WIDGET_CLASS (parent_class)->unrealize)
    (* GTK_WIDGET_CLASS (parent_class)->unrealize) (widget);
}


/**
 * kp_calendar_view_new:
 * @type: A #KPViewModelType
 * @d: day
 * @m: month
 * @y: year
 *
 * Create a new #KPCalendarView
 * 
 * Returns: A #KPCalendarView
 */
GtkWidget *
kp_calendar_view_new (KPViewModelType type, guint d, guint m, guint y)
{
  KPCalendarViewPrivateData *p_data;
  GtkWidget *widget;
  GDate *date;
  g_return_val_if_fail (type < KP_VIEW_MODEL_TYPE_N, NULL);

  widget = g_object_new (kp_calendar_view_get_type (), NULL);

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (widget);
  p_data->type = type;

  if (d == 0 || m == 0 || y == 0) {
    date = g_date_new ();
    g_date_set_time (date, time (NULL));
  }
  else
    date = g_date_new_dmy (d, m, y);

  p_data->year = g_date_get_year (date);
  p_data->month = g_date_get_month (date);
  p_data->day = g_date_get_day (date);

  if (cvt_data[type].number_func)
    cvt_data[type].number_func (KP_CALENDAR_VIEW (widget));

  update (KP_CALENDAR_VIEW (widget));
  g_date_free (date);
  
  find_index (KP_CALENDAR_VIEW (widget), &p_data->active_x, &p_data->active_y);
 
  return widget;
}

  
GType
kp_calendar_view_get_active_mark_type (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  KPCalendarEntry *entry;
  const gchar *mark;
  guint d, m, y;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  
  kp_calendar_view_get_date (cv, &d, &m, &y);
  g_return_val_if_fail (g_date_valid_dmy (d, m, y), G_TYPE_INVALID);

  mark = kp_calendar_view_get_current_mark (cv);
  if (mark)
    entry = kp_training_log_get_entry (p_data->log, d, m, y, mark);
  else
    return G_TYPE_INVALID;
  
  if (entry == NULL)
    return G_TYPE_INVALID;
  
  return G_OBJECT_TYPE (entry);  
}


static void
kp_calendar_view_get_dmy (KPViewModel *view, guint *d, guint *m, guint *y)
{
  kp_calendar_view_get_date (KP_CALENDAR_VIEW (view), d, m, y);
}

static void
kp_calendar_view_set_dmy (KPViewModel *view, guint d, guint m, guint y)
{
  kp_calendar_view_set_date (KP_CALENDAR_VIEW (view), d, m, y);
  gtk_widget_queue_draw (GTK_WIDGET (view));
}

static void
add_mark (KPCalendarEntry *entry, KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  KPDate date;
  gboolean already_in_table;
  GSList *list = NULL;
  guint key;

  kp_calendar_entry_get_date (entry, &date);
  
  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  g_return_if_fail (KP_IS_CALENDAR_ENTRY (entry));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  key = get_hash_key_dmy (date.d, date.m, date.y);
  list = g_hash_table_lookup (entries_table, GUINT_TO_POINTER (key));

  already_in_table = (list != NULL);
  list = g_slist_append (list, entry);

  kp_debug ("KPCalendarView has %u marks for this day.\n",
             g_slist_length (list));
  
  if (!already_in_table)
    g_hash_table_insert (entries_table, GUINT_TO_POINTER (key), list);

  p_data->marks_need_update = TRUE;
  update (cv);
}

static void
kp_calendar_view_set_log (KPViewModel *view, KPTrainingLog *log)
{
  KPCalendarViewPrivateData *p_data;
  g_return_if_fail (KP_IS_CALENDAR_VIEW (view));
  g_return_if_fail (KP_IS_TRAINING_LOG (log));

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (view);
  p_data->log = log;

  kp_debug ("Connecting signals.");
  log_connect_signals (log, KP_CALENDAR_VIEW (view));

  kp_training_log_foreach (log, (GFunc) add_mark, view);

  p_data->marks_need_update = TRUE;
  update (KP_CALENDAR_VIEW (view));
}
  
static void
kp_calendar_view_unset_log (KPViewModel *view)
{
  KPCalendarViewPrivateData *p_data;
  g_return_if_fail (KP_IS_CALENDAR_VIEW (view));

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (view);
  
  log_disconnect_signals (p_data->log, KP_CALENDAR_VIEW (view));
  kp_calendar_view_clear_marks (KP_CALENDAR_VIEW (view));
  p_data->log = NULL;
}


static void
kp_calendar_view_activate (KPViewModel *model)
{
  KPCalendarViewPrivateData *p_data;
  g_return_if_fail (KP_IS_CALENDAR_VIEW (model));

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (model);
  p_data->view_is_active = TRUE;
   
  update (KP_CALENDAR_VIEW (model));

  kp_debug ("Activated.");
}


static void
kp_calendar_view_deactivate (KPViewModel *model)
{
  KPCalendarViewPrivateData *p_data;
  g_return_if_fail (KP_IS_CALENDAR_VIEW (model));

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (model);
  p_data->view_is_active = FALSE;
  
  kp_debug ("Deactivated.");

}

static void
log_connect_signals (KPTrainingLog *log, KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (cv));

  g_return_if_fail (KP_IS_TRAINING_LOG (log));

  if (p_data->signals_connected)
    return;
  else
    p_data->signals_connected = TRUE;
  
  g_signal_connect (G_OBJECT (log), "entry-removed",
                    G_CALLBACK (log_entry_removed), cv);
  g_signal_connect (G_OBJECT (log), "new-entry-added",
                    G_CALLBACK (log_entry_added), cv);
}


static void
log_disconnect_signals (KPTrainingLog *log, KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (cv));

  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  
  if (p_data->signals_connected == FALSE)
    return;
  else
    p_data->signals_connected = FALSE;
  
  g_signal_handlers_disconnect_by_func (log, log_entry_removed, cv);
  g_signal_handlers_disconnect_by_func (log, log_entry_added, cv);
  g_signal_handlers_disconnect_by_func (log, log_entry_changed, cv);
}

  
static void
log_entry_removed (KPTrainingLog *log, guint d, guint m, guint y,
                   const gchar *mark_str, KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  KPCalendarEntry *entry;
  GSList *list;
  GSList *days_list;
  guint key;
  guint len;
    
  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));

  kp_debug ("Removing mark: %s\n", mark_str);
 
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  entry = g_object_get_data (G_OBJECT (log), "entry");
  g_return_if_fail (KP_IS_CALENDAR_ENTRY (entry));

  key = get_hash_key_dmy (d, m, y);

  days_list = g_hash_table_lookup (entries_table, GUINT_TO_POINTER (key));
  list = days_list;

  g_return_if_fail (days_list != NULL);

  while (days_list) {
    if (days_list->data == entry) {
      break;
    }
    
    days_list = days_list->next;
  }
  if (!days_list)
    return;
  
  len = g_slist_length (list);

  if (len == 1) {
    list->data = NULL;
    g_slist_free (list);
    g_hash_table_remove (entries_table, GUINT_TO_POINTER (key));
  } else {
    list = g_slist_remove_link (list, days_list);

    g_hash_table_insert (entries_table, GUINT_TO_POINTER (key), list);

    days_list->data = NULL;
    g_slist_free_1 (days_list);
  }

  p_data->marks_need_update = TRUE;
  update (cv);
}


static void
log_entry_added (KPTrainingLog *log, KPCalendarEntry *entry, KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  gboolean already_in_table;
  GSList *list = NULL;
  KPDate date;
  guint key;
  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
 
  g_assert (KP_IS_CALENDAR_ENTRY (entry));
  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  kp_calendar_entry_get_date (entry, &date);
  key = get_hash_key_dmy (date.d, date.m, date.y);
  list = g_hash_table_lookup (entries_table, GUINT_TO_POINTER (key));

  already_in_table = (list != NULL);
  list = g_slist_append (list, entry);

  kp_debug ("KPCalendarView has %u marks for this day.\n",
             g_slist_length (list));
  
  if (!already_in_table)
    g_hash_table_insert (entries_table, GUINT_TO_POINTER (key), list);

  p_data->marks_need_update = TRUE;
  update (cv);
}

static void
log_entry_changed (KPTrainingLog *log, guint d, guint m, guint y,
                   const gchar *mark, const gchar *old_mark,
                   KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (cv));

  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));

  kp_debug ("Mark changed, updating.. \"%s\"", mark);
 
  p_data->marks_need_update = TRUE;
  update (cv);
  
  /*
  kp_calendar_view_remove_mark (cv, d, m, y, old_mark);
  kp_calendar_view_add_mark (cv, d, m, y, 0, 0, mark, TRUE);
  */
}

static void
kp_calendar_view_set_view_type (KPViewModel *view, KPViewModelType type)
{
  kp_calendar_view_set (KP_CALENDAR_VIEW (view), type);
  gtk_widget_queue_draw (GTK_WIDGET (view));
}


static KPViewModelType
kp_calendar_view_get_view_type (KPViewModel *view)
{
  KPCalendarViewPrivateData *p_data;

  g_return_val_if_fail (KP_IS_CALENDAR_VIEW (view), KP_VIEW_MODEL_TYPE_N);
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (view));
  g_return_val_if_fail (p_data != NULL, KP_VIEW_MODEL_TYPE_N);
  
  return p_data->type;
}

static gchar *
kp_calendar_view_get_icon_name (KPViewModel *view)
{
  return g_strdup ("calendar.png");
}


/**
 * kp_calendar_view_get_end_date:
 * @cv: A #KPCalendarView
 *
 * Find out what is the latest date in the current view.
 * 
 * Returns: a #GDate that represents last day in the current view.
 */
GDate *
kp_calendar_view_get_end_date (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  GDate *date;
  guint current;
  guint max;
  guint i,j;

  date = g_date_new ();
  
  g_return_val_if_fail (KP_IS_CALENDAR_VIEW (cv), NULL);
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  max = 0;
  
  for (i=0; i < CVT (p_data).rows; i++)
    for (j=0; j < CVT (p_data).cols; j++) {
      /* In yearview there can be 0s too */
      if (CVT (p_data).days[i][j] > 0) {
        g_date_set_day (date, CVT (p_data).days[i][j]);
        g_date_set_month (date, CVT (p_data).months[i][j]);
        g_date_set_year (date, CVT (p_data).years[i][j]);

        current = g_date_get_julian (date);
        if (current > max)
          max = g_date_get_julian (date);
      }
    }
  g_date_set_julian (date, max);
  g_return_val_if_fail (g_date_valid (date), NULL);

  return date;
}

/**
 * kp_calendar_view_get_start_date:
 * @cv: A #KPCalendarView
 *
 * Find out what is the earliest date in the current view.
 * 
 * Returns: a #GDate that represents first day in the current view.
 */
GDate *
kp_calendar_view_get_start_date (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  GDate *date;
  guint current;
  guint min;
  guint i,j;

  g_return_val_if_fail (KP_IS_CALENDAR_VIEW (cv), NULL);
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  min = G_MAXUINT;
  date = g_date_new ();
  
  for (i=0; i < CVT (p_data).rows; i++)
    for (j=0; j < CVT (p_data).cols; j++) {
      /* In yearview there can be 0s too */
      if (CVT (p_data).days[i][j] > 0) {
        g_date_set_day (date, CVT (p_data).days[i][j]);
        g_date_set_month (date, CVT (p_data).months[i][j]);
        g_date_set_year (date, CVT (p_data).years[i][j]);

        current = g_date_get_julian (date);
        if (current < min)
          min = g_date_get_julian (date);
      }
    }
  g_date_set_julian (date, min);
  g_return_val_if_fail (g_date_valid (date), NULL);
  
  return date;
}


/**
 * kp_calendar_view_get_date:
 * @cv: A #KPCalendarView
 * @d: A location to store day number or NULL
 * @m: A location to store month number or NULL
 * @y: A location to store year number or NULL
 *
 * Finds out the date of the currently selected day. 
 */
void
kp_calendar_view_get_date (KPCalendarView * cv, guint * d, guint * m, guint * y)
{
  KPCalendarViewPrivateData *p_data;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  if (d)
    *d = p_data->day;
  if (m)
    *m = p_data->month;
  if (y)
    *y = p_data->year;
}


/**
 * kp_calendar_view_set_date:
 * @cv: A #KPCalendarView
 * @d: Day number between 1 and 31
 * @m: Month number between 1 and 12
 * @y: Year number in format yyyy
 *
 * Sets the active day in the view to the day represented by 
 * @d, @m and @y.
 */
void
kp_calendar_view_set_date (KPCalendarView *cv, guint d, guint m, guint y)
{
  KPCalendarViewPrivateData *p_data;
  GDate *date;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  date = g_date_new_dmy (d, m, y);
  g_return_if_fail (g_date_valid (date));

  /* Only update marks if we have to. That way this doesn't break
   * the thing with changing the showing workout by clicking it. */
  if (d != p_data->day || m != p_data->month || p_data->year)
    p_data->marks_need_update = TRUE;
  
  p_data->day = d;
  p_data->month = m;
  p_data->year = y;
  update (cv);
  g_date_free (date);
}

void
kp_calendar_view_remove_day_from_hash (gpointer key, GSList *list, 
                                       gpointer data)
{
  GSList *tmp = list;
  
  while (tmp) {
    if (tmp->data) 
      g_free (tmp->data);
    tmp = tmp->next;
  }
  g_slist_free (list);
}

void
kp_calendar_view_clear_marks (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  kp_debug ("Removing marks from KPCalendarView.\n");

  g_hash_table_destroy (entries_table);
  entries_table = g_hash_table_new_full (g_direct_hash, NULL, NULL, NULL);
  
  g_hash_table_foreach (marks_table, (GHFunc)kp_calendar_view_remove_day_from_hash,
                        NULL);
  g_hash_table_destroy (marks_table);
  marks_table = g_hash_table_new_full (g_direct_hash, NULL, NULL, NULL);
    
  p_data->marks_need_update = TRUE;
  update (cv);
}


/**
 * kp_calendar_view_set:
 * @cv: A #KPCalendarView
 * @type: A #KPViewModelType
 *
 * Set the view type of this KPCalendarView. There are currently 
 * some views available: year, month, week and day view.
 */
void
kp_calendar_view_set (KPCalendarView *cv, KPViewModelType type)
{
  KPCalendarViewPrivateData *p_data;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  
  if (type != p_data->type) {
    p_data->marks_need_update = TRUE;
    p_data->view_type_updated = TRUE;
    p_data->type = type;
    update (cv);
  }
}

/**
 * kp_calendar_view_attatch_popup_menu:
 * @cv: A #KPCalendarView
 * @menu: A #GtkMenu
 *
 * Attach a GtkMenu to KPCalendarView so when KPCalendarView
 * receives "popup-menu" event, this menu will be shown.
 * This function could be useful if #KPCalendarView would be used
 * in some other application in the future..
 */
void
kp_calendar_view_attach_popup_menu (KPCalendarView *cv, GtkMenu *menu)
{
  KPCalendarViewPrivateData *p_data;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  p_data->popup_menu = GTK_WIDGET (menu);
}


void
kp_calendar_view_open_to_window (KPViewModelType type, guint d, guint m,
                                 guint y)
{
  GtkWidget *window;
  GtkWidget *calendar;

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_position (GTK_WINDOW (window), GTK_WIN_POS_MOUSE);

  calendar = kp_calendar_view_new (type, d, m, y);
  gtk_container_add (GTK_CONTAINER (window), calendar);
  gtk_widget_show_all (window);
}

gboolean
kp_calendar_view_get_active (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  
  return (p_data->active_x >= 0 && p_data->active_y >= 0);
}

void
kp_calendar_view_set_active (KPCalendarView *cv, gboolean activate)
{
  KPCalendarViewPrivateData *p_data;
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
 
  if (activate)
    find_index (cv, &p_data->active_x, &p_data->active_y);
  else {
    p_data->active_x = -1;
    p_data->active_y = -1;
    update (cv);
  }
}

static void
kp_calendar_view_set_title_date (KPCalendarView * cv)
{
  KPCalendarViewPrivateData *p_data;
  struct tm tm;
  GDate *date;
  gchar buf[255];
  gchar *format;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  date = g_date_new_dmy (p_data->day, p_data->month, p_data->year);

  switch (p_data->type) {
  case KP_VIEW_MODEL_TYPE_WEEK:
    format = _("Week %V %Y");
    break;
  case KP_VIEW_MODEL_TYPE_MONTH:
    format = "%B %Y";
    break;
  case KP_VIEW_MODEL_TYPE_YEAR:
    format = "%Y";
    break;
  case KP_VIEW_MODEL_TYPE_ALL_TIME:
  case KP_VIEW_MODEL_TYPE_DAY:
    format = "%x";
    break;
    /* This would be a bug */
  default:
    format = "";                /* stupid compiler.. */
    g_assert_not_reached ();
  }
  g_date_to_struct_tm (date, &tm);
  strftime (buf, sizeof (buf) - 1, format, &tm);
  g_snprintf(p_data->title_str,
             sizeof (p_data->title_str) - 1,
            "<span size=\"x-large\" weight=\"bold\">%s</span>",
             buf);
  g_date_free (date);
}

  
static guint
get_number_of_entry_in_day (GSList *list, const gchar *entry_str)
{
  KPCalendarEntry *entry;
  GSList *node;
  const gchar *str;
  guint n;


  for (n = 1, node = list; node; node = node->next, n++) {
    if (KP_IS_CALENDAR_ENTRY (node->data)) {
      entry = KP_CALENDAR_ENTRY (node->data);
      str = kp_calendar_entry_to_string (entry);
      
      if (strcmp (str, entry_str) == 0) {
        return n;
      }
    }

  }
  return 0;
}


/**********************************************************************
* Marks handling ..
*********************************************************************/

/**
 * Updates the state of the current view's.
 */
static void
update_marks (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  KPCalendarEntry *entry;
  GSList *list;
  GSList *node;
  guint key;
  guint i, j;
  guint current;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  if (!p_data->marks_need_update)
    return;

  for (i = 0; i < CVT (p_data).rows; i++)
    for (j = 0; j < CVT (p_data).cols; j++) {
      key = get_hash_key_dmy (CVT (p_data).days[i][j], 
                              CVT (p_data).months[i][j], 
                              CVT (p_data).years[i][j]);
    
      /*current = get_number_of_entry_in_day (list, CVT (p_data).marks[row][col]) - 1;*/

      list = g_hash_table_lookup (entries_table, GUINT_TO_POINTER (key));

      if (list && KP_IS_CALENDAR_ENTRY (list->data)) {
        
        if (!p_data->view_type_updated && CVT (p_data).marks[i][j]) {
          current = get_number_of_entry_in_day (list, CVT (p_data).marks[i][j]) - 1;
        } else
          current = 0;
        
        if (current) {
          node = g_slist_nth (list, current);
          if (!node)
            node = list;  /* Just for to be sure */
        } else
          node = list;
          
        entry = KP_CALENDAR_ENTRY (node->data);
        CVT (p_data).marks[i][j] = g_strdup (kp_calendar_entry_to_string (entry));
      } else {
        if (list)
          g_warning ("Entry: %p\n", list->data);
        CVT (p_data).marks[i][j] = NULL;
      }
    }
  p_data->marks_need_update = FALSE;
  p_data->view_type_updated = FALSE;
}


void
kp_calendar_view_remove_mark (KPCalendarView *cv, guint d, guint m, guint y,
                              const gchar *mark)
{
  KPCalendarViewPrivateData *p_data;
  GSList *list = NULL;
  GSList *days_list;
  guint key;
  guint len;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  key = get_hash_key_dmy (d, m, y);
  
  days_list = g_hash_table_lookup (marks_table, GUINT_TO_POINTER (key));
  list = days_list;

  g_return_if_fail (days_list != NULL);

  while (days_list) {
    if (strcmp (days_list->data, mark) == 0)
      break;
    
    days_list = days_list->next;
  }
  if (!days_list)
    return;
  
  len = g_slist_length (list);

  kp_debug ("The list of this day contains %u entries.\n", len);
  
  if (len == 1) {
    g_slist_free (list);
    g_hash_table_remove (marks_table, GUINT_TO_POINTER (key));
  } else {
    list = g_slist_remove_link (list, days_list);

    kp_debug ("The new list contains %u marks.", g_slist_length (list));
   
    g_hash_table_insert (marks_table, GUINT_TO_POINTER (key), list);
    days_list->data = NULL;
    g_slist_free_1 (days_list);
  }
  p_data->marks_need_update = TRUE;
  update (cv);
}


/** 
 * Add mark to a given date. If h or/and m are set, they are used
 * to decide the order of marks in the same day.
 */
void
kp_calendar_view_add_mark (KPCalendarView *cv, guint d, guint m, guint y,
                           guint h, guint min, const gchar *str,
                           gboolean update_marks)
{
  KPCalendarViewPrivateData *p_data;
  gboolean            already_in_table;
  GSList             *list = NULL;
  guint               key;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  key = get_hash_key_dmy (d, m, y);
  list = g_hash_table_lookup (marks_table, GUINT_TO_POINTER (key));

  already_in_table = (list != NULL);
  list = g_slist_append (list, g_strdup (str));

  kp_debug ("KPCalendarView has %u marks for this day.\n",
             g_slist_length (list));
  
  if (!already_in_table)
    g_hash_table_insert (marks_table, GUINT_TO_POINTER (key), list);

  if (update_marks) {
    p_data->marks_need_update = TRUE;
    update (cv);
  }
}

G_CONST_RETURN gchar*
kp_calendar_view_get_current_mark (KPCalendarView *cv) 
{
  KPCalendarViewPrivateData *p_data;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  g_return_val_if_fail (p_data != NULL, NULL);
  g_return_val_if_fail ((guint)p_data->active_x <= CVT (p_data).cols, NULL);
  g_return_val_if_fail ((guint)p_data->active_y <= CVT (p_data).rows, NULL);

  if (!kp_calendar_view_get_active (cv))
    return NULL;

  return CVT (p_data).marks[p_data->active_y][p_data->active_x];
}


void
kp_calendar_view_remove_current_mark (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  const gchar *mark;
  GSList *days_list;
  GSList *list;
  guint key;
  guint len;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  
  g_return_if_fail (kp_calendar_view_get_active (cv));
  
  key = get_hash_key_dmy (p_data->day, p_data->month, p_data->year);
  
  mark = CVT (p_data).marks[p_data->active_y][p_data->active_x];
  
  g_return_if_fail (mark != NULL);
  
  days_list = g_hash_table_lookup (marks_table, GUINT_TO_POINTER (key));

  g_return_if_fail (days_list != NULL);

  list = g_slist_nth (days_list, g_slist_index (days_list, mark));
  len = g_slist_length (days_list);

  if (len == 1) {
    g_slist_free (days_list);
    days_list = NULL;
    g_hash_table_remove (marks_table, GUINT_TO_POINTER (key));
  } else {
    days_list = g_slist_remove_link (days_list, list);
    g_hash_table_replace (marks_table, GUINT_TO_POINTER (key), days_list);
    g_slist_free_1 (list);
  }
  
  p_data->marks_need_update = TRUE;
  update (cv);
}


/*
20030810

2003 * 10000 = 20030000
+ 08 * 100 = 800
+ 10 * 1 = 10
= 20030810
*/
static              guint
get_hash_key_dmy (guint d, guint m, guint y)
{
  return (y * 10000 + m * 100 + d);
}

/**********************************************************************
* WEEKLY VIEW
*********************************************************************/

static void
all_time_num (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  guint cols, rows;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  cols = CVT (p_data).cols;
  rows = CVT (p_data).rows;

  if (CVT (p_data).days == NULL) {
    CVT (p_data).years = (guint **) kp_array_2d_alloc (1, 1, sizeof (guint *));
    CVT (p_data).months = (guint **) kp_array_2d_alloc (1, 1, sizeof (guint *));
    CVT (p_data).days = (guint **) kp_array_2d_alloc (1, 1, sizeof (guint *));
    CVT (p_data).marks = (gchar ***) kp_array_2d_alloc (1, 1, sizeof (gchar *));
  }
  CVT (p_data).years[0][0] = p_data->year;
  CVT (p_data).months[0][0] = p_data->month;
  CVT (p_data).days[0][0] = p_data->day;
}


static void
day_num (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  guint               cols, rows;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  cols = CVT (p_data).cols;
  rows = CVT (p_data).rows;

  if (CVT (p_data).days == NULL) {
    CVT (p_data).years = (guint **) kp_array_2d_alloc (1, 1, sizeof (guint *));
    CVT (p_data).months = (guint **) kp_array_2d_alloc (1, 1, sizeof (guint *));
    CVT (p_data).days = (guint **) kp_array_2d_alloc (1, 1, sizeof (guint *));
    CVT (p_data).marks = (gchar ***) kp_array_2d_alloc (1, 1, sizeof (gchar *));
  }
  CVT (p_data).years[0][0] = p_data->year;
  CVT (p_data).months[0][0] = p_data->month;
  CVT (p_data).days[0][0] = p_data->day;
}

static void
year_num (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  guint               type = KP_VIEW_MODEL_TYPE_YEAR;
  guint             **months = cvt_data[type].months;
  guint             **days = cvt_data[type].days;
  guint               cols = cvt_data[type].cols;
  guint               rows = cvt_data[type].rows;
  guint               ndays_in_prev_month;
  guint               ndays_in_month;
  guint               first_day, day;
  guint               row, col, i;
  guint               month;
  guint               year;
  guint               s_row = 0;
  guint               s_col = 0;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  month = p_data->month;
  year = p_data->year;

  if (days == NULL) {
    g_return_if_fail (months == NULL);

    CVT (p_data).marks = (gchar ***) kp_array_2d_alloc (rows, cols, sizeof (gchar *));
    CVT (p_data).years = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint *));
    CVT (p_data).months = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint *));
    CVT (p_data).days = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint *));
  }

  for (i = 1; i <= 12; i++) {

    if (i <= 4)
      s_row = 0;
    if (i > 4 && i <= 8)
      s_row = 1;
    if (i > 8 && i <= 12)
      s_row = 2;

    s_col = i - s_row * 4 - 1;

    /*  x x x (1,0) (2,0) (3,0)
     *  x x x (1,1) (2,1) (3,1)
     *  x x x
     *  x x x
     */

    ndays_in_month = kp_get_month_len (i, kp_leap (year));
    first_day = kp_day_of_week (year, i, 1);

    /* Week starts on monday, see below */
    first_day--;

    /* FIXME: this should be handled.. */
    /*  if (calendar->display_flags & CALENDAR_WEEK_START_MONDAY)
       first_day--;
       else
       first_day %= 7;
     */

    /* Compute days of previous month */
    if (i > 1)
      ndays_in_prev_month = kp_get_month_len (i - 1, kp_leap (year));
    else
      ndays_in_prev_month = kp_get_month_len (12, kp_leap (year));

    day = ndays_in_prev_month - first_day + 1;
    row = 0;
    if (first_day > 0) {
      for (col = 0; col < first_day; col++) {
        CVT (p_data).days[s_row * 6 + row][s_col * 7 + col] = 0;
        CVT (p_data).months[s_row * 6 + row][s_col * 7 + col] = (i == 1) ?
          12 : i - 1;
        CVT (p_data).years[s_row * 6 + row][s_col * 7 + col] = (i == 1) ? 
          p_data->year - 1 : p_data->year;
        day++;
      }
    }
    /* Compute days of current month */
    col = first_day;
    for (day = 1; day <= ndays_in_month; day++) {
      CVT (p_data).days[s_row * 6 + row][s_col * 7 + col] = day;        /*day */
      CVT (p_data).months[s_row * 6 + row][s_col * 7 + col] = i;
      CVT (p_data).years[s_row * 6 + row][s_col * 7 + col] = p_data->year;

      col++;
      if (col == 7) {
        row++;
        col = 0;
      }
    }
    /* Compute days of next month */
    day = 1;
    for (; row <= 5; row++) {
      for (; col <= 6; col++) {
        CVT (p_data).days[s_row * 6 + row][s_col * 7 + col] = 0;
        CVT (p_data).months[s_row * 6 + row][s_col * 7 + col] =
          (i == 12) ? 1 : i + 1;
        CVT (p_data).years[s_row * 6 + row][s_col * 7 + col] = 
          (i == 12) ? p_data->year + 1 : p_data->year;
        day++;
      }
      col = 0;
    }
  }
}

static void
week_num (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  guint               year;
  guint               cols, rows;
  guint               i, j;
  GDate              *date;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  cols = CVT (p_data).cols;
  rows = CVT (p_data).rows;

  year = p_data->year;
  date = g_date_new_dmy (p_data->day, p_data->month, p_data->year);
  g_date_subtract_days (date, g_date_get_weekday (date) - 1);

  g_return_if_fail (year > 1900);
  g_return_if_fail (year < 2200);

  if (CVT (p_data).days == NULL) {
    CVT (p_data).marks = (gchar ***) kp_array_2d_alloc (rows, cols, sizeof (gchar *));
    CVT (p_data).years = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint *));
    CVT (p_data).months = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint *));
    CVT (p_data).days = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint *));
  }
  for (i = 0; i < rows; i++) {
    for (j = 0; j < cols; j++) {
      
      CVT (p_data).years[i][j] = g_date_get_year (date);
      CVT (p_data).months[i][j] = g_date_get_month (date);
      CVT (p_data).days[i][j] = g_date_get_day (date);

      g_date_add_days (date, 1);
    }
  }
  g_date_free (date);
}

/**********************************************************************
* MONTHLY VIEW
*********************************************************************/

static void
month_num (KPCalendarView * cv)
{
  KPCalendarViewPrivateData *p_data;
  guint               type = KP_VIEW_MODEL_TYPE_MONTH;
  guint               cols = cvt_data[type].cols;
  guint               rows = cvt_data[type].rows;
  guint               ndays_in_prev_month;
  guint               ndays_in_month;
  guint               first_day, day;
  guint               row, col;
  guint               month;
  guint               year;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  month = p_data->month;
  year = p_data->year;

  if (CVT (p_data).days == NULL) {
    CVT (p_data).marks = (gchar ***) kp_array_2d_alloc (rows, cols, sizeof (gchar *));
    CVT (p_data).years = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint *));
    CVT (p_data).months = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint *));
    CVT (p_data).days = (guint **) kp_array_2d_alloc (rows, cols, sizeof (guint *));
  }
  ndays_in_month = kp_get_month_len (month, kp_leap (year));
  first_day = kp_day_of_week (year, month, 1);

  /* Week starts on monday, see below */
  first_day--;

  /* FIXME: this should be handled.. */
  /*  if (calendar->display_flags & CALENDAR_WEEK_START_MONDAY)
     first_day--;
     else
     first_day %= 7;
   */

  /* Compute days of previous month */
  if (month > 1)
    ndays_in_prev_month = kp_get_month_len (month - 1, kp_leap (year));
  else
    ndays_in_prev_month = kp_get_month_len (12, kp_leap (year));

  /* This algorithm to count day numbers is taken from
   * GtkCalendar */

  day = ndays_in_prev_month - first_day + 1;
  row = 0;
  if (first_day > 0) {
    for (col = 0; col < first_day; col++) {
      CVT (p_data).days[row][col] = 0;
      CVT (p_data).months[row][col] = 0;
      CVT (p_data).years[row][col] = 0;
      day++;
    }
  }

  /* Compute days of current month */
  col = first_day;
  for (day = 1; day <= ndays_in_month; day++) {
    CVT (p_data).days[row][col] = day;
    CVT (p_data).months[row][col] = month;
    CVT (p_data).years[row][col] = p_data->year;

    col++;
    if (col == 7) {
      row++;
      col = 0;
    }
  }
  /* Compute days of next month */
  day = 1;
  for (; row <= 5; row++) {
    for (; col <= 6; col++) {
      CVT (p_data).days[row][col] = 0;
      CVT (p_data).months[row][col] = 0;
      CVT (p_data).years[row][col] = 0;
      day++;
    }
    col = 0;
  }
}



static GdkPixbuf *
get_icon_for_entry (KPCalendarView *cv, guint col, guint row)
{
  KPCalendarViewPrivateData *p_data;
  GdkPixbuf *buf, *tmp;
  GSList *list, *node;
  GType type;
  guint key;
  guint n, current;
  gchar *file;
  static GHashTable *table = NULL;
 
  g_return_val_if_fail (KP_IS_CALENDAR_VIEW (cv), NULL);
  
  if (table == NULL)
    table = g_hash_table_new (g_int_hash, g_direct_equal);
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  
  if (CVT (p_data).marks[row][col] == NULL)
    return NULL;
        
  key = get_hash_key_dmy (CVT (p_data).days[row][col],
                          CVT (p_data).months[row][col],
                          CVT (p_data).years[row][col]);
    
  list = g_hash_table_lookup (entries_table, GUINT_TO_POINTER (key));
  n = (list) ? g_slist_length (list) : 0;
       
  if (n > 1) {
    current = get_number_of_entry_in_day (list, CVT (p_data).marks[row][col]) - 1;
    node = g_slist_nth (list, current);
  } else
    node = list;

  g_return_val_if_fail (node != NULL, NULL);
  g_return_val_if_fail (G_IS_OBJECT (node->data), NULL);
  type = G_OBJECT_TYPE (G_OBJECT (node->data));

  if (!(buf = g_hash_table_lookup (table, GINT_TO_POINTER (type)))) {
    file = g_build_filename (KIPINA_PIXMAP_DIR, 
                             kp_calendar_entry_get_icon_name (
                               KP_CALENDAR_ENTRY (node->data)), NULL);
    
    tmp = gdk_pixbuf_new_from_file (file, NULL);
    buf = gdk_pixbuf_scale_simple (tmp, 16, 16, GDK_INTERP_HYPER);
    g_object_unref (tmp);
    g_object_ref (G_OBJECT (buf));
    g_hash_table_insert (table, GINT_TO_POINTER (type), buf);
  } 
  
  return buf;
}

  
static void
basic_mark (KPCalendarView *cv, GString *buffer, guint col, guint row)
{
  KPCalendarViewPrivateData *p_data;
  GSList *list;
  guint key;
  guint n, current;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  if (p_data->type == KP_VIEW_MODEL_TYPE_ALL_TIME)
    CVT (p_data).marks[0][0] = "Not supported";
  
  if (CVT (p_data).marks[row][col] != NULL) {
        
    key = get_hash_key_dmy (CVT (p_data).days[row][col],
                            CVT (p_data).months[row][col],
                            CVT (p_data).years[row][col]);
    
    list = g_hash_table_lookup (entries_table, GUINT_TO_POINTER (key));

    n = (list) ? g_slist_length (list) : 0;
       
    if (n > 1) {
      current = get_number_of_entry_in_day (list, CVT (p_data).marks[row][col]);
      g_string_printf (buffer, "%u.%u [%u/%u]\n%s",
                       CVT (p_data).days[row][col],
                       CVT (p_data).months[row][col],
                       current, n,
                       CVT (p_data).marks[row][col]);
    }
    else {
      g_string_printf (buffer, "%u.%u\n%s",
                       CVT (p_data).days[row][col],
                       CVT (p_data).months[row][col],
                       CVT (p_data).marks[row][col]);
    }
  }
  else if (CVT (p_data).days[row][col] != 0)
    g_string_printf (buffer, "%u.%u",
                     CVT (p_data).days[row][col],
                     CVT (p_data).months[row][col]);
  else
    g_string_printf (buffer, "%s", "");
}      

static void
year_mark (KPCalendarView *cv, GString *buffer, guint col, guint row)
{
  KPCalendarViewPrivateData *p_data;
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  if (CVT (p_data).days[row][col] > 0)
    g_string_printf (buffer, "%u", CVT (p_data).days[row][col]);
  else
    g_string_printf (buffer, "%s", "");
}

/*
* Update GUI.
*/
static void
update (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  static guint n = 0;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  if (!p_data->view_is_active)
    return;

  if (CVT (p_data).number_func)
    CVT (p_data).number_func (cv);
  
  update_marks (cv);

  kp_calendar_view_set_title_date (cv);
 
  gtk_widget_queue_draw (GTK_WIDGET (cv));
  
  if (kp_calendar_view_get_active (cv))
    find_index (cv, &p_data->active_x, &p_data->active_y);

  kp_debug ("KPCalendarView updated, n = %u.", ++n);
}


/**********************************************************************
 * CALLBACKs
 **********************************************************************/

static void
kp_calendar_view_destroy (GtkObject *object)
{
  KPCalendarViewPrivateData *p_data;
  guint i;

  g_return_if_fail (KP_IS_CALENDAR_VIEW (object));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (object));
  if (KP_CALENDAR_VIEW (object)->private_data == NULL)
    return;

  for (i = 0; i < KP_VIEW_MODEL_TYPE_N; i++) {
    
    if (p_data->cvt[i].days != NULL) {
      kp_array_2d_free (p_data->cvt[i].days,
                        p_data->cvt[i].rows,
                        p_data->cvt[i].cols);
      p_data->cvt[i].days = NULL;
    }
    if (p_data->cvt[i].months) {
      kp_array_2d_free (p_data->cvt[i].months,
                        p_data->cvt[i].rows,
                        p_data->cvt[i].cols);
      p_data->cvt[i].months = NULL;
    }
#if 0
    /* TODO: Make 2d array for strings */
    if (p_data->cvt[i].marks) {
      kp_array_2d_free (p_data->cvt[i].marks,
                        p_data->cvt[i].rows,
                        p_data->cvt[i].cols);
      p_data->cvt[i].marks = NULL;
    }
#endif
  }

  /* g_hash_table_destroy (marks_table); */

  if (p_data)
    g_free (p_data);
  
  KP_CALENDAR_VIEW (object)->private_data = NULL;
}



static GdkColor *
get_color (const gchar *color, GtkWidget *widget)
{
  GdkColor *c;
 
  if (!allocated_colors)
    allocated_colors = g_hash_table_new_full (g_direct_hash, g_str_equal,
                                             (GDestroyNotify) gdk_color_free,
                                              g_free);    
  
  c = g_hash_table_lookup (allocated_colors, color);
  if (c)
    return c;
  
  c = g_malloc (sizeof *c);
  gdk_color_parse ((color) ? color : "black", c);
  
  if (gdk_colormap_alloc_color (gtk_widget_get_colormap (widget), c, FALSE, TRUE))  
    g_hash_table_insert (allocated_colors, (gpointer)color, c);
  
  return c;
}


static gboolean
is_active_cell (KPCalendarView *cv, guint row, guint col)
{
  KPCalendarViewPrivateData *p_data;

  if (!kp_calendar_view_get_active (cv))
    return FALSE;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
   
  if (CVT (p_data).type == KP_VIEW_MODEL_TYPE_DAY)
    return FALSE;

  return (CVT (p_data).days[row][col] == p_data->day
      &&  CVT (p_data).months[row][col] == p_data->month
      &&  CVT (p_data).years[row][col] == p_data->year);
}

static gboolean
paint_day (GtkWidget *widget, GdkEventExpose *event, PangoLayout *layout,
           GdkColor *color)
{
  KPCalendarViewPrivateData *p_data;
  static GString *buffer = NULL;
  GdkPixbuf *icon;
  GSList *lines;
  GdkGC *gc;
  PangoRectangle rec;
  guint l;
  gint h;
  guint height, line_height;
  GDate *date;
  struct tm tm;
  gchar buf[64];
  gchar *format;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (widget));
  
  if (!buffer)
    buffer = g_string_sized_new (128);
  
  gc = widget->style->fg_gc[GTK_WIDGET_STATE (widget)];

  pango_layout_set_text (layout, "foo", -1);
  pango_layout_get_pixel_extents (layout, &rec, NULL);
  line_height = rec.height;

  height = widget->allocation.height - p_data->header_height;

  /* Set grid color */
  gdk_gc_set_foreground (gc, &p_data->grid_color);

  /* Draw title background */
  gdk_draw_rectangle (widget->window, widget->style->bg_gc[GTK_STATE_ACTIVE],
                      TRUE, 0, 0, widget->allocation.width,
                      p_data->header_height);
  
  /* Draw title label */
  format = "%A %x";
  date = g_date_new_dmy (p_data->day, p_data->month, p_data->year);
  g_date_to_struct_tm (date, &tm);
  strftime (buf, sizeof (buf) - 1, format, &tm);
  g_date_free (date);

  g_string_printf (buffer, "<b>%s</b>", buf);
  pango_layout_set_markup (layout, buffer->str, -1);
  gdk_draw_layout (widget->window, gc, 3, 3, layout);
  

  /* Draw horizontal line (__) */
  gdk_draw_line (widget->window, gc, 0, p_data->header_height,
                 widget->allocation.width, p_data->header_height);

  CVT (p_data).mark_func (KP_CALENDAR_VIEW (widget), buffer, 0, 0);
   
  pango_layout_set_markup (layout, buffer->str, -1);

  pango_layout_get_pixel_size (layout, NULL, &h);

  /* Paint the icon */
  icon = get_icon_for_entry (KP_CALENDAR_VIEW (widget), 0, 0);
  if (icon)
    gdk_draw_pixbuf (widget->window, NULL, icon, 0, 0, 
                     widget->allocation.width - 16 - 2, 
                     p_data->header_height + 2, 16, 16,
                     GDK_RGB_DITHER_NORMAL, 0, 0);


  /* Paint the text */
  gdk_gc_set_foreground (gc, &p_data->day_num_color);

  lines = pango_layout_get_lines (layout);
  for (l=line_height; l < height && lines && lines->data; lines = lines->next) {
    gdk_draw_layout_line (widget->window, gc, 2 ,
                          p_data->header_height + l + 2, lines->data);
    l += line_height + p_data->line_spacing;
  }

  gdk_gc_set_foreground (gc, get_color ("black", widget));
  gdk_window_end_paint (widget->window);

  return TRUE;
}

static gboolean
paint_week (GtkWidget *widget, GdkEventExpose *event, PangoLayout *layout,
            GdkColor *color)
{
  KPCalendarViewPrivateData *p_data;
  static GString *buffer = NULL;
  GSList *lines;
  GdkGC *gc;
  PangoRectangle      rec;
  GdkPixbuf *icon;
  guint i, j, x, y, l;
  gint h;
  guint width, height, line_height, extra;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (widget));
  
  if (!buffer)
    buffer = g_string_sized_new (128);
  
  gc = widget->style->fg_gc[GTK_WIDGET_STATE (widget)];

  gdk_window_begin_paint_region (widget->window, event->region);
  gdk_window_set_background (widget->window, get_color ("#FFFFFF", widget));
  
  pango_layout_set_text (layout, "foo", -1);
  pango_layout_get_pixel_extents (layout, &rec, NULL);
  line_height = rec.height;

  /* Set grid color */
  gdk_gc_set_foreground (gc, &p_data->grid_color);

  /* Calc how much extra space there is */
  extra = widget->allocation.width % 4;
  width = widget->allocation.width / 4;

  pango_layout_set_width (layout, width * 1000);

  height = (widget->allocation.height - 2 * p_data->header_height) / 2;
 
  /* Draw title backgrounds */
  for (i=0; i < 2; i++) {
    y = (i == 0) ? 0 : p_data->header_height + height;
    gdk_draw_rectangle (widget->window, widget->style->bg_gc[GTK_STATE_ACTIVE],
                        TRUE, 0, y, widget->allocation.width,
                        p_data->header_height);
  }
  
  /* Draw vertical lines (|) */
  for (i=1; i <= 3; i++)
    gdk_draw_line (widget->window, gc, i * width, 0,
                   i * width, widget->allocation.height);

  /* Draw weekday labels */
  for (i=0; i < 7; i++) {
    
    y = (i > 3) ? p_data->header_height + height + 2 : 2;
    x = (i > 3) ? (i - 4) * width + 2 : i * width + 2;
    
    g_string_printf (buffer, "<b>%s</b>", _(weekdays[i]));
    pango_layout_set_markup (layout, buffer->str, -1);
    gdk_draw_layout (widget->window, gc, x, y, layout);
  }

  /* Draw line above the week numbers */
  y = p_data->header_height + height;
  gdk_draw_line (widget->window, gc, 0, y, widget->allocation.width, y);

  for (i=0; i < 2; i++) {
    y =  (i + 1) * p_data->header_height + i * height;
    
    /* Draw horizontal line (__) */
    gdk_draw_line (widget->window, gc, 0, y, widget->allocation.width, y);

    for (j=0,extra = widget->allocation.width % 4; j < 4; j++) {

      x = j * (widget->allocation.width / 4);
  
      /* leave 8th cell empty */
      if ((i == 1 && j >= 3) != TRUE) {
        CVT (p_data).mark_func (KP_CALENDAR_VIEW (widget), buffer, j, i);
   
        pango_layout_set_markup (layout, buffer->str, -1);

        /* Paint active day with different color */
        if (is_active_cell (KP_CALENDAR_VIEW (widget), i, j))
        {
          gdk_gc_set_foreground (gc, SELECTED_BG_COLOR (widget));
          gdk_draw_rectangle (widget->window, gc, TRUE,
                              x + (j > 0),
                              y + 1,
                              width + extra * (j == 3) - (j > 0),
                              height - (i == 0));
        }
      } else 
        pango_layout_set_markup (layout, "", -1);
     
      /* Paint the icon, if there is an entry and if it's not the "8th" cell */
      icon = get_icon_for_entry (KP_CALENDAR_VIEW (widget), j, i);
      if (icon && j != 3)
        gdk_draw_pixbuf (widget->window, NULL, icon, 0, 0, x + width - 16 - 2, y + 2, 16, 16,
                         GDK_RGB_DITHER_NORMAL, 0, 0);

      pango_layout_get_pixel_size (layout, NULL, &h);

      /* Paint the text */
      gdk_gc_set_foreground (gc, &p_data->day_num_color);

      lines = pango_layout_get_lines (layout);
      for (l=line_height; l < height && lines && lines->data; lines = lines->next) {
        gdk_draw_layout_line (widget->window, gc, x + 2 , y + l + 2, lines->data);
        l += line_height + p_data->line_spacing;
      }

      gdk_gc_set_foreground (gc, get_color ("black", widget));
    }
  }
  gdk_window_end_paint (widget->window);

  return TRUE;
}
static gboolean
paint_month (GtkWidget *widget, GdkEventExpose *event, PangoLayout *layout,
             GdkColor *color)
{
  KPCalendarViewPrivateData *p_data;
  static GString *buffer = NULL;
  GSList *lines;
  GdkGC *gc;
  PangoRectangle      rec;
  guint i, j, x, y, l;
  gint h;
  guint width, height, line_height, extra;
  GdkPixbuf *icon;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (widget));
  
  if (!buffer)
    buffer = g_string_sized_new (128);
  
  gc = widget->style->fg_gc[GTK_WIDGET_STATE (widget)];

  gdk_window_begin_paint_region (widget->window, event->region);
  gdk_window_set_background (widget->window, get_color ("#FFFFFF", widget));
  
  pango_layout_set_text (layout, "foo", -1);
  pango_layout_get_pixel_extents (layout, &rec, NULL);
  line_height = rec.height;

  /* Draw title background */
  gdk_draw_rectangle (widget->window, widget->style->bg_gc[GTK_STATE_ACTIVE],
                      TRUE, 0, 0, widget->allocation.width,
                      p_data->header_height);
  
  /* Set grid color */
  gdk_gc_set_foreground (gc, &p_data->grid_color);

  /* Calc how much extra space there is */
  extra = widget->allocation.width % 7;
  width = widget->allocation.width / 7;

  /* Draw vertical lines (|) */
  for (i=1; i < 7; i++)
    gdk_draw_line (widget->window, gc, i * width, 0,
                   i * width, widget->allocation.height);

  /* Draw weekday labels */
  for (i=0; i < 7; i++) {
    g_string_printf (buffer, "<b>%s</b>", _(weekdays[i]));
    pango_layout_set_markup (layout, buffer->str, -1);
    gdk_draw_layout (widget->window, gc, i * width + 3, 2, layout);
  }

  height = (widget->allocation.height - p_data->header_height) / 6;

  for (i=0; i < 6; i++) {
    y = p_data->header_height + i * height;
    /* Draw horizontal line (__) */
    gdk_draw_line (widget->window, gc, 0, y, widget->allocation.width, y);
    
    for (j=0,extra = widget->allocation.width % 7; j < 7; j++) {

      x = j * (widget->allocation.width / 7);
      CVT (p_data).mark_func (KP_CALENDAR_VIEW (widget), buffer, j, i);
      
      pango_layout_set_markup (layout, buffer->str, -1);
      pango_layout_get_pixel_size (layout, NULL, &h);

      /* Paint active day with different color */
      if (is_active_cell (KP_CALENDAR_VIEW (widget), i, j))
      {
        gdk_gc_set_foreground (gc, SELECTED_BG_COLOR (widget));
        gdk_draw_rectangle (widget->window, gc, TRUE,
                            x + (j > 0),
                            y + 1,
                            width - (j > 0) + extra * (j == 6),
                            height);
      }
      /* Paint the icon */
      icon = get_icon_for_entry (KP_CALENDAR_VIEW (widget), j, i);
      if (icon)
        gdk_draw_pixbuf (widget->window, NULL, icon, 0, 0, x + width - 16 - 2,
                         y + 2, 16, 16,
                         GDK_RGB_DITHER_NORMAL, 0, 0);
 
      /* Paint the text */
      gdk_gc_set_foreground (gc, &p_data->day_num_color);

      lines = pango_layout_get_lines (layout);
      for (l=line_height; (l + 2) < height && lines && lines->data; lines = lines->next) {
        gdk_draw_layout_line (widget->window, gc, x + 2, y + l + 2, lines->data);
        l += line_height + p_data->line_spacing;
      }
      gdk_gc_set_foreground (gc, get_color ("black", widget));
    }
  }
  gdk_window_end_paint (widget->window);

  return TRUE;
}

static gboolean
paint_year (GtkWidget *widget, GdkEventExpose *event, PangoLayout *layout,
            GdkColor *color)
{
  KPCalendarViewPrivateData *p_data;
  static GString *buffer = NULL;
  GdkGC *gc;
  PangoRectangle rec;
  guint i, j, w, h, x, y;
  guint row, col;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (widget));
  
  if (!buffer)
    buffer = g_string_sized_new (128);
  
  gc = widget->style->fg_gc[GTK_WIDGET_STATE (widget)];

  w = widget->allocation.width / 4;
  h = (widget->allocation.height - 3 * p_data->header_height) / 3;

  pango_layout_set_width (layout, w * 1000);
  pango_layout_set_text (layout, "foo", -1);
  pango_layout_set_spacing (layout, 0);
  pango_layout_get_pixel_extents (layout, &rec, NULL);
  
  for (i=0; i < 3; i++) {
    y = i * h + (i + 1) * p_data->header_height;

    /* Draw title backgrounds */
    gdk_draw_rectangle (widget->window, widget->style->bg_gc[GTK_STATE_ACTIVE],
                        TRUE, 0, y - p_data->header_height,
                        widget->allocation.width, p_data->header_height);
    /* Draw title bottom lines */
    gdk_draw_line (widget->window, gc, 0, y, widget->allocation.width, y);

    if (i > 0) {
      /* Draw title top lines */
      gdk_draw_line (widget->window, gc, 0, y - p_data->header_height,
                     widget->allocation.width, y - p_data->header_height);
    }

    /* Draw month names */
    for (j=0; j < 4; j++) {
      g_string_printf (buffer, "<b>%s</b>", _(months[i * 4 + j]));
      pango_layout_set_markup (layout, buffer->str, -1);
      gdk_draw_layout (widget->window, gc, j * w + 2,
                       y - p_data->header_height + 2, layout);
    }
  }

  pango_layout_set_markup (layout, "", -1);
 
  /* Draw vertical grid lines */
  for (i=1; i <= 3; i++) 
    gdk_draw_line (widget->window, gc, i * w, 0, i * w, widget->allocation.height);

  /* 28 x 18 */

  guint m_x;
  guint m_y;
  
  w = widget->allocation.width / 4;
  h = (widget->allocation.height - p_data->header_height * 3) / 3;
        
  for (i=0, row=0; i < 18; i++, row++) {
    m_y = i / 6;
    for (j=0, col=0; j < 28; j++, col++) {
      m_x = j / 7;

      x = m_x * w + (col % 7) * (w / 7);
      y = m_y * h + (row % 6) * (h / 6) + (i / 6 + 1) * p_data->header_height;

      /* Paint active day with different color */
      if (is_active_cell (KP_CALENDAR_VIEW (widget), row, col))
      {
        gdk_gc_set_foreground (gc, SELECTED_BG_COLOR (widget));
        gdk_draw_rectangle (widget->window, gc, TRUE,
                            x + (j > 0),
                            y + 1,
                            w / 7 - 1,h / 6);
      }
 
      /* Draw an ellipse */
      gdk_gc_set_foreground (gc, get_color ("#CC9999", widget));

      if (CVT (p_data).marks[row][col])
        gdk_draw_arc (widget->window, gc, FALSE, x+1, y+2, 18, 16, 0, 23040);
      
      /* Day number */
      gdk_gc_set_foreground (gc, get_color ("black", widget));
      year_mark (KP_CALENDAR_VIEW (widget), buffer, col, row);
      pango_layout_set_markup (layout, buffer->str, -1);

      /* If day number is below 10, move it to the middle of the ring */
      if (CVT (p_data).days[row][col] < 10) {
        pango_layout_get_pixel_extents (layout, &rec, NULL);
        x += rec.width / 2;
      }
      
      gdk_draw_layout (widget->window, gc, x + 2, y + 2, layout);
    }
  }

  gdk_gc_set_foreground (gc, get_color ("black", widget));
  gdk_window_end_paint (widget->window);

  return TRUE;
}


static gboolean
kp_calendar_view_expose (GtkWidget *widget, GdkEventExpose *event)
{
  KPCalendarViewPrivateData *p_data;
  static GdkColor    *color = NULL;
  GdkGC              *gc;
  PangoAttrList      *attrs;
  PangoAttribute     *attr_size;
  PangoLayout        *layout;
  guint               w;

  if (!GTK_WIDGET_DRAWABLE (widget))
    return FALSE;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (widget));
  
  gc = widget->style->fg_gc[GTK_WIDGET_STATE (widget)];

  gdk_window_begin_paint_region (widget->window, event->region);
  gdk_window_set_background (widget->window, get_color ("#FFFFFF", widget));
  attrs = pango_attr_list_new ();
  attr_size = pango_attr_size_new ((CVT (p_data).default_font_size * 10000.0));
  pango_attr_list_insert (attrs, attr_size);
  
  if (!color) {
    color = g_malloc (sizeof *color);
    gdk_color_parse ("black", color);
    gdk_color_alloc (gtk_widget_get_colormap (widget), color);
  }

  layout = gtk_widget_create_pango_layout (widget, "foo");

  gdk_gc_set_foreground (gc, get_color ("#FFFFFF", widget));
  gdk_gc_set_background (gc, get_color ("#FFFFFF", widget));
  gdk_draw_rectangle (widget->window, gc, TRUE, 0, 0,
                      widget->allocation.width-1,
                      widget->allocation.height-1);
  gdk_gc_set_foreground (gc, get_color ("black", widget));

  w = widget->allocation.width / CVT (p_data).cols;
  
  pango_layout_set_width (layout, (guint)w * 1000);
  pango_layout_set_wrap (layout, PANGO_WRAP_CHAR);
  pango_layout_set_attributes (layout, attrs);

  pango_attr_list_unref (attrs);

  switch (CVT (p_data).type)
  {  
    case KP_VIEW_MODEL_TYPE_MONTH:
    paint_month (widget, event, layout, color);
    break;
    
    case KP_VIEW_MODEL_TYPE_WEEK:
    paint_week (widget, event, layout, color);
    break;
  
    case KP_VIEW_MODEL_TYPE_DAY:
    paint_day (widget, event, layout, color);
    break;

    case KP_VIEW_MODEL_TYPE_YEAR:
    paint_year (widget, event, layout, color);
    break;
  
    case KP_VIEW_MODEL_TYPE_ALL_TIME:
    pango_layout_set_markup (layout,
                             "<big>View type (All time) can't be shown "
                             "in the calendar!</big>",
                             -1);
    gdk_draw_layout (widget->window, gc, 2, 2, layout);   
    break;

    default:
    g_return_val_if_reached (TRUE);
  }

  return TRUE;
}

static gboolean
show_popup_menu (KPCalendarView *cv, GdkEventButton *event)
{
  KPCalendarViewPrivateData *p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  
  kp_entries_popup_prepare_dynamic (p_data->popup_data, 
                                    KP_VIEW_POPUP_MODEL (cv));
  
  gtk_menu_popup (GTK_MENU (p_data->popup_data->menu),
                  NULL, NULL, NULL, NULL,
                 (event) ? event->button : 0,
                  gdk_event_get_time ((GdkEvent *) event));

  return TRUE;
}

static void
year_coordinates_to_cell (KPCalendarView *cv, guint x, guint y, gint *col, gint *row)
{
  KPCalendarViewPrivateData *p_data;
  guint height;
  guint width;
  guint mw;
  guint w;
  guint hh;
  guint h;
  guint mon;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  height = GTK_WIDGET (cv)->allocation.height;
  width = GTK_WIDGET (cv)->allocation.width;
  hh = p_data->header_height;
  h = (height - (hh * 3)) / 3;
  w = height / 4;

  /* 1st header */
  if (y < hh)
    goto error;

  /* 2nd header */
  if ((y > hh + h) && (y < hh + h + hh))
    goto error;
  
  /* 3rd header */
  if (y > (2 * hh + 2 * h) && (y < (3 * hh + 2 * h)))
    goto error;

  mw = (width / 4);
  mon = (x / (width / 4));
  *col = mon * 7 + ((x % mw) /(mw / 7)); 

  y -= ((y / (hh + h)) + 1) * hh;
  *row = (y - (y / height) * hh) / ((height - 3 * hh) / 18);

  return;

error:
  *row = -1;
  *col = -1;
}
static void
month_coordinates_to_cell (KPCalendarView *cv, guint x, guint y, gint *col, gint *row)
{
  KPCalendarViewPrivateData *p_data;
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  if (y < p_data->header_height) {
    *row = -1;
    *col = -1;
    return;
  }

  y -= p_data->header_height;
  *row = y / ((GTK_WIDGET (cv)->allocation.height - p_data->header_height) / 6);
  *col = x / (GTK_WIDGET (cv)->allocation.width / 7);
}

static void
week_coordinates_to_cell (KPCalendarView *cv, guint x, guint y, gint *col, gint *row)
{
  KPCalendarViewPrivateData *p_data;
  guint h;
  
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  h = (GTK_WIDGET (cv)->allocation.height - 2 * p_data->header_height) / 2;

  if (y < p_data->header_height
   ||(y > p_data->header_height + h && y < p_data->header_height * 2 + h))
  {
    *row = *col = -1;
    return;
  }
 
  *row = (y > (2 * p_data->header_height + h));
  *col = x / (GTK_WIDGET (cv)->allocation.width / 4);

#if 0
  /* This is for historical reasons (week is 1 x 7) */
  *row = *col + ((*row > 0) * 4);
  *col = 0; 

  if (*row == 7)
    *col = *row = -1;
#endif
}

static gboolean
kp_calendar_view_button_press (GtkWidget *widget, GdkEventButton *event)
{
  KPCalendarViewPrivateData *p_data;
  KPCalendarEntry *entry;
  GSList *list_tmp;
  GSList *list = NULL;
  guint current, len;
  KPDate date;
  guint key;
  gint row;
  gint col;
  
  g_return_val_if_fail (KP_IS_CALENDAR_VIEW (widget), TRUE);
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (widget));

  switch (CVT (p_data).type) {
    case KP_VIEW_MODEL_TYPE_WEEK: 
      week_coordinates_to_cell (KP_CALENDAR_VIEW (widget), event->x, event->y,
                               &col, &row);
      break;

    case KP_VIEW_MODEL_TYPE_MONTH:
      month_coordinates_to_cell (KP_CALENDAR_VIEW (widget), event->x, event->y,
                                &col, &row);
      break;

    case KP_VIEW_MODEL_TYPE_YEAR:
      year_coordinates_to_cell (KP_CALENDAR_VIEW (widget), event->x, event->y,
                                &col, &row);
      break;

    default:
      row = get_row_from_y (KP_CALENDAR_VIEW (widget), event->y);
      col = get_col_from_x (KP_CALENDAR_VIEW (widget), event->x);
  }
  
  /* Clicked pixel isn't in any row or column */
  if (row < 0 || col < 0)
    return FALSE;

  p_data->active_x = col;
  p_data->active_y = row;

  if (!GTK_WIDGET_HAS_FOCUS (widget))
    gtk_widget_grab_focus (widget);

  if (event->button == 3) {
    return show_popup_menu (KP_CALENDAR_VIEW (widget), event);
  }
 
  /* In yearview, empty columns are marked as '0' in **days */
  if (CVT (p_data).days[row][col] > 0 && event->type == GDK_BUTTON_PRESS) {

    p_data->month = CVT (p_data).months[row][col];
    p_data->year = CVT (p_data).years[row][col];
    p_data->day = CVT (p_data).days[row][col];
   
    kp_date_set_dmy (&date, p_data->day, p_data->month, p_data->year);
    g_signal_emit (G_OBJECT (widget),
                   kp_calendar_view_signals[DAY_SELECTED_SIGNAL], 0, &date);
        
    key = get_hash_key_dmy (p_data->day, p_data->month, p_data->year);
    list = g_hash_table_lookup (entries_table, GUINT_TO_POINTER (key));
  
    len = (list) ? g_slist_length (list) : 0;
    if (len > 1) {
      current = get_number_of_entry_in_day (list, CVT (p_data).marks[row][col]);
      current--;
      
      if (current >= (len - 1))
        current = 0;
      else
        current++;

      list_tmp = g_slist_nth (list, current);

      if (list_tmp) {
        entry = KP_CALENDAR_ENTRY (list_tmp->data);
        CVT (p_data).marks[row][col] = g_strdup (kp_calendar_entry_to_string (entry));
        
        current = get_number_of_entry_in_day (list, CVT (p_data).marks[row][col]);
      }
    } else {
      if (len && KP_IS_CALENDAR_ENTRY (list->data)) {
        entry = KP_CALENDAR_ENTRY (list->data); 
        CVT (p_data).marks[row][col] = g_strdup (kp_calendar_entry_to_string (entry));
      }
    }
    update (KP_CALENDAR_VIEW (widget));
  }
  return FALSE;
}

static gdouble
row_height (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  guint               ah;
  guint               skip_p_y;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  skip_p_y = (CVT (p_data).skip_padding_n_y < 1)
    ? 1 : CVT (p_data).skip_padding_n_y;

  ah = (GTK_WIDGET (cv)->allocation.height -
        - CVT (p_data).margin_top - CVT (p_data).margin_bottom
        - ((CVT (p_data).rows / skip_p_y) - 1) * CVT (p_data).padding_y);

  return ((gdouble) ah / (gdouble) CVT (p_data).rows);
}

/*
 * Returns the row which point is in.
 * If the point isn't in any row, returns -1.
 */
static              gint
get_row_from_y (KPCalendarView *cv, gdouble y)
{
  KPCalendarViewPrivateData *p_data;
  gdouble             y_row_start;
  gdouble             height;
  gdouble             skip_y;
  guint               i;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  height = row_height (cv);

  for (i = 0; i < CVT (p_data).rows; i++) {
    skip_y = (CVT (p_data).skip_padding_n_y == 0) ? 0
      : (i / CVT (p_data).skip_padding_n_y) * CVT (p_data).padding_y;

    y_row_start = CVT (p_data).margin_top + i * height + skip_y;

    if ((y > y_row_start) && (y < (y_row_start + height))) {
      return i;
    }
  }
  return -1;
}

static              gdouble
col_width (KPCalendarView *cv)
{
  KPCalendarViewPrivateData *p_data;
  guint               aw;
  guint               skip_p_x;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  skip_p_x = (CVT (p_data).skip_padding_n_x < 1)
    ? 1 : CVT (p_data).skip_padding_n_x;

  aw = (GTK_WIDGET (cv)->allocation.width
        - CVT (p_data).margin_left - CVT (p_data).margin_right
        - ((CVT (p_data).cols / skip_p_x) - 1) * CVT (p_data).padding_x);

  return ((gdouble) aw / (gdouble) CVT (p_data).cols);
}

/*
 * Returns the column which point is in.
 * If the point isn't in any column, returns -1.
 */
static              gint
get_col_from_x (KPCalendarView * cv, gdouble x)
{
  KPCalendarViewPrivateData *p_data;
  gdouble             x_col_start;
  gdouble             width;
  guint               skip_x;
  guint               i;

  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
  width = col_width (cv);

  for (i = 0; i < CVT (p_data).cols; i++) {
    skip_x = (CVT (p_data).skip_padding_n_x == 0) ? 0
      : (i / CVT (p_data).skip_padding_n_x) * CVT (p_data).padding_x;

    x_col_start = CVT (p_data).margin_left + i * width + skip_x;

    if ((x > x_col_start) && (x < (x_col_start + width))) {
      return i;
    }
  }
  return -1;
}

static gboolean
kp_calendar_view_key_press (GtkWidget *widget, GdkEventKey *event)
{
  KPCalendarViewPrivateData *p_data;
  KPDate date;
  guint x, y;
  
  g_return_val_if_fail (KP_IS_CALENDAR_VIEW (widget), TRUE);
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (KP_CALENDAR_VIEW (widget));

  x = p_data->active_x;
  y = p_data->active_y;

  switch (event->keyval)
  {
    case GDK_KP_Left:
    case GDK_Left:
      if (x > 0) {
        x--; 
      } else {
        x = CVT (p_data).cols - 1;
      }
    break;

    case GDK_KP_Right:
    case GDK_Right:
      if (x < (CVT (p_data).cols - 1))
        x++;
      else
        x = 0;
    break;

    case GDK_KP_Up:
    case GDK_Up:
      if (y > 0)
        y--;
      else
        y = CVT (p_data).rows - 1;
    break;

    case GDK_KP_Down:
    case GDK_Down:
      if (y < (CVT (p_data).rows - 1))
        y++;
      else
        y = 0;
    break;

    case GDK_KP_Home:
    case GDK_Home:
      find_first (KP_CALENDAR_VIEW (widget), &x, &y);
    break;
    
    case GDK_KP_End:
    case GDK_End:
      find_last (KP_CALENDAR_VIEW (widget), &x, &y);
    break;
  }
  /* Not a day, just an empty cell */
  if (CVT (p_data).days[y][x] == 0) 
    return TRUE;

  p_data->active_x = x;
  p_data->active_y = y;
  p_data->day = CVT (p_data).days[p_data->active_y][p_data->active_x];
  p_data->month = CVT (p_data).months[p_data->active_y][p_data->active_x];
  p_data->year = CVT (p_data).years[p_data->active_y][p_data->active_x];

  update (KP_CALENDAR_VIEW (widget));
  
  kp_date_set_dmy (&date, p_data->day, p_data->month, p_data->year);
  g_signal_emit (G_OBJECT (widget),
                 kp_calendar_view_signals[DAY_SELECTED_SIGNAL], 0, &date);
  
  return TRUE;
}


static void
find_last (KPCalendarView *cv, guint *x, guint *y)
{
  KPCalendarViewPrivateData *p_data;
  guint i, j;
  guint n;
  
  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  if (CVT (p_data).type == KP_VIEW_MODEL_TYPE_YEAR)
    n = 12;
  else
    n = 0;

  for (i = CVT (p_data).rows; i > n; i--)
    for (j = CVT (p_data).cols; j > 0; j--)
      if (CVT (p_data).days[i-1][j-1] > 0) {
        *x = j - 1;
        *y = i - 1;
        return;
      }
  g_assert_not_reached ();
}


static void
find_first (KPCalendarView *cv, guint *x, guint *y)
{
  KPCalendarViewPrivateData *p_data;
  guint i, j;
  
  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);

  for (i = 0; i < CVT (p_data).rows; i++)
    for (j = 0; j < CVT (p_data).cols; j++)
      if (CVT (p_data).days[i][j] > 0) {
        *x = j;
        *y = i;
        return;
      }
  g_assert_not_reached ();
}

static void
find_index (KPCalendarView *cv, gint *x, gint *y)
{
  KPCalendarViewPrivateData *p_data;
  guint i, j;
  
  g_return_if_fail (KP_IS_CALENDAR_VIEW (cv));
  p_data = KP_CALENDAR_VIEW_PRIVATE_DATA (cv);
 
  for (i=0; i < CVT (p_data).rows; i++)
    for (j=0; j < CVT (p_data).cols; j++)
      if (CVT (p_data).days[i][j] == p_data->day
       && CVT (p_data).months[i][j] == p_data->month
       && CVT (p_data).years[i][j] == p_data->year) {

        *y = i;
        *x = j;
        return;
      }
  g_assert_not_reached ();
}

