#include <string.h>

#include <gconf/gconf-client.h>
#include <glade/glade.h>

#include "../kipina-i18n.h"
#include "../kpsettings.h"

#include "kpviewmodel.h"
#include "kpstatusbar.h"
#include "kpguiutils.h"

  
void
kp_gui_info_message (GtkWindow *parent, const gchar *format, ...)
{
  GtkWidget *widget;
  gchar buf[512];
  va_list args;

  va_start (args, format);
  vsnprintf (buf, sizeof (buf)-1, format, args);
  va_end (args);
  widget = gtk_message_dialog_new (parent,
                                   GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                   GTK_MESSAGE_INFO,
                                   GTK_BUTTONS_CLOSE,
                                   buf);

  gtk_dialog_run (GTK_DIALOG (widget));
  gtk_widget_destroy (widget);  
}
void
kp_gui_err_message (GtkWindow *parent, const gchar *format, ...)
{
  GtkWidget *widget;
  gchar buf[512];
  va_list args;

  va_start (args, format);
  vsnprintf (buf, sizeof (buf)-1, format, args);
  va_end (args);
  widget = gtk_message_dialog_new (parent,
                                   GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                   GTK_MESSAGE_ERROR,
                                   GTK_BUTTONS_CLOSE,
                                   buf);

  gtk_dialog_run (GTK_DIALOG (widget));
  gtk_widget_destroy (widget);  
}


GladeXML *
kp_gui_load (const gchar *module, const gchar * root)
{
  GladeXML *xml;
  gchar *filename;
    
  filename = g_strdup_printf ("%s/%s.glade", KIPINA_GLADE_DIR, module);

  xml = glade_xml_new (filename, root, NULL);
      
  if (!xml)
    g_error ("Can't load ui from file \"%s\", this is a BUG!", filename);

  g_free (filename);

  return xml;
}

/**
 * This functions is used to connect all signal handlers in an array
 * of GuiModuleSignalsData -definitions.
 * 
 * This way we can declare our signal handlers as static instead of
 * declaring them global functions for glade_xml_signal_autoconnect().
 **/
void 
kp_gui_module_signals_connect (GladeXML *xml, GuiModuleSignalsData *gmd)
{
  GtkWidget *widget;
  guint i=0;
  
  while (gmd[i].glade_widget_name) {
    widget = glade_xml_get_widget (xml, gmd[i].glade_widget_name);
    
    if (!widget)
      g_error ("Widget '%s' not found! This is a BUG!", gmd[i].glade_widget_name);
    
    g_signal_connect (G_OBJECT (widget), gmd[i].signal,
                      G_CALLBACK (gmd[i].callback), gmd[i].data);
    i++;
  }
}

/**
 * The same as above, but use pass @data to all callback functions as user data.
 **/
void 
kp_gui_module_signals_connect_data (GladeXML *xml, GuiModuleSignalsData *gmd, gpointer data)
{
  GtkWidget *widget;
  guint i=0;
  
  while (gmd[i].glade_widget_name) {
    widget = glade_xml_get_widget (xml, gmd[i].glade_widget_name);
    
    if (!widget)
      g_error ("Widget '%s' not found! This is a BUG!", gmd[i].glade_widget_name);
    
    g_signal_connect (G_OBJECT (widget), gmd[i].signal,
                      G_CALLBACK (gmd[i].callback), data);
    i++;
  }
}

/**
 * kp_gui_get_option_menu_active:
 * @menu: A #GtkOptionMenu
 *
 * Get the text of the currently selected item of the GtkOptionMenu.
 * 
 * Returns: The text of the active item or NULL if something goes wrong.
 * If the value is not NULL, it must be freed by the caller.
 */
gchar *
kp_gui_get_option_menu_active (GtkOptionMenu *menu)
{
  GtkWidget *label;

  g_return_val_if_fail (GTK_IS_OPTION_MENU (menu), NULL);
 
  if (GTK_BIN (menu)->child)
    label = gtk_bin_get_child (GTK_BIN (menu));
  else {
    if (GTK_IS_BIN (menu->menu_item)
     && GTK_IS_LABEL (GTK_BIN (menu->menu_item)->child))
      label = GTK_BIN (menu->menu_item)->child;
    else
      return NULL;
  }
 
  return g_strdup (gtk_label_get_text (GTK_LABEL (label)));
}

 
gint
kp_gui_get_combo_box_index (GtkComboBox *box, const gchar *key)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *str;
  gint i;

  model = gtk_combo_box_get_model (box);

  if (gtk_tree_model_get_iter_first (model, &iter) == FALSE)
    return -1;
  
  i = 0;
  do {
    gtk_tree_model_get (model, &iter, 0, &str, -1);
    
    if (strcmp (str, key) == 0)
      return i;
    i++;
  }
  while (gtk_tree_model_iter_next (model, &iter));
  return -1;  
}


GtkResponseType
kp_gui_ask_if_user_wants_to_save (GtkWindow *window)
{
  GtkWidget *dialog;
  gint response;

  g_return_val_if_fail (GTK_IS_WINDOW (window), GTK_RESPONSE_NONE);

  dialog = gtk_message_dialog_new_with_markup (window,
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_WARNING,
                            GTK_BUTTONS_NONE,
                           "<big><b>Save the changes to document before "
                           "closing?</b></big>\n\nIf you don't save, changes will be "
                           "permanently lost.");
    
  gtk_dialog_add_buttons (GTK_DIALOG (dialog),
                         "Close _without Saving", GTK_RESPONSE_CLOSE,
                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                          GTK_STOCK_SAVE, GTK_RESPONSE_OK,
                          NULL);
  
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);

  response = gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);

  return response;
}


gboolean
kp_gui_ask_if_user_wants_to_replace (GtkWindow *window, const gchar *file)
{
  GtkWidget *dialog;
  gint response;

  g_return_val_if_fail (GTK_IS_WINDOW (window), GTK_RESPONSE_NONE);

  dialog = gtk_message_dialog_new_with_markup (window,
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_QUESTION,
                            GTK_BUTTONS_NONE,
                           "<big><b>A file named %s already exists.</b></big>\n\n"
                           "Do you want to replace it with the one you are saving?",
                            file);
    
  gtk_dialog_add_buttons (GTK_DIALOG (dialog), 
                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                       _("Replace"), GTK_RESPONSE_OK,
                          NULL);
  
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);

  response = gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);

  return response == GTK_RESPONSE_OK;
}

  
GtkResponseType
kp_gui_ask_if_user_wants_to_save_old_log (GtkWindow *window)
{
  GtkWidget *dialog;
  gint response;

  g_return_val_if_fail (GTK_IS_WINDOW (window), GTK_RESPONSE_NONE);

  dialog = gtk_message_dialog_new_with_markup (window,
                            GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_QUESTION,
                            GTK_BUTTONS_NONE,
                         _("<big><b>Do you want to save the current log before "
                           "creating a new one?</b></big>\n\n"
                           "If you say no, the changes you made will be lost!"));
    
  gtk_dialog_add_buttons (GTK_DIALOG (dialog),
                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                          GTK_STOCK_NO, GTK_RESPONSE_NO,
                          GTK_STOCK_YES, GTK_RESPONSE_YES,
                          NULL);
  
  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);

  response = gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);

  return response;
}


gint
kp_gui_get_yes_no_cancel (GtkWindow *window, const gchar *message)
{
  GtkWidget *dialog;
  gint response;

  g_return_val_if_fail (GTK_IS_WINDOW (window), GTK_RESPONSE_NONE);

  dialog = gtk_message_dialog_new (window,
                                   GTK_DIALOG_MODAL |
                                   GTK_DIALOG_DESTROY_WITH_PARENT,
                                   GTK_MESSAGE_QUESTION,
                                   GTK_BUTTONS_YES_NO,
                                   message);

  gtk_dialog_add_button (GTK_DIALOG (dialog),
                         GTK_STOCK_CANCEL,
                         GTK_RESPONSE_CANCEL);

  gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);

  response = gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);

  return response;
}

/**
 * kp_gui_report_error:
 * @window: Parent window for the dialog
 * @bar: A #KPStatusbar, can be NULL if msg is not wanted
 * to be shown in the statusbar
 * @msg: Message to show to the user
 *
 * Show this error message to the user.
 */
void
kp_gui_report_error (GtkWindow *window, KPStatusbar *bar, const gchar *msg)
{
  GtkWidget *dialog;

  if (KP_IS_STATUSBAR (bar))
    kp_statusbar_set_message (bar, msg);
  
  dialog = gtk_message_dialog_new (GTK_WINDOW (window),
                                   GTK_DIALOG_DESTROY_WITH_PARENT,
                                   GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, msg);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}


/**
 * This function never returns NULL or other value which
 * can be considered as error. So, caller of this function
 * must not check the return value.
 * 
 * If some error happens, it is a bug
 * and program execution will be terminated.
 **/
GtkWidget *
kp_gui_get_widget (GladeXML *xml, const gchar *widget_name)
{
  GtkWidget *widget;
  
  if (!xml || !widget_name)
    g_assert_not_reached ();
  
  widget = glade_xml_get_widget (xml, widget_name);
  if (GTK_IS_WIDGET (widget))
    return widget;

  g_error ("%s(): widget '%s' not found! This is a BUG!",
           __PRETTY_FUNCTION__, widget_name);

  /* Prevent a compiler warning */
  return NULL;
}


gchar *
kp_gui_get_dir (GtkWindow *parent)
{
  GtkWidget *dialog;
  char *filename = NULL;

  dialog = gtk_file_chooser_dialog_new (_("Select a Directory"),
                                        parent,
                                        GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                        GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                                        NULL);

  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
  {
    filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
  }
  gtk_widget_destroy (dialog);

  return filename;
}

  
gchar *
kp_gui_get_file_to_open (GtkWindow *parent)
{
  static GtkWidget *dialog = NULL;
  GConfClient *client;
  char *filename = NULL;
  gchar *default_dir;

#define KEY_DEFAULT_DIR "/apps/kipina/preferences/default_dir"

  client = gconf_client_get_default ();

  dialog = gtk_file_chooser_dialog_new (_("Open File"),
                                        parent,
                                        GTK_FILE_CHOOSER_ACTION_OPEN,
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                        GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                                        NULL);
  
  if ((default_dir = gconf_client_get_string (client, KEY_DEFAULT_DIR, NULL))) {
    if (strlen (default_dir) != 0) {
      if (!gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (dialog),
                                                default_dir))
        kp_debug ("Couldn't set the file chooser current dir: \"%s\"!", default_dir);
    }
  }

  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT) {
    filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
  }
  gtk_widget_destroy (dialog);

  return filename;
}

gchar *
kp_gui_get_file_to_save (GtkWindow *parent)
{
  GtkWidget *dialog;
  char *filename = NULL;

  dialog = gtk_file_chooser_dialog_new (_("Save File"),
                                        parent,
                                        GTK_FILE_CHOOSER_ACTION_SAVE,
                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                        GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
                                        NULL);

  if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
  {
    filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
  }
  gtk_widget_destroy (dialog);

  return filename;
}


void
kp_gui_open_file_dialog (GCallback handler)
{
  GtkFileSelection *sel;
  GtkWidget *file_selection;

  file_selection = gtk_file_selection_new (_("Please select a file!"));
  gtk_window_set_position (GTK_WINDOW (file_selection), GTK_WIN_POS_MOUSE);

  sel = GTK_FILE_SELECTION (file_selection);

  g_signal_connect (GTK_OBJECT (sel->ok_button), "clicked",
                    G_CALLBACK (handler), file_selection);

  g_signal_connect_swapped (GTK_OBJECT (sel->ok_button), "clicked",
                            G_CALLBACK (gtk_widget_destroy), file_selection);

  g_signal_connect_swapped (GTK_OBJECT (sel->cancel_button), "clicked",
                            G_CALLBACK (gtk_widget_destroy), file_selection);

  gtk_widget_show (file_selection);
}

void
kp_gui_destroy_widget_passed (GtkWidget *widget, gpointer data)
{
  if (GTK_IS_WIDGET (data))
    gtk_widget_destroy (GTK_WIDGET (data));
}


/**
 * kp_gui_get_dates_for_view_type:
 * @date: Date to get start and end for
 * @type: View type
 * @start: Location that must store valid pointer to #GDate
 * @end: Location that must store valid pointer to #GDate
 *
 * Get start and end date for the @date and @type.
 */
void
kp_gui_get_dates_for_view_type (GDate *date, KPViewModelType type,
                                GDate **start, GDate **end,
                                KPTrainingLog *log)
{
  guint d, m, y;
  guint n_items;
  guint week;
  KPDate *d1;
  KPDate *d2;
  
  g_return_if_fail (g_date_valid (date));
  g_return_if_fail (start != NULL);
  g_return_if_fail (end != NULL);

  d = g_date_get_day (date);
  m = g_date_get_month (date);
  y = g_date_get_year (date);
  
  switch (type)
  {
    case KP_VIEW_MODEL_TYPE_DAY:
      g_date_set_dmy (*start, d, m, y);
      g_date_set_dmy (*end, d, m, y);
      break;

    case KP_VIEW_MODEL_TYPE_WEEK:
      if (kp_week_of_year (&week, &y, m, d)) {
        g_date_set_dmy (*start, d, m, y);
        g_date_subtract_days (*start, g_date_get_weekday (date) - 1);
        g_date_set_julian (*end, g_date_get_julian (*start));
        g_date_add_days (*end, 7-1);
        break;
      } else
        g_return_if_reached ();
    
    case KP_VIEW_MODEL_TYPE_MONTH:
      n_items = kp_get_month_len (m, kp_leap (y));
      g_date_set_dmy (*start, 1, m, y);
      g_date_set_dmy (*end, n_items, m, y);
      break;
  
    case KP_VIEW_MODEL_TYPE_YEAR:
      g_date_set_dmy (*start, 1, 1, y);
      g_date_set_dmy (*end, 31, 12, y);
      break;

    case KP_VIEW_MODEL_TYPE_ALL_TIME:
      g_return_if_fail (KP_IS_TRAINING_LOG (log));

      d1 = kp_training_log_get_earliest_date (log);
      d2 = kp_training_log_get_latest_date (log);
      
      g_date_set_dmy (*start, d1->d, d1->m, d1->y);
      g_date_set_dmy (*end, d2->d, d2->m, d2->y);

      kp_date_free (d1);
      kp_date_free (d2);
      
      break;
      
    default:
      g_return_if_reached ();
  }
}

/**
 * kp_get_combo_box_first_entry:
 * @box: A #GtkComboBox
 *
 * Get the first item in the combo box.
 *
 * Returns: A newly-allocated string or NULL if the box is empty.
 */
gchar *
kp_get_combo_box_first_entry (GtkComboBox *box)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *entry;

  g_return_val_if_fail (GTK_IS_COMBO_BOX (box), NULL);

  model = gtk_combo_box_get_model (box);
  g_return_val_if_fail (GTK_IS_TREE_MODEL (model), NULL);
  
  if (gtk_tree_model_get_iter_first (model, &iter)) {
    gtk_tree_model_get (model, &iter, 0, &entry, -1);
    return entry;
  }

  return NULL;
}

GdkPixbuf *
kp_get_icon_as_pixbuf (const gchar *icon)
{
  GdkPixbuf *buf;
  gchar *name;

  name = kp_get_icon_filename (icon);
  buf = gdk_pixbuf_new_from_file_at_scale (name, 16, 16, TRUE, NULL);
  g_free (name);

  return buf;
}


GtkImage *
kp_get_icon_as_image (const gchar *icon)
{
  GtkWidget *image;
  GdkPixbuf *buf;
  gchar *name;

  name = kp_get_icon_filename (icon);
  buf = gdk_pixbuf_new_from_file_at_scale (name, 16, 16, TRUE, NULL);
  image = gtk_image_new_from_pixbuf (buf);
  g_free (name);

  return GTK_IMAGE (image);
}

gchar *
kp_get_icon_filename (const gchar *icon)
{
  return g_strdup_printf ("%s/%s", KIPINA_PIXMAP_DIR, icon);
}


gchar *
kp_color_to_markup (GdkColor *color)
{
  gchar *str;
  gchar *rgb;
  
  rgb = g_strdup_printf ("#%02x%02x%02x",
                         color->red / 256,
                         color->green / 256,
                         color->blue / 256);
  
  str = g_strdup_printf ("<span weight=\"bold\" color=\"%s\">%s</span>",
                           rgb, rgb);
  g_free (rgb);

  return str;
}


gchar *
kp_color_str_to_markup (const gchar *color)
{
  GdkColor c;
  gdk_color_parse (color, &c);

  return kp_color_to_markup (&c);
}


void
kp_color_from_markup (const gchar *markup, GdkColor *color)
{
  gchar *str;
  
  if (!pango_parse_markup (markup, -1, 0, NULL, &str, NULL, NULL))
    str = g_strdup ("black");

  gdk_color_parse (str, color);
  g_free (str);
}


