#include <math.h>

#include "page-year.h"
#include "kphtmlutil.h"
#include "kphtmlwriter.h"

#include "../kpworkoutmodel.h"
#include "../kipina-i18n.h"

static KPHtmlNaviData navi_data = 
{
  1, 0, 0,
  { NULL, NULL, NULL, NULL, NULL },
  { NULL, NULL, NULL, NULL, NULL }
};


static void
stats_table_year (KPHtmlWriter *writer, FILE *fp, guint year)
{
  KPLogStatItem *item;
  const gchar *name;
  guint msec;
  gchar *str;
  guint m;

  kp_html_h1 (fp, "%s %u", _("Year"), year);
  kp_html_table_head (fp, _("Month"), 
                          _("Total Distance"), 
                          _("Total Duration"),
                          _("Avg Pace"),
                          _("Avg Distance"),
                          _("Avg Duration"),
                          _("N Workouts"),
                          _("Other Entries"), 
                             NULL);
  
  for (m=0; m < 12; m++) {
    item = kp_log_stat_get_month_stats (writer->stat, year, m+1, NULL, NULL);
    g_return_if_fail (item != NULL);

    if (item->n_entries == 0)
      continue;
    
    name = kp_get_month_name (m);
   
    kp_html_table_row_start (fp);

    /* Year */
    kp_html_table_cell (fp, "<a href=\"month-%u-%u.html\"><b>%s</b></a>\n", 
                          year, m+1, _(name));
    
    /* Distance */
    kp_html_table_cell (fp, "%.1f", item->tot_di);

    /* Duration */
    str = kp_date_mseconds_to_std_string (item->tot_du / 1000 * 1000);
    kp_html_table_cell (fp, "%s", str);
    g_free (str);

    /* Avg Pace */
    if (item->avg_di != 0 && item->avg_du != 0) {
      /* Get rid of the milliseconds */
      msec = ((guint)(floor(((gdouble)item->avg_du / (gdouble) item->avg_di) + 0.5))) / 1000;
      str = kp_date_mseconds_to_std_string (msec * 1000);
      kp_html_table_cell (fp, "%s", str);
      g_free (str);
    }
    else
      kp_html_table_cell (fp, "%c", '?');

    /* Avg Distance */
    kp_html_table_cell (fp, "%.1f", item->avg_di);

    /* Avg Duration */
    str = kp_date_mseconds_to_std_string (item->avg_du / 1000 * 1000);
    kp_html_table_cell (fp, "%s", str);
    g_free (str);
    
    /* N Workouts */
    kp_html_table_cell (fp, "%u", item->n_workouts);
 
    /* Other entries */   
    kp_html_table_cell (fp, "%u", item->n_entries - item->n_workouts);

    kp_html_table_row_end (fp);
    kp_log_stat_item_free (item);
  }

  kp_html_table_foot (fp);
}
  
  
void
page_year (KPHtmlWriter *writer, KPTrainingLog *log, guint year, GError **err)
{
  GError *tmp_error;
  FILE *fp;
  gchar *f;
  gchar *ft;

  ft = g_strdup_printf ("Year %u", year);
  f = g_strdup_printf ("year-%u.html", year);
 
  /* Year page */
  tmp_error = NULL;
  fp = kp_html_writer_open (writer, f, &tmp_error);
  
  if (fp == NULL) {
    g_propagate_error (err, tmp_error);
    return;
  }
 
  kp_html_navi (fp, &navi_data, year, 0, 0);

  g_free (ft); 
  g_free (f);
  
  stats_table_year (writer, fp, year);
  kp_html_writer_close (fp);
}


