/*
 *  Klavier : Virtual keyboard for K Destop Environement
 *
 *  This program (klavier) is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 *  This program (klavier) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public
 * License along with this program (klavier); if not, write to the Free
 * Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA. 
 *
 * Author : Sebastien HUSS <sebastien.huss@laposte.net>
 *
 */


/*
 * Includes
 */

//#include <stdio.h>
#include <iostream>
using namespace std;
#include <stdlib.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xlibint.h>
#include <X11/Xutil.h>
#include <X11/keysymdef.h>
#include <X11/keysym.h>
#include <X11/extensions/XTest.h>
#include <X11/Xos.h>
#include <X11/Xproto.h>
#include <ctype.h>


#include "sendkey.h"

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif
/*
 * Constructor
 */

sendKey::sendKey(Display *disp)
{
//	this->display		= XOpenDisplay("");
	this->display		= disp;
	this->listKeyCode	= NULL;
	this->useXTest		= initXTest();
	getModifierList();
	getKeyList();
}

/*
 * Destructor
 */
sendKey::~sendKey()
{
	if (this->listKeyCode != NULL)
		Xfree(this->listKeyCode);
	closeXTest();
//	XCloseDisplay(this->display);
}

void	sendKey::sendChar(char *car)
{
	// first lookup the char
	KeySym	ks	= XStringToKeysym(car);
	KeyCode kc;
	if (ks>0)
	{
		kc	= XKeysymToKeycode(this->display, ks);
		if (kc<=0)
		{
			cerr << "KeyCode not found for " << car << " (" << (int)ks << ")" << endl;
			//fprintf(stderr, "KeyCode not found for %s (%d)\n", car, (int)ks);
			return;
		}
		if (this->listKeyCode[kc].normal == ks)
		{
			this->send(kc);
		}
		else if (this->listKeyCode[kc].shift == ks)
		{
			this->send(this->shift.kc, sendKey::keyDown);
			this->send(kc);
			this->send(this->shift.kc, sendKey::keyUp);
		}
		else if (this->listKeyCode[kc].altgr == ks)
		{
			this->send(this->altgr.kc, sendKey::keyDown);
			this->send(kc);
			this->send(this->altgr.kc, sendKey::keyUp);
		}
		else
		{
			cerr << "Cannot locate the modifier to get the char " << car << " with the keycode " << (int)kc << endl;
		}
	}
	else
	{
		// have to find the good one :
		if (!strcmp(car,"="))
			this->sendChar((char*)"equal");
		else if (!strcmp(car,""))
			this->sendChar((char*)"eacute");
		// TODO: finish this thing to match needs
		else
			cerr << "KeySym not found for char : " << car << endl;
	}
}
void	sendKey::send(KeyCode kc, keyDirection kd)
{
	if (this->display == NULL)
		return;

	if (this->useXTest)
	{
		switch (kd)
		{
			case keyDown:
				XTestFakeKeyEvent(this->display, (unsigned int) kc, TRUE, 0);
				break;
			case keyUp:
				XTestFakeKeyEvent(this->display, (unsigned int) kc, FALSE, 0);
				break;
			case keyDownUp:
			default:
				XTestFakeKeyEvent(this->display, (unsigned int) kc, TRUE, 0);
				XTestFakeKeyEvent(this->display, (unsigned int) kc, FALSE, 0);
				break;
		}
		XSync(this->display, false);
	}
	else
		cerr << "Cannot use XTest Extention ?" << endl;
}

sendKey::keyValue*	sendKey::getKeyValue(KeyCode kc)
{
	if (this->listKeyCode == NULL)
		return NULL;
	return	&(this->listKeyCode[(int)kc]);
}

char*	sendKey::keysymToString(KeySym ks)
{
	switch(ks)
	{
		case XK_Shift_L:	return "Shift";
		case XK_Shift_R:	return "Shift";
		case XK_Control_L:	return "Ctrl";
		case XK_Control_R:	return "Ctrl";
		case XK_Caps_Lock:	return "Caps\nLock";
		case XK_Alt_L:		return "Alt";
		case XK_Alt_R:		return "AltGr";
		case XK_Super_L:	return "Win";
		case XK_Super_R:	return "Win";
		case XK_Tab:		return "->|";
		case XK_BackSpace:	return "<-";
		case XK_KP_0:		return "0";
		case XK_KP_1:		return "1";
		case XK_KP_2:		return "2";
		case XK_KP_3:		return "3";
		case XK_KP_4:		return "4";
		case XK_KP_5:		return "5";
		case XK_KP_6:		return "6";
		case XK_KP_7:		return "7";
		case XK_KP_8:		return "8";
		case XK_KP_9:		return "9";
		case XK_Clear:		return "cls";
		case XK_Return:		return "<!";
		case XK_Pause:		return "Pause";
		case XK_Scroll_Lock:	return "Scroll\nLock";
		case XK_Escape:		return "Esc";
		case XK_Delete:		return "Del";
		case XK_Home:		return "Home";
		case XK_Left:		return "<";
		case XK_Up:		return "^";
		case XK_Right:		return ">";
		case XK_Down:		return "v";
		case XK_Num_Lock:	return "Num\nLock";
		case XK_KP_Equal:	return "=";
		case XK_KP_Multiply:	return "*";
		case XK_KP_Add:		return "+";
		case XK_KP_Separator:	return "|";
		case XK_KP_Subtract:	return "-";
		case XK_KP_Decimal:	return ".";
		case XK_KP_Divide:	return "/";
		case XK_dead_circumflex: return "^";
		case XK_dead_tilde:	return "~";
		case XK_space:		return "";
		case XK_exclam:		return "!";
		case XK_quotedbl:	return "\"";
		case XK_dollar:		return "$";
		case XK_percent:	return "%";
		case XK_ampersand:	return "&";
		case XK_apostrophe:	return "'";
		case XK_parenleft:	return "(";
		case XK_parenright:	return ")";
		case XK_asterisk:	return "*";
		case XK_plus:		return "+";
		case XK_comma:		return "-";
		case XK_minus:		return "-";
		case XK_slash:		return "/";
		case XK_bracketleft:	return "[";
		case XK_backslash:	return "\\";
		case XK_bracketright:	return "]";
		case XK_asciicircum: 	return "^";
		case XK_underscore:	return "_";
		case XK_agrave:		return "";
		case XK_eacute:		return "";
		case XK_egrave:		return "";
		case XK_grave:		return "`";
		case XK_braceleft:	return "{";
		case XK_bar:		return "|";
		case XK_braceright:	return "}";
		case XK_equal:		return "=";
		case XK_section:	return "";
		case XK_asciitilde:	return "~";
		case XK_degree:		return "";
		case XK_twosuperior:	return "";
		case XK_mu:		return "";
		case XK_paragraph:	return "";
		case XK_ccedilla:	return "";
		case XK_Sys_Req:	return "";
		case XK_notsign:	return "";
		case XK_onesuperior:	return "";
		case XK_numbersign:	return "#";
		case XK_at:		return "@";
		case XK_Terminate_Server:return "";
		case XK_Prior:		return "";
		case XK_dead_diaeresis:	return "";
		case XK_sterling:	return "";
		case XK_currency:	return "";
		case XK_KP_Delete:	return "Supp";
		case XK_End:		return "End";
		case XK_Next:		return ">>";
		case XK_ugrave:		return "";
		case XK_dead_grave:	return "`";
		case XK_less:		return "<";
		case XK_greater:	return ">";
		case XK_question:	return "?";
		case XK_dead_acute:	return "";
		case XK_semicolon:	return ",";
		case XK_period:		return "";
//		case XK_horizconnector:return "";
		case XK_colon:		return ":";
		case XK_KP_Enter:	return "Enter";
		case XK_ISO_Level3_Shift:	return "AltGr";
		case XK_Meta_L:		return "";
		case XK_Menu:		return "Menu";
		case XK_ISO_Left_Tab:	return "";
		case XK_periodcentered:	return "";
		case XK_dead_belowdot:	return "";
		case XK_yen:		return "";
		case XK_plusminus:	return "";
		case XK_onehalf:	return "?";
		case XK_Aring:		return "";
		case XK_aring:		return "";
		case XK_acute:		return "?";
		case XK_ae:		return "";
		case XK_adiaeresis:	return "";
		case XK_Adiaeresis:	return "";
		case XK_odiaeresis:	return "";
		case XK_Odiaeresis:	return "";
		case XK_Print:		return "Print\nScreen";
		case 269024800:		return ""; //XK_XF86_Ungrab
		case 269024801:		return ""; //XK_XF86_ClearGrab
		case 269024803:		return ""; //XK_XF86_Prev_VMode
		case 269024802:		return ""; //XK_XF86_Next_VMode
		case 2211:		return ""; //XK_horizconnector

		default:
			/*
			string s = XKeysymToString(ks);
			if (s.length() != 1)
				printf("case %d:return \"\"; //XK_%s\n", (int)ks, XKeysymToString(ks));
			else
				printf ("%s\n", s);
			*/
			return XKeysymToString(ks);
	}
}


bool sendKey::getModifierList()
{
	XModifierKeymap *modifiers;
	KeyCode *modifierMap;
	KeyCode kc;
	KeySym ks;
	int m_index;
	int m_key;

	if (this->display == NULL)
		return false;

	modifiers = XGetModifierMapping(this->display);
	modifierMap = modifiers->modifiermap;

	this->alt.found		= false;
	this->shift.found	= false;
	this->ctrl.found	= false;
	this->caps.found	= false;
	this->numlock.found	= false;
	this->altgr.found	= false;

	for (m_index = 0; m_index < 8; m_index++)
	{

		for (m_key = 0; m_key < modifiers->max_keypermod; m_key++)
		{
			kc = modifierMap[m_index * modifiers->max_keypermod + m_key]; 

			if (kc != 0)
			{
				ks = XKeycodeToKeysym(this->display, kc, 0);
				switch (ks)
				{

					case XK_Alt_R:
					case XK_Alt_L:
						this->alt.index 	= m_index;
						this->alt.ks		= ks;
						this->alt.kc		= kc;
						this->alt.found		= true;
						break;

					case XK_Shift_R:
					case XK_Shift_L:
						this->shift.index	= m_index;
						this->shift.ks		= ks;
						this->shift.kc		= kc;
						this->shift.found	= true;
						break;

					case XK_Control_R:
					case XK_Control_L:
						this->ctrl.index	= m_index;
						this->ctrl.ks		= ks;
						this->ctrl.kc		= kc;
						this->ctrl.found	= true;
						break;

					case XK_Caps_Lock:
						this->caps.index	= m_index;
						this->caps.ks		= ks;
						this->caps.kc		= kc;
						this->caps.found	= true;
						break;

					case XK_Num_Lock:
						this->numlock.index	= m_index;
						this->numlock.ks	= ks;
						this->numlock.kc	= kc;
						this->numlock.found	= true;
						break;

					case XK_ISO_Level3_Shift:
						this->altgr.index	= m_index;
						this->altgr.ks		= ks;
						this->altgr.kc		= kc;
						this->altgr.found	= true;
						break;

				}

				break;
			}
		}
	}

	if (!this->altgr.found)
	{
		this->altgr.ks		= XKeycodeToKeysym(this->display,113,0);
		this->altgr.kc		= (KeyCode)113;
		this->altgr.found	= true;
	}
	XFreeModifiermap(modifiers);
	return true;
}

bool sendKey::getKeyList()
{
	int minKeycode, maxKeycode, keysymsPerKeycode;
	KeySym *keymap;

	if (this->display == NULL)
		return false;

	XDisplayKeycodes(this->display, &minKeycode, &maxKeycode);
	keymap = XGetKeyboardMapping(this->display, minKeycode, 
				     (maxKeycode - minKeycode + 1), 
				     &keysymsPerKeycode);

	if (this->listKeyCode != NULL)
		Xfree(this->listKeyCode);
	this->listKeyCode = (keyValue*)Xmalloc((maxKeycode+1) * sizeof(keyValue));
	for(int i=0;i < (maxKeycode - minKeycode + 1); i++)
	{
		this->listKeyCode[i+minKeycode].normal = keymap[(i*keysymsPerKeycode)];
		if (this->shift.found)
			this->listKeyCode[i+minKeycode].shift = keymap[((i*keysymsPerKeycode) + 1)];
		else
			this->listKeyCode[i+minKeycode].shift = 0;
		if (this->altgr.found)
			this->listKeyCode[i+minKeycode].altgr = keymap[((i*keysymsPerKeycode) + 2)];
		else
			this->listKeyCode[i+minKeycode].altgr = 0;
	}

	return true;
}

bool sendKey::initXTest()
{
	int event, error;
	int major, minor;

	if (this->display == NULL)
		return false;
	else
	{
		// does the display have the Xtest-extension?

		if (!XTestQueryExtension(this->display, &event, &error, &major, &minor))
		{
        		// nope, extension not supported

			cerr << "XTest extension not supported on server \"" << DisplayString(this->display) << "\"" << endl;
			//fprintf(stderr, "XTest extension not supported on server \"%s\"\n", DisplayString(this->display));

			return false;
		}

		// sync the server
//		XSync(this->display, FALSE);

		return true;
	}
}

void sendKey::closeXTest()
{
	if (this->display == NULL || !this->useXTest)
		return;

	// discard and even flush all events on the remote display

	XTestDiscard(this->display);

	XFlush(this->display);
	this->useXTest	= false;
}
