/***************************************************************************
                          chatview.h  -  description
                             -------------------
    begin                : Wed Jan 15 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CHATVIEW_H
#define CHATVIEW_H

#include <qpoint.h>
#include <qstringlist.h>
#include <qtimer.h>
#include <qwidget.h>

#include <kconfig.h>
#include <khtml_part.h>
#include <kurl.h>

#include "chatmessage.h"
#include "chatviewinterface.h"

// Forward declarations
class CurrentAccount;
class KMessTest;
class ChatMessageStyle;
class ChatMessageView;


/**
 * The view widget of a chat with a contact with message display and entry.
 * @author Mike K. Bennett
 */
class ChatView : public ChatViewInterface
{
  Q_OBJECT

  friend class KMessTest;

  public:
    // The constructor
                          ChatView(QWidget *parent=0, const char *name=0);
    // The destructor
                         ~ChatView();
    // Copy the currently selected text
    void                  editCopy();
    // Initialize the object
    bool                  initialize();
    // Return the encoding of the HTML document.
    QString               getEncoding() const;
    // Return the HTML source of the page.
    QString               getHtml() const;
    // Return the selected text in the message area.
    QString               getSelectedText(bool asHtml) const;
    // Whether or not the message area is empty
    bool                  isEmpty() const;
    // Update the messages which contain custom emoticons
    void                  updateCustomEmoticon( const QString &handle, const QString &code );
    // Change the text zoom factor
    void                  changeZoomFactor( bool increase );
    // Restore the window properties (called by the parent ChatWindow)
    void                  readProperties( KConfig *config );
    // Save the window properties (called by the parent ChatWindow)
    void                  saveProperties( KConfig *config );

  public slots:
    // Insert an emoticon into the message editor
    void                  insertEmoticon( QString emoticonText );
    // Scroll to the bottom of the chat browser
    void                  scrollChatToBottom();
    // Add the given message to the message browser.
    void                  showMessage( const ChatMessage &message );
    // Show a dialog to save the chat.
    void                  showSaveChatDialog();
    // Save the chat to the given file
    void                  saveChatToFile(const QString &path );

  private: // Private methods
    // Delete the newline behind the message edit's cursor.
    void                  deleteNewlineAtCursor();
    // Event filter to detect special actions in the message editor.
    bool                  eventFilter( QObject *obj, QEvent *event );
    // Initialize the current account
    bool                  initializeCurrentAccount();
    // Send a message via the server
    void                  sendMessage();

  private slots: // Private slots
    // The user pressed return in the message editor, so send the message
    void                  enterPressed();
    // The message text changed, so the user is typing
    void                  messageTextChanged();
    // The user clicked the new line button so insert a new line in the editor
    void                  newLineClicked();
    // The user clicked send, so send the message
    void                  sendClicked();
    // Open a dialog to add a new custom emoticon seen in the chat
    void                  slotAddNewEmoticon();
    // The user clicked a kmess internal link in the ChatMessageView
    void                  slotSendAppCommand();
    // The user clicked the "copy address" or "copy email" option in the context menu
    void                  slotCopyAddress();
    // The user clicked the "copy text" option in the context menu.
    void                  slotCopyChatText();
    // Open a new url clicked in the khtml widget
    void                 slotOpenURLRequest(const KURL &url, const KParts::URLArgs &args);
    // The user clicked the "select all" option in the context menu.
    void                  slotSelectAllChatText();
    // The user right clicked at the KHTMLPart to show a popup.
    void                  slotShowContextMenu(const QString &url, const QPoint &point);
    // The user clicked the "visit address" or "send email" option in the context menu, or clicked a link in the ChatMessageView
    void                  slotVisitAddress();
    // Update the editor's font to match the account's font
    void                  updateEditorFont();

  private: // Private attributes
    QPtrList<ChatMessage> chatMessages_;
    // The embedded khtml part
    ChatMessageView      *chatMessageView_;
    // The XSL transformation handler
    ChatMessageStyle     *chatStyle_;
    // URL address which has been right-clicked
    KURL                  chatViewClickedUrl_;
    // A pointer to the instance of the current account
    CurrentAccount       *currentAccount_;
    // Stores the current level of zoom
    int                   currentZoom_;
    // Whether or not typing messages should be sent (true except the very moment a text message is sent)
    bool                  doSendTypingMessages_;
    // Whether or not the object was initialized
    bool                  initialized_;
    // The last messages sent by the same contact. This is used to combine them
    QPtrList<ChatMessage> lastContactMessages_;
    // The message bar's height
    int                   messageBarHeight_;
    // The messages that contain custom emoticons
    QStringList           pendingEmoticonTags_;
    // The sidebar's width
    int                   sidebarWidth_;
    // A timer to time whether a "user is typing" message should be sent.
    QTimer                userTypingTimer_;

  private slots:
    // The color in the text box changed.
    void                  editorColorChanged(const QColor &color);
    // Update the chat style
    void                  updateChatStyle();

  signals: // Public signals
    // Signal that the user wants to add a new custom emoticon copying it from a friend's
    void                 addEmoticon(QString handle, QString shortcut);
    // Signal that there's an application command
    void                  appCommand(QString cookie, QString contact, QString method);
    // Signal that the given text should be sent as a message
    void                  sendMessageToContact(QString text);
    // Signal that the user wants to transfer some files
    void                  sendFiles( QStringList fileList );
    // Signal that the user is typing
    void                  userIsTyping();
};

#endif
