/***************************************************************************
                          contact.cpp  -  description
                             -------------------
    begin                : Sun Jan 5 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "contact.h"

#include <qdir.h>
#include <qpixmap.h>
#include <qstringlist.h>

#include <kdebug.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kstddirs.h>

#include "contactextension.h"
#include "../currentaccount.h"
#include "../kmessdebug.h"
#include "../msnobject.h"
#include "../specialgroups.h"

#ifdef KMESSDEBUG_CONTACT
  #define KMESSDEBUG_CONTACT_GENERAL
#endif



// The constructor
Contact::Contact(QString handle, QString friendlyName, int lists, QString groupIds, QString guid)
 : ContactBase(handle, friendlyName),
   allowed_(false),
   blocked_(false),
   friend_(false),
   guid_(guid),
   lastStatus_("FLN"),
   movingFrom_(""),
   msnObject_(0),
   pending_(false),
   reverse_(false),
   status_("FLN")
{
#ifdef KMESSTEST
  ASSERT( ! handle_.isEmpty() );
  ASSERT( ! friendlyName_.isEmpty() );
  ASSERT( lists != 0 );
  ASSERT( groupIds.isEmpty() || groupIds.length() > 8 );
#endif

  if(lists & MSN_LIST_FRIEND)  friend_  = true;
  if(lists & MSN_LIST_ALLOWED) allowed_ = true;
  if(lists & MSN_LIST_BLOCKED) blocked_ = true;
  if(lists & MSN_LIST_REVERSE) reverse_ = true;
  if(lists & MSN_LIST_PENDING) pending_ = true;

  if( groupIds.length() > 8 )  // make sure it's not an int or some other invalid parameter (should be a GUID)
  {
    groupIds_  = QStringList::split(",", groupIds);
  }

  // The extension stores additional properties not found in
  // the official client/protocol, but only found in KMess.
  // Pictures are stored there because you can assign one
  // (KMess implemented this before the official client supported
  // picture transfers). Names can be aliased, which is why there're
  // also there.
  extension_ = new ContactExtension( handle );

  connect ( extension_, SIGNAL(        changedFriendlyName() ),
            this,         SLOT( forwardChangedFriendlyName() ) );
  connect ( extension_, SIGNAL(             changedPicture() ),
            this,         SLOT(      forwardChangedPicture() ) );
}



// The destructor
Contact::~Contact()
{
#ifdef KMESSDEBUG_CONTACT_GENERAL
  kdDebug() << "DESTROYED Contact [handle=" << handle_ << "]" << endl;
#endif

  delete extension_;
  if(msnObject_) delete msnObject_;
}



// Add the groupID to the list of current groups
void Contact::addGroupId(const QString &groupId)
{
  // Check whether contact is already added to the group.
  if( groupIds_.contains(groupId) )
  {
    kdWarning() << "Contact::addGroupId: " << handle_ << " is already a member of group " << groupId << "." << endl;
    return;
  }
  // Protect against invalid use, causes problems in KMessView.
  if( groupId.isEmpty() )
  {
    kdWarning() << "Contact::addGroupId: Group-ID is empty (contact=" << getHandle() << ")!" << endl;
    return;
  }

  // Append to groups
  groupIds_.append(groupId);

#ifdef KMESSDEBUG_CONTACT_GENERAL
  kdDebug() << "Contact::addGroupId: groups of " << handle_ << " are '" << groupIds_.join(",") << "'" << endl;
#endif

  emit changedGroup(this);
}


// Check if this is a new contact
bool Contact::checkIfContactAddedUser()
{
#ifdef KMESSDEBUG_CONTACT_GENERAL
  kdDebug() << "Contact " << handle_ << " - Check if new." << endl;
#endif

  // Check if the contact is a new contact
  return ( reverse_ && ! (friend_ || blocked_ || allowed_ ) );
}



// Slot to pass on the changedFriendlyName()
void Contact::forwardChangedFriendlyName()
{
  emit changedFriendlyName();
}



// Forward the "changed picture" signal
void Contact::forwardChangedPicture()
{
  emit changedPicture();
}



// Return the path to the contact's picture
QString Contact::getContactPicturePath() const
{
  QString picturePath = extension_->getContactPicturePath();

  if( picturePath.isEmpty() )
  {
    return getContactDefaultPicturePath();
  }
  else
  {
    return picturePath;
  }
}



// Return the current media type
const QString & Contact::getCurrentMediaType() const
{
  return currentMediaType_;
}



// Return the current media type
const QString & Contact::getCurrentMediaString() const
{
  return currentMediaString_;
}



// Return a pointer to the extension class
ContactExtension *Contact::getExtension() const
{
  return extension_;
}


// Return the contact's friendly name
QString Contact::getFriendlyName() const
{
  if ( extension_->getUseAlternativeName() )
  {
    return extension_->getAlternativeName();
  }
  else
  {
    return friendlyName_;
  }
}



// Return the contact's previous status
const QString &Contact::getLastStatus() const
{
  return lastStatus_;
}



// Return the contact's globally unique identifier
const QString &Contact::getGuid() const
{
  return guid_;
}



// Return the group ID's the contact is added to
QStringList Contact::getGroupIds() const
{
  return groupIds_;
}


// Return the MSNObject
const MsnObject* Contact::getMsnObject() const
{
  return msnObject_;
}



// Return the contact's personal message
const QString & Contact::getPersonalMessage() const
{
  return personalMessage_;
}



// Get the source group for an imminent move operation
const QString & Contact::getPrepareMove() const
{
  return movingFrom_;
}



// Return the contact's status
QString Contact::getStatus() const
{
  return status_;
}



// Return the contact's true friendly name, regardless of extension
QString Contact::getTrueFriendlyName() const
{
  return friendlyName_;
}



// Whether or not the contact is allowed
bool Contact::isAllowed() const
{
  return allowed_;
}



// Whether or not the contact is blocked
bool Contact::isBlocked() const
{
  return blocked_;
}



// Whether or not the contact is on the friends list
bool Contact::isFriend() const
{
  return friend_;
}



// Whether or not the contact is on the pending list
bool Contact::isPending() const
{
  return pending_;
}



// Whether or not the contact is on the reverse list
bool Contact::isReverse() const
{
  return reverse_;
}



// Setup and load an MSN Object
void Contact::loadMsnObject( QString msnObject )
{
  if( ! msnObject )
  {
    return;
  }

  // Contact removed his picture.
  if( msnObject.isEmpty() )
  {
#ifdef KMESSDEBUG_CONTACT_GENERAL
    kdDebug() << "Contact " << getHandle() << " - MSN Object Removed!" << endl;
#endif

    delete msnObject_;
    msnObject_ = 0;
    emit changedMsnObject( this );
  }

  // Contact set his picture, or changed it.
  if( msnObject_ )
  {
    // Actually not changed...
    if( ! msnObject_->hasChanged( msnObject ) )
    {
      return;
    }

    // Delete previous object.
    delete msnObject_;
    msnObject_= 0;
  }

#ifdef KMESSDEBUG_CONTACT_GENERAL
  kdDebug() << "Contact " << getHandle() << " - MSN Object Changed!" << endl;
#endif

  msnObject_ = new MsnObject( msnObject );
  emit changedMsnObject( this );
}



// Remove the group from the list of group IDs
void Contact::removeGroupId(const QString &groupId)
{
  int removeCount = groupIds_.remove(groupId);
  if(removeCount == 0)
  {
    kdWarning() << "Contact::addGroupId: " << handle_ << " was not registered to group " << groupId << "." << endl;
  }

#ifdef KMESSDEBUG_CONTACT_GENERAL
  kdDebug() << "Contact::removeGroupId: groups of " << handle_ << " are '" << groupIds_.join(",") << "'" << endl;
#endif

  emit changedGroup(this);
}



// Set whether or not the contact is allowed
void Contact::setAllowed(bool allowed)
{
  if( allowed_ != allowed )
  {
    allowed_ = allowed;
    emit changedList(this);
  }
}



// Set whether or not the contact is blocked
void Contact::setBlocked(bool blocked)
{
  if( blocked_ != blocked )
  {
    blocked_ = blocked;
    emit changedList(this);
  }
}



// Set whether or not the contact is on the friends list
void Contact::setFriend(bool isFriend)
{
  if( friend_ != isFriend )
  {
    friend_ = isFriend;

    // If the contact is no longer at the FL, remove from all groups
    if(! isFriend)
    {
      groupIds_.clear();
      setStatus("FLN", false);
    }

    emit changedList(this);
  }
}



// Change the contact's friendly name
void Contact::setFriendlyName(QString newName)
{
  if( ! newName.isEmpty() )
  {
    friendlyName_ = newName;
//    updateItemIdentifierAndPixmap();
    emit changedFriendlyName();
  }
}



// Change the GUID of the contact, this is only known when the contact is added to the friends list.
void Contact::setGuid(const QString &guid)
{
  guid_ = guid;
}



// Set whether or not the user is on a given list
void Contact::setList(const QString list, bool isMember)
{
#ifdef KMESSTEST
  ASSERT( ( list == "FL" ) || ( list == "AL" ) || ( list == "BL" ) || ( list == "RL" ) );
#endif
#ifdef KMESSDEBUG_CONTACT_GENERAL
  kdDebug() << "Contact::setList: old =" << (allowed_ ? " AL" : "")
                                         << (blocked_ ? " BL" : "")
                                         << (friend_  ? " FL" : "")
                                         << (reverse_ ? " RL" : "") << endl;
#endif

  if ( list == "FL" )
  {
    setFriend( isMember );
  }
  else if ( list == "AL" )
  {
    setAllowed( isMember );
  }
  else if ( list == "BL" )
  {
    setBlocked( isMember );
  }
  else if ( list == "RL" )
  {
    setReverse( isMember );
  }

#ifdef KMESSDEBUG_CONTACT_GENERAL
  kdDebug() << "Contact::setList: new =" << (allowed_ ? " AL" : "")
                                         << (blocked_ ? " BL" : "")
                                         << (friend_  ? " FL" : "")
                                         << (reverse_ ? " RL" : "") << endl;
#endif
}



// Set the personal message of the contact
void Contact::setPersonalStatus( const QString &message, const QString &mediaType, const QString &mediaString )
{
  if( personalMessage_    != message
  ||  currentMediaType_   != mediaType
  ||  currentMediaString_ != mediaString )
  {
    personalMessage_    = message;
    currentMediaType_   = mediaType;
    currentMediaString_ = mediaString;
    emit changedPersonalMessage(this);
  }
}



// Prepare moving the contact to another group from the specified one
void Contact::setPrepareMove( QString from )
{
  // It's the MSN Notification Server the one who actually performs the 
  // move, not us. This method only allows to understand if a received
  // "add contact" command is subsequent to a move or not.
  if( ! from.isEmpty() )
  {
    movingFrom_ = from;
  }
  else
  {
    movingFrom_ = QString::null;
  }
  
}



// Set whether or not the contact is on the reverse list
void Contact::setReverse(bool reverse)
{
  if( reverse_ != reverse )
  {
    reverse_ = reverse;
    emit changedList(this);
  }
}



// Set the contact's status
void Contact::setStatus(const QString status, bool showBaloon)
{
  bool isValidStatus = ( status == "AWY" ) ||
                       ( status == "BRB" ) ||
                       ( status == "BSY" ) ||
                       ( status == "FLN" ) ||
                       ( status == "IDL" ) ||
                       ( status == "LUN" ) ||
                       ( status == "NLN" ) ||
                       ( status == "PHN" );

#ifdef KMESSTEST
  ASSERT( isValidStatus );
#endif

  if(isValidStatus)
  {
    lastStatus_ = status_;
    status_ = status;

#ifdef KMESSDEBUG_CONTACT_GENERAL
    kdDebug() << "Contact " << handle_ << " went from " << lastStatus_ << " to " << status_ << "." << endl;
#endif

    // Signal the UI
    emit changedStatus();

    // Check if the contact went offline
    if( status_ == "FLN" && lastStatus_ != "FLN" )
    {
#ifdef KMESSDEBUG_CONTACT_GENERAL
      kdDebug() << "Contact went offline." << endl;
#endif
      emit contactOffline( this, showBaloon );
    }
    // Check if the contact went online
    else if( status_ != "FLN" && lastStatus_ == "FLN" )
    {
      // showBaloon is here to avoid showing notification when we connect
#ifdef KMESSDEBUG_CONTACT_GENERAL
      kdDebug() << "Contact went online." << endl;
#endif
      emit contactOnline( this, showBaloon );
    }
  }
}

#include "contact.moc"
