/***************************************************************************
                          msnswitchboardconnection.h  -  description
                             -------------------
    begin                : Fri Jan 24 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef MSNSWITCHBOARDCONNECTION_H
#define MSNSWITCHBOARDCONNECTION_H

#include <qfont.h>
#include <qstringlist.h>
#include <qvaluelist.h>
#include <qstring.h>
#include <qcstring.h>
#include <qpair.h>
#include <qtimer.h>

#include "msnconnection.h"
#include "mimemessage.h"
#include "../chat/chatmaster.h"


// Forward declarations
class VoiceConversation;
class ChatInformation;
class KMessTest;
class MsnObject;
class P2PMessage;
class ContactBase;
class ChatMessage;


/**
 * A connection to a switchboard server, over which chats are conducted.
 *
 * @author Mike K. Bennett
 * @ingroup NetworkCore
 */
class MsnSwitchboardConnection : public MsnConnection
{
  Q_OBJECT

  friend class KMessTest;

  public:
    // The supported applications
    enum ApplicationType { GNOMEMEETING, FILETRANSFER, MSNREMOTEDESKTOP, VOICECONVERSATION };
    // The constructor
                         MsnSwitchboardConnection();
    // The destructor
                        ~MsnSwitchboardConnection();
    // Close the connection, only for emergaincy situations
    virtual void         closeConnection();
    // Clean up, close the connection, destroy this object
    void                 closeConnectionLater(bool autoDelete = false);
    // Make a list of the contacts in the chat
    QStringList          getContactsInChat() const;
    // Return the first contact the chat started with.
    const QString&       getFirstContact() const;
    // Return the last contact who left the chat.
    const QString&       getLastContact() const;
    // Return whether the user started the chat
    bool                 getUserStartedChat() const;
    // Initialize the object
    bool                 initialize();
    // Invite a contact into the chat
    void                 inviteContact(QString handle);
    // Check whether the switchboard is buzy (has too many pending messages)
    bool                 isBusy() const;
    // Check if a certain contact is in the chat
    bool                 isContactInChat(const QString& handle) const;
    // Check if all contacts left
    bool                 isEmpty() const;
    // Check if only the given contact is in the chat
    bool                 isExclusiveChatWithContact(const QString& handle) const;
    // Check whether the switchboard is currently not connected, nor trying to connect
    bool                 isInactive() const;
    // Start a switchboard connection
    void                 connectTo( const ChatInformation &chatInfo );
    // Send a nudge to a contact
    void                 sendNudge();
    // Deliver a message for an Application object.
    void                 sendApplicationMessage( const MimeMessage &message );

  public slots: // Public slots
    // Send a message to the contact(s)
    void                 sendChatMessage(QString text);
    // Send a "ping" to avoid MSN closing the connection
    void                 sendKeepAlive();
    // The user is typing so send a typing message
    void                 sendTypingMessage();

  protected slots: // Protected slots
    // The socket connected, so send the version command.
    void                 connectionSuccess();

  private slots:
    // An ApplicationList object indicated it aborted all it's applications.
    void                 slotApplicationsAborted(const QString &contact);
    // The connection has gone inactive, close it
    void                 timeoutConnection();

  private:  // private structs
    struct UnAckedMessage
    {
      uint        time;
      AckType     ackType;
      MimeMessage message;
    };

  private: // Private methods
    // Initialize or restart the last activity timer, for keep alive messages
    void                 activity();
    // Clean the old unacked messages
    void                 cleanUnackedMessages();
    // Do what's required when a contact joins the chat
    void                 contactJoined(const QString& handle, const QString& friendlyName, const uint capabilities);
    // Remove a contact from the list of contacts in the chat
    void                 contactLeft(const QString& handle);
    // Convert an html format (#RRGGBB) color to an msn format (BBGGRR) color
    void                 convertHtmlColorToMsnColor(QString &color) const;
    // Convert and msn format color (BBGGRR) to an html format (#RRGGBB) color
    void                 convertMsnColorToHtmlColor(QString &color) const;
    // Get a font from the message
    void                 getFontFromMessageFormat(QFont &font, const MimeMessage& message) const;
    // Get an html font color from the message
    void                 getFontColorFromMessageFormat(QString &color, const MimeMessage& message) const;
    // Received a positive delivery message.
    void                 gotAck(const QStringList& command);
    // Received notification that a contact is no longer in session.
    void                 gotBye(const QStringList& command);
    // Received the initial roster information for new contacts joining a session.
    void                 gotIro(const QStringList& command);
    // Received notification of a new client in the session.
    void                 gotJoi(const QStringList& command);
    // Received a negative acknowledgement of the receipt of a message.
    void                 gotNak(const QStringList& command);
    // Received notification of the termination of a client-server session.
    void                 gotOut(const QStringList& command);
    // Received a client-server authentication message.
    void                 gotUsr(const QStringList& command);
    // Parse a regular command
    void                 parseCommand(const QStringList& command);
    // Parse a datacast message (e.g. nudge or voice clip)
    void                 parseDatacastMessage(const QString &contactHandle, const MimeMessage &message);
    // Parse an emoticon message
    void                 parseEmoticonMessage(const QString &contactHandle, const QString &messageBody);
    // Parse a message command
    void                 parseMessage(const QStringList& command, const MimeMessage &message);
    // Parse a payload command
    virtual void         parsePayloadMessage(const QStringList &command, const QByteArray &payload);
    // Send a client caps message to the contacts
    void                 sendClientCaps();
    // Send a message to the contact(s), or leave it pending until a connection is restored
    void                 sendMimeMessageWhenReady(AckType ackType, const MimeMessage &message);
    // Send messages that weren't sent because a contact had to be re-called
    void                 sendPendingMessages();
    // Store a message for later acknowledgement
    void                 storeMessageForAcknowledgement(int ack, AckType ackType, const MimeMessage& message);

  private: // Private attributes
    // Whether this object is aborting all it's applications
    bool                 abortingApplications_;
    // The number of ACK messages pending
    int                  acksPending_;
    // The authorization used to connect to the switchboard server
    QString              authorization_;
    // Whether this object should be deleted after finishing closeConnectionLater()
    bool                 autoDeleteLater_;
    // The id of the chat, used for authenticating a contact-started chat
    QString              chatId_;
    // Whether this switchboard is connected to a chatwindow or it's a background connection.
    bool                 backgroundConnection_;
    // Whether this object is closing it's connection using closeConnectionLater()
    bool                 closingConnection_;
    // Current status of connection to the switchboard server
    enum ConnectionState
    {
      SB_DISCONNECTED,
      SB_CONNECTING,
      SB_AUTHORIZING,
      SB_INVITING_CONTACTS,
      SB_CHAT_STARTED,
      SB_CONTACTS_LEFT,
      SB_REQUESTING_CHAT
    } connectionState_;
    // A list of the contacts in the chat
    QStringList          contactsInChat_;
    // The last contact that was participating to the chat.
    QString              firstContact_;
    // Whether or not the object has been initialized
    bool                 initialized_;
    // The last contact that last the chat.
    QString              lastContact_;
    // A list of contacts that have been invited when the switchboard wasn't ready yet
    QStringList          pendingInvitations_;
    // A list of custom emoticon which have already been sent to the contacts
    // NOTE: Behavior changed since WLM 8+, emoticon data must be sent every time.
//     QStringList          sentEmoticons_;
    // A list of messages that should be sent to the contact when connection is reestablished
    QPtrList< QPair<AckType,MimeMessage> > pendingMessages_;
    // Queue of sent messages waiting ACK or NAK
    QMap< int, UnAckedMessage > unAckedMessages_;
    // Whether the user (true) or the contact (false) started the chat
    bool                 userStartedChat_;
    // Timer to keep track of the last activity, to manage keep alive messages.
    QTimer              *keepAliveTimer_;
    // Timer to keep track of last activities, to manage keep alive messages.
    int                  keepAlivesRemaining_;

  signals: // Public signals
    // Signal that a contact joined (or was already in) the chat
    void                 contactJoinedChat( QString handle, QString friendlyName );
    // Signal that a contact left the chat
    void                 contactLeftChat( QString handle, bool isChatIdle );
    // Signal that a contact is typing
    void                 contactTyping( QString handle, QString friendlyName );
    // Signal a chat message
    void                 chatMessage( const ChatMessage &message );
    // The switchboard received a Mime message, requesting processing by ChatMaster
    void                 gotMessage( const MimeMessage &message, const QString &handle );
    // The switchboard received a P2P message, requesting processing by ChatMaster
    void                 gotMessage( const P2PMessage &message, const QString &handle );
    // The switchboard received a msn object (wink, voice clip)
    void                 gotMsnObject( const QString &msnObjectData, const QString &handle );
    // The switchboard received a nudge (buzzer)
    void                 receivedNudge( const QString &handle );
    // The switchboard is ready to send more application messages
    void                 readySend();
    // Request the Chat Master to link this switchboard to a chat window
    void                 requestChatWindow( MsnSwitchboardConnection *switchboard );
    // Signal that this class needs a new chat connection
    void                 requestNewSwitchboard( QString contactHandle );
    // Signal that the delivery of a message has failed
    void                 sendingFailed( QString handle );
    // The switchboard is closing, remove it from the Chat Master references
    void                 deleteMe( MsnSwitchboardConnection *object );
};

#endif
