/***************************************************************************
                          tcpconnectionbase.h -  description
                             -------------------
    begin                : Tue Sep 03 2006
    copyright            : (C) 2005 by Diederik van der Boor
    email                : vdboor --at-- codingdomain.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef TCPCONNECTIONBASE_H
#define TCPCONNECTIONBASE_H

#include "kmessbuffer.h"

#include <qobject.h>
#include <qtimer.h>
#include <qmutex.h>
#include <qbuffer.h>

class KExtendedSocket;
class KSSL;

/**
 * @brief A base for socket handling.
 *
 * This class implements a base for connection handling.
 *
 * It implements the following features:
 * - buffering
 * - async-connect
 * - SSL support
 * - timeout detection
 * - signal handling
 * 
 * Internally, KExtendedSocket is used for compatibility with older KDE versions.
 *
 * The following methods are especially useful for derived classes:
 * - openConnection()
 * - closeConnection()
 * - getAvailableBytes()
 * - readBlock()
 * - peekBlock()
 * - writeBlock()
 *
 * Derived classes should overwrite the following methods to implement the event handling:
 * - slotConnectionEstablished()
 * - slotConnectionFailed()
 * - slotDataReceived()
 * - closeConnection()
 * - closeSockets()
 *
 * @author Diederik van der Boor
 * @ingroup NetworkCore
 */
class TcpConnectionBase : public QObject
{
  Q_OBJECT

  public:
    // The constructor
                           TcpConnectionBase(QObject *parent = 0, const char *name = 0);
    // The destructor
    virtual               ~TcpConnectionBase();

    // Close the connection
    virtual void           closeConnection();
    // Return true if a connection is active
    bool                   isConnected() const;
    // Get the remote ip the socket is connected with.
    QString                getRemoteIp() const;
    // Get the remote port the socket is connected with.
    QString                getRemotePort() const;
    // Return the error description
    QString                getSocketError() const;
    // Connect to a host
    bool                   openConnection(const QString &host, int port, bool ssl = false, bool async = true);

  protected slots:
    /**
     * @brief Called when the connection is established
     */
    virtual void           slotConnectionEstablished() = 0;

    // Called when the connection could not be made.
    virtual void           slotConnectionFailed();

    /**
     * @brief Called when data is received from the socket.
     *
     * Derived classes can use getBytesAvailable(), peekBlock(), readBlock() and readAll()
     * to read and process the incoming data.
     * Note the method should read all socket data unless it's known more data will arrive.
     * Otherwise, the application is ignoring incoming data too long.
     */
    virtual void           slotDataReceived() = 0;

  private slots:
    // A timeout occured to connect a socket
    void                   slotConnectionTimeout();
    // The connection was established
    void                   slotSocketConnected();
    // The connection could not be made.
    void                   slotSocketFailed(int error);
    // The socket can read new data
    void                   slotSocketReadyRead();
    // The socket can write new data
    void                   slotSocketReadyWrite();

  protected: // Protected methods
    // Close the sockets
    virtual bool           closeSockets();
    // Return the number of bytes which are already received in the local buffer
    int                    getAvailableBytes() const;
    // Return the connect timeout
    int                    getConnectTimeout() const;
    // Peek data from the socket
    int                    peekBlock( char *buffer, const int size );
    // Peek data from the socket (uses the QByteArray size() as block size)
    int                    peekBlock( QByteArray *buffer, const int maxSize = 0, const int offset = 0 );
    // Return everything that's in the read buffer.
    QByteArray             readAll();
    // Read data from the socket
    int                    readBlock( char *buffer, const int size );
    // Read data from the socket (uses the QByteArray size() as block size)
    int                    readBlock( QByteArray *buffer, const int maxSize = 0, const int offset = 0 );
    // Set the connect timeout
    void                   setConnectTimeout( int seconds );
    // Stop the timeout detection
    void                   stopTimeoutDetection();
    // Write data to the socket
    bool                   writeBlock( const char *block, const int size );
    // Write data to the socket
    bool                   writeBlock( const QByteArray &block );

  private:  // Private methods
    // Convert a QByteArray to data/size fields.
    bool                   parseBuffer( QByteArray *buffer, const int maxSize, const int bufferOffset, char **bufferStart, int *blockSize );

  private: // Private attributes
    /// The location openConnection() is connecting to, for debugging
    QString                connectingTo_;
    /// The connect timeout
    int                    connectTimeout_;
    /// The timeout handling for openConnection()
    QTimer                 connectionTimer_;
    /// The buffer for read data
    KMessBuffer            readBuffer_;
    /// The socket over which data is sent or received.
    KExtendedSocket       *socket_;
    /// The SSL presentation wrapper
    KSSL                  *ssl_;
    /// Whether an timeout occured
    bool                   timeout_;
};

#endif
