/***************************************************************************
                          notificationcontactstatus.cpp - notifies when a
                            contact changes their msn status
                             -------------------
    begin                : Saturday July 28 2007
    copyright            : (C) 2007 by Valerio Pilo
    email                : amroth@coldshock.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "notificationcontactstatus.h"

#include <qpixmap.h>
#include <qfile.h>
#include <qstylesheet.h>

#include <kdebug.h>
#include <klocale.h>

#include <kaudioplayer.h>
#include <knotifyclient.h>

#include "../kmessdebug.h"



// Class constructor
NotificationContactStatus::NotificationContactStatus( NotificationManager* manager )
  : enableNotifyOnline_(true)
  , enableNotifyOffline_(false)
  , enableNotifyStatus_(false)
  , manager_(manager)
{
}



// Trigger this event's action
void NotificationContactStatus::activate( QStringList stringList, QObject */*object*/ )
{
#ifdef KMESSDEBUG_NOTIFICATIONCONTACTSTATUS
  kdDebug() << "NotificationContactStatus::activate() - Starting chat." << endl;
#endif

  if( stringList[1] == "FLN" )
  {
    return;
  }

  emit startChat( stringList[0], ChatInformation::CONNECTION_CHAT );
}



// Obtains a status string like "contact has come online" for a MSN status code
QString NotificationContactStatus::getStatusString( QString status )
{
  if     ( status == "AWY" ) return i18n( "<b>%1</b> has gone away" );
  else if( status == "BRB" ) return i18n( "<b>%1</b> will be right back" );
  else if( status == "BSY" ) return i18n( "<b>%1</b> is now busy" );
  else if( status == "FLN" ) return i18n( "<b>%1</b> has logged out" );
  else if( status == "HDN" ) return i18n( "<b>%1</b> has become invisible" ); // Is this one even possible?!
  else if( status == "IDL" ) return i18n( "<b>%1</b> has gone idle" );
  else if( status == "LUN" ) return i18n( "<b>%1</b> is out for lunch" );
  else if( status == "PHN" ) return i18n( "<b>%1</b> is on the phone" );
  else                       return i18n( "<b>%1</b> is now online" );
}



// Notify the user about this event (a contact has changed its status)
void NotificationContactStatus::notify( Contact *contact, bool isNeeded )
{
  if( contact == 0 )
  {
    kdWarning() << "NotificationContactStatus::notify() - Contact is null!" << endl;
    return;
  }

  if( ! KMESS_NULL(contact->getExtension()) && ! contact->getExtension()->getNotifyOnline() )
  {
#ifdef KMESSDEBUG_NOTIFICATIONCONTACTSTATUS
    kdDebug() << "NotificationContactStatus::notify() - Skipping, notification disabled for " << contact->getHandle() << "." << endl;
#endif
    return;
  }

  // If notification for this event is disabled, don't show a balloon.
  if( ! isNeeded )
  {
#ifdef KMESSDEBUG_NOTIFICATIONCONTACTSTATUS
    kdDebug() << "NotificationContactStatus::notify() - Not needed, skipping." << endl;
#endif
    return;
  }


  QString text;
  QString status = contact->getStatus();
  QString lastStatus = contact->getLastStatus();

#ifdef KMESSDEBUG_NOTIFICATIONCONTACTSTATUS
  kdDebug() << "NotificationContactStatus::notify() - Contact: " << contact->getHandle() << ", status: " << lastStatus << "->" << status << " - "
            << "Online: "  << ( enableNotifyOnline_  ? "true" : "false" ) << ", "
            << "Offline: " << ( enableNotifyOffline_ ? "true" : "false" ) << ", "
            << "Status: "  << ( enableNotifyStatus_  ? "true" : "false" ) << endl;
#endif

  // When contacts go online, whichever may be their status, just show an 'is now online'.
  if( lastStatus == "FLN" && status != "FLN" && enableNotifyOnline_ )
    text = getStatusString( "NLN" );
  else
    text = getStatusString( status );

  // The resulting string is a i18n'ed string, with the contact friendly name as argument.
  text = text.arg( QStyleSheet::escape( contact->getFriendlyName() ) );



  // Play a sound
  QString soundPath = contact->getExtension()->getContactSoundPath();
  QFile   soundFile( soundPath );

  // Find what kind of event is this, and skip the notification if it's a disabled event
  bool notifyUser = true;
       if( lastStatus == "FLN" && status != "FLN" ) notifyUser = enableNotifyOnline_;
  else if( lastStatus != "FLN" && status == "FLN" ) notifyUser = enableNotifyOffline_;
  else                                              notifyUser = enableNotifyStatus_;

  if( ! notifyUser )
  {
#ifdef KMESSDEBUG_NOTIFICATIONCONTACTSTATUS
    kdDebug() << "NotificationContactStatus::notify() - Event popup disabled, skipping." << endl;
#endif

    return;
  }

  // If a custom audio file exists, play it
  if ( soundFile.exists() )
  {
#ifdef KMESSDEBUG_NOTIFICATIONCONTACTSTATUS
    kdDebug() << "NotificationContactStatus::notify() - Play sound " << soundPath << endl;
#endif

    // Play the custom sound instead of the generic sound
    KAudioPlayer::play( soundPath );
  }
  else
  {
#ifdef KMESSDEBUG_NOTIFICATIONCONTACTSTATUS
    kdDebug() << "NotificationContactStatus::notify() - Play the default sound." << endl;
#endif

    QString statusText;

    if( lastStatus == "FLN" )
    {
      statusText = "contact online";
    }
    else if( status == "FLN" )
    {
      statusText = "contact offline";
    }
    else
    {
      statusText = "contact status";
    }

#if KDE_IS_VERSION(3,2,0)
    KNotifyClient::event( manager_->getTrayWidget()->winId(), statusText, text );
#else
    KNotifyClient::event( statusText, text );
#endif
  }

  // Display a popup for this event
  PassivePopup::PopupProperties props;
  props.source = contact->getHandle();
  props.message = text;
  props.icon = contact->getContactPicturePath();
  props.status = status;
  props.destMessages << contact->getHandle() << status;
  props.destObject = 0;

  manager_->notify( this, props );
}



// Enable or disable the contact offline notifications
void NotificationContactStatus::setEnabled( bool enableNotifyOnline, bool enableNotifyOffline, bool enableNotifyStatus )
{
  enableNotifyOnline_  = enableNotifyOnline;
  enableNotifyOffline_ = enableNotifyOffline;
  enableNotifyStatus_  = enableNotifyStatus;
}



#include "notificationcontactstatus.moc"
