/* This file is part of the KDE project
 * Copyright (C) 2007 Thomas Zander <zander@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "KWPageSettingsDialog.h"
#include "KWPageLayout.h"
#include "KWPagePreview.h"
#include "KWDocument.h"
#include "commands/KWPagePropertiesCommand.h"

#include <KoLayoutVisitor.h>

#include <QTimer>

//#include <KDebug>

KWPageSettingsDialog::KWPageSettingsDialog(QWidget *parent, KWDocument *document, KWPage *page)
    : KDialog(parent),
    m_document(document),
    m_page(page),
    m_visited (false)
{
    Q_ASSERT(document);
    Q_ASSERT(page);
    m_layout = m_page->pageLayout();
    QWidget *widget = new QWidget(this);
    setMainWidget(widget);
    QHBoxLayout *lay = new QHBoxLayout(widget);
    lay->setMargin(0);
    widget->setLayout(lay);

    m_pageLayoutWidget = new KWPageLayout(widget, m_layout);
    m_pageLayoutWidget->showUnitchooser(false);
    m_pageLayoutWidget->forSinglePage(m_page);
    m_pageLayoutWidget->setStartPageNumber(m_document->startPage());
    m_pageLayoutWidget->setTextDirection(m_page->directionHint());
    m_pageLayoutWidget->layout()->setMargin(0);
    lay->addWidget(m_pageLayoutWidget);

    KWPagePreview *prev = new KWPagePreview(widget);
    prev->setPageLayout(m_layout);
    lay->addWidget(prev);

    connect (m_pageLayoutWidget, SIGNAL(layoutChanged(const KoPageLayout&)),
            prev, SLOT(setPageLayout(const KoPageLayout&)));
    connect (m_pageLayoutWidget, SIGNAL(layoutChanged(const KoPageLayout&)),
            this, SLOT(setPageLayout(const KoPageLayout&)));
}

void KWPageSettingsDialog::setPageLayout(const KoPageLayout &layout) {
    m_layout = layout;
}

void KWPageSettingsDialog::accept() {
    if(m_pageLayoutWidget->marginsForDocument()) {
        // TODO
    }
    else {
        KoText::Direction newDir = m_pageLayoutWidget->textDirection();
        KWPagePropertiesCommand *cmd = new KWPagePropertiesCommand(m_document, m_page, m_layout, newDir);
        m_document->addCommand(cmd);
    }

    m_document->setStartPage(m_pageLayoutWidget->startPageNumber());

    KDialog::accept();
    deleteLater();
}

void KWPageSettingsDialog::reject() {
    KDialog::reject();
    deleteLater();
}

void KWPageSettingsDialog::showEvent (QShowEvent *e) {
    KDialog::showEvent(e);
    if(m_visited) return;
    m_visited = true;
    QTimer::singleShot(0, this, SLOT(visit()));
}

void KWPageSettingsDialog::visit() {
    KoLayoutVisitor visitor;
    visitor.visit(m_pageLayoutWidget);
    visitor.relayout();
}

void KWPageSettingsDialog::showTextDirection(bool on) {
    m_pageLayoutWidget->showTextDirection(on);
}

