<?php
/*
 *  Copyright (c) 2005 Intra2net AG
 *  Copyright (c) 2006 erfrakon Partnerschaftsgesellschaft
 *
 *    Written by Thomas Jarosch <thomas.jarosch@intra2net.com>
 *    Written by Martin Konold <martin.konold@erfrakon.de>
 *
 *  This  program is free  software; you can redistribute  it and/or
 *  modify it  under the terms of the GNU  General Public License as
 *  published by the  Free Software Foundation; either version 2, or
 *  (at your option) any later version.
 *
 *  This program is  distributed in the hope that it will be useful,
 *  but WITHOUT  ANY WARRANTY; without even the  implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  General Public License for more details.
 *
 *  You can view the  GNU General Public License, online, at the GNU
 *  Project's homepage; see <http://www.gnu.org/licenses/gpl.html>.
 */

 /* Class to efficently cache kolab events stored on an IMAP server */

class FreeBusyIMAPCache {
  var $version;       // internal version of format
  var $store_prefix;  // prefix to prepend to all file operations
  var $owner;         // folder owner
  var $foldername;    // folder name
  var $cache_modified;      // indicates if we have to writeout the cache
  var $cache;         // cache data

  function FreeBusyIMAPCache($store_prefix, &$owner, &$foldername) {
    $this->store_prefix = $store_prefix;
    $this->owner = $owner;
    $this->foldername = $foldername;
    $this->version = 1;
    $this->reset_cache();
  }
  
  function reset_cache() {
    $this->cache = array();
    $this->cache["version"] = $this->version;
    $this->cache["uidvalidity"] = -1;
    $this->cache["uidnext"] = -1;
    $this->cache["incidences-for"] = "";
    $this->cache["imap2fb"] = array();
    $this->cache_modified = true;
  }

  function check_folder_changed($uidvalidity, $uidnext, $incidences_for, &$new_uids) {
    $changed = false;
    // uidvalidity changed?
    if ($uidvalidity != $this->cache["uidvalidity"]) {
      myLog("uidvalidity changed (old: ".$this->cache["uidvalidity"].", new: $uidvalidity), clearing cache", RM_LOG_DEBUG);
      $this->reset_cache();
      $changed = true;
    }

    // uidnext changed?
    if ($uidnext != $this->cache["uidnext"]) {
      myLog("uidnext on folder changed (old: ".$this->cache["uidnext"].", new: ".$uidnext.")", RM_LOG_DEBUG);
      $changed = true;
    }

    // incidences-for changed?
    if ($incidences_for != $this->cache["incidences-for"]) {
      myLog("incidences-for changed (old: ".$this->cache["incidences-for"].", new: $incidences_for), clearing cache", RM_LOG_DEBUG);
      $this->reset_cache();
      $changed = true;
    }

    $this->cache["uidvalidity"] = $uidvalidity;
    $this->cache["uidnext"] = $uidnext;
    $this->cache["incidences-for"] = $incidences_for;

    // deleted a message?
    $old_uids = array_keys($this->cache["imap2fb"]);
    while(list($key, $old_uid) = each($old_uids)) {
      if (!in_array($old_uid, $new_uids)) {
        unset($this->cache["imap2fb"][$old_uid]);
        $this->cache_modified = true;
        $changed = true;
      }
    }

    if (!$changed)
      myLog("check_changed: folder didn't change", RM_LOG_DEBUG);
    return $changed;
  }

  function check_uid_exists($uid) {
    return array_key_exists($uid, $this->cache["imap2fb"]);
  }

  function add_empty_imap2fb(&$imap_uid) {
    $this->cache["imap2fb"][$imap_uid] = array();
    $this->cache_modified = true;
  }

  function add_imap2fb(&$imap_uid, $fb_start, $fb_end, $fb_duration, $fb_extra) {
    /*
         Internal imap2fb array structure:
         0..n IMAP uid
         |----------- 0..n free/busy periods
                      |----------- start
                      |----------- end
                      |----------- duration
                      |----------- extra
    */
    myLog("added event to store: uid: $imap_uid, start: $fb_start, end: $fb_end, duration: $fb_duration", RM_LOG_DEBUG);

    $store = array();
    $store["start"] = $fb_start;
    $store["end"] = $fb_end;
    $store["duration"] = $fb_duration;
    $store["extra"] = $fb_extra;

    $this->cache["imap2fb"][$imap_uid][] = $store;
    $this->cache_modified = true;
  }

  function output_fb(&$vFb) {
    reset($this->cache["imap2fb"]);
    while(list($uid, $periods) = each($this->cache["imap2fb"]))
      while(list($key, $period) = each($periods))
        $vFb->addBusyPeriod('BUSY', $period["start"], $period["end"], $period["duration"], $period["extra"]);
  }


  function compute_filename() {
    $folder_parts = explode('/', $this->foldername);
    unset($folder_parts[0]);
    $folder_storename = join('/', $folder_parts);

    $folder_storename = str_replace(".", "^", $folder_storename);
    $folder_storename = str_replace("\0", "", $folder_storename);

    if( ereg( '(.*)@(.*)', $this->owner, $regs ) ) {
      $domain = $regs[2].'/';
	   $domain = str_replace(".", "^", $domain);
	   $domain = str_replace("\0", "", $domain);
    } else $domain = '';
	
    $full_path = $this->store_prefix.$domain.$folder_storename.".imapcache";
    return $full_path;
  }

  function cache_load() {
    $filename = $this->compute_filename();

    myLog("Trying to load file: $filename", RM_LOG_DEBUG);

    if (!is_readable($filename))
      return false;

    $this->cache = unserialize(file_get_contents($filename));

    // Delete disc cache if it's from an old version
    if ($this->cache["version"] != $this->version) {
      myLog("Version mismatch (got: ".$this->cache["version"].", current: ".$this->version.", dropping cache", RM_LOG_WARN);
      $this->reset_cache();
    } 
    else $this->cache_modified = false;
    return true;
  }

  function cache_store($force=false) {
    if ($this->cache_modified || $force) {
      $filename = $this->compute_filename();
      myLog("Trying to save cache to file: $filename", RM_LOG_DEBUG);

      if (!$this->mkdirhier(dirname($filename))) {
        myLog("can't create director hierachy: ".dirname($filename), RM_LOG_ERROR);
        return;
      }

      $tmpname = tempnam(dirname($this->store_prefix), 'imapcache');
      $fp = fopen($tmpname, 'w');
      if(!$fp) return false;
        if (fwrite($fp, serialize($this->cache)) === false) {
          fclose ($fp);
          myLog("can't write to file: $tmpname. Out of discspace?", RM_LOG_ERROR);
          return;
        }

      if(!rename($tmpname, $filename)) {
         myLog("can't rename $tmpname to $filename", RM_LOG_ERROR);
         return false;
      }
      fclose($fp);
      $this->cache_modified = false;
    } 
    else {
      myLog("IMAPcache unmodified, not saving", RM_LOG_DEBUG);
    }	
  }

  function cache_delete() {
    unlink($this->compute_filename());
    $this->reset_cache();
  }

  function mkdirhier( $dirname ) {
    $base = substr($dirname,0,strrpos($dirname,'/'));
    $base = str_replace(".", "^", $base);
    if( !empty( $base ) && !is_dir( $base ) ) {
      if( !$this->mkdirhier( $base ) ) return false;
    }
    if( !file_exists( $dirname ) ) return mkdir( $dirname, 0755 );
    return true;
  }
};

?>
