//                          -*- mode: C++ -*-
//
// Copyright(C) 2005,2006,2007 Stefan Siegl <stesie@brokenpipe.de>
// Copyright(C) 2006 Martin Albrecht <malb@informatik.uni-bremen.de>
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef KOPETESILC_SILCCONTACTMANAGER_H
#define KOPETESILC_SILCCONTACTMANAGER_H

#include <qptrlist.h>
#include <kdemacros.h>
#include <qobject.h>

//#include "silccontact.h"

class SilcAccount;
class SilcBuddyContact;
class SilcChannelContact;
class SilcServerContact;
class SilcBuddyContactData;
class SilcContact;

// include libsilc stuff directly into a separate namespace
// we cannot just include silcaccount.h, since it does rely on 
// SilcContactManager class ....
namespace SilcTK {
#include <silc.h>
#include <silcclient.h>
}

namespace Kopete {
  class OnlineStatus;
  class MetaContact;
}

/**
 * @brief list of SilcContact instances
 *
 * @author Stefan Siegl <stesie@brokenpipe.de>
 */
template <class T>
class SilcContactList : public QPtrList<SilcContact> {
public:
  /**
   * @brief compare function needed by QPtrList
   */
  int compareItems(QPtrCollection::Item item1, QPtrCollection::Item item2);

  /** 
   * @brief look for the SilcContact with the provided name
   */
  T *lookup(const QString &name);

  SilcBuddyContact *lookupByFingerprint(const QString &fingerprint);
  SilcBuddyContact *lookupById(const QString &id);

  /**
   * @brief mark all registered SilcContact instances with the provided
   * Kopete::OnlineStatus
   */
  void setStatus(Kopete::OnlineStatus status);
};



/**
 * @brief create, destroy and otherwise care for SilcContact instances
 *
 * SilcContactManager is mainly responsible for creating instances of
 * SilcChannelContact and SilcBuddyContact.  Every SilcAccount has such a
 * manager throughout it's lifetime.
 *
 * On connection establishment and shutdown as well as while being connected
 * it updates the online status of the buddies, etc.
 *
 * Furthermore it listens for incoming private messages (from SilcEngine
 * driver) and delivers them to the corresponding SilcBuddyContact (which is
 * to be created if it is not already existing)
 *
 * @author Stefan Siegl <stesie@brokenpipe.de>
 */
class SilcContactManager : public QObject {
  Q_OBJECT;

public:
  SilcContactManager(SilcAccount *);
  ~SilcContactManager();

  /**
   * @brief create new SilcChannelContact (if not already existing)
   *
   * createChannel calls lookupChannel to check whether there already is a
   * SilcChannelContact with the specified name.  If there is not, it will
   * allocate a new one, with the specified Kopete::MetaContact set as it's
   * parent.
   *
   * If you do not specify a Kopete::MetaContact, a new (temporary, i.e. not
   * displayed in the contact list) one will be created for you.
   *
   * @param name - the name of the channel
   * @param meta - the meta contact to set as parent (or NULL)
   */
  SilcChannelContact *createChannel(const QString &name,
				    Kopete::MetaContact *meta = 0, 
				    SilcTK::SilcChannelEntry = 0);

  /**
   * @brief find an already existing SilcChannelContact
   *
   * NULL is returned, in case the channel is yet unknown
   *
   * @param name - the name of the channel to seek for
   */
  SilcChannelContact *lookupChannel(const QString &name);

  /**
   * @brief create new SilcBuddyContact (if not already existing)
   *
   * createBuddy calls lookupBuddy to check whether there already is a
   * SilcBuddyContact with the specified name.  If there is not, it will
   * allocate a new one, with the specified Kopete::MetaContact set as it's
   * parent.
   *
   * If you do not specify a Kopete::MetaContact, a new (temporary, i.e. not
   * displayed in the contact list) one will be created for you.
   *
   * @param name - the name of the (remote) user
   * @param entry - the SilcTK structure for this client
   * @param meta - the meta contact to set as parent (or NULL)
   */
 
  SilcBuddyContact *createBuddy(const QString &name,
                                Kopete::MetaContact *meta = NULL,
                                SilcTK::SilcClientEntry entry = NULL);

  SilcBuddyContact *createBuddy(Kopete::MetaContact *meta,
				const QString &finger);

  /**
   * pops the buddy identified by nickname from the pending buddies
   * list
   *
   * returns NULL if nothing is found
   * 
   * @param nickname - nickname to identify buddy
   */

  SilcBuddyContactData *popPendingBuddy(const QString &nickname);

  /**
   * @brief find an already existing SilcBuddyContact
   *
   * NULL is returned, in case the buddy is not known yet
   *
   * @param name - the nickname of the buddy to seek for
   * @deprecated  You should not identify a buddy by it's nickname, use
   * it's fingerprint instead.
   */
  KDE_DEPRECATED SilcBuddyContact *lookupBuddy(const QString &name);


/**
   * @brief find an already existing SilcBuddyContact
   *
   * NULL is returned, in case the buddy is not known yet
   *
   * @param name - the fingerprint of the buddy to seek for
   */

  SilcBuddyContact * lookupBuddyByFingerprint(const QString &finger);

  /**
   * @brief adds a buddy to the Buddy List
   *
   * false is returned if the buddy is already in the listens
   *
   * @param buddy - the buddy to add
   */

   bool addBuddy(SilcBuddyContact *buddy);

  /**
   * @brief find an already existing SilcBuddyContact by it's ID
   *
   * @param id - the id, including the `buddy.' part
   */
  SilcBuddyContact *lookupBuddyById(const QString &id);

  /**
   * @brief return a pointer back to the SilcAccount class
   */
  inline SilcAccount *account(void) const { return _account; }

  /**
   * @brief remove buddy from every channel he was joined in to
   */
  void buddySignedOff(SilcBuddyContact *buddy,
		      const QString &message = QString::null);

  /**
   */
  void setOnlineStatus(SilcBuddyContact *, const Kopete::OnlineStatus &);

  SilcServerContact *createServer(const QString &hostname);

  /**
   * @brief add all buddies to the watch list (or remove them again)
   */
  void watchAllBuddies(bool watch);

protected:
  /**
   * @brief list of registered channels
   */
  SilcContactList<SilcChannelContact> _channels;

  /**
   * @brief list of registered users
   */
  SilcContactList<SilcBuddyContact> _buddies;

  /**
   * @brief list of pending users
   */
  QPtrList<SilcBuddyContactData> _pending_buddies;

private:
  /**
   * @brief pointer back to the SilcAccount (our parent)
   */
  SilcAccount *_account;

  static void getClientsCallback(SilcTK::SilcClient,
				 SilcTK::SilcClientConnection,
				 SilcTK::SilcStatus status,
				 SilcTK::SilcDList celist,
				 void *context);

  unsigned int _outstandingWhoisRequests;

private slots:
  /** 
   * @brief connection to the SILC network successfully established
   *
   * Mark all registered SilcChannelContact instances 'online' and fire
   * /WHOIS requests for buddies to learn about their online status.
   */
  void slotConnected(void);

  /**
   * @brief connection to the SILC network was reset
   *
   * Mark all buddies and channels offline.
   */
  void slotDisconnected(void);
};

#endif // KOPETESILC_SILCCONTACTMANAGER_H
