/**********************************************************************
** Copyright (C) 2000 Trolltech AS.  All rights reserved.
**
** This file is part of Qt Designer.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include "config.h"
#include <qfile.h>
#include <qfileinfo.h>
#include <qtextstream.h>

Config::Config( const QString &fn )
    : filename( fn )
{
    git = groups.end();

		m_bChanged = false;
    m_bOk = read();
}

Config::~Config()
{
    write();
}

bool Config::ok()
{
	return m_bOk;
}

void Config::setGroup( const QString &gname )
{
    QMap< QString, ConfigGroup>::Iterator it = groups.find( gname );
    if ( it == groups.end() ) {
	ConfigGroup grp;
	git = groups.insert( gname, grp );
	return;
    }
    git = it;
}

void Config::writeEntry( const QString &key, const QString &value )
{
    if ( git == groups.end() ) {
	qWarning( "no group set" );
	return;
    }
		m_bChanged = true;
    ( *git ).insert( key, value );
}

void Config::writeEntry( const QString &key, int num )
{
    QString s;
    s.setNum( num );
    writeEntry( key, s );
}

void Config::writeEntry( const QString &key, double num )
{
    QString s;
    s.setNum( num );
    writeEntry( key, s );
}

void Config::writeEntry( const QString &key, bool b )
{
    QString s;
    s.setNum( ( int )b );
    writeEntry( key, s );
}

void Config::writeEntry( const QString &key, const QStringList &lst, const QChar &sep )
{
    QString s;
    QStringList::ConstIterator it = lst.begin();
    for ( ; it != lst.end(); ++it )
	s += *it + sep;
    writeEntry( key, s );
}

QString Config::readEntry( const QString &key, const QString &deflt )
{
    if ( git == groups.end() ) {
	qWarning( "no group set" );
	return deflt;
    }
    ConfigGroup::Iterator it = ( *git ).find( key );
    if ( it != ( *git ).end() )
	return *it;
    else
	return deflt;
}

double Config::readNumEntry( const QString &key, double deflt )
{
    QString s = readEntry( key );
    if ( s.isEmpty() )
	return deflt;
    else
	return s.toDouble();
}

int Config::readNumEntry( const QString &key, int deflt )
{
    QString s = readEntry( key );
    if ( s.isEmpty() )
	return deflt;
    else
	return s.toInt();
}

bool Config::readBoolEntry( const QString &key, bool deflt )
{
    QString s = readEntry( key );
    if ( s.isEmpty() )
	return deflt;
    else
	return (bool)s.toInt();
}

QStringList Config::readListEntry( const QString &key, const QChar &sep )
{
    QString s = readEntry( key );
    if ( s.isEmpty() )
	return QStringList();
    else
	return QStringList::split( sep, s );
}

/*!
	Removes the group "grp" from the group list.

	\sa removeEntry, clearGroup
*/
void Config::removeGroup(const QString &grp)
{
	groups.remove(grp);
}

/*!
	Removes the entry "key" from the current group.
	Does nothing if no group is set or the entry is not existing.

	\sa removeGroup, clearGroup
*/
void Config::removeEntry(const QString &key)
{
    if ( git == groups.end() ) {
	qWarning( "no group set" );
	return;
    }
		( *git ).remove( key );
}

/*!
	Clears the current group, does nothing if no group is set.
	Use removeGroup to remove the group itself;

	\sa removeEntry, removeGroup
*/
void Config::clearGroup()
{
    if ( git == groups.end() ) {
	qWarning( "no group set" );
	return;
    }
    ( *git ).clear();
}

/*!
	Writes by the default the file specified in the constructor if no attribute is used,
	otherwise it writes the file "fn".

	Return true and does nothing if we have not used any writeEntry function except
	we write to a different file then orignialy opened.
	setGroup alone does nothing, so its not possible to make an empty group or overwrite
	an file that has been read only during operation.
*/
bool Config::write( const QString &fn )
{
		if (fn.isEmpty() && m_bChanged==false)
	return true;

    if ( !fn.isEmpty() )
	filename = fn;

    QFile f( filename );
    if ( !f.open( IO_WriteOnly ) ) {
	qWarning( "could not open for writing '%s'", filename.latin1() );
	git = groups.end();
	return false;
    }

    QTextStream s( &f );
    QMap< QString, ConfigGroup >::Iterator g_it = groups.begin();
    for ( ; g_it != groups.end(); ++g_it ) {
	s << "[" << g_it.key() << "]" << "\n";
	ConfigGroup::Iterator e_it = ( *g_it ).begin();
	for ( ; e_it != ( *g_it ).end(); ++e_it )
	    s << e_it.key() << " = " << *e_it << "\n";
    }

    f.close();

		return true;
}

/*!
	Reads the entire file speciefied in the constructor.

	Returns false if the file does not exist or the file can't be opened by some reason.

	Note: Prints out qWarnings if an error occurs in debug mode
*/
bool Config::read()
{
    if ( !QFileInfo( filename ).exists() ) {
 	qWarning( "'%s' doesn't exist", filename.latin1() );
	git = groups.end();
	return false;
    }

    QFile f( filename );
    if ( !f.open( IO_ReadOnly ) ) {
	qWarning( "could not open for reading '%s'", filename.latin1() );
	git = groups.end();
	return false;
    }

    QTextStream s( &f );

    QString line;
    while ( !s.atEnd() ) {
	line = s.readLine();
	parse( line );
    }

    f.close();

    return true;
}

/*!
	Parces a single line
*/
void Config::parse( const QString &l )
{
    QString line = l.stripWhiteSpace();
    if ( line[ 0 ] == QChar( '[' ) ) {
	QString gname = line;
	gname = gname.remove( 0, 1 );
	if ( gname[ (int)gname.length() - 1 ] == QChar( ']' ) )
	    gname = gname.remove( gname.length() - 1, 1 );
	ConfigGroup grp;
	git = groups.insert( gname, grp );
    } else {
	if ( git == groups.end() ) {
	    qWarning( "line '%s' out of group", line.latin1() );
	    return;
	}
	QStringList lst = QStringList::split( '=', line );
	if ( lst.count() != 2 && line.find( '=' ) == -1 ) {
	    qWarning( "corrupted line '%s' in group '%s'",
		      line.latin1(), git.key().latin1() );
	    return;
	}
	( *git ).insert( lst[ 0 ].stripWhiteSpace(), lst[ 1 ].stripWhiteSpace() );
    }
}
