// =============================================================================
//
//      --- kvi_userdialog.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviUserDialog"

#include <qframe.h>
#include <qlayout.h>

#include "kvi_app.h"
#include "kvi_boolselector.h"
#include "kvi_iconloader.h"
#include "kvi_locale.h"
#include "kvi_maintoolbar.h"
#include "kvi_options.h"
#include "kvi_stringselector.h"
#include "kvi_userdialog.h"

/*
	@quickhelp: KviUserDialog
	@widget: User options dialog
		Here you set your basic IRC preferences:<br>
		Edit your nickname in the <a href="qh_qwidget_nicknametab.kvihelp">nickname tab</a>.<br>
		Edit your default user mode in the <a href="qh_qwidget_defumodetab.kvihelp">default user mode tab</a>.<br>
*/

KviUserDialog::KviUserDialog(QWidget *parent)
	: KviTabDialog(parent, "userdialog", false, _i18n_("KVIrc: User Configuration"))
{
	setIcon(KviIconLoader::loadIcon(tb_configUser));

	/*
		@quickhelp: QWidget_NicknameTab
		@widget: Nickname tab
			The options in this tab are strictly required to connect to an IRC server.<br>
			The <b>nickname</b> is a name that the IRC server will use to identify you.<br>
			Your nickname should be <b>unique</a> in the whole IRC network that you are going to connect to.<br>
			So it may happen that the nickname you have chosen is already used by someone else.<br>
			For this reason you can specify an <b>alternative nickname</b>.<br>
			The <b>user name</b> will be declared to the server at connect time.<br>
			The server may choose to accept this one, or to check your real username by
			connecting to your ident service.<br>
			The <b>real name</b> will be shown to other users when they will perform a <a href="whois.kvihelp">WHOIS</a> on you.<br>
			You can put any (reasonably short) string here.<br>
	*/

	//
	// Nickname tab...
	//
	QFrame      *tab = addPage(_CHAR_2_QSTRING(_i18n_("&Identity")));
	QGridLayout *g   = new QGridLayout(tab, 4, 1, 10, 4);

	KviStringSelector *c;
	c = new KviStringSelector(tab, _i18n_("Nickname:"), &(g_pOptions->m_szNickName1));
	g->addWidget(c, 0, 0);
	c = new KviStringSelector(tab, _i18n_("Alt. nickname:"), &(g_pOptions->m_szNickName2));
	g->addWidget(c, 1, 0);
	c = new KviStringSelector(tab, _i18n_("Username:"), &(g_pOptions->m_szUserName));
	g->addWidget(c, 2, 0);
	c = new KviStringSelector(tab, _i18n_("Real name:"), &(g_pOptions->m_szRealName));
	g->addWidget(c, 3, 0);

	/*
		@quickhelp: QWidget_DefUModeTab
		@widget: Default user mode tab
			Here you can choose the default user modes that KVIrc will attempt to set for you at connect time.<br>
			When you are <b>invisible</b>, other users will be not able to "see" you unless they know your nickname.<br>
			The generic <b>+s</b> flag makes you become a recipient for the server notices.
			Many servers have also other flags that allow you to choose the type of the notices you want to receive.<br>
			Note: Some servers will automatically set for you some user modes and not all modes may be supported.<br>
	*/

	//
	// Protection tab
	//
	tab = addPage(_CHAR_2_QSTRING(_i18n_("Default &Modes")));
	g   = new QGridLayout(tab, 3, 1, 10, 4);

	KviBoolSelector *b;
	b = new KviBoolSelector(tab, _i18n_("Invisible (Mode +i)"), &(g_pOptions->m_bModeIOnConnect));
	g->addWidget(b, 0, 0);
	b = new KviBoolSelector(tab, _i18n_("Receive server notices (Mode +s)"), &(g_pOptions->m_bModeSOnConnect));
	g->addWidget(b, 1, 0);
	b = new KviBoolSelector(tab, _i18n_("Receive wallops (Mode +w)"), &(g_pOptions->m_bModeWOnConnect));
	g->addWidget(b, 2, 0);

	setButtonHelpText(_CHAR_2_QSTRING(_i18n_("What is &This?")));
	connect(this, SIGNAL(helpClicked()), g_pApp, SLOT(slot_whatIsThisRequest()));
	setButtonOK();
	setButtonCancel();
}

KviUserDialog::~KviUserDialog()
{
	// Nothing here
}

void KviUserDialog::done(int r)
{
	KviTabDialog::done(r);
	if( r == Accepted ) {
		KviBoolSelector::commitAll(this);
		KviStringSelector::commitAll(this);
		// Convert spaces in the username to underscores to prevent sending
		// wrong "USER ..." commands
		(g_pOptions->m_szUserName).replaceAll(' ', "_");
	}
	emit finished((r == Accepted));
}

void KviUserDialog::closeEvent(QCloseEvent *)
{
	emit finished(false);
}

#include "m_kvi_userdialog.moc"
