###
### $Rev: 48 $
### $Release: 0.6.1 $
### copyright(c) 2005 kuwata-lab all rights reserved.
###

require 'kwalify/messages'
require 'kwalify/errors'
require 'kwalify/types'
require 'kwalify/rule'

module Kwalify

   ##
   ## ex.
   ##   schema = YAML.load_file('schema.yaml')
   ##   validator = Kwalify::Validator.new(schema)
   ##   document = YAML.load_file('document.yaml')
   ##   error_list = validator.validate(document)
   ##   unless error_list.empty?
   ##     error_list.each do |error|
   ##       puts "- [#{error.path}] #{error.message}"
   ##     end
   ##   end
   ##
   class Validator
      include Kwalify::ErrorHelper

      def initialize(hash, &block)
         @rule  = Rule.new(hash)
         @block = block
      end
      attr_reader :rule


      def _inspect
         @rule._inspect
      end


      def validate(value)
         path = "";  errors = [];  done = {}
         _validate(value, @rule, path, errors, done)
         return errors
      end


      protected


      def validate_hook(value, rule, path, errors)
      end


      def _validate(value, rule, path, errors, done)
         if Types.collection?(value)
            return if done[value.__id__]     # avoid infinite loop
            done[value.__id__] = true
         end
         if rule.required && value == nil
            #* key=:required_novalue  msg="value required but none."
            errors << validate_error(:required_novalue, rule, path, value)
            return
         end
         if rule.type_class && value != nil && !value.is_a?(rule.type_class)
            #* key=:type_unmatch  msg="not a %s."
            errors << validate_error(:type_unmatch, rule, path, value, [Kwalify.word(rule.type)])
            return
         end
         #
         n = errors.length
         if rule.sequence
            _validate_sequence(value, rule, path, errors, done)
         elsif rule.mapping
            _validate_mapping(value, rule, path, errors, done)
         else
            _validate_scalar(value, rule, path, errors, done)
         end
         return unless errors.length == n
         #
         validate_hook(value, rule, path, errors)
         @block.call(value, rule, path, errors) if @block
      end


      private


      def _validate_scalar(value, rule, path, errors, done)
         assert_error("rule.sequence.class==#{rule.sequence.class.name} (expected NilClass)") if rule.sequence
         assert_error("rule.mapping.class==#{rule.mapping.class.name} (expected NilClass)") if rule.mapping
         if rule.assert_proc
            unless rule.assert_proc.call(value)
               #* key=:assert_failed  msg="assertion expression failed (%s)."
               errors << validate_error(:assert_failed, rule, path, value, [rule.assert])
            end
         end
         if rule.enum
            unless rule.enum.include?(value)
               keyname = File.basename(path)
               keyname = 'enum' if keyname =~ /\A\d+\z/
               #* key=:enum_notexist  msg="invalid %s value."
               errors << validate_error(:enum_notexist, rule, path, value, [keyname])
            end
         end
         #
         return if value == nil
         #
         if rule.pattern
            unless value.to_s =~ rule.regexp
               #* key=:pattern_unmatch  msg="not matched to pattern %s."
               errors << validate_error(:pattern_unmatch, rule, path, value, [rule.pattern])
            end
         end
         if rule.range
            assert_error("value.class=#{value.class.name}") unless Types.scalar?(value)
            if rule.range['max'] && rule.range['max'] < value
               #* key=:range_toolarge  msg="too large (> max %s)."
               errors << validate_error(:range_toolarge, rule, path, value, [rule.range['max'].to_s])
            end
            if rule.range['min'] && rule.range['min'] > value
               #* key=:range_toosmall  msg="too small (< min %s)."
               errors << validate_error(:range_toosmall, rule, path, value, [rule.range['min'].to_s])
            end
            if rule.range['max-ex'] && rule.range['max-ex'] <= value
               #* key=:range_toolargeex  msg="too large (>= max %s)."
               errors << validate_error(:range_toolargeex, rule, path, value, [rule.range['max-ex'].to_s])
            end
            if rule.range['min-ex'] && rule.range['min-ex'] >= value
               #* key=:range_toosmallex  msg="too small (<= min %s)."
               errors << validate_error(:range_toosmallex, rule, path, value, [rule.range['min-ex'].to_s])
            end
         end
         if rule.length
            assert_error("value.class=#{value.class.name}") unless value.is_a?(String) || value.is_a?(Text)
            len = value.to_s.length
            if rule.length['max'] && rule.length['max'] < len
               #* key=:length_toolong  msg="too long (length %d > max %d)."
               errors << validate_error(:length_toolong, rule, path, value, [len, rule.length['max']])
            end
            if rule.length['min'] && rule.length['min'] > len
               #* key=:length_tooshort  msg="too short (length %d < min %d)."
               errors << validate_error(:length_tooshort, rule, path, value, [len, rule.length['min']])
            end
            if rule.length['max-ex'] && rule.length['max-ex'] <= len
               #* key=:length_toolongex  msg="too long (length %d >= max %d)."
               errors << validate_error(:length_toolongex, rule, path, value, [len, rule.length['max-ex']])
            end
            if rule.length['min-ex'] && rule.length['min-ex'] >= len
               #* key=:length_tooshortex  msg="too short (length %d <= min %d)."
               errors << validate_error(:length_tooshortex, rule, path, value, [len, rule.length['min-ex']])
            end
         end
      end


      def _validate_sequence(list, seq_rule, path, errors, done)
         assert_error("seq_rule.sequence.class==#{seq_rule.sequence.class.name} (expected Array)") unless seq_rule.sequence.is_a?(Array)
         assert_error("seq_rule.sequence.length==#{seq_rule.sequence.length} (expected 1)") unless seq_rule.sequence.length == 1
         return if list == nil
         rule = seq_rule.sequence[0]
         list.each_with_index do |val, i|
            _validate(val, rule, "#{path}/#{i}", errors, done)   ## validate recursively
         end
         if rule.type == 'map'
            unique_keys = []
            rule.mapping.keys.each do |key|
               map_rule = rule.mapping[key]
               unique_keys << key if map_rule.unique || map_rule.ident
            end
            unique_keys.each do |key|
               table = {}
               list.each_with_index do |map, i|
                  val = map[key]
                  next if val == nil
                  curr_path = "#{path}/#{i}/#{key}"
                  if table[val]
                     #* key=:value_notunique  msg="is already used at '%s'."
                     errors << validate_error(:value_notunique, rule, "#{path}/#{i}/#{key}", val, "#{path}/#{table[val]}/#{key}")
                  else
                     table[val] = i
                  end
               end
            end if !unique_keys.empty?
         elsif rule.unique
            table = {}
            list.each_with_index do |val, i|
               next if val == nil
               if table[val]
                  #  #* key=:value_notunique  msg="is already used at '%s'."
                  errors << validate_error(:value_notunique, rule, "#{path}/#{i}", val, "#{path}/#{table[val]}")
               else
                  table[val] = i
               end
            end
         end
      end


      def _validate_mapping(hash, map_rule, path, errors, done)
         assert_error("map_rule.mapping.class==#{map_rule.mapping.class.name} (expected Hash)") unless map_rule.mapping.is_a?(Hash)
         return if hash == nil
         map_rule.mapping.each do |key, rule|
            if rule.required && !hash.key?(key)
               #* key=:required_nokey  msg="key '%s:' is required."
               errors << validate_error(:required_nokey, rule, path, hash, [key])
            end
         end
         hash.each do |key, val|
            rule = map_rule.mapping[key]
            unless rule
               #* key=:key_undefined  msg="key '%s' is undefined."
               errors << validate_error(:key_undefined, rule, "#{path}/#{key}", hash, ["#{key}:"])
               ##* key=:key_undefined  msg="undefined key."
               #errors << validate_error(:key_undefined, rule, "#{path}/#{key}", "#{key}:")
            else
               _validate(val, rule, "#{path}/#{key}", errors, done)   ## validate recursively
            end
         end
      end

   end

end
