/***************************************************************************
 *   Copyright (C) 2004 by Matthias Reif                                   *
 *   matthias.reif@informatik.tu-chemnitz.de                               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "kprofilewidget.h"

#include "wwidget.h"

KProfileWidget::KProfileWidget( WWidget* wWidget, QWidget *parent, const char *name ) : QWidget( parent, name )
{
	this->wWidget = wWidget;
	KIconLoader iconLoader;
	loadButton = new KPushButton( iconLoader.loadIconSet( "fileopen", KIcon::Small ), i18n( "Load" ), this );
	saveButton = new KPushButton( iconLoader.loadIconSet( "filesave", KIcon::Small ), i18n( "Save" ), this );
	removeButton = new KPushButton( iconLoader.loadIconSet( "editdelete", KIcon::Small ), i18n( "Remove" ), this );
	
	connect( loadButton, SIGNAL( clicked() ), this, SLOT( loadSlot() ) );
	connect( saveButton, SIGNAL( clicked() ), this, SLOT( saveSlot() ) );
	connect( removeButton, SIGNAL( clicked() ), this, SLOT( removeSlot() ) );

	removeButton->adjustSize();
	saveButton->resize( removeButton->size() );
	loadButton->resize( removeButton->size() );
	
	loadButton->setEnabled( false );
	removeButton->setEnabled( false );

	listView = new KListView( this );
	listView->setRootIsDecorated( false );
	listView->setSelectionModeExt( KListView::Single );
	listView->move( KWaveControl::SPACING, KWaveControl::SPACING );
	listView->addColumn( i18n( "Name" ) );
	listView->addColumn( i18n( "ESSID" ) );
	listView->addColumn( i18n( "IP" ) );
	listView->addColumn( i18n( "Gateway" ) );
	listView->addColumn( i18n( "Mode" ) );
	listView->addColumn( i18n( "Channel" ) );
	listView->addColumn( i18n( "Encryption" ) );
	listView->addColumn( i18n( "Key" ) );
	connect( listView, SIGNAL( selectionChanged() ), this, SLOT( selectionChangedSlot() ) );

	restoreProfiles();
}


KProfileWidget::~KProfileWidget()
{
}


/*!
    \fn KProfileWidget::resizeEvent ( QResizeEvent * )
 */
void KProfileWidget::resizeEvent ( QResizeEvent * )
{
	loadButton->move( this->width() - KWaveControl::SPACING - removeButton->width(), KWaveControl::SPACING );
	saveButton->move( this->width() - KWaveControl::SPACING - removeButton->width(), loadButton->y() + loadButton->height() + KWaveControl::SPACING );
	removeButton->move( this->width() - KWaveControl::SPACING - removeButton->width(), saveButton->y() + saveButton->height() + KWaveControl::SPACING );
	
	listView->resize( this->width() - loadButton->width() - 3 * KWaveControl::SPACING, this->height() - 2 * KWaveControl::SPACING );
}


/*!
	\fn KProfileWidget::loadSlot()
 */
void KProfileWidget::loadSlot()
{
	QString name = listView->selectedItem()->text( 0 );
	KGlobal::config()->setGroup( "kwavecontrol_profiles" );
	QStringList list = KGlobal::config()->readListEntry( name );
	if( list.size() >= 4 )
	{
		Profile profile;
		profile.essid = list[ 0 ];
		profile.ip = list[ 1 ];
		profile.mode = list[ 2 ].toInt();
		profile.frequency = list[ 3 ].toDouble();
		// just for compatibility
		if( list.size() >= 5 )
		{
			profile.gateway = list[ 4 ];
		}
		else
		{
			profile.gateway = "";
		}
		// just for compatibility
		if( list.size() >= 8 )
		{
			profile.security = list[ 5 ].toInt();
			profile.key = list[ 6 ];
			profile.keyIndex = list[ 7 ].toInt();
		}
		else
		{
			profile.security = -1;
			profile.key = "";
			profile.keyIndex = -1;
		}
		wWidget->setProfile( profile );
	}
}


/*!
	\fn KProfileWidget::saveSlot()
 */
void KProfileWidget::saveSlot()
{
	bool ok;
	QString name = KInputDialog::getText( i18n( "Profile name" ), i18n( "Enter name for new profile:" ), QString::null, &ok );
	if( ok && ( name != "kwavecontrol_profile_names" ) )
	{
		Profile profile = wWidget->getProfile();
/*		cout << profile.essid << endl;
		cout << profile.ip << endl;
		cout << profile.mode << endl;
		cout << profile.frequency << endl;*/
		
		QStringList list;
		list.push_back( profile.essid );
		list.push_back( profile.ip );
		list.push_back( QString::number( profile.mode ) );
		list.push_back( QString::number( profile.frequency ) );
		list.push_back( profile.gateway );
		list.push_back( QString::number( profile.security ) );
		list.push_back( profile.key );
		list.push_back( QString::number( profile.keyIndex ) );
		
		KGlobal::config()->setGroup( "kwavecontrol_profiles" );
		KGlobal::config()->writeEntry( name, list, ',', true, false, false );
		
		list = KGlobal::config()->readListEntry( "kwavecontrol_profile_names" );
		list.push_back( name );
		KGlobal::config()->writeEntry( "kwavecontrol_profile_names", list, ',', true, false, false );
	}
	
	restoreProfiles();
}


/*!
	\fn KProfileWidget::removeSlot()
 */
void KProfileWidget::removeSlot()
{
	QString name = listView->selectedItem()->text( 0 );
	
	int result = KMessageBox::questionYesNo( this, i18n( "Do you realy want to remove the profile" ) + " \"" + name + "\"?", i18n( "Remove profile" ) );
	if( result == KMessageBox::Yes )
	{
		KGlobal::config()->setGroup( "kwavecontrol_profiles" );
		KGlobal::config()->deleteEntry( name );
		
		QStringList profiles;
		KGlobal::config()->setGroup( "kwavecontrol_profiles" );
		profiles = KGlobal::config()->readListEntry( "kwavecontrol_profile_names" );
		profiles.remove( name );
		KGlobal::config()->writeEntry( "kwavecontrol_profile_names", profiles );

		listView->clearSelection();
	}
	restoreProfiles();
}


/*!
	\fn KProfileWidget::selectionChangedSlot()
 */
void KProfileWidget::selectionChangedSlot()
{
	loadButton->setEnabled( listView->selectedItem() );
	removeButton->setEnabled( listView->selectedItem() );
}


/*!
    \fn KProfileWidget::restoreProfiles()
 */
void KProfileWidget::restoreProfiles()
{
	listView->clear();

	QStringList profiles, profileValues;
	KGlobal::config()->setGroup( "kwavecontrol_profiles" );
	profiles = KGlobal::config()->readListEntry( "kwavecontrol_profile_names" );
	
	unsigned int i;
	for( i = 0; i < profiles.size(); i++ )
	{
		profileValues = KGlobal::config()->readListEntry( profiles[ i ] );
		KListViewItem* profileItem = new KListViewItem( listView, profiles[ i ] );

		profileItem->setText( 1, profileValues[ 0 ] );
		profileItem->setText( 2, profileValues[ 1 ] );

		// gateway: compatibility
		if( profileValues.size() >= 5 )
			profileItem->setText( 3, profileValues[ 4 ] );

		profileItem->setText( 4, QString( iw_operation_mode[ profileValues[ 2 ].toInt() ] ) );
		
		double channel = profileValues[ 3 ].toDouble();
		if( channel < WCard::KILO )
			profileItem->setText( 5, QString::number( channel ) );
		else
			profileItem->setText( 5, WWidget::frequency2QString( profileValues[ 3 ].toDouble() ) );

		if( profileValues.size() >= 8 )
		{
			switch( profileValues[ 5 ].toInt() )
			{
				case IW_ENCODE_DISABLED:
					profileItem->setText( 6, "off" );
					break;
				case IW_ENCODE_OPEN:
					profileItem->setText( 6, "open" );
					break;
				case IW_ENCODE_RESTRICTED:
					profileItem->setText( 6, "restricted" );
					break;
			}
			profileItem->setText( 7, profileValues[ 6 ] );
// 			profileItem->setText( 8, profileValues[ 7 ] );
		}


		listView->insertItem( profileItem );
	}
}


#include "kprofilewidget.moc"
