// LabPlot : Plot.cc

#include <math.h>
#include <iostream>
#include <kdebug.h>
#include <klocale.h>
#include <kconfig.h>
#include "Plot.h"
#include "Plot2DSurface.h"
#include "PlotQWT3D.h"

#include "parser_extern.h"

using namespace std;

//! general Plot class
Plot::Plot(Worksheet *w)
	: worksheet(w)
{
	kdDebug()<<"Plot::Plot()"<<endl;
	graphlist = new GraphList();

	KConfig *config = w->getMainWin()->Config();
	aspect_enabled = config->readBoolEntry("AspectRatio",false);

	bgcolor = config->readColorEntry("BackgroundColor",&Qt::white);
	gbgcolor = config->readColorEntry("GraphBackgroundColor",&Qt::white);

	// title (richtext title)
	if(worksheet==0) kdDebug()<<"no Worksheet defined!"<<endl;
	QFont font = worksheet->getMainWin()->defaultFont();
	title = new Label(i18n("Title"),font,QColor(Qt::black));
	title->setPosition(0.4,0.04);

	font.setPointSize((int)(0.4*font.pointSize()));
	legend.setFont(font);
	
	position.setPoint(config->readDoubleNumEntry("Position X",0.0),config->readDoubleNumEntry("Position Y",0.0));
	size.setPoint(config->readDoubleNumEntry("Size X",1.0),config->readDoubleNumEntry("Size Y",1.0));
	p1.setPoint(.15,.15);
	p2.setPoint(.95,.85);

	baseline = config->readDoubleNumEntry("YBaseline",0);
	baseline_enabled = config->readBoolEntry("YBaselineEnabled",false);
	xbaseline = config->readDoubleNumEntry("XBaseline",0);
	xbaseline_enabled = config->readBoolEntry("XBaselineEnabled",false);

	region= new LRange(config->readDoubleNumEntry("RegionMin",0.0),config->readDoubleNumEntry("RegionMax",0.0));
	region_enabled = config->readBoolEntry("RegionEnabled",false);

	transparent = config->readBoolEntry("Transparent",false);
	clipoffset = config->readNumEntry("ClipOffset",10);

	marks_enabled = config->readBoolEntry("MarkerEnabled",false);
	markx = new LRange(config->readDoubleNumEntry("MarkerXMin",0.0),config->readDoubleNumEntry("MarkerXMax",1.0));
	marky = new LRange(config->readDoubleNumEntry("MarkerYMin",0.0),config->readDoubleNumEntry("MarkerYMax",1.0));

	// fill
	fill_enabled = config->readBoolEntry("FillEnabled",false);
	filltype = config->readNumEntry("FillType",0);
	fillg1 = config->readNumEntry("FillG1",1);
	fillg2 = config->readNumEntry("FillG2",2);
	fillbrush = QBrush(config->readColorEntry("FillColor",&Qt::red),
		(Qt::BrushStyle)config->readNumEntry("FillBrush",Qt::SolidPattern));
	kdDebug()<<"Plot OK"<<endl;
}

// read default axis settings
void Plot::readAxisSettings(Axis *a, int type, int item) {
	KConfig *config = worksheet->getMainWin()->Config();
	config->setGroup( "Axes" );
	
	// default values for different plots
	bool e=true, borderenabled=true, gridenabled=false;
	switch (type) {
	case P2D:
		if(item>1) {
			e = false;
			borderenabled = false;
		};break;
	case P3D:
		if(item>2) {
			e = false;
		}; break;
	case PPIE: case PTERNARY:
		gridenabled = true;
		break;
	default:break;
	}
	
	QString entry = QString("PlotType %1 Axis %2 ").arg(type).arg(item); 

	a->Enable(config->readBoolEntry(entry+"Enabled",e));
	a->setPosition(config->readNumEntry(entry+"Position"));
	a->setScale((TScale) (config->readNumEntry(entry+"Scale",LINEAR)));
	a->setScaling(config->readNumEntry(entry+"Scaling",1));
	a->setShift(config->readNumEntry(entry+"Shift"));
	// dont set range -> automatic

	a->getLabel()->readSettings(config,entry);
	
	a->setTickPos(config->readNumEntry(entry+"TickPosition"));
	a->setTickType(config->readNumEntry(entry+"TickStyle",1));
	a->enableMajorTicks(config->readBoolEntry(entry+"MajorTicksEnabled",true));
	a->setMajorTicks(config->readDoubleNumEntry(entry+"MajorTicks",-1));
	a->setMajorTickWidth(config->readNumEntry(entry+"MajorTickWidth",1));
	a->enableMinorTicks(config->readBoolEntry(entry+"MinorTicksEnabled",true));
	a->setMinorTicks(config->readNumEntry(entry+"MinorTicks",3));
	a->setMinorTickWidth(config->readNumEntry(entry+"MinorTickWidth",1));
	a->setTickColor(config->readColorEntry(entry+"TickColor",&Qt::black));

	if(type == PQWT3D) {
		a->setMajorTickLength(config->readDoubleNumEntry(entry+"MajorTickLength",0.05));
		a->setMinorTickLength(config->readDoubleNumEntry(entry+"MinorTickLength",0.05));
	}

	a->enableTickLabel(config->readBoolEntry(entry+"TickLabelEnabled",true));
	QFont font(QString("Adobe Times"),12);
	a->setTickLabelFont(config->readFontEntry(entry+"TickLabelFont",&font));
	a->setTickLabelColor(config->readColorEntry(entry+"TickLabelColor",&Qt::black));
	a->setTickLabelFormat((TFormat)config->readNumEntry(entry+"TickLabelFormat",(int)AUTO));
	a->setDateTimeFormat(config->readEntry(entry+"DateTimeFormat","auto"));
	a->setTickLabelPrecision(config->readNumEntry(entry+"TickLabelPrecision",3));
	a->setTickLabelPosition(config->readNumEntry(entry+"TickLabelPosition",15));
	a->setTickLabelPrefix(config->readEntry(entry+"TickLabelPrefix"));
	a->setTickLabelSuffix(config->readEntry(entry+"TickLabelSuffix"));
	a->setTickLabelRotation(config->readNumEntry(entry+"TickLabelRotation"));
	
	a->enableMajorGrid(config->readBoolEntry(entry+"MajorGridEnabled",gridenabled));
	a->enableMinorGrid(config->readBoolEntry(entry+"MinorGridEnabled",gridenabled));
	a->setMajorGridWidth(config->readNumEntry(entry+"MajorGridWidth",1));
	a->setMajorGridType((Qt::PenStyle)config->readNumEntry(entry+"MajorGridStyle",(int)Qt::DashLine));
	a->setMajorGridColor(config->readColorEntry(entry+"MajorGridColor",&Qt::black));
	a->setMinorGridWidth(config->readNumEntry(entry+"MinorGridWidth",1));
	a->setMinorGridType((Qt::PenStyle)config->readNumEntry(entry+"MinorGridStyle",(int)Qt::DotLine));
	a->setMinorGridColor(config->readColorEntry(entry+"MinorGridColor",&Qt::black));
	
	a->enableBorder(config->readBoolEntry(entry+"BorderEnabled",borderenabled));
	a->setBorderColor(config->readColorEntry(entry+"BorderColor",&Qt::black));
	a->setBorderWidth(config->readNumEntry(entry+"BorderWidth",1));
}

void Plot::sortPoints(QPointArray pa,int s, int e) {
	// bubble sort
/*	QPoint tmp;
	for(unsigned int i=a;i<e;i++) {
		for(unsigned int j=0;j<pa.size()-1;j++) {
			if(pa[j].x()>pa[j+1].x()) {
				tmp=pa[j];
				pa[j]=pa[j+1];
				pa[j+1]=tmp;
			}
		}
	}
*/
	// quick sort
	if(e>s+1) {
		QPoint tmp, mid = pa[(s+e)/2];

		int i=s, j=e;
		while(i<j) {
			while(pa[i].x() < mid.x()) i++;
			while(mid.x() < pa[j].x()) j--;

			if(i<j) {
				tmp = pa[i];
				pa[i] = pa[j];
				pa[j] = tmp;
				i++;j--;
			}
		}

		sortPoints(pa,s,j);
		sortPoints(pa,i,e);
	}
}

void Plot::drawStyle(QPainter *p, Style *style, Symbol *symbol, QPointArray pa, int xmin, int xmax, int ymin, int ymax) {
	kdDebug()<<"Plot::drawStyle()"<<endl;
	bool filled = style->isFilled();
	QColor c = style->FillColor();
 	QPen pen( style->Color(), style->Width(),(Qt::PenStyle)style->PenStyle() );
	p->setPen(pen);
	QBrush brush(c,(Qt::BrushStyle)style->Brush());

	// calculate baseline
	double min = actrange[1].rMin();
	double max = actrange[1].rMax();
	double minx = actrange[0].rMin();
	double maxx = actrange[0].rMax();

	int basey = ymax - (int) ((baseline-min)/(max-min)*(double)(ymax-ymin));
	int basex = xmin + (int) ((xbaseline-minx)/(maxx-minx)*(double)(xmax-xmin));
	int bw = style->BoxWidth();
	int rmin = xmin + (int) ((region->rMin()-minx)/(maxx-minx)*(double)(xmax-xmin));
	int rmax = xmin + (int) ((region->rMax()-minx)/(maxx-minx)*(double)(xmax-xmin));
	//kdDebug()<<"BASEX = "<<basex<<endl;
	//kdDebug()<<"BASEY = "<<basey<<endl;

//	kdDebug()<<"POINTARRAY "<<i<<'='<<pa[i].x()<<' '<<pa[i].y()<<endl;

	switch(style->Type()) {
	case 0:	// line
		if (filled) {
			p->setPen(Qt::NoPen);
			p->setBrush(brush);
		
			QPointArray fillpa(pa.size()+2);
			if(fabs((double)(rmax-rmin)) > 0) {	//draw only region
				int index=0,newindex=1;
				while(pa[index].x() < rmin)
					index++;
				fillpa[0] = QPoint(pa[index].x(),basey);
				while(pa[index].x() < rmax)
					fillpa[newindex++] = pa[index++];
				fillpa[newindex] = QPoint(pa[index-1].x(),basey);
				fillpa.resize(newindex+1);
			}
			else {
				fillpa[0] = QPoint(pa[0].x(),basey);
				for(unsigned int i=0;i<pa.size();i++)
					fillpa[i+1]=pa[i];
				fillpa[pa.size()+1]=QPoint(pa[pa.size()-1].x(),basey);
			}
			p->drawPolygon(fillpa);
			p->setPen(pen);
		}
		p->drawPolyline(pa);
		break;
	case 2:	{ // steps	: only for 2d
		QPointArray stepspa(2*pa.size());
		// start
		stepspa[0]=pa[0];		
		stepspa[1]=QPoint((pa[0].x()+pa[1].x())/2,pa[0].y());
		// end
		stepspa[2*pa.size()-2]=QPoint((int)((pa[pa.size()-2].x()+pa[pa.size()-1].x())/2.0),pa[pa.size()-1].y());
		stepspa[2*pa.size()-1]=pa[pa.size()-1];
		for(unsigned int i=1;i<pa.size()-1;i++) {
			stepspa[2*i]=QPoint((int)((pa[i-1].x()+pa[i].x())/2.0),pa[i].y());		// left
			stepspa[2*i+1]=QPoint((int)((pa[i+1].x()+pa[i].x())/2.0),pa[i].y());		// right
		}
		if (filled) {
			QPointArray fillpa(stepspa.size()+2);
			p->setPen(Qt::NoPen);
			p->setBrush(brush);
			if(fabs((double)(rmax-rmin)) > 0) {	// draw only region
				int index=0,newindex=1;
				while(stepspa[index].x() < rmin)
					index++;
				fillpa[0] = QPoint(stepspa[index].x(),basey);
				while(stepspa[index].x() < rmax)
					fillpa[newindex++] = stepspa[index++];
				fillpa[newindex] = QPoint(stepspa[index-1].x(),basey);
				fillpa.resize(newindex+1);
			}
			else {
				fillpa[0]=QPoint(stepspa[0].x(),basey);
				for(unsigned int i=0;i<stepspa.size();i++)
					fillpa[i+1]=stepspa[i];
				fillpa[stepspa.size()+1]=QPoint(stepspa[stepspa.size()-1].x(),basey);
			}
			p->drawPolygon(fillpa);
			p->setPen(pen);
		}
		p->drawPolyline(stepspa);
		}; break;
	case 3:	// boxes	: only for 2d
		for(unsigned int i=0;i<pa.size();i++) {
			QPointArray boxpa(4);
			
			if(style->AutoBoxWidth()) {
				if(i==0) {
					boxpa[0]=QPoint(pa[i].x()-(int)((pa[i+1].x()-pa[i].x())/2.0),pa[i].y());
					boxpa[1]=QPoint(pa[i].x()+(int)((pa[i+1].x()-pa[i].x())/2.0),pa[i].y());
					boxpa[2]=QPoint(pa[i].x()+(int)((pa[i+1].x()-pa[i].x())/2.0),basey);
					boxpa[3]=QPoint(pa[i].x()-(int)((pa[i+1].x()-pa[i].x())/2.0),basey);
				}
				else if(i==pa.size()-1) {
					boxpa[0]=QPoint(pa[i].x()-(int)((pa[i].x()-pa[i-1].x())/2.0),pa[i].y());
					boxpa[1]=QPoint(pa[i].x()+(int)((pa[i].x()-pa[i-1].x())/2.0),pa[i].y());
					boxpa[2]=QPoint(pa[i].x()+(int)((pa[i].x()-pa[i-1].x())/2.0),basey);
					boxpa[3]=QPoint(pa[i].x()-(int)((pa[i].x()-pa[i-1].x())/2.0),basey);
				}
				else {
					boxpa[0]=QPoint(pa[i].x()-(int)((pa[i].x()-pa[i-1].x())/2.0),pa[i].y());
					boxpa[1]=QPoint(pa[i].x()+(int)((pa[i+1].x()-pa[i].x())/2.0),pa[i].y());
					boxpa[2]=QPoint(pa[i].x()+(int)((pa[i+1].x()-pa[i].x())/2.0),basey);
					boxpa[3]=QPoint(pa[i].x()-(int)((pa[i].x()-pa[i-1].x())/2.0),basey);
				}
			}
			else {
				boxpa[0]=QPoint(pa[i].x()-bw/2,pa[i].y());
				boxpa[1]=QPoint(pa[i].x()+bw/2,pa[i].y());
				boxpa[2]=QPoint(pa[i].x()+bw/2,basey);
				boxpa[3]=QPoint(pa[i].x()-bw/2,basey);
			}

			p->setBrush(Qt::NoBrush);
			if (filled)
				p->setBrush(brush);
			p->drawPolygon(boxpa);
		}
		break;
	case 4:	// impulses	: only for 2d
		for (unsigned int i=0;i<pa.size();i++)
			p->drawLine(pa[i].x(),pa[i].y(),pa[i].x(),basey);
		break;
	case 5:	// yboxes	:only for 2d
		for(unsigned int i=0;i<pa.size();i++) {
			QPointArray boxpa(4);
			
			if(style->AutoBoxWidth()) {
				if(i==0) {
					boxpa[0]=QPoint(pa[i].x(),(int)(pa[i].y()-(pa[i+1].y()-pa[i].y())/2.0));
					boxpa[1]=QPoint(pa[i].x(),(int)(pa[i].y()+(pa[i+1].y()-pa[i].y())/2.0));
					boxpa[2]=QPoint(basex,pa[i].y()+(int)((pa[i+1].y()-pa[i].y())/2.0));
					boxpa[3]=QPoint(basex,pa[i].y()- (int)((pa[i+1].y()-pa[i].y())/2.0));
				}
				else if(i==pa.size()-1) {
					boxpa[0]=QPoint(pa[i].x(),pa[i].y()-(int)((pa[i].y()-pa[i-1].y())/2.0));
					boxpa[1]=QPoint(pa[i].x(),pa[i].y()+(int)((pa[i].y()-pa[i-1].y())/2.0));
					boxpa[2]=QPoint(basex,pa[i].y()+(int)((pa[i].y()-pa[i-1].y())/2.0));
					boxpa[3]=QPoint(basex, pa[i].y()-(int)((pa[i].y()-pa[i-1].y())/2.0));
				}
				else {
					boxpa[0]=QPoint(pa[i].x(),pa[i].y()-(int)((pa[i].y()-pa[i-1].y())/2.0));
					boxpa[1]=QPoint(pa[i].x(),pa[i].y()+(int)((pa[i+1].y()-pa[i].y())/2.0));
					boxpa[2]=QPoint(basex,pa[i].y()+(int)((pa[i+1].y()-pa[i].y())/2.0));
					boxpa[3]=QPoint(basex, pa[i].y()-(int)((pa[i].y()-pa[i-1].y())/2.0));
				}
			}
			else {
				boxpa[0]=QPoint(pa[i].x(),pa[i].y()-bw/2);
				boxpa[1]=QPoint(pa[i].x(),pa[i].y()+bw/2);
				boxpa[2]=QPoint(basex, pa[i].y()+bw/2);
				boxpa[3]=QPoint(basex, pa[i].y()-bw/2);
			}
			
			p->setBrush(Qt::NoBrush);
			if (filled)
				p->setBrush(brush);
			p->drawPolygon(boxpa);
		}
		break;
	}

	// draw symbol
	for (unsigned int i=0;i<pa.size();i++)
		symbol->draw(p,pa[i].x(),pa[i].y());
}


//! draw errorbars for x-y-dy, x-y-dx-dy and x-y-dy1-dy2
// at (x,y) from xleft to xright and ybottom to ytop
void Plot::drawErrorBar(QPainter *p, QPointArray pa, QPointArray hpa, QPointArray vpa) {
	for(unsigned int i=0;i<pa.size();i++) {
		int x=pa[i].x(), y=pa[i].y();
		int xleft=hpa[i].x(), xright=hpa[i].y();
		int ytop=vpa[i].x(),ybottom=vpa[i].y();
		
		if (xleft != x) {
			p->drawLine(xleft,y,x,y);
			p->drawLine(xleft,y-2,xleft,y+2);
		}
		if (xright != x) {
			p->drawLine(x,y,xright,y);
			p->drawLine(xright,y-2,xright,y+2);
		}
		if (ytop != y) {
			p->drawLine(x,y,x,ytop);
			p->drawLine(x-2,ytop,x+2,ytop);
		}
		if (ybottom != y) {
			p->drawLine(x,y,x,ybottom);
			p->drawLine(x-2,ybottom,x+2,ybottom);
		}
	}
}

QDomElement Plot::savePlotXML(QDomDocument doc) {
	QDomElement plottag = doc.createElement( "Plot" );
	plottag.setAttribute("type",QString::number(type));

	QDomElement tag = doc.createElement( "Background" );
   	plottag.appendChild( tag );
  	QDomText t = doc.createTextNode( bgcolor.name() );
    	tag.appendChild( t );
	tag = doc.createElement( "GraphBackground" );
   	plottag.appendChild( tag );
  	t = doc.createTextNode( gbgcolor.name() );
    	tag.appendChild( t );
	tag = doc.createElement( "Transparent" );
   	plottag.appendChild( tag );
  	t = doc.createTextNode( QString::number(transparent) );
    	tag.appendChild( t );
	tag = doc.createElement( "ClipOffset" );
   	plottag.appendChild( tag );
  	t = doc.createTextNode( QString::number(clipoffset) );
    	tag.appendChild( t );

	tag = doc.createElement( "XActRange" );
	tag.setAttribute("min",QString::number(actrange[0].rMin()));
	tag.setAttribute("max",QString::number(actrange[0].rMax()));
   	plottag.appendChild( tag );
	tag = doc.createElement( "YActRange" );
	tag.setAttribute("min",QString::number(actrange[1].rMin()));
	tag.setAttribute("max",QString::number(actrange[1].rMax()));
   	plottag.appendChild( tag );
	tag = doc.createElement( "ZActRange" );
	tag.setAttribute("min",QString::number(actrange[2].rMin()));
	tag.setAttribute("max",QString::number(actrange[2].rMax()));
   	plottag.appendChild( tag );
	
	tag = doc.createElement( "Baseline" );
	tag.setAttribute("x",QString::number(xbaseline));
	tag.setAttribute("y",QString::number(baseline));
	tag.setAttribute("x_enabled",QString::number(xbaseline_enabled));
	tag.setAttribute("y_enabled",QString::number(baseline_enabled));
   	plottag.appendChild( tag );

	tag = doc.createElement( "Region" );
	tag.setAttribute("min",QString::number(region->rMin()));
	tag.setAttribute("max",QString::number(region->rMax()));
	tag.setAttribute("enabled",QString::number(region_enabled));
   	plottag.appendChild( tag );

	tag = doc.createElement( "Position" );
	tag.setAttribute("x",QString::number(position.X()));
	tag.setAttribute("y",QString::number(position.Y()));
   	plottag.appendChild( tag );
	tag = doc.createElement( "Size" );
	tag.setAttribute("x",QString::number(size.X()));
	tag.setAttribute("y",QString::number(size.Y()));
   	plottag.appendChild( tag );
	tag = doc.createElement( "PlotArea" );
	tag.setAttribute("xmin",QString::number(p1.X()));
	tag.setAttribute("xmax",QString::number(p2.X()));
	tag.setAttribute("ymin",QString::number(p1.Y()));
	tag.setAttribute("ymax",QString::number(p2.Y()));
   	plottag.appendChild( tag );

	tag = doc.createElement( "AspectRatio" );
   	plottag.appendChild( tag );
  	t = doc.createTextNode( QString::number(aspect_enabled) );
    	tag.appendChild( t );

	tag = doc.createElement( "Marks" );
	tag.setAttribute("enabled",QString::number(marks_enabled));
	tag.setAttribute("xmin",QString::number(markx->rMin()));
	tag.setAttribute("xmax",QString::number(markx->rMax()));
	tag.setAttribute("ymin",QString::number(marky->rMin()));
	tag.setAttribute("ymax",QString::number(marky->rMax()));
   	plottag.appendChild( tag );

	tag = doc.createElement( "Fill" );
	tag.setAttribute("enabled",QString::number(fill_enabled));
	tag.setAttribute("type",QString::number(filltype));
	tag.setAttribute("firstgraph",QString::number(fillg1));
	tag.setAttribute("secondgraph",QString::number(fillg2));
	tag.setAttribute("color",fillbrush.color().name());
	tag.setAttribute("brush",QString::number(fillbrush.style()));
   	plottag.appendChild( tag );

	tag = title->saveXML(doc);
   	plottag.appendChild( tag );
	tag = legend.saveXML(doc);
   	plottag.appendChild( tag );

	// save aditional plot stuff
	saveXML(doc,plottag);

	graphlist->saveXML(doc,plottag);

	return plottag;
}

void Plot::save(QTextStream *t) {
	kdDebug()<<"Plot::save()"<<endl;
	*t<<bgcolor.name()<<endl;
	*t<<gbgcolor.name()<<endl;
	*t<<transparent<<endl;
	*t<<clipoffset<<endl;

	for (int i=0;i<3;i++)
		*t<<actrange[i].rMin()<<' '<<actrange[i].rMax()<<endl;

	// save baselines
	*t<<baseline<<' '<<baseline_enabled<<' '<<xbaseline<<' '<<xbaseline_enabled<<endl;

	//save region
	*t<<region->rMin()<<' '<<region->rMax()<<' '<<region_enabled<<endl;

	*t<<position.X()<<' '<<position.Y()<<endl;
	*t<<size.X()<<' '<<size.Y()<<endl;
	*t<<p1.X()<<' '<<p1.Y()<<endl;
	*t<<p2.X()<<' '<<p2.Y()<<endl;
	
	*t<<aspect_enabled<<endl;
	
	//marks
	*t<<marks_enabled<<' '<<markx->rMin()<<' '<<markx->rMax()<<' '<<marky->rMin()<<' '<<marky->rMax()<<endl;

	// fill
	*t<<(int)fill_enabled<<' '<<filltype<<endl;
	*t<<fillg1<<' '<<fillg2<<endl;
	*t<<fillbrush.color().name()<<' '<<fillbrush.style()<<endl;

	title->save(t);
	legend.save(t);
	saveAxes(t);

	if (type == PSURFACE) {
		Plot2DSurface *plot = (Plot2DSurface *)this;
		*t<<plot->densityEnabled()<<' '<<plot->contourEnabled()<<endl;
		*t<<plot->Number()<<' '<<endl;
		*t<<plot->ContourColor().name()<<endl;
		*t<<plot->Mesh()<<' '<<plot->ColoredContour()<<' '<<plot->Brush()<<' ';
		*t<<plot->Relative()<<' '<<plot->Threshold()<<endl;
#ifdef HAVE_GL
		Qwt3D::ColorVector cv = plot->getColorVector();
		*t<<cv.size()<<endl;
		for(unsigned int i=0;i<cv.size();i++)
			*t<<cv[i].r<<' '<<cv[i].g<<' '<<cv[i].b<<' '<<cv[i].a<<endl;
#endif
	}
#ifdef HAVE_GL
	else if (type == PQWT3D) {
		PlotQWT3D *plot = (PlotQWT3D *)this;
		
		*t<<(int) plot->PlotStyle()<<endl;
		*t<<(int) plot->CoordinateStyle()<<endl;
		*t<<(int) plot->FloorStyle()<<endl;
		*t<<(int) plot->aspectRatio()<<endl;
		*t<<plot->isolines()<<endl;
		*t<<(int) plot->mouseDisabled()<<endl;
		for(int i=0;i<12;i++)
			*t<<plot->getAxis(i)->majorTickLength()<<' '<<plot->getAxis(i)->minorTickLength()<<endl;
		*t<<(int) plot->Resolution()<<endl;

		Qwt3D::ColorVector cv = plot->getColorVector();
		*t<<cv.size()<<endl;
		for(unsigned int i=0;i<cv.size();i++)
			*t<<cv[i].r<<' '<<cv[i].g<<' '<<cv[i].b<<' '<<cv[i].a<<endl;
	}
#endif

	// dump graphs
	QProgressDialog *progress = new QProgressDialog( i18n("Saving Project ..."), i18n("Cancel"), 100,
		worksheet->getMainWin(), "progress", true );
	progress->setMinimumDuration(1000);

	unsigned int i;
	for (i=0;i < graphlist->Number();i++) {
		switch (graphlist->getStruct(i)) {
		case GRAPH2D:
			*t<<"Graph2D "<<i<<endl;
			graphlist->getGraph2D(i)->save(t,progress);
			break;
		case GRAPH3D:
			*t<<"Graph3D "<<i<<endl;
			graphlist->getGraph3D(i)->save(t,progress);
			break;
		case GRAPHM:
			*t<<"GraphM "<<i<<endl;
			graphlist->getGraphM(i)->save(t,progress);
			break;
		case GRAPH4D:
			*t<<"Graph4D "<<i<<endl;
			graphlist->getGraph4D(i)->save(t,progress);
			break;
		case GRAPHIMAGE:
			*t<<"GraphIMAGE "<<i<<endl;
			graphlist->getGraphIMAGE(i)->save(t);
			break;
		case GRAPHL:
			*t<<"GraphL "<<i<<endl;
			graphlist->getGraphL(i)->save(t,progress);
			break;
		default: break;
		}
	}
	*t<<"EOG "<<i<<endl;
}

#ifdef HAVE_GL
Qwt3D::ColorVector Plot::convertOldColormap(int index) {
	Qwt3D::ColorVector cv;
	cv.clear();
		
	Qwt3D::RGBA rgb;

	switch(index) {
	case 0: 
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			i>127?rgb.r = (2*(i-128))/255.0:rgb.r = 0;
			rgb.g = 0;
			i>127?rgb.b = 0:rgb.b = (2*(127-i))/255.0;
			cv.push_back(rgb);
		}
		break;
	case 1:
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			rgb.r=0;
			i>127?rgb.g = (2*(i-128))/255.0:rgb.g = 0;
			i>127?rgb.b = 0:rgb.b = (2*(127-i))/255.0;
			cv.push_back(rgb);
		}
		break;
	case 2:
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			i>127?rgb.r = 0:rgb.r = (2*(127-i))/255.0;
			rgb.g=0;
			i>127?rgb.b = (2*(i-128))/255.0:rgb.b=0;
			cv.push_back(rgb);
		}
		break;
	case 3:
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			rgb.r=0;
			i>127?rgb.g = 0:rgb.g = (2*(127-i))/255.0;
			i>127?rgb.b = (2*(i-128))/255.0:rgb.b=0;
			cv.push_back(rgb);
		}
		break;
	case 4:
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			i>127?rgb.r = 0:rgb.r = (2*(127-i))/255.0;
			i>127?rgb.g = (2*(i-128))/255.0:rgb.g=0;
			rgb.b=0;
			cv.push_back(rgb);
		}
		break;
	case 5:
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			rgb.r=i/255.;
			rgb.g=0;
			rgb.b=0;
			cv.push_back(rgb);
		}
		break;
	case 6:
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			rgb.r=0;
			rgb.g=i/255.;
			rgb.b=0;
			cv.push_back(rgb);
		}
		break;
	case 7:
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			rgb.r=0;
			rgb.g=0;
			rgb.b=i/255.;
			cv.push_back(rgb);
		}
		break;
	case 8:
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			rgb.r=i/255.;
			rgb.g=i/255.;
			rgb.b=i/255.;
			cv.push_back(rgb);
		}
		break;
	case 9:
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			rgb.r=i/255.;
			rgb.g=i/255.;
			rgb.b=0;
			cv.push_back(rgb);
		}
		break;
	case 10:
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			rgb.r=i/255.;
			rgb.g=0;
			rgb.b=i/255.;
			cv.push_back(rgb);
		}
		break;
	case 11:
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			rgb.r=0;
			rgb.g=i/255.;
			rgb.b=i/255.;
			cv.push_back(rgb);
		}
		break;
	case 12:
		for(int i=0;i<255;i++) {
			QColor c = QColor(255-i,255,255,QColor::Hsv);
			rgb.a = 1;
			rgb.r=c.red()/255.;
			rgb.g=c.green()/255.;
			rgb.b=c.blue()/255.;
			cv.push_back(rgb);
		}
		break;
	case 13:
		for(int i=0;i<255;i++) {
			QColor c = QColor(255-i,255,200,QColor::Hsv);
			rgb.a = 1;
			rgb.r=c.red()/255.;
			rgb.g=c.green()/255.;
			rgb.b=c.blue()/255.;
			cv.push_back(rgb);
		}
		break;
	case 14:
		for(int i=0;i<255;i++) {
			QColor c = QColor(255-i,100,255,QColor::Hsv);
			rgb.a = 1;
			rgb.r=c.red()/255.;
			rgb.g=c.green()/255.;
			rgb.b=c.blue()/255.;
			cv.push_back(rgb);
		}
		break;
	case 15:
		for(int i=0;i<255;i++) {
			QColor c = QColor(255-i,100,200,QColor::Hsv);
			rgb.a = 1;
			rgb.r=c.red()/255.;
			rgb.g=c.green()/255.;
			rgb.b=c.blue()/255.;
			cv.push_back(rgb);
		}
		break;
	case 16:
		for(int i=0;i<255;i++) {
			QColor c = QColor(i,255,255,QColor::Hsv);
			rgb.a = 1;
			rgb.r=c.red()/255.;
			rgb.g=c.green()/255.;
			rgb.b=c.blue()/255.;
			cv.push_back(rgb);
		}
		break;
	case 17:
		for(int i=0;i<255;i++) {
			QColor c = QColor(i,255,200,QColor::Hsv);
			rgb.a = 1;
			rgb.r=c.red()/255.;
			rgb.g=c.green()/255.;
			rgb.b=c.blue()/255.;
			cv.push_back(rgb);
		}
		break;
	case 18:
		for(int i=0;i<255;i++) {
			QColor c = QColor(i,100,255,QColor::Hsv);
			rgb.a = 1;
			rgb.r=c.red()/255.;
			rgb.g=c.green()/255.;
			rgb.b=c.blue()/255.;
			cv.push_back(rgb);
		}
		break;
	case 19:
		for(int i=0;i<255;i++) {
			QColor c = QColor(i,100,200,QColor::Hsv);
			rgb.a = 1;
			rgb.r=c.red()/255.;
			rgb.g=c.green()/255.;
			rgb.b=c.blue()/255.;
			cv.push_back(rgb);
		}
		break;
	case 20:
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			rgb.r=(255.-i)/255.;
			rgb.g=(255.-i)/255.;
			rgb.b=(255.-i)/255.;
			cv.push_back(rgb);
		}
		break;
	}
	
	return cv;
}
#endif

void Plot::openPlotXML(QDomNode node) {
	while(!node.isNull()) {
		QDomElement e = node.toElement();
//		kdDebug()<<"PLOT TAG = "<<e.tagName()<<endl;
//		kdDebug()<<"PLOT TEXT = "<<e.text()<<endl;

		if(e.tagName() == "Background")
			bgcolor = QColor(e.text());
		else if(e.tagName() == "GraphBackground")
			gbgcolor = QColor(e.text());
		else if(e.tagName() == "Transparent")
			transparent = (bool) e.text().toInt();
		else if(e.tagName() == "ClipOffset")
			clipoffset = e.text().toInt();

		else if(e.tagName() == "XActRange")
			actrange[0].setRange(e.attribute("min").toDouble(),e.attribute("max").toDouble());
		else if(e.tagName() == "YActRange")
			actrange[1].setRange(e.attribute("min").toDouble(),e.attribute("max").toDouble());
		else if(e.tagName() == "ZActRange")
			actrange[2].setRange(e.attribute("min").toDouble(),e.attribute("max").toDouble());

		else if(e.tagName() == "Baseline") {
			xbaseline_enabled = (bool) e.attribute("x_enabled").toInt();
			baseline_enabled = (bool) e.attribute("y_enabled").toInt();
			xbaseline = e.attribute("x").toDouble();
			baseline = e.attribute("y").toDouble();
		}
		else if(e.tagName() == "Region") {
			region_enabled = (bool) e.attribute("enabled").toInt();
			region->setRange(e.attribute("min").toDouble(),e.attribute("max").toDouble());
		}
		else if(e.tagName() == "Position")
			position.setPoint(e.attribute("x").toDouble(),e.attribute("y").toDouble());
		else if(e.tagName() == "Size")
			size.setPoint(e.attribute("x").toDouble(),e.attribute("y").toDouble());
		else if(e.tagName() == "PlotArea") {
			p1.setPoint(e.attribute("xmin").toDouble(),e.attribute("xmax").toDouble());
			p2.setPoint(e.attribute("ymin").toDouble(),e.attribute("ymax").toDouble());
		}
		else if(e.tagName() == "AspectRatio")
			aspect_enabled = (bool) e.text().toInt();
		else if(e.tagName() == "Marks") {
			marks_enabled = (bool) e.attribute("enabled").toInt();
			markx->setRange(e.attribute("xmin").toDouble(),e.attribute("ymin").toDouble());
			marky->setRange(e.attribute("xmax").toDouble(),e.attribute("ymax").toDouble());
		}
		else if(e.tagName() == "Fill") {
			fill_enabled = (bool) e.attribute("enabled").toInt();
			filltype = e.attribute("type").toInt();
			fillg1 = e.attribute("firstgraph").toInt();
			fillg2 = e.attribute("secondgraph").toInt();
			fillbrush = QBrush(e.attribute("color"),(Qt::BrushStyle) e.attribute("brush").toInt());
		}
		else if(e.tagName() == "Label")
			title->openXML(e.firstChild());		
		else if(e.tagName() == "Legend")
			legend.openXML(e.firstChild());
		else if(e.tagName() == "Graph") {
			switch(e.attribute("type").toInt()) {
			case GRAPH2D: {
				Graph2D *g = new Graph2D();
				g->openXML(e.firstChild());
				worksheet->addGraph2D(g,type);
				}; break;
			case GRAPH3D: {
				Graph3D *g = new Graph3D();
				g->openXML(e.firstChild());
				worksheet->addGraph3D(g,type);
				}; break;
			case GRAPH4D: {
				Graph4D *g = new Graph4D();
				g->openXML(e.firstChild());
				worksheet->addGraph4D(g);
				}; break;
			case GRAPHM: {
				GraphM *g = new GraphM();
				g->openXML(e.firstChild());
				worksheet->addGraphM(g,type);
				}; break;
			case GRAPHIMAGE: {
				GraphIMAGE *g = new GraphIMAGE();
				g->openXML(e.firstChild());
				worksheet->addGraphIMAGE(g);
				}; break;
			case GRAPHL: {
				GraphL *g = new GraphL();
				g->openXML(e.firstChild());
				worksheet->addGraphL(g,type);
				}; break;
			}
		}
		
		// plot type stuff
		openXML(e);

		node = node.nextSibling();
	}
}

void Plot::open(QTextStream *t, int version) {
	kdDebug()<<"Plot::open()"<<endl;

	QString family, color;
	double x,y;

	*t>>color;
	bgcolor = QColor(color);
	if(version > 3) {
		*t>>color;
		gbgcolor = QColor(color);

		// not used in newer versions
		if (version<11)
			*t>>x>>x>>x>>x;
		else {
			int tmp;
			*t>>tmp;
			transparent = (bool) tmp;
			if(version>13) {
				*t>>clipoffset;
			}
		}

		for (int i=0;i<3;i++) {
			*t>>x>>y;	// means min / max
			actrange[i].setMin(x);
			actrange[i].setMax(y);
			kdDebug()<<"	ActRange"<<i<<" min/max = "<<x<<' '<<y<<endl;
		}
		
		if(version>18) {
			int e1,e2;
			*t>>baseline>>e1>>xbaseline>>e2;
			baseline_enabled=(bool)e1;
			xbaseline_enabled=(bool)e2;
		}

		//region
		if (version>9) {
			int e;
			*t>>x>>y>>e;
			region->setMin(x);
			region->setMax(y);
			region_enabled=e;
			kdDebug()<<"Region : "<<x<<' '<<y<<' '<<e<<endl;
		}

		// position & size
		if(version>10) {
			*t>>x>>y;
			position.setPoint(x,y);
			*t>>x>>y;
			size.setPoint(x,y);
			*t>>x>>y;
			p1.setPoint(x,y);
			*t>>x>>y;
			p2.setPoint(x,y);
			
			if(version>22) {
				int e;
				*t>>e;
				aspect_enabled=e;
			}

			if(version>19) {
				int e;
				double xmin, xmax, ymin,ymax;
				*t>>e>>xmin>>xmax>>ymin>>ymax;
				marks_enabled=e;
				markx->setMin(xmin);
				markx->setMax(xmax);
				marky->setMin(ymin);
				marky->setMax(ymax);
			}
			if(version > 23) {
				QString c;
				int v;

				*t>>v>>filltype;
				fill_enabled = bool(v);
				*t>>fillg1>>fillg2;
					
				*t>>c>>v;
				fillbrush = QBrush(c,(Qt::BrushStyle) v);
			}

		}
		else {
			position.setPoint(0,0);
			size.setPoint(1,1);
			p1.setPoint(.11,.15);
			p2.setPoint(.95,.85);
		}
	}

	// title
	kdDebug()<<"Opening title ..."<<endl;
	title->open(t,version);

	// legend
	kdDebug()<<"Opening legend ..."<<endl;
	legend.open(t,version);

	// axes
	kdDebug()<<"Opening axes ..."<<endl;
	openAxes(t,version);

	// plot type specific stuff
	if (type == PSURFACE && version > 5) {
		Plot2DSurface *plot = (Plot2DSurface *)this;
		int de, ce, n, p;
		QString tmp;
		*t>>de>>ce;
		plot->enableDensity(de);
		plot->enableContour(ce);
		if(version>22)
			*t>>n;
		else {	
			*t>>n>>p;
#ifdef HAVE_GL
			plot->setColorVector(convertOldColormap(p));
#endif
		}
		plot->setNumber(n);
		if (version>12) {
			*t>>tmp;
			plot->setContourColor(QColor(tmp));
			*t>>de>>ce>>n>>p>>tmp;
			plot->setMesh(de);
			plot->setColoredContour(ce);
			plot->setBrush(n);
			plot->setRelative(p);
			plot->setThreshold(tmp.toDouble());
		}
		if(version > 22) {
#ifdef HAVE_GL
			//read colorvector
			Qwt3D::ColorVector cv;
			cv.clear();
			Qwt3D::RGBA rgb;
#endif

			int size;
			*t>>size;
			for(int i=0;i<size;i++) {
#ifdef HAVE_GL
				*t >> rgb.r >> rgb.g >> rgb.b>>rgb.a;
				cv.push_back(rgb);
#else
				int r,g,b;	// over read values
				*t>>r>>g>>b;
#endif
			}
#ifdef HAVE_GL
			plot->setColorVector(cv);
#endif
		}
	}
	else if (type == PQWT3D && version > 17) {
		PlotQWT3D *plot = (PlotQWT3D *)this;
		int pstyle, cstyle, fstyle, ar, iso, mouse, resolution;
		double maj[12], min[12];
		*t>>pstyle;
		*t>>cstyle;
		*t>>fstyle;
		*t>>ar;
		*t>>iso;
		if(version > 20) {
			*t>>mouse;
			for(int i=0;i<12;i++)
				*t>>maj[i]>>min[i];
			*t>>resolution;
			kdDebug()<<"	RESOLUTION = "<<resolution<<endl;
		}
#ifdef HAVE_GL
		plot->setPlotStyle((Qwt3D::PLOTSTYLE) pstyle);
		plot->setCoordinateStyle((Qwt3D::COORDSTYLE) cstyle);
		plot->setFloorStyle((Qwt3D::FLOORSTYLE) fstyle);
		plot->setAspectRatio(ar);
		plot->setIsolines(iso);
		if(version > 20) {
			plot->disableMouse(mouse);
			for(int i=0;i<12;i++) {
				plot->getAxis(i)->setMajorTickLength(maj[i]);
				plot->getAxis(i)->setMinorTickLength(min[i]);
			}
			plot->setDataResolution(resolution);
		}

		//read colorvector
		Qwt3D::ColorVector cv;
		cv.clear();
		Qwt3D::RGBA rgb;
#endif
		
		int size;
		*t>>size;
		for(int i=0;i<size;i++) {
#ifdef HAVE_GL
			*t >> rgb.r >> rgb.g >> rgb.b>>rgb.a;
			cv.push_back(rgb);
#else
			int r,g,b;	// over read values
			*t>>r>>g>>b;
#endif
		}
#ifdef HAVE_GL
		plot->setColorVector(cv);
#endif
	}

	// get the data
	QString gstring;
	kdDebug()<<"Opening Graph ..."<<endl;
	QProgressDialog *progress = new QProgressDialog( i18n("Opening Project ..."), i18n("Cancel"), 100,
		worksheet->getMainWin(), "progress", true );
	progress->setMinimumDuration(1000);

	int n;
	*t>>gstring>>n;
	kdDebug()<<"string = "<<gstring<<" / n = "<<n<<endl;
	while (!strncmp("Graph",gstring,5) ) {
		kdDebug()<<" GRAPH "<<gstring<<' '<<n<<endl;

		if (gstring == "Graph2D") {
			Graph2D *g = new Graph2D();
			g->open(t,version,progress);
			worksheet->addGraph2D(g,type);
		}
		else if (gstring == "Graph3D") {
			Graph3D *g = new Graph3D();
			g->open(t,version,progress);
			worksheet->addGraph3D(g,type);
		}
		else if (gstring == "Graph4D") {
			Graph4D *g = new Graph4D();
			g->open(t,version,progress);
			worksheet->addGraph4D(g);
		}
		else if (gstring == "GraphM") {
			GraphM *g = new GraphM();
			g->open(t,version,progress);
			worksheet->addGraphM(g,type);
		}
		else if (gstring == "GraphIMAGE") {
			GraphIMAGE *g = new GraphIMAGE();
			g->open(t,version,progress);
			worksheet->addGraphIMAGE(g);
		}
		else if (gstring == "GraphL") {
			GraphL *g = new GraphL();
			g->open(t,version,progress);
			worksheet->addGraphL(g,type);
		}
		*t>>gstring>>n;
		kdDebug()<<" GRAPH LINE :  "<<gstring<<' '<<n<<endl;
	}
	kdDebug()<<"Plot::open() OK"<<endl;
}

// called from Plot<Type> with all axes
void Plot::saveAxis(QTextStream *t, Axis *axis) {
	kdDebug()<<"Plot::saveAxis()"<<endl;
	*t<<axis->Scale()<<endl;

	//Grid & Border
	*t<<axis->MajorGridEnabled()<<' '<<axis->BorderEnabled()<<' '<<axis->Enabled()<<endl;
	*t<<axis->MinorGridEnabled()<<endl;
	*t<<axis->majorGridColor().name()<<endl;
	*t<<axis->minorGridColor().name()<<endl;
	*t<<(int)axis->MajorGridType()<<' '<<(int)axis->MinorGridType()<<' '<<axis->borderWidth()<<endl;
	*t<<axis->majorGridWidth()<<' '<<axis->minorGridWidth()<<' '<<axis->majorTickWidth()<<' '<<axis->minorTickWidth()<<endl;

	axis->getLabel()->save(t);

	*t<<axis->Position()<<endl;

	*t<<axis->TickPos()<<endl;
	*t<<axis->Scaling()<<' '<<axis->Shift()<<endl;
	*t<<axis->tickType()<<endl;
	*t<<axis->tickLabelEnabled()<<endl;
	*t<<axis->TickLabelPrefix()<<endl;
	*t<<axis->TickLabelSuffix()<<endl;
	*t<<axis->TickLabelRotation()<<endl;
	*t<<axis->TickLabelPosition()<<endl;

	//Ticks
	QFont tf = axis->TickLabelFont();
	*t<<tf.family()<<endl;
	*t<<tf.pointSize()<<' '<<tf.weight()<<' '<<tf.italic()<<endl;
	*t<<axis->MajorTicks()<<' '<<axis->MinorTicks()<<endl;
	*t<<axis->MajorTicksEnabled()<<' '<<axis->MinorTicksEnabled()<<endl;
	*t<<axis->TickColor().name()<<endl;
	*t<<axis->TickLabelColor().name()<<endl;
	*t<<axis->BorderColor().name()<<endl;
	*t<<axis->TickLabelFormat()<<endl;
	*t<<axis->TickLabelPrecision()<<endl;
	*t<<axis->DateTimeFormat()<<endl;
}

// called from Plot<Type> with all axes
void Plot::openAxis(QTextStream *t,int version,Axis *axis) {
	kdDebug()<<"Plot::openAxis()"<<endl;
	QString family, color, bgcolor;
	int pointsize, weight, italic, trans;
	double x,y;

	QString l;
	double major;
	int minor;
	int majore,minore;
	int ge,be,e;
	int boxed=0;
	double rotation=0;
	int is_texlabel=0;

	int s=0;
	if(version > 7)
		*t>>s;
	axis->setScale((TScale)s);
	kdDebug()<<"SCALE "<<s<<endl;

	*t>>ge>>be>>e;
	axis->enableBorder(be);
	axis->enableMajorGrid(ge);
	kdDebug()<<"BORDER/MAJORGRID "<<be<<' '<<ge<<endl;
	
	if(version > 14) {
		*t>>ge;
		axis->enableMinorGrid(ge);
	}

	axis->Enable(e);
	if (version > 3) {
		*t>>color;
		axis->setMajorGridColor(QColor(color));
		kdDebug()<<"MAJOR GRID COLOR = "<<color<<endl;
		if(version > 18) {
			int e1,e2,e3,e4;
			*t>>color;
			kdDebug()<<"MINOR GRID COLOR = "<<color<<endl;
			axis->setMinorGridColor(QColor(color));
			*t>>e1>>e2>>e3;
			axis->setMajorGridType((Qt::PenStyle)e1);
			axis->setMinorGridType((Qt::PenStyle)e2);
			axis->setBorderWidth(e3);
			*t>>e1>>e2>>e3>>e4;
			axis->setMajorGridWidth(e1);
			axis->setMinorGridWidth(e2);
			axis->setMajorTickWidth(e3);
			axis->setMinorTickWidth(e4);
			kdDebug()<<"GRID WIDTHs : "<<e1<<e2<<e3<<e4<<endl;
		}
	}
	
	l=t->readLine();	// needed. don't know why ...

	// open axis label
	l=t->readLine();

	kdDebug()<<"Label = "<<l<<endl;

	// Label::open() partly duplicated here !!!
	if (version > 6) {	// new order
		family=t->readLine();
		*t>>pointsize>>weight>>italic;
		*t>>color;
		*t>>x>>y;
		if(version > 8)
			*t>>boxed;
		if(version > 16)
			*t>>rotation;
		if(version > 20)
			*t>>is_texlabel;
		if (version > 21) {
			*t>>bgcolor;
			*t>>trans;
		}
	}
	else if (version > 3) {
		*t>>color;
		*t>>x>>y;
		axis->getLabel()->setPosition(x,y);

		t->readLine();
		family=t->readLine();
		*t>>pointsize>>weight>>italic;
	}
	else {
		*t>>family>>pointsize>>weight>>italic;
	}

	kdDebug()<<"axis (Font) "<<family<<' '<<pointsize<<endl;

	Label *label = new Label(l,QFont(family,pointsize,weight,italic),QColor(color));
	label->setPosition(x,y);
	label->setBoxed(boxed);
	label->setRotation(rotation);
	label->setTeXLabel(is_texlabel);
	label->setBackgroundColor(bgcolor);
	label->setTransparent(trans);
	axis->setLabel(label);

	if (version > 22) {
		int pos;
		*t>>pos;
		axis->setPosition(pos);
	}

	if (version > 10) {
		*t>>x;
		axis->setTickPos((int)x);
		kdDebug()<<"	TIC POSITION = "<<x<<endl;
		*t>>x>>y;
		axis->setScaling(x);
		axis->setShift(y);
		kdDebug()<<"	SCALING/SHIFT = "<<x<<' '<<y<<endl;
		if(version > 22) {
			int e;
			*t>>e;
			axis->setTickType(e);
		}
		if(version > 21) {
			int e;
			*t>>e;
			axis->enableTickLabel(e);
		}
		t->readLine();
		QString tmpprefix=t->readLine();
		axis->setTickLabelPrefix(tmpprefix);
		kdDebug()<<"Prefix = "<<tmpprefix<<endl;
		QString tmpsuffix=t->readLine();
		axis->setTickLabelSuffix(tmpsuffix);
		kdDebug()<<"Suffix = "<<tmpsuffix<<endl;
		if(version > 14) {
			*t>>x;
			axis->setTickLabelRotation(x);
			if(version > 16) {
				*t>>x;
				axis->setTickLabelPosition((int)x);
			}
			t->readLine();
		}
	}

	// tics
	if (version > 3) {
		if (version<11)
			t->readLine();
		family=t->readLine();
		*t>>pointsize>>weight>>italic;
		axis->setTickLabelFont( QFont(family,pointsize,weight,italic));
	}

	kdDebug()<<"axis (Ticks) "<<family<<pointsize<<endl;

	*t>>major>>minor;
	axis->setMajorTicks(major);
	axis->setMinorTicks(minor);
	*t>> majore>>minore;
	axis->enableMajorTicks(majore);
	axis->enableMinorTicks(minore);
	if (version > 3) {
		*t>>color;
		axis->setTickColor(QColor(color));
		*t>>color;
		axis->setTickLabelColor(QColor(color));
		*t>>color;
		axis->setBorderColor(QColor(color));
	}
	if (version > 4) {
		int tmp;
		*t>>tmp;
		axis->setTickLabelFormat((TFormat)tmp);
		*t>>tmp;
		axis->setTickLabelPrecision(tmp);
	}
	if(version >11) {
		QString tmp;
		t->readLine();
		tmp = t->readLine();
		axis->setDateTimeFormat(tmp);
	}

	kdDebug()<<"OK Axis : "<<l<<endl;
}

void Plot::autoScaleX() {
	TScale scale = ((Plot2D *)this)->getAxis(0)->Scale();
	double min=range[0].rMin(), max=range[0].rMax();

	worksheet->checkRanges(scale,&min,&max);
	
	actrange[0].setMin(min);
	actrange[0].setMax(max);
}

void Plot::autoScaleY() {
	TScale scale = ((Plot2D *)this)->getAxis(1)->Scale();
	double min=range[1].rMin(), max=range[1].rMax();

	worksheet->checkRanges(scale,&min,&max);
	
	actrange[1].setMin(min);
	actrange[1].setMax(max);
}

void Plot::autoScaleZ() {
	TScale scale = ((Plot2D *)this)->getAxis(2)->Scale();
	double min=range[2].rMin(), max=range[2].rMax();

	worksheet->checkRanges(scale,&min,&max);
	
	actrange[2].setMin(min);
	actrange[2].setMax(max);
}
	
//! build the tic label string according to atlf
QString Plot::TicLabel(int atlf, int prec, QString dtf, double value) {
	QString label;

	switch(atlf) {
	case AUTO:
		label = QString::number(value,'g',prec);
		break;
	case NORMAL:
		label = QString::number(value,'f',prec);
		break;
	case SCIENTIFIC:
		label = QString::number(value,'e',prec);
		break;
	case POWER10:
		label = "10<span style=\"vertical-align:super\">"+ QString::number(log10(value),'g',prec)+"</span>";
		break;
	case POWER2:
		label = "2<span style=\"vertical-align:super\">"+ QString::number(log2(value),'g',prec)+"</span>";
		break;
	case POWERE:
		label = "e<span style=\"vertical-align:super\">"+ QString::number(log(value),'g',prec)+"</span>";
		break;
	case FSQRT:
		label = "sqrt("+ QString::number(value*value,'g',prec) + ")";
		break;
	case TIME: {
		QTime time;
		time=time.addMSecs((int) (value*1000));
			
		QString format;
		if(fabs(value)<1)
			format="z";
		else if(fabs(value)<10) {
			format="s.zzz";
			if (prec==0)
				format="s";
			else if (prec==1) {
				// round to 100 ms
				int ms=time.msec();
				time=time.addMSecs(-ms);
				ms = 100*(int)rint(ms/100);
				time=time.addMSecs(ms);
			}
			else if (prec==2) {
				// round to 10 ms
				int ms=time.msec();
				time=time.addMSecs(-ms);
				ms = 10*(int)rint(ms/10);
				time=time.addMSecs(ms);
			}
		}
		else if (fabs(value)<3600) {
			format = "m:ss";
			if (prec==0) {
				int s=time.second();
				// round to full minute
				time=time.addSecs(-s);
				if(s>=30)
					time=time.addSecs(60);
				format="m";
			}
			else if (prec==1) {
				// round to 10 seconds
				int s=time.second();
				time=time.addSecs(-s);
				s = 10*(int)rint(s/(int)10);
				time=time.addSecs(s);
			}
		}
		else {
			// TODO : round minutes
			format="h:mm:ss";
		}
			
		// overwrite auto format
		if (dtf != i18n("auto"))
			format = dtf;
		label=time.toString(format);
		kdDebug()<<"VALUE in Time Format : "<<label<<endl;
		}
		break;
	case DATE: {
		QDate date(1970,1,1);
		date=date.addDays((int) value);
		QString format("dd.MM.yyyy");
		if (dtf != i18n("auto"))
			format = dtf;
			label=date.toString(format);
			kdDebug()<<"VALUE in Date Format ( "<<format<<") : "<<label<<endl;
		}
		break;
	case DATETIME: {
		QDate date(1970,1,1);
		QDateTime datetime(date);
//		kdDebug()<<"value = "<<(int) value<<endl;
		datetime=datetime.addSecs((int)value);
		QString format("dd.MM.yyyy h:mm:ss");
		if (dtf != i18n("auto"))
			format = dtf;
		label = datetime.toString(format);
//		kdDebug()<<"VALUE in DateTime Format ( "<<format<<") : "<<label<<endl;
		}
		break;
	case DEGREE:
		label = QString::number(180/M_PI*value,'f',prec)+'';
		break;
	}

	return label;
}

//! get the tic label value from the string according to atlf (range of axes)
double Plot::TicLabelValue(int atlf, QString string) {
	double value=0;
	
	switch(atlf) {
	case AUTO:
	case NORMAL:
	case SCIENTIFIC:
	case POWER10:
	case POWER2:
	case POWERE:
	case FSQRT:
		value = parse((char *)(string.latin1()));	// parse input
		break;
	case TIME: {
		QTime time;
		time = time.fromString(string);
		value = -1.0/1000.0*time.msecsTo(QTime()); // for reading msecs
		} break;
	case DATE: {
		QDate date;
		date = date.fromString(string,Qt::ISODate);              // must be yyyy-MM-dd
		//kdDebug()<<"DATE : "<<date.toString()<<endl;
		//kdDebug()<<"\tdaysTo1970 : "<<date.daysTo(QDate(1970,1,1))<<endl;
		value = -1.0*date.daysTo(QDate(1970,1,1));
		}
		break;
	case DATETIME: {
                QDateTime datetime;
                datetime = datetime.fromString(string,Qt::ISODate);
//              kdDebug()<<"DATETIME : "<<datetime.toString()<<endl;
//              kdDebug()<<"\tsecsTo1970 : "<<-datetime.secsTo(QDate(1970,1,1))<<endl;
                value = -1.0*datetime.secsTo(QDate(1970,1,1));
		}
		break;
	case DEGREE:
		string.remove('');
		value = M_PI/180.0*string.toDouble();
		break;
	}

	return value;
}

void Plot::shiftRight() {
	TScale scale = ((Plot2D *)this)->getAxis(0)->Scale();
	double r1=actrange[0].rMin(), r2=actrange[0].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);

	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1-r; nr2 = r2-r; break;
	case LOG10: case LOG2: case LN: nr1 = r1/r; nr2 = r2/r; break;
	}

	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[0].setMin(nr1);
	actrange[0].setMax(nr2);
}

void Plot::shiftLeft() {
	TScale scale = ((Plot2D *)this)->getAxis(0)->Scale();
	double r1=actrange[0].rMin(), r2=actrange[0].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1+r; nr2 = r2+r; break;
	case LOG10: case LOG2: case LN: nr1 = r1*r; nr2 = r2*r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[0].setMin(nr1);
	actrange[0].setMax(nr2);
}

void Plot::shiftUp() {
	TScale scale = ((Plot2D *)this)->getAxis(1)->Scale();
	double r1=actrange[1].rMin(), r2=actrange[1].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1-r; nr2 = r2-r; break;
	case LOG10: case LOG2: case LN: nr1 = r1/r; nr2 = r2/r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[1].setMin(nr1);
	actrange[1].setMax(nr2);
}
	
void Plot::shiftDown() {
	TScale scale = ((Plot2D *)this)->getAxis(1)->Scale();
	double r1=actrange[1].rMin(), r2=actrange[1].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1+r; nr2 = r2+r; break;
	case LOG10: case LOG2: case LN: nr1 = r1*r; nr2 = r2*r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[1].setMin(nr1);
	actrange[1].setMax(nr2);
}

void Plot::scaleXUp() {
	TScale scale = ((Plot2D *)this)->getAxis(0)->Scale();
	double r1=actrange[0].rMin(), r2=actrange[0].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1+r; nr2 = r2-r; break;
	case LOG10: case LOG2: case LN: nr1 = r1*r; nr2 = r2/r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[0].setMin(nr1);
	actrange[0].setMax(nr2);
}

void Plot::scaleXDown() {
	TScale scale = ((Plot2D *)this)->getAxis(0)->Scale();
	double r1=actrange[0].rMin(), r2=actrange[0].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1-r; nr2 = r2+r; break;
	case LOG10: case LOG2: case LN: nr1 = r1/r; nr2 = r2*r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[0].setMin(nr1);
	actrange[0].setMax(nr2);
}

void Plot::scaleYUp() {
	TScale scale = ((Plot2D *)this)->getAxis(1)->Scale();
	double r1=actrange[1].rMin(), r2=actrange[1].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1+r; nr2 = r2-r; break;
	case LOG10: case LOG2: case LN: nr1 = r1*r; nr2 = r2/r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[1].setMin(nr1);
	actrange[1].setMax(nr2);
}
	
void Plot::scaleYDown() {
	TScale scale = ((Plot2D *)this)->getAxis(1)->Scale();
	double r1=actrange[1].rMin(), r2=actrange[1].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1-r; nr2 = r2+r; break;
	case LOG10: case LOG2: case LN: nr1 = r1/r; nr2 = r2*r; break;
	}

	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[1].setMin(nr1);
	actrange[1].setMax(nr2);
}

void Plot::scaleZUp() {
	// does this work ?
	TScale scale = ((Plot2D *)this)->getAxis(2)->Scale();
	double r1=actrange[2].rMin(), r2=actrange[2].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1+r; nr2 = r2-r; break;
	case LOG10: case LOG2: case LN: nr1 = r1*r; nr2 = r2/r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[2].setMin(nr1);
	actrange[2].setMax(nr2);
}

void Plot::scaleZDown() {
	// does this work ?
	TScale scale = ((Plot2D *)this)->getAxis(2)->Scale();
	double r1=actrange[2].rMin(), r2=actrange[2].rMax();
	
	worksheet->checkRanges(scale,&r1,&r2);
	
	double r=worksheet->partRanges(scale,r1,r2);
	double nr1, nr2;
	switch(scale) {
	case LINEAR: case SQRT: case SX2: nr1 = r1-r; nr2 = r2+r; break;
	case LOG10: case LOG2: case LN: nr1 = r1/r; nr2 = r2*r; break;
	}
	
	// just checking
	worksheet->checkRanges(scale,&nr1,&nr2);

	actrange[2].setMin(nr1);
	actrange[2].setMax(nr2);
}

void Plot::zoomIn() { 
	TScale xscale = ((Plot2D *)this)->getAxis(0)->Scale();
	TScale yscale = ((Plot2D *)this)->getAxis(1)->Scale();
	TScale zscale = ((Plot2D *)this)->getAxis(2)->Scale();
	double x1=actrange[0].rMin(), x2=actrange[0].rMax();
	double y1=actrange[1].rMin(), y2=actrange[1].rMax();
	double z1=actrange[2].rMin(), z2=actrange[2].rMax();
	
	worksheet->checkRanges(xscale,&x1,&x2);
	worksheet->checkRanges(yscale,&y1,&y2);
	worksheet->checkRanges(zscale,&z1,&z2);
	
	double x=worksheet->partRanges(xscale,x1,x2);
	double y=worksheet->partRanges(yscale,y1,y2);
	double z=worksheet->partRanges(zscale,z1,z2);
	double nx1, nx2, ny1, ny2, nz1, nz2;
	switch(xscale) {
	case LINEAR: case SQRT: case SX2: nx1 = x1+x; nx2 = x2-x;  break;
	case LOG10: case LOG2: case LN: nx1 = x1*x; nx2 = x2/x; break;
	}
	switch(yscale) {
	case LINEAR: case SQRT: case SX2: ny1 = y1+y; ny2 = y2-y; break;
	case LOG10: case LOG2: case LN: ny1 = y1*y; ny2 = y2/y; break;
	}
	switch(zscale) {
	case LINEAR: case SQRT: case SX2: nz1 = z1+z; nz2 = z2-z; break;
	case LOG10: case LOG2: case LN: nz1 = z1*z; nz2 = z2/z; break;
	}
	
	// just checking
	worksheet->checkRanges(xscale,&nx1,&nx2);
	worksheet->checkRanges(yscale,&ny1,&ny2);
	worksheet->checkRanges(zscale,&nz1,&nz2);

	actrange[0].setMin(nx1);actrange[0].setMax(nx2);
	actrange[1].setMin(ny1);actrange[1].setMax(ny2);
	actrange[2].setMin(nz1);actrange[2].setMax(nz2);
}

void Plot::zoomOut() { 
	TScale xscale = ((Plot2D *)this)->getAxis(0)->Scale();
	TScale yscale = ((Plot2D *)this)->getAxis(1)->Scale();
	TScale zscale = ((Plot2D *)this)->getAxis(2)->Scale();
	double x1=actrange[0].rMin(), x2=actrange[0].rMax();
	double y1=actrange[1].rMin(), y2=actrange[1].rMax();
	double z1=actrange[2].rMin(), z2=actrange[2].rMax();
	
	worksheet->checkRanges(xscale,&x1,&x2);
	worksheet->checkRanges(yscale,&y1,&y2);
	worksheet->checkRanges(zscale,&z1,&z2);
	
	double x=worksheet->partRanges(xscale,x1,x2);
	double y=worksheet->partRanges(yscale,y1,y2);
	double z=worksheet->partRanges(zscale,z1,z2);
	double nx1, nx2, ny1, ny2, nz1, nz2;
	switch(xscale) {
	case LINEAR: case SQRT: case SX2: nx1 = x1-x; nx2 = x2+x; break;
	case LOG10: case LOG2: case LN: nx1 = x1/x; nx2 = x2*x; break;
	}
	switch(yscale) {
	case LINEAR: case SQRT: case SX2: ny1 = y1-y; ny2 = y2+y; break;
	case LOG10: case LOG2: case LN: ny1 = y1/y; ny2 = y2*y; break;
	}
	switch(zscale) {
	case LINEAR: case SQRT: case SX2: nz1 = z1-z; nz2 = z2+z; break;
	case LOG10: case LOG2: case LN: nz1 = z1/z; nz2 = z2*z; break;
	}
	
	// just checking
	worksheet->checkRanges(xscale,&nx1,&nx2);
	worksheet->checkRanges(yscale,&ny1,&ny2);
	worksheet->checkRanges(zscale,&nz1,&nz2);

	actrange[0].setMin(nx1);actrange[0].setMax(nx2);
	actrange[1].setMin(ny1);actrange[1].setMax(ny2);
	actrange[2].setMin(nz1);actrange[2].setMax(nz2);
}

// find y value for x in data mode
Point Plot::dataValue(double x) {
	double y=0;

	double xmin=actrange[0].rMin(), xmax=actrange[0].rMax();
	double ymin=actrange[1].rMin(), ymax=actrange[1].rMax();
	
	//TODO : which graph
	int graph=0;
	
	if(x<p1.X())
		return Point(p1.X(),p2.Y());
	else if( x>p2.X())
		return Point(p2.X(),p2.Y());

//	kdDebug()<<"	X-VALUE for datamode = "<<x<<endl;

	// convert from world coordinates
	switch(type){
	case P2D:
		switch(((Plot2D *)this)->getAxis(0)->Scale()) {
		case LINEAR:	x = xmin+(x-p1.X())*(xmax-xmin)/(p2.X()-p1.X()); break;
		case LOG10: 	x = pow(10,log10(xmin)+(x-p1.X())*log10(xmax/xmin)/(p2.X()-p1.X())); break;
		case LOG2: 	x = pow(2,log2(xmin)+(x-p1.X())*log2(xmax/xmin)/(p2.X()-p1.X())); break;
		case LN: 		x = pow(M_E,log(xmin)+(x-p1.X())*log(xmax/xmin)/(p2.X()-p1.X())); break;
		case SQRT:
			//TODO
			break;
		case SX2:
			//TODO
			break;
		}
	default: break;
	}

//	kdDebug()<<"	X for datamode = "<<x<<endl;
	
	GRAPHType s = graphlist->getStruct(graph);
	
	switch(s) {
	case GRAPH2D: {
		Graph2D *g = graphlist->getGraph2D(graph);
		Point *ptr = g->Data();
		
		for(int i=0;i<g->Number()-1;i++) {
			if(ptr[i].X() < x && ptr[i+1].X() > x) {
				x = ptr[i].X();
				y = ptr[i].Y();
				break;
			}
		}
		}; break;
	case GRAPH3D: {
		Graph3D *g = graphlist->getGraph3D(graph);
		Point3D *ptr = g->Data();
		
		for(int i=0;i<g->Number()-1;i++) {
			if(ptr[i].X() < x && ptr[i+1].X() > x) {
				x = ptr[i].X();
				y = ptr[i].Y();
				break;
			}
		}
		}; break;
	case GRAPH4D: {
		Graph4D *g = graphlist->getGraph4D(graph);
		Point4D *ptr = g->Data();
		
		for(int i=0;i<g->Number()-1;i++) {
			if(ptr[i].X() < x && ptr[i+1].X() > x) {
				x = ptr[i].X();
				y = ptr[i].Y();
				break;
			}
		}
		}; break;
	default: break;
	}
	
	worksheet->getMainWin()->message("( " + QString::number(x) + " / " + QString::number(y) + " )");
	
	// convert to world coordinates
//	kdDebug()<<"	Y for datamode = "<<y<<endl;
	switch(type){
	case P2D:
		switch(((Plot2D *)this)->getAxis(1)->Scale()) {
		case LINEAR:	y = p2.Y()-(y-ymin)/(ymax-ymin)*(p2.Y()-p1.Y()); break;
		case LOG10:	y = p2.Y()-(log10(y)-log10(ymin))/log10(ymax/ymin)*(p2.Y()-p1.Y()); break;
		case LOG2:	y = p2.Y()-(log2(y)-log2(ymin))/log2(ymax/ymin)*(p2.Y()-p1.Y()); break;
		case LN:		y = p2.Y()-(log(y)-log(ymin))/log(ymax/ymin)*(p2.Y()-p1.Y()); break;
		case SQRT:
			//TODO
			break;
		case SX2:
			//TODO
			break;
		}
		switch(((Plot2D *)this)->getAxis(0)->Scale()) {
		case LINEAR:	x = p1.X()+(x-xmin)/(xmax-xmin)*(p2.X()-p1.X()); break;
		case LOG10:	x = p1.X()+(log10(x)-log10(xmin))/log10(xmax/xmin)*(p2.X()-p1.X()); break;
		case LOG2:	x = p1.X()+(log2(x)-log2(xmin))/log2(xmax/xmin)*(p2.X()-p1.X()); break;
		case LN:		x = p1.X()+(log(x)-log(xmin))/log(xmax/xmin)*(p2.X()-p1.X()); break;
		case SQRT:
			//TODO
			break;
		case SX2:
			//TODO
			break;
		}
	default: break;
	}
//	kdDebug()<<"	Y-VALUE for datamode = "<<y<<endl;
	
	return Point(x,y);
}
	
//! calcuate tic number for auto tics
int Plot::autoTicks(double min, double max) {
//	kdDebug()<<"Plot::autoTicks : min/max = "<<min<<' '<<max<<endl;
	if(max-min==0)
		return -1;
	int floorvalue = (int) floor(log10(max-min));
	int tics = (int)((max-min)/pow(10.0,(double)floorvalue));

	// just to make sure
	if(tics<=0)
		return -1;
	
	// always between 4 and 10 tics
	while(tics<4)
		tics *= 2;

	return tics;	
}
