/* vim: set noet ts=4:
 *
 * Copyright (c) 2002-2007 Martin A. Godisch <martin@godisch.de>.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any later
 * version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 51 Franklin
 * St, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#include <dictd.h>
#include <charset.h>
#include <latrine.h>
#include <memory.h>
#include <stdio.h>
#include <data.h>

static gzFile dict = NULL;

/* returns  0: success
 * returns -1: failure
 */
int open_dictd(const char *file)
{
	assert(dict == NULL);
	if ((dict = gzopen(file, "rb")) == NULL) {
		errmsg("gzopen: %s: %s", file, errno == 0 ? zError(Z_MEM_ERROR) : strerror(errno));
		return -1;
	}
	return 0;
}

/* returns  1: success
 * returns  0: no more words available
 * returns -1: failure
 */
int read_dictd(struct word *w)
{
	static char buffer[BUFSIZE] = {0};
	char *s;
	int n;
	struct Tlang *l = NULL;

	assert(dict != NULL);
	memset(w->lang, 0, sizeof(w->lang)); /* for FREE() calls below */

	while (*buffer == 0) {
		if (gzgets(dict, buffer, sizeof(buffer) - 1) == Z_NULL) {
			if (gzeof(dict))
				return 0;
			else {
				s = (char*)gzerror(dict, &n);
				errmsg("gzgets: %s", n == Z_ERRNO ? strerror(errno) : s);
				return -1;
			}
		}
		if (*buffer == '\t' || *buffer == ' ') {
			/* FIXME: errmsg: first entry without lang[0] */
			*buffer = 0;
		}
		if (*buffer == '\n' || *buffer == '#')
			*buffer = 0;
	}

	/*** read lang[0] ***/

	while (*buffer != '\t' && *buffer != ' ') {

		if (*buffer != 0) {
			if ((s = index(buffer, '\n')) != NULL)
				*s = 0;
			assert(index(buffer, '\n') == NULL);
			utf2local(buffer, sizeof(buffer));
			if (w->lang[0] == NULL)
				l = w->lang[0] = MALLOC(sizeof(struct Tlang));
			else {
				l->next = MALLOC(sizeof(struct Tlang));
				l = l->next;
			}
			l->c = STRDUP(buffer);
		}

		if (gzgets(dict, buffer, sizeof(buffer) - 1) == Z_NULL) {
			if (gzeof(dict)) {
				/* FIXME: errmsg: last entry without lang[1] */
				free_langlist(w);
				return 0;
			} else {
				s = (char*)gzerror(dict, &n);
				errmsg("gzgets: %s", n == Z_ERRNO ? strerror(errno) : s);
				free_langlist(w);
				return -1;
			}
		}
		if (*buffer == '\n' || *buffer == '#')
			*buffer = 0;
	}

	/*** read lang[1] ***/

	while (*buffer == '\t' || *buffer == ' ' || *buffer == 0) {

		if (*buffer != 0) {
			if ((s = index(buffer, '\n')) != NULL)
				*s = 0;
			assert(index(buffer, '\n') == NULL);
			utf2local(buffer, sizeof(buffer));
			s = buffer;
			while (*s == '\t' || *s == ' ')
				s++;
			if (w->lang[1] == NULL)
				l = w->lang[1] = MALLOC(sizeof(struct Tlang));
			else {
				l->next = MALLOC(sizeof(struct Tlang));
				l = l->next;
			}
			l->c = STRDUP(s);
		}
		/* FIXME: errmsg: too many lines */

		if (gzgets(dict, buffer, sizeof(buffer) - 1) == Z_NULL) {
			if (gzeof(dict))
				break;
			else {
				s = (char*)gzerror(dict, &n);
				errmsg("gzgets: %s", n == Z_ERRNO ? strerror(errno) : s);
				free_langlist(w);
				return -1;
			}
		}
		if (*buffer == '\n' || *buffer == '#')
			*buffer = 0;
	}

	if (strncmp(w->lang[0]->c, "00-", 3) == 0) {
		free_langlist(w);
		return(read_dictd(w));
	}

	if (debug) {
		fprintf(debug, "read_dictd: ");
		for (l = w->lang[0]; l != NULL; l = l->next)
			fprintf(debug, "|%s", l->c);
		fprintf(debug, "| -> ");
		for (l = w->lang[1]; l != NULL; l = l->next)
			fprintf(debug, "|%s", l->c);
		fprintf(debug, "|\n");
	}

	return 1;
}

/* returns  0: success
 * returns -1: failure
 */
int close_dictd(void)
{
	int ret;

	assert(dict != NULL);
	ret  = gzclose(dict);
	dict = NULL;
	return ret;
}
