/*
 * dmnt.c -- Digital Unix mount functions for lslk
 *
 * V. Abell
 * Purdue University Computing Center
 *
 * Chris Eleveld <chris@sector7.com>
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell.
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */
#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dmnt.c,v 1.3 99/11/10 15:00:47 abe Exp $";
#endif


#include "lslk.h"


/*
 * getmntdev() - get device number from mount structure
 */

dev_t
getmntdev(v)
	struct vnode *v;		/* vnode pointer */
{
	fsid_t *f;
	struct l_fsid *fp;
	int i;
	struct mount m;
	struct l_vfs *vfs;
/*
 * If the vnode lacks a mount structure pointer, return a device number
 * of zero.
 */
	if (!v->v_mount)
	    return((dev_t)0);
/*
 * Search current VFS structure chain.
 */
	for (vfs = Vfs; vfs; vfs = vfs->next) {
	    if (v->v_mount == vfs->mount)
		return(vfs->dev);
	}
/*
 * Read the vnode's mount structure.
 */
	if (kread((KA_T)v->v_mount, (char *)&m, sizeof(m)))
	    return((dev_t)0);
	f = &m.m_stat.f_fsid;
/*
 * Search the local mount table for an fsid match.
 */
	for (i = 0; i < NMnt; i++) {
	    if (!(fp = (struct l_fsid *)Mnt[i].priv))
		continue;
	    if (f->val[0] == fp->fsid.val[0] && f->val[1] == fp->fsid.val[1])
		break;
	}
	if (i >= NMnt)
	    return((dev_t)0);
/*
 * Allocate a new local VFS structure and fill it in for this vnode.
 */
	if ((vfs = (struct l_vfs *)malloc(sizeof(struct l_vfs)))
	== (struct l_vfs *)NULL) {
	    (void) fprintf(stderr, "%s: no space for l_vfs structure\n", Pn);
	    Exit(1);
	}
	vfs->dev = Mnt[i].dev;
	vfs->mount = v->v_mount;
	vfs->next = Vfs;
	Vfs = vfs;
	return(vfs->dev);
}


/*
 * readmnt() -- read mount table information
 */

int
readmnt()
{
	char c;
	dev_t dev;
	char *dn = (char *)NULL;
	int err, i, n, na, nl, v;
	int numfs;
	MALLOC_S len;
	struct l_fsid *lf;
	char *m, *o, *oo, *s, *ln;
	struct statfs *mp;
	struct stat sb;
/*
 * Get the number of mounted file systems and read the mount table.
 */
	numfs = getmntinfo(&mp, MNT_WAIT);
	if (numfs < 0) {
	    (void) fprintf(stderr, "%s: getmntinfo: %s\n", Pn, strerror(errno));
	    return(1);
	}
	for (i = na = 0; i < numfs; i++, mp++) {

	/*
	 * Skip all filesystems without inodes:
	 */
	     switch (mp->f_type) {
		case MOUNT_NONE:
		case MOUNT_PROCFS:
		    continue;

# if	defined(MOUNT_NFS3)
		case MOUNT_NFS3:
# endif	/* defined(MOUNT_NFS3) */

		case MOUNT_NFS:

		/*
		 * The mount-from name of some unmounted file systems under
		 * auto mounter control end with ``:(pid<n>):'' -- where <n>
		 * is the PID of the auto mounter process.
		 */
		     if ((ln = strchr(mp->f_mntfromname, ':')) != NULL) {
			if (strncmp(ln+1, "(pid", 4) == 0 && isdigit(*(ln+5))) {
			    for (ln += 6; *ln && isdigit(*ln); ln++)
				/*EMPTY*/;
			    if (*ln == ')' && *(ln+1) == '\0')
				continue;
			}
		     }
		/*
		 * Another auto mounter mount-from name form is "amd:<n>" --
		 * where <n> is the PID of the auto mounter process.
		 */
		    if (strncmp(mp->f_mntfromname, "amd:", 4) == 0
		    &&  isdigit(mp->f_mntfromname[4])) {
			ln = &mp->f_mntfromname[5];
			while (*ln && isdigit(*ln))
			    ln++;
			if (*ln == ':' && *(ln+1) == '\0')
			    continue;
		    }

		case MOUNT_UFS:
		case MOUNT_S5FS:
		case MOUNT_CDFS:
		case MOUNT_MSFS:
		case MOUNT_ADDON:
		case MOUNT_FDFS:
		case MOUNT_FFM:
		case MOUNT_EFS:
		case MOUNT_MFS:
		case MOUNT_PC:
		case MOUNT_DFS:
		    break;
		default:
		    if (Owarn)
			(void) fprintf(stderr,
			    "%s: unknown filesystem type: %d\n",
			    Pn, mp->f_type);
		    continue;
	     }
	/*
	 * Interpolate a possible symbolic directory link.
	 */
	    if (dn)
		(void) free((MALLOC_P *)dn);
	    if ((dn = (char *)malloc((MALLOC_S)strlen(mp->f_mntonname) + 1))
	    == NULL) {
		err = 1;
		break;
	    }
	    (void) strcpy(dn, mp->f_mntonname);
	    if ((ln = Readlink(dn)) == NULL) {
		if (!Owarn) {
		    (void) fprintf(stderr,
			"      Output information may be incomplete.\n");
		}
		if (ln != dn) {
		    (void) free((MALLOC_P *)dn);
		    dn = ln;
		}
	    }
	/*
	 * verify that path to mount point starts at /
	 */
	    if (*dn != '/') {
		if (Owarn)
		    (void) fprintf(stderr,
			"%s: WARNING: illegal mount point (%s)\n",
			Pn, mp->f_mntonname);
		continue;
	    }
	/*
	 * Stat (safely) the mount point.
	 */
	    if (statsafely(dn, &sb)) {
		(void) fprintf(stderr, "%s: can't stat(%s): %s\n",
		    Pn, mp->f_mntonname, strerror(errno));
		(void) fprintf(stderr,
		    "      Output information may be incomplete.\n");
		continue;
	    }
	    dev = sb.st_dev;
	/*
	 * See if there's room in the local mount table for another entry.
	 */
	    if (NMnt >= na) {

	    /*
	     * Increase the space allocated to the local mount table and
	     * reserve space for the increase.
	     */
		na += 10;
		len = (MALLOC_S)(na * sizeof(struct mounts));
		if (Mnt)
		    Mnt = (struct mounts *)realloc((MALLOC_P *)Mnt, len);
		else
		    Mnt = (struct mounts *)malloc(len);
		if (!Mnt) {
		    (void) fprintf(stderr,
			"%s: no space for %d mount structures at %s: %s\n",
			Pn, na, mp->f_mntfromname, mp->f_mntonname);
		    return(1);
		}
	    }
	/*
	 * Allocate space for what's been mounted (fsname) and where
	 * it's been mounted (dir).
	 */
	    if (!(m = (char *)malloc(strlen(mp->f_mntonname) + 1))) {
		(void) fprintf(stderr, "%s: no room for directory: %s\n",
		    Pn, mp->f_mntonname);
		return(1);
	    }
	    (void) strcpy(m, mp->f_mntonname);
	    if (!(s = (char *)malloc(strlen(mp->f_mntfromname) + 1))) {
		(void) fprintf(stderr, "%s: no room for fsname: %s\n",
		    Pn, mp->f_mntfromname);
		return(1);
	    }
	    (void) strcpy(s, mp->f_mntfromname);
	/*
	 * Allocate space for a local file system ID structure and fill it.
	 */
	    if ((lf = (struct l_fsid *)malloc(sizeof(struct l_fsid)))
	    == (struct l_fsid *)NULL) {
		(void) fprintf(stderr,
		    "%s: can't allocate lfsid struct for %s: %s\n",
		    Pn, mp->f_mntonname, mp->f_mntfromname);
		Exit(1);
	    }
	    lf->fsid = mp->f_fsid;
	/*
	 * Create the new local mount information entry.
	 */
	    Mnt[NMnt].dev = dev;
	    Mnt[NMnt].sp = s;
	    Mnt[NMnt].mntp = m;
	    Mnt[NMnt++].priv = (caddr_t)lf;
	}
	return(0);
}
