#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include "keys.h"


#define MAXLINE	300
#define MAXKEY	40


static char *keystr[] = {
	"c",
	"z",
	"a",
	"s",
	"d",
	"w",
	"Up",
	"Left",
	"Down",
	"Right",
	"Control",
	"space",
	NULL
};

static int default_pc_map[MAXKEY] =
	{ 46, 44, 30, 31, 32, 17, 103, 105, 108, 106, 29, 57 };

static int default_ppc_map[MAXKEY] =
	{  8,  6,  0,  1,  2, 13,  62,  59,  61,  60, 54, -1 };

static int key_up_vals[MAXKEY] = {
	KEYBOARD_KEY_UP_C,
	KEYBOARD_KEY_UP_Z,
	KEYBOARD_KEY_UP_A,
	KEYBOARD_KEY_UP_S,
	KEYBOARD_KEY_UP_D,
	KEYBOARD_KEY_UP_W,
	KEYBOARD_KEY_UP_ARR_U,
	KEYBOARD_KEY_UP_ARR_L,
	KEYBOARD_KEY_UP_ARR_D,
	KEYBOARD_KEY_UP_ARR_R,
	KEYBOARD_KEY_UP_CTRL,
	KEYBOARD_KEY_UP_SPACE
};


static int key_down_vals[MAXKEY] = {
	KEYBOARD_KEY_DOWN_C,
	KEYBOARD_KEY_DOWN_Z,
	KEYBOARD_KEY_DOWN_A,
	KEYBOARD_KEY_DOWN_S,
	KEYBOARD_KEY_DOWN_D,
	KEYBOARD_KEY_DOWN_W,
	KEYBOARD_KEY_DOWN_ARR_U,
	KEYBOARD_KEY_DOWN_ARR_L,
	KEYBOARD_KEY_DOWN_ARR_D,
	KEYBOARD_KEY_DOWN_ARR_R,
	KEYBOARD_KEY_DOWN_CTRL,
	KEYBOARD_KEY_DOWN_SPACE
};


static int keymap[MAXKEY];


/* Example line of dumpkeys output:
 * "keycode  46 = +c                +C                cent  ..."
 */

int get_config_from_dumpkeys( void )
{
	FILE *fpin;
	char line[MAXLINE];
	char *ptr;
	int codenum;
	int i, ln;

	for( i=0; i<MAXKEY; i++ )
		keymap[i] = 0;

	if( (fpin = popen( "dumpkeys", "r" )) == NULL ) {
		return 0;
	}

	while( fgets(line, MAXLINE, fpin) != NULL ) {
		ptr = line;
		while( *ptr && isspace(*ptr) ) ptr++;		/* skip WS */
		if( *ptr && (*ptr == '!' || *ptr == '#') )
			continue;								/* skip comment lines */
		if( strncasecmp( ptr, "keycode", 7 ) )
			continue;								/* only keycode lines */
		ptr += 7;									/* skip "keycode" */
		while( *ptr && isspace(*ptr) ) ptr++;		/* skip WS */
		if( sscanf( ptr, "%d", &codenum ) != 1 )	/* grab keycode number */
			continue;
		while( *ptr && isdigit(*ptr) ) ptr++;		/* skip number */
		while( *ptr && isspace(*ptr) ) ptr++;		/* skip WS */
		if( *ptr != '=' )
			continue;
		ptr++;										/* skip '=' */
		while( *ptr && isspace(*ptr) ) ptr++;		/* skip WS */
		while( *ptr && *ptr == '+' ) ptr++;			/* skip '+' (optional) */
		for( i=0; keystr[i]; i++ ) {
			ln = strlen(keystr[i]);
			if( !strncasecmp( ptr, keystr[i], ln ) && isspace(*(ptr+ln)) ) {
				if( keymap[i] )						/* not already set */
					continue;
				keymap[i] = codenum;
				break;
			}
		}
	}

	if( ferror(fpin) ) {
		return 0;
	}
	if( pclose(fpin) == -1 ) {
		return 0;
	}

	/* Got all keys? */
	for( i=0; keystr[i]; i++ )
		if( !keymap[i] )
			return 0;

	return 1;
}


void copy_config_from_pc_default( void )
{
	int i;
	for( i=0; keystr[i]; i++ )
		keymap[i] = default_pc_map[i];
}


void copy_config_from_ppc_default( void )
{
	int i;
	for( i=0; keystr[i]; i++ )
		keymap[i] = default_ppc_map[i];
}


void display_config( void )
{
	int i;
	for( i=0; keystr[i]; i++ ) {
		printf( "%12s = %d\n", keystr[i], keymap[i] );
	}
}


int translate_raw_key( int key )
{
	int i;
	for( i=0; keystr[i]; i++ ) {
		if( key == keymap[i] )
			return key_down_vals[i];
		if( key == RELEASE(keymap[i]) )
			return key_up_vals[i];
	}
	return 0;
}


#if 0
int main( void )
{
	if( get_config_from_dumpkeys() )
		printf( "Got config successfully!\n" );

	display_config();
	return 1;
}
#endif
